import requests
import json
import os

from si_tests.logger import logger
from si_tests import settings
from si_tests.clients.k8s import K8sCluster


class ScaleLabTrendClient:
    _k8sclient: K8sCluster = None
    _mgmt_k8sclient: K8sCluster = None

    _report_id: str = ''

    def __init__(self, k8sclient=None, mgmt_k8sclient=None):
        if k8sclient:
            self._k8sclient = k8sclient

        if mgmt_k8sclient:
            self._mgmt_k8sclient = mgmt_k8sclient

    def uploadReport(self, env_type, env_name, path):
        logger.info("ScaleLabTrends upload Rally report: {}".format(path))

        with open(path, 'r') as f:
            resp = requests.post(
                url=settings.RALLY_UPLOAD_SCALE_LAB_TRENDS_URL + "/api/report",
                data={
                    'env_type': env_type,
                    'env_name': env_name,
                },
                files={
                    'report': f
                }
            )
            resp.raise_for_status()

            self._report_id = resp.json()['report_id']

    def saveBareMetalHosts(self, namespace=None, upload=False):
        # fields = ['kind', 'metadata', 'spec']

        logger.info("ScaleLabTrends save BareMetalHosts")

        bmh_dir = os.path.join(settings.ARTIFACTS_DIR, "upload/BareMetalHosts/")
        os.makedirs(bmh_dir)
        bmh_list = self._mgmt_k8sclient.kaas_baremetalhosts.list(namespace=namespace)
        for bmh in bmh_list:
            data = {k: bmh.data[k] for k in bmh.model.attribute_map.keys()}

            logger.info("Save BareMetalHost: {}".format(bmh.name))
            with open(os.path.join(bmh_dir, bmh.name), 'w') as f:
                json.dump(data, f, default=str)

            if upload:
                logger.info("Upload BareMetalHost: {}".format(bmh.name))
                resp = requests.post(
                    url=settings.RALLY_UPLOAD_SCALE_LAB_TRENDS_URL + "/api/report/" + self._report_id + "/file",
                    files={
                        'bmh': json.dumps(data, default=str)
                    }
                )
                resp.raise_for_status()

    def saveMachines(self, namespace=None, upload=False):
        # fields = ['kind', 'metadata', 'spec']

        logger.info("ScaleLabTrends save Machines")

        machine_dir = os.path.join(settings.ARTIFACTS_DIR, "upload/Machines/")
        os.makedirs(machine_dir)
        machine_list = self._mgmt_k8sclient.kaas_machines.list(namespace=namespace)
        for machine in machine_list:
            data = {k: machine.data[k] for k in machine.model.attribute_map.keys()}

            logger.info("Save Machine: {}".format(machine.name))
            with open(os.path.join(machine_dir, machine.name), 'w') as f:
                json.dump(data, f, default=str)

            if upload:
                logger.info("Upload Machine: {}".format(machine.name))
                resp = requests.post(
                    url=settings.RALLY_UPLOAD_SCALE_LAB_TRENDS_URL + "/api/report/" + self._report_id + "/file",
                    files={
                        'machine': json.dumps(data, default=str)
                    }
                )
                resp.raise_for_status()

    def saveCluster(self, namespace=None, upload=False):
        # fields = ['kind', 'metadata', 'spec']

        logger.info("ScaleLabTrends save Cluster")

        cluster_dir = os.path.join(settings.ARTIFACTS_DIR, "upload/Cluster/")
        os.makedirs(cluster_dir)
        cluster_list = self._mgmt_k8sclient.kaas_clusters.list(namespace=namespace)
        for cluster in cluster_list:
            data = {k: cluster.data[k] for k in cluster.model.attribute_map.keys()}

            logger.info("Save Cluster: {}".format(cluster.name))
            with open(os.path.join(cluster_dir, cluster.name), 'w') as f:
                json.dump(data, f, default=str)

            if upload:
                logger.info("Upload Cluster: {}".format(cluster.name))
                resp = requests.post(
                    url=settings.RALLY_UPLOAD_SCALE_LAB_TRENDS_URL + "/api/report/" + self._report_id + "/file",
                    files={
                        'cluster': json.dumps(data, default=str)
                    }
                )
                resp.raise_for_status()

    def saveOpenStackDeployment(self, namespace=None, upload=False):
        # fields = ['kind', 'metadata', 'spec']

        logger.info("ScaleLabTrends save OpenStackDeployment")

        osdpl_dir = os.path.join(settings.ARTIFACTS_DIR, "upload/OpenStackDeployment/")
        os.makedirs(osdpl_dir)
        osdpl_list = self._k8sclient.openstackdeployment.list(namespace=namespace)
        for osdpl in osdpl_list:
            data = {k: osdpl.data[k] for k in osdpl.model.attribute_map.keys()}

            logger.info("Save OpenStackDeployment: {}".format(osdpl.name))
            with open(os.path.join(osdpl_dir, osdpl.name), 'w') as f:
                json.dump(data, f, default=str)

            if upload:
                logger.info("Upload OpenStackDeployment: {}".format(osdpl.name))
                resp = requests.post(
                    url=settings.RALLY_UPLOAD_SCALE_LAB_TRENDS_URL + "/api/report/" + self._report_id + "/file",
                    files={
                        'osdpl': json.dumps(data, default=str)
                    }
                )
                resp.raise_for_status()
