import os
import yaml
import json
import time

from numpy import percentile
from si_tests.fixtures.kubectl import create_secret
from si_tests import logger
from si_tests import settings
from si_tests.utils import templates, utils, scale_lab_trend_client, waiters


LOG = logger.logger


def add_label_daemonset_node(kubectl_client, rally_label_key, rally_label_value):
    label_selector_daemonset = ("stacklight!=enabled,!node-role.kubernetes.io/master," +
                                f"{rally_label_key}!={rally_label_value}")
    nodes_daemonset = kubectl_client.nodes.list(label_selector=label_selector_daemonset)
    label_daemonset = {'rally': 'daemonset'}
    for node in nodes_daemonset:
        node.add_label(label_daemonset)


def _get_rally_node(nodesList, rally_label):
    for node in nodesList:
        if node.exists_labels(rally_label):
            return node
    return nodesList[0]


def prepare_rally_node(kubectl_client, rally_label, rally_taint, node=None):
    if node:
        rally_node = kubectl_client.nodes.get(name=node)
        LOG.info('Pre-selected node {}'.format(rally_node.get_name()))
    else:
        label_selector = 'stacklight!=enabled,!node-role.kubernetes.io/master'
        nodes = kubectl_client.nodes.list(label_selector=label_selector)
        rally_node = _get_rally_node(nodes, rally_label)
        LOG.info('Selected node {}'.format(rally_node.get_name()))
    if not rally_node.exists_labels(rally_label):
        rally_node.add_label(rally_label)
    if not rally_node.exists_taint(rally_taint):
        rally_node.add_taint(rally_taint)


def create_kubeconfig_secret(kubectl_client, content, namespace, name):
    config = ''
    if settings.INTERNAL_K8S_ENDPOINT_ENABLED:
        INTERNAL_K8S_ENDPOINT = 'https://kubernetes.default.svc.cluster.local:443'
        kubeconfig = yaml.load(content, Loader=yaml.SafeLoader)
        for cluster in kubeconfig['clusters']:
            cluster['cluster']['server'] = INTERNAL_K8S_ENDPOINT
        config = yaml.dump(kubeconfig)
    else:
        config = content
    create_secret(kubectl_client, config, namespace, name)


def wait_rally_finished(pod):
    LOG.info("Waiting for results ...")
    return pod.read().status.phase == 'Running'


def test_rally(kaas_manager):
    """Test for running Rally in a Kubernetes or OpenStack environment
    and uploading results to Scale Lab Trends.

     Scenario:
        - OpenStack:
            1. Create ConfigMap with rally configurations.
            2. Create rally pod and start testing.
            3. Wait for the results of rally.
            4. Copy the results of the rally.
            5. Upload the results to Scale Lab Trends. (optional)
        - Kubernetes:
            1. Create rally namespace in the target cluster.
            2. Create kubeconfig secret in the target cluster.
            3. Create rally pod and start testing.
            4. Make isolated node for run rally on them.
            5. Add labels for the daemonset scenarios.
            6. Create ConfigMap with rally configurations.
            7. Create rally pod and start testing.
            8. Wait for the results of rally.
            9. Copy the results of the rally.
            10. Get the steal time on the control nodes. (optional)
            11. Upload the results to Scale Lab Trends. (optional)
    """
    # General rally params
    target_cluster_namespace_name = settings.TARGET_NAMESPACE
    platform = settings.RALLY_PLATFORM
    configmap_name = settings.RALLY_CONFIGMAP_NAME
    pod_name = settings.RALLY_POD_NAME
    image = settings.RALLY_IMAGE
    nginx_image = settings.RALLY_POD_NGINX_IMAGE
    debug_mode = settings.RALLY_DEBUG_MODE
    scenarios_repo = settings.RALLY_SCENARIOS_REPO
    scenarios_branch = settings.RALLY_SCENARIOS_BRANCH
    enable_scenarios = settings.RALLY_ENABLE_SCENARIOS
    skip_scenarios = settings.RALLY_SKIP_SCENARIOS
    task_args = settings.RALLY_TASK_ARGS
    env_type = settings.RALLY_ENV_TYPE
    env_name = settings.RALLY_ENV_NAME
    upload_report = settings.RALLY_UPLOAD_SCALE_LAB_TRENDS
    save_trend_reports = settings.RALLY_SAVE_TREND_REPORTS
    label_key = settings.RALLY_LABEL_KEY
    label_value = settings.RALLY_LABEL_VALUE
    ns = settings.RALLY_NAMESPACE
    timeout = settings.RALLY_WAIT_TIMEOUT
    parallel_tasks = settings.RALLY_PARALLEL_TASKS
    results_container_name = "nginx-results"
    # Params for k8s
    node_name = settings.RALLY_NODE_NAME
    toleration_key = settings.RALLY_TOLERATION_KEY
    toleration_value = settings.RALLY_TOLERATION_VALUE
    toleration_effect = settings.RALLY_TOLERATION_EFFECT
    artifacts_dir = settings.ARTIFACTS_DIR

    # Collect data and prepare for test
    if settings.RALLY_ON_DEDICATED_CLUSTER:
        # Cluster with rally:
        deploy_cluster_name = settings.RALLY_DEPLOY_CLUSTER_NAME
        deploy_cluster_namespace = kaas_manager.get_namespace(settings.RALLY_DEPLOY_CLUSTER_NAMESPACE)
        deploy_cluster = deploy_cluster_namespace.get_cluster(deploy_cluster_name)
        kubectl_client = deploy_cluster.k8sclient
        # Target cluster for testing:
        target_cluster_name = settings.TARGET_CLUSTER
        target_cluster_namespace = kaas_manager.get_namespace(target_cluster_namespace_name)
        target_cluster = target_cluster_namespace.get_cluster(target_cluster_name)
    else:
        deploy_cluster_name = target_cluster_name = settings.TARGET_CLUSTER
        target_cluster_namespace = kaas_manager.get_namespace(target_cluster_namespace_name)
        target_cluster = target_cluster_namespace.get_cluster(deploy_cluster_name)
        kubectl_client = target_cluster.k8sclient

    # Acquire target cluster kubeconfig
    _, tg_kubeconfig = target_cluster.get_kubeconfig_from_secret()

    rally_pod_options = {
        'RALLY_POD_NAME': pod_name,
        'RALLY_IMAGE': image,
        'RALLY_POD_NGINX_IMAGE': nginx_image,
        'RALLY_DEBUG_MODE': str(debug_mode).lower(),
        'RALLY_SCENARIOS_REPO': scenarios_repo,
        'RALLY_SCENARIOS_BRANCH': scenarios_branch,
        'RALLY_CONFIGMAP_NAME': configmap_name,
        'RALLY_PLATFORM': platform,
        'RALLY_LABEL_KEY': label_key,
        'RALLY_LABEL_VALUE': label_value,
        'RALLY_TOLERATION_KEY': toleration_key,
        'RALLY_TOLERATION_VALUE': toleration_value,
        'RALLY_TOLERATION_EFFECT': toleration_effect,
        'RALLY_RESULTS_CONTAINER_NAME': results_container_name,
        'RALLY_PARALLEL_TASKS': parallel_tasks
    }
    if platform == 'kubernetes':
        # Create ns in target cluster
        if not kubectl_client.namespaces.present(name=ns):
            LOG.info("Creating Namespace '{0}'".format(ns))
            kubectl_client.namespaces.create(
                name=ns,
                body={'metadata': {'name': ns}})

        # Create secret in target cluster
        LOG.debug("Creating target cluster kubeconfig secret")
        secret_name = "kubeconfig-" + target_cluster_name
        if kubectl_client.secrets.present(name=secret_name, namespace=ns):
            LOG.warning("Secret '{0}' already exists. Deleting.."
                        .format(secret_name))
            kubectl_client.secrets.get(name=secret_name, namespace=ns).delete()
        create_kubeconfig_secret(kubectl_client, tg_kubeconfig, ns, secret_name)

        # Add kubeconfig option to rally pod
        rally_pod_options['KUBECONFIG_SECRET_NAME'] = secret_name

        prepare_rally_node(
            kubectl_client=kubectl_client,
            rally_label={label_key: label_value},
            rally_taint={
                "key": toleration_key,
                "value": toleration_value,
                "effect": toleration_effect
            },
            node=node_name)
        add_label_daemonset_node(kubectl_client, label_key, label_value)

    LOG.info(f"Create ConfigMap {configmap_name}")
    config_map_config = yaml.load(
        templates.render_template(
            settings.RALLY_CONFIGMAP_YAML,
            {
                'RALLY_CONFIGMAP_NAME': configmap_name
            }
        ),
        Loader=yaml.SafeLoader
    )
    if enable_scenarios:
        config_map_config['data']['enable-scenarios'] = enable_scenarios

    if skip_scenarios:
        config_map_config['data']['skip-scenarios'] = skip_scenarios

    if task_args:
        d = yaml.load(task_args, Loader=yaml.SafeLoader)
        s = yaml.load(config_map_config['data']['task-params.yaml'], Loader=yaml.SafeLoader)
        utils.merge_dicts(s, d)
        config_map_config['data']['task-params.yaml'] = yaml.dump(s)

    if kubectl_client.configmaps.present(name=configmap_name, namespace=ns):
        LOG.warning(f"ConfigMap '{configmap_name}' already exists. Deleting..")
        kubectl_client.configmaps.get(name=configmap_name, namespace=ns).delete()

    kubectl_client.configmaps.create(name=configmap_name, namespace=ns, body=config_map_config)

    # Prepare rally pod template
    pod_config = yaml.load(
        templates.render_template(settings.RALLY_POD_YAML, rally_pod_options),
        Loader=yaml.SafeLoader
    )

    LOG.info(f"Create Pod {pod_name}")
    LOG.debug(pod_config)
    if kubectl_client.pods.present(name=pod_name, namespace=ns):
        LOG.warning(f"Pod '{pod_name}' already exists. Deleting..")
        kubectl_client.pods.get(name=pod_name, namespace=ns).delete()

    pod = kubectl_client.pods.create(name=pod_name, namespace=ns, body=pod_config)

    try:
        if timeout > 24*60*60:  # One day
            interval = 1800
        else:
            interval = 120
        start = time.time()
        waiters.wait(lambda: wait_rally_finished(pod),
                     timeout=timeout, interval=interval)
        end = time.time()
    except Exception:
        LOG.error("Task failed")
        raise
    finally:
        pod.cp_from_pod(source_dir='/var/rally/results', compression='-z', container=results_container_name)

        # Get steal time during the rally test
        if settings.RALLY_GET_STEAL_TIME:
            LOG.info('Get steal time from control nodes')
            control_nodes = [x.get_k8s_node_name() for x in target_cluster.get_machines(machine_type="control")]
            result = {}
            steal_time_values_per_node = {x: [] for x in control_nodes}
            LOG.debug(f"Control nodes: {control_nodes}")
            query = f"rate(node_cpu_seconds_total{{mode='steal',node=~'{'|'.join(control_nodes)}'}}[1m])"
            LOG.debug(query)
            prometheus_response = target_cluster.prometheusclient.get_query_range(
                query=query,
                start=start,
                end=end,
                step=60)
            LOG.debug(prometheus_response)

            # Prepare steal time metrics
            for metrics in prometheus_response:
                if metrics['metric'].get('node'):
                    for value in metrics['values']:
                        steal_time_values_per_node[metrics['metric']['node']].append(float(value[1]))

            # Get 95 percentile
            LOG.info('Steal time on nodes')
            for node_name, list_of_values in steal_time_values_per_node.items():
                steal_time_percentile = percentile(list_of_values, 95)
                result[node_name] = steal_time_percentile
                LOG.info(f"{node_name}: {steal_time_percentile}")

            with open(os.path.join(artifacts_dir, 'steal_time.json'), 'w') as f:
                json.dump(result, f)

        # Uploading rally results to scale lab trends
        if save_trend_reports or upload_report:
            sltc = scale_lab_trend_client.ScaleLabTrendClient(
                k8sclient=target_cluster.k8sclient,
                mgmt_k8sclient=kaas_manager.api
            )

            if upload_report:
                sltc.uploadReport(
                    env_type=env_type,
                    env_name=env_name,
                    path=os.path.join(artifacts_dir, 'rally-report.json')
                )

            sltc.saveCluster(namespace=target_cluster_namespace_name, upload=upload_report)
            sltc.saveMachines(namespace=target_cluster_namespace_name, upload=upload_report)
            if platform == 'openstack':
                sltc.saveBareMetalHosts(namespace=target_cluster_namespace_name, upload=upload_report)
                sltc.saveOpenStackDeployment(namespace='openstack', upload=upload_report)

        if platform == 'kubernetes':
            # Checking all pods are in consistent state after test run
            target_cluster.check.check_k8s_pods()

    LOG.info("Done")
