import copy
import time

from si_tests.deployments.utils import commons
from si_tests.managers import openstack_manager

# NOTE(mkarpin): Tests in this file may do changes to nova availability zones,
#                this may affect how workload migrations work. Avoid adding
#                here session fixtures which boot workloads on nodes. Also
#                make sure there are no workloads on environment before
#                running these tests.


def test_node_controller_computes_az(openstack_client_manager, os_manager):
    """Test node controller maintenance parallel computes with respect AZ settings.
    The test requires at least 3 compute nodes.

    1) Configure openstack-controller
       nwl_parallel_max_compute = 2, respect_nova_az = True
    2) Left 2 computes to AZ nova, create rack1 AZ and add rest of hosts there.
    3) Create nmr for 1 compute in nova AZ
    4) Wait nwl is inactive
    5) Create nmr for 1 compute in rack1 AZ
    6) Ensure nwl is active for a while
    7) Create nmr for 2nd compute in nova AZ
    8) Wait when nwl for 2nd compute in nova AZ is inactive
    9) Remove nmr for computes from nova AZ
    10) Check nmr for compute in rack1 is inactive
    11) Check nmr for compute in nova is active
    12) Remove all nmrs
    13) Check all nwl are active
    14) Cleanup
    """

    compute_concurrency = 2
    compute_nodes = os_manager.api.nodes.list(label_selector="openstack-compute-node=enabled")
    if os_manager.is_tf_enabled:
        # Exclude tf-vrouter gateway nodes in case of TF:
        compute_node_names = [n.name for n in compute_nodes if
                              n.data['metadata']['labels'].get(
                                  'openstack-gateway') != "enabled"]
    else:
        compute_node_names = [n.name for n in compute_nodes]

    nova_az_node_names = compute_node_names[:2]
    rack1_az_node_names = compute_node_names[2:]

    # NOTE(vsaienko): we need +1 node up to concurrency to check that nwl still
    # active while we have nmr for it.
    assert (
        len(compute_node_names) >= compute_concurrency + 1
    ), f"Environment has compute nodes less node than required minimum {compute_concurrency + 1}"
    commons.LOG.info("Step 1: Configure respect AZ openstack-controller settings")
    om = openstack_manager.OpenStackManager()
    os_config_orig = om.operator_config
    os_config = copy.deepcopy(os_config_orig)
    os_config["maintenance"]["respect_nova_az"] = "True"
    om.operator_config = os_config
    nmr_api = os_manager.api.nodemaintenancerequests
    nwl_api = os_manager.api.nodeworkloadlocks
    time.sleep(60)

    commons.LOG.info("Step 2: Create AZ setup")
    rack1_cmp1 = rack1_az_node_names[0]
    openstack_client_manager.aggregate.create(["rack1", "--zone", "rack1"])
    openstack_client_manager.aggregate.add_host("rack1", rack1_cmp1)

    commons.LOG.info("Step 3: Create nmr for cmp1 nova AZ")
    cmp1, cmp2 = nova_az_node_names
    nmr_api.create_node_maintenance_request(cmp1, f"nmr-si-test-{cmp1}", scope="drain")
    commons.LOG.info("Step 4: Wait nwl for cmp1 nova AZ is inactive")
    nwl_api.wait_nwl_state(f"openstack-{cmp1}", "inactive")

    commons.LOG.info("Step 5: Create nmr for cmp1 rack1 AZ")
    nmr_api.create_node_maintenance_request(
        rack1_cmp1,
        f"nmr-si-test-{rack1_cmp1}",
        scope="drain",
    )
    commons.LOG.info("Step 6: Ensure nwl for cmp1 rack1 AZ is still active")
    # Make sure number of inactive locks is not changed
    for i in range(1, 3):
        commons.LOG.info(f"Checking state of nwl cmp1 rack1 is active, attempt {i}")
        assert nwl_api.check_nwl_state(f"openstack-{rack1_cmp1}", "active"), "The nwl cmp1 rack1 is inactive."
        time.sleep(60)

    commons.LOG.info("Step 7: Create nmr for cmp2 nova AZ")
    nmr_api.create_node_maintenance_request(cmp2, f"nmr-si-test-{cmp2}", scope="drain")
    commons.LOG.info("Step 8: Wait nwl for cmp2 nova AZ is inactive")
    nwl_api.wait_nwl_state(f"openstack-{cmp2}", "inactive")

    assert nwl_api.check_nwl_state(f"openstack-{rack1_cmp1}", "active"), "The nwl cmp1 rack1 is inactive."

    commons.LOG.info("Step 9: Remove nmr for cmp1,cmp2 nova AZ")
    for host in nova_az_node_names:
        nmr_api.delete_node_maintenance_request(f"nmr-si-test-{host}")
        nwl_api.wait_nwl_state(f"openstack-{host}", "active")
    commons.LOG.info("Step 10: Ensure nwl for cmp1,cmp2 nova AZ are active")
    commons.LOG.info("Step 11: Ensure nwl for cmp1 rack1 AZ is inactive")

    nwl_api.wait_nwl_state(f"openstack-{rack1_cmp1}", "inactive")
    commons.LOG.info("Step 12: Remove nmr for cmp1 rack1 AZ")
    nmr_api.delete_node_maintenance_request(f"nmr-si-test-{rack1_cmp1}")

    commons.LOG.info("Step 13: Ensure nwl for cmp1 rack1 AZ is active")

    nwl_api.wait_nwl_state(f"openstack-{rack1_cmp1}", "active")

    commons.LOG.info("Step 14: Cleanup")
    om.operator_config = os_config_orig
    openstack_client_manager.aggregate.remove_host("rack1", rack1_cmp1)
    openstack_client_manager.aggregate.delete(["rack1"])
