#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.managers.workaroundcheck_manager import wa_phys429_hack_mgmt_deploy_on_vsphere

LOG = logger.logger


class TestProvisionBootstrapv2MgmtBMCluster():
    def test_bootstrapv2_bm_mgmt(self, show_step):
        """Bootstrap V2: start bootstrap cluster and deploy baremetal

        Scenario:
            1. Erase bootstrap V2 cluster if present
            2. Prepare bootstrap V2 cluster
            3. Start bootstrap V2 management cluster deployment
            4. Create default MCC users, if requested
            5. Check versions
            6. Wait for all pods to be Running and Ready
            7. Check that all expected pods exist
            8. Download bootstrap artifacts
            9. Check LDAP integration if enabled and required
            10.Check correct runtime
        """
        if settings.BM_DEPLOY_MGMT_ON_VSPHERE:
            bootstrap = wa_phys429_hack_mgmt_deploy_on_vsphere()
        else:
            show_step(1)
            bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
            bootstrap.step_001_erase_env_before_bv2()

            # step 002: Prepare bootstrap V2 cluster
            show_step(2)
            LOG.info("Initialize bootstrap manager")
            bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
        # All env vars should be already inside bootstrap.env
        if not bootstrap.setup_bootstrapv2_cluster(envs_string='true'):
            pytest.fail("Failed to setup bootstrap_v2 cluster")

        show_step(3)
        env_config_name = bootstrap.bm_determine_env_config_name()
        if "virtual" in env_config_name:
            bootstrap.step_004_deploy_kaas_bm_mgmt_cluster()
        else:
            bootstrap.step_004_deploy_kaas_bm_cluster_physical(env_config_name)

        # step 004: Create default MCC users, if requested
        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(4)
            bootstrap.step_004_create_default_mcc_users(custom_bootstrap_path=settings.KAAS_BOOTSTRAP_TARGET_DIR)

        if settings.MCC_UPGRADE_AUTO_DELAY_ENABLED:
            LOG.info("Enabling auto-delay for mcc upgrade")
            bootstrap.step_006_enable_mcc_upgrade_auto_delay()
        # step 005: Check versions
        show_step(5)
        bootstrap.step_005_check_versions()

        # step 006: Wait for all pods to be Running and Ready
        show_step(6)
        bootstrap.step_006_wait_for_pods()

        # step 007: Check that all expected pods exist
        show_step(7)
        bootstrap.step_006_postdeployment_checks()

        # step 008: Download bootstrap artifacts
        show_step(8)
        bootstrap.step_007_download_bootstrap_artifacts()

        # step 009: Check LDAP integration if enabled and required
        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(9)
            bootstrap.check_keystone_ldap_integration()

        # step 010: Check runtime
        if settings.DESIRED_RUNTIME:
            show_step(10)
            bootstrap.compare_cluster_runtime_with_desired()
