import pytest

import json
from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, k8s_utils

LOG = logger.logger


@pytest.mark.os
@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures('store_cluster_kubeconfig')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KCM_CLUSTER_DEPLOYMENT_NAME}"])
def test_create_os_standalone_cluster(kcm_manager, show_step, _):
    """Deploy managed clusterdeployment on OpenStack

    Scenario:
        1. Collect env data
        2. Create or get NS and check cluster templates presence
        3. Create auth
        4. Prepare clusterdeployment config
        5. Create clusterdeployment
        6. Wait for clusterdeployment progress to finish
        7. Check expected pods, its statuses and k8s nodes
        8. Set default storageclass if not present
        9. Sync CoreDNS Corefile with management cluster
    """
    show_step(1)
    # TODO(va4st): Move to separated test with auth actions, determine auth parameters dynamically
    cluster_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    namespace_name = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE
    auth_suffix = f"ksi-openstack-auth-{utils.gen_random_string(4)}"
    # https://github.com/kubernetes/cloud-provider-openstack/blob/master/charts/
    # cinder-csi-plugin/templates/storageclass.yaml#L5
    # https://github.com/k0rdent/kcm/blob/main/templates/cluster/openstack-standalone-cp/
    # templates/k0scontrolplane.yaml#L149
    storageclass = 'csi-cinder-sc-delete'
    auth_secret_name = f"{auth_suffix}-secret"
    cred_name = f"{auth_suffix}-cred"

    show_step(2)
    ns = kcm_manager.get_or_create_namespace(namespace_name)
    clustertemplate = (settings.KCM_CLUSTER_TEMPLATE_NAME or
                       ns.get_latest_template_by_prefix('openstack-standalone', strategy='chart'))
    available_templates = [tmpl.name for tmpl in ns.get_cluster_templates()]
    assert clustertemplate in available_templates, (f"Clustertemplate {clustertemplate} not found in namespace "
                                                    f"{ns.name}. Available templates: {available_templates}")
    utils.check_expected_pods_template_exists(clustertemplate)

    show_step(3)
    auth_sec = ns.create_os_auth_secret(name=auth_secret_name)
    # NOTE(va4st): For OpenStack there no separated auth. Only secret required
    cred = ns.create_credential(name=cred_name, cred_object=auth_sec)
    ns.create_sveltos_resource_template_cm_using_secret(auth_sec, provider='openstack')

    show_step(4)
    # TODO(va4st): make configurable through settings and add parameters verification before testrun
    config = ns.get_cluster_config(provider='openstack', os_auth_secret=auth_sec)

    child_labels = None

    if kcm_manager.kof.kof_ms_deployed():
        child_labels = {
            'k0rdent.mirantis.com/kof-cluster-role': 'child'
        }
        if kcm_manager.kof.kof_is_deployed():
            # update specific labels for Istio regional
            labels = {
                'k0rdent.mirantis.com/istio-role': 'child'
            }
        else:
            labels = {
                'k0rdent.mirantis.com/kof-storage-secrets': 'true'
            }
        utils.merge_dicts(child_labels, labels)

        if not kcm_manager.kof.kof_is_deployed() and kcm_manager.source == 'custom-enterprise':
            annotations = {
                'k0rdent.mirantis.com/kof-collectors-values': json.dumps({
                    "kof": {
                        "logs": {
                            "tls_options": {
                                "insecure_skip_verify": True
                            }
                        },
                        "metrics": {
                            "tls_options": {
                                "insecure_skip_verify": True
                            }
                        },
                        "traces": {
                            "tls_options": {
                                "insecure_skip_verify": True
                            }
                        }
                    }
                }, indent=2)
            }
            config['clusterAnnotations'] = annotations

    show_step(5)
    cld = ns.create_cluster_deployment(name=cluster_name,
                                       template=clustertemplate,
                                       credential=cred,
                                       config=config,
                                       labels=child_labels)
    LOG.info(f"Created clusterdeployment {cld.namespace}/{cld.name}")

    show_step(6)
    cld.check.check_cluster_readiness()

    show_step(7)
    cld.check.check_actual_expected_pods()
    cld.check.check_k8s_pods()
    cld.check.check_k8s_nodes()

    show_step(8)
    if not cld.is_default_storageclass_present():
        sc = cld.get_storageclass_by_name(storageclass)
        sc.make_default()
        assert sc.is_default, f"StorageClass {storageclass} is not default"

    show_step(9)
    # TODO(va4st): Need a better way to determine the needs for CoreDNS sync (e.g some marker in mgmt config maybe?)
    if (not kcm_manager.kof.kof_is_deployed() and kcm_manager.kof.kof_ms_deployed()
            and kcm_manager.source == 'custom-enterprise'):
        corefile = k8s_utils.get_corefile(kcm_manager.api)
        k8s_utils.override_coredns_configmap(cld.k8sclient, corefile)
        coredns_deployment = cld.k8sclient.deployments.get(name='coredns', namespace='kube-system')
        coredns_deployment.rollout_restart()
