#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import datetime
import os
import time

import exec_helpers
import yaml
from kubernetes.client import V1Eviction, V1ObjectMeta
from kubernetes.client.rest import ApiException
from kubernetes.stream import stream
from retry import retry
from urllib3.exceptions import MaxRetryError
from websocket._exceptions import WebSocketBadStatusException

# import si_tests.utils.templates as template_utils
import si_tests.utils.waiters as helpers
from si_tests import logger
from si_tests import settings
from si_tests.clients.k8s.base import K8sBaseManager
from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.utils import exceptions
from si_tests.utils.helpers import skip_rc_retry, relogin_401

LOG = logger.logger


class K8sPod(K8sNamespacedResource):
    resource_type = 'pod'

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_pod(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_pod(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_pod(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_pod(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, async_del=False, timeout=120, **kwargs):
        uid = self.uid
        LOG.debug(f"Deleting pod {self.namespace}/{self.name} with UID: {uid}")
        self._manager.api.delete_namespaced_pod(
            self.name, self.namespace, **kwargs)
        if not async_del:
            helpers.wait(lambda: not self.exists(),
                         timeout=timeout,
                         timeout_msg='Timeout deleting pod {0}/{1}'
                         .format(self.namespace, self.name))

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind, metadata,
        spec, status or items
        """
        return self.read().to_dict()

    @property
    def containers(self):
        return [c for c in self.data.get('spec', {}).get('containers', [])]

    @property
    def containers_resources(self):
        containers_map = {
            'pod_name': self.name, 'containers': []}
        for container in self.containers:
            containers_map['containers'].append(
                {'name': container.get('name'),
                 'resources': container.get('resources')})
        return containers_map

    @property
    def job_name(self):
        """Name of job which spawned pod

        :return: Job name if exists. If not - None
        """
        pod_meta = self.read().to_dict()['metadata']
        if pod_meta.get('labels', {}).get('job-name', None):
            return pod_meta['labels']['job-name']
        else:
            return None

    @skip_rc_retry((ApiException, MaxRetryError, WebSocketBadStatusException), delay=20, tries=8)
    def exists(self):
        pods = self._manager.list_raw(self.namespace).to_dict()['items']
        pod = [x for x in pods if x['metadata']['uid'] == self.uid]
        return True if pod else False

    def cp_from_pod(self, source_dir, source_file='.', destination=settings.ARTIFACTS_DIR,
                    fragment_size=100000, fragment_timeout=600, compression='', container=None):
        """Method that provides kubectl cp functionality from pod to localhost.

        :param source_dir: directory to use as a root of the archive
        :param source_file: file/directory name or mask to archive and copy
        :param destination: local directory to store files and directories
        :param fragment_size: number of base64-encoded lines transmitted at once
        :param fragment_timeout: timeout in seconds for transmit a single fragment
        :param compression: flag for 'tar' application that will be added for compressing the tar archive.
                            Please ensure that the pod contains the compression application
        :param container: container to execute command

        Basically it archives source using kubectl exec, write binary
        stdout to local tar file using base64 and unpack it to destination dir.

        tar.base64 archive is transferred in small portions of data, to retry
        only that portion in case of error.
        """
        pod_name = self.read().metadata.name

        timestamp = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
        tmp_tar_base64_name = f"test_{pod_name}_{timestamp}.tar.base64"
        remote_tar_base64 = f"/tmp/{tmp_tar_base64_name}"
        local_tar_base64 = f"{destination}{tmp_tar_base64_name}"

        LOG.info("Copy files from the pod {0}/{1}:{2}/{3} "
                 "to the local directory {4}"
                 .format(self.name, self.namespace, source_dir,
                         source_file, destination))

        if source_file == '.':
            exists = self.check_non_empty_folder(source_dir, container=container)
        else:
            exists = self.check_file_exists(os.path.join(source_dir, source_file),
                                            container=container)

        if not exists:
            LOG.error("There is nothing to copy")
            return None

        try:
            # Command for pack and code files to tar.base64
            error_msg = 'create_tar_base64_failed'
            res = self.exec(['/bin/sh', '-c',
                             f'cd {source_dir}; tar -C {source_dir} {compression} -c {source_file} | base64 > '
                             f'{remote_tar_base64} && sync || echo "{error_msg}"'], container=container).strip()
            if error_msg in res:
                raise Exception(f"Error appeared while archiving and encoding {source_dir}/{source_file} "
                                f"into {remote_tar_base64} in the pod {self.namespace}/{self.name}")

            error_msg = 'split_tar_base64_failed'
            res = self.exec(['/bin/sh', '-c',
                             f'cd {source_dir}; '
                             f'split -a 4 -l {fragment_size} {remote_tar_base64} {remote_tar_base64}- 2>&1'
                             f' && sync || echo "{error_msg}"'], container=container).strip()
            if error_msg in res:
                raise Exception(f"Error appeared while splitting {source_dir}/{remote_tar_base64} "
                                f"into {fragment_size}-lines fragments in the pod {self.namespace}/{self.name}:\n"
                                f"{res}")

            error_msg = 'ls_tar_base64_failed'
            total_files_str = self.exec(['/bin/sh', '-c',
                                         f'ls -1 {remote_tar_base64}-* || echo "{error_msg}"'], container=container)
            if error_msg in res:
                raise Exception(f"Error appeared while listing files from {remote_tar_base64}-* :\n{res}")
            LOG.debug(f"File {source_dir}/{remote_tar_base64} was splitted into the following files:\n"
                      f"{total_files_str}")
            total_files = total_files_str.splitlines()

            total_lines = self._get_remote_file_lines_count(remote_tar_base64, container=container)
            LOG.info(f"Transmitting base64 encoded file with total {total_lines} lines")

            with open(local_tar_base64, "w") as tar_buffer:
                start_line = 0
                for base64_file in sorted(total_files):
                    fragment_lines = self._get_remote_file_lines_count(base64_file, container=container)

                    LOG.info(f"Processing base64 encoded fragment from line {start_line}, size {fragment_lines} lines")
                    tar_pos = tar_buffer.tell()
                    self._stream_partial_file(
                        tar_buffer,
                        tar_pos,
                        base64_file,
                        local_tar_base64,
                        fragment_lines,
                        fragment_timeout,
                        container=container)

                    start_line += fragment_lines

                tar_buffer.seek(0)
        finally:
            # Remove remote tar.base64 file
            res = self.exec(['/bin/sh', '-c', f'rm {remote_tar_base64} || true'], container=container)

        # Decode and unpack files from tar.base64 to 'destination'
        exec_helpers.Subprocess().check_call(f"base64 -d {local_tar_base64} | tar -C {destination} {compression} -x")
        # Remove local tar.base64 file
        exec_helpers.Subprocess().check_call(f"rm {local_tar_base64}")

    @retry((ApiException, Exception, exceptions.TimeoutError), delay=2, tries=3, logger=LOG)
    def _stream_partial_file(self, tar_buffer, tar_pos, remote_filepath, local_filepath,
                             fragment_lines, fragment_timeout, container=None):

        remote_md5 = self._get_remote_file_md5sum(remote_filepath, container=container)
        exec_command = ['/bin/sh', '-c', f'cat {remote_filepath}']
        tar_buffer.seek(tar_pos)

        if not container:
            container = self.containers[0]['name']
            LOG.info(f"Container name is not defined. Using container {container} "
                     f"by default for executing cmd")

        resp = stream(self._manager.api.connect_get_namespaced_pod_exec,
                      self.name, self.namespace,
                      command=exec_command,
                      container=container,
                      stderr=True, stdin=True,
                      stdout=True, tty=False,
                      _preload_content=False,
                      _request_timeout=1800)

        start_time = time.time()
        while resp.is_open():
            resp.update(timeout=1)
            if resp.peek_stdout():
                out = resp.read_stdout()
                tar_buffer.write(out)
            if resp.peek_stderr():
                LOG.error("STDERR: %s" % resp.read_stderr())
            if start_time + fragment_timeout < time.time():
                timeout_msg = (
                    f"Timeout {fragment_timeout}s reached while copying file '{remote_filepath}' lines")
                raise exceptions.TimeoutError(timeout_msg)
        resp.close()
        tar_buffer.flush()

        # Get only just written lines from the local file, always 'fragment_lines' number from the end
        local_md5 = self._get_local_fragment_md5sum(local_filepath, fragment_lines, fragment_lines)
        if local_md5 != remote_md5:
            raise Exception(f"Checksum mismatch while copying file '{remote_filepath}'. "
                            f"Remote md5sum: {remote_md5}  Local md5sum: {local_md5}")

    def _get_remote_file_lines_count(self, remote_filepath, container=None):
        res = self.exec(['/bin/sh', '-c', f'wc -l {remote_filepath}'], container=container).split()[0]
        LOG.debug(f"Remote file '{remote_filepath}' has {res} lines")
        return int(res)

    def _get_remote_file_md5sum(self, remote_filepath, container=None):
        LOG.debug(f"Getting remote md5sum for {remote_filepath}")
        _res = self.exec(['/bin/sh', '-c',
                          f'cat {remote_filepath} | md5sum'],
                         container=container)
        res = _res.split()[0]
        if not res:
            raise Exception(f"Failed to calculate md5sum for file {remote_filepath} in {self.namespace}/{self.name}")
        return res

    def _get_local_fragment_md5sum(self, local_filepath, tailpos, headpos):
        LOG.debug(f"Getting local md5sum from {tailpos} size {headpos}")
        res = exec_helpers.Subprocess().check_call(f'tail -n {tailpos} {local_filepath} | head -n {headpos} | md5sum')
        md5sum = res.stdout_str.split()[0]
        if not md5sum:
            raise Exception(
                f"Failed to calculate md5sum for local file {local_filepath} from {tailpos} size {headpos}")
        return md5sum

    def get_restarts_number(self):
        restarts = 0
        status = self.data.get('status')
        if status:
            # The container_statuses key may have value None
            for cont in status.get('container_statuses') or []:
                restarts += cont.get('restart_count', 0)
        return restarts

    def check_file_exists(self, filepath='', container=None):
        res = self.exec(
            ['/bin/sh', '-c',
             'test -f {0} && echo "1" || echo "0"'.format(filepath)],
            container=container).strip()
        return True if res != "0" else False

    def check_folder_exists(self, dir_path='', container=None):
        """Check if directory in pod exists
        """
        res = self.exec(
            ['/bin/sh', '-c',
             'test -d {0}/ && echo "1" || echo "0"'.format(dir_path)],
            container=container).strip()
        return True if res != "0" else False

    def check_non_empty_folder(self, path='', container=None):
        """Check if folder in pod is non-empty
        """
        res = self.exec(
            ['/bin/sh', '-c',
             'test -d {0}/ && ls -A {0} | wc -l || echo "0"'.format(path)],
            container=container).strip()
        return True if res != "0" else False

    @skip_rc_retry((ApiException, MaxRetryError, WebSocketBadStatusException),
                   delay=20, tries=8)
    @relogin_401
    def exec(self, command, container=None, stderr=True, stdin=False,
             stdout=True, tty=False, _preload_content=True,
             _request_timeout=120, **kwargs):
        """
        :param container: container to execute command
        :param stderr: redirect stderr. Defaults to True
        :param stdin: redirect stdin. Defaults to False
        :param stdout: redirect stdout. Defaults to True
        :param tty: allocate tty. Defaults to False
        :param _preload_content: read/decode response data. Defaults to True
        # https://github.com/kubernetes-client/python/issues/10850
        :param _request_timeout: timeout for request. Defaults to 120
        """
        if not container:
            container = self.containers[0]['name']
            LOG.debug(f"Container name is not defined. Using container {container} "
                      f"by default for executing cmd")

        resp = stream(self._manager.api.connect_get_namespaced_pod_exec,
                      self.name, self.namespace, command=command, container=container,
                      stderr=stderr, stdin=stdin, stdout=stdout, tty=tty,
                      _preload_content=_preload_content,
                      _request_timeout=_request_timeout, **kwargs)
        return resp

    def wait_phase(self, phases, log_path=settings.ARTIFACTS_DIR,
                   timeout=500, interval=3):
        if isinstance(phases, str):
            phases = [phases]

        try:
            helpers.wait(lambda: self.read().status.phase in phases,
                         timeout=timeout, interval=interval,
                         timeout_msg='Timeout waiting, pod {0} phase is not '
                                     'in "{1}"'.format(self.name, phases))
        except Exception:
            pod_describe = str(self.read())
            pod_name = self.read().metadata.name
            timestamp = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
            logfile = '{path}pod-describe-{pod_name}-{timestamp}.log'.format(
                path=log_path,
                pod_name=pod_name,
                timestamp=timestamp
            )
            with open(logfile, 'w') as f:
                f.write(pod_describe)
            LOG.error(pod_describe)
            raise
        return self

    def are_containers_ready(self):
        """Check that containers in the pod are ready

        Return True if all containers are ready, False if any container is not ready"""

        status = self.data.get('status')
        if status and status.get('container_statuses'):
            statuses = [x.get('ready') or False for x in status.get('container_statuses') or []]
        else:
            statuses = [False]
        return False not in statuses

    def wait_ready(self, log_path=settings.ARTIFACTS_DIR,
                   timeout=360, interval=3):
        def _statuses():
            status = self.data.get('status')
            if status and status.get('container_statuses'):
                statuses = [x.get('ready') or False for x in status.get('container_statuses') or []]
            else:
                statuses = [False]
            return statuses
        try:
            helpers.wait(self.are_containers_ready,
                         timeout=timeout, interval=interval,
                         timeout_msg='Timeout waiting, pod {0} is '
                                     'not Ready yet'.format(self.name))
        except Exception:
            pod_describe = str(self.read())
            pod_name = self.read().metadata.name
            timestamp = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
            logfile = '{path}pod-describe-{pod_name}-{timestamp}.log'.format(
                path=log_path,
                pod_name=pod_name,
                timestamp=timestamp
            )
            with open(logfile, 'w') as f:
                f.write(pod_describe)
            LOG.error(pod_describe)
            raise
        return self

    @skip_rc_retry((ApiException, MaxRetryError, WebSocketBadStatusException), delay=20, tries=8)
    @relogin_401
    def get_logs(self, container=None, since_seconds=None):
        pod_name = self.read().metadata.name
        logs = self._manager.api.read_namespaced_pod_log(
            name=pod_name,
            namespace=self.read().metadata.namespace,
            container=container,
            since_seconds=since_seconds
        )
        return logs

    def get_pvc_names(self):
        pod_volumes = self.data['spec'].get('volumes', [])
        pvc_names = {}
        for volume in pod_volumes:
            if volume.get('persistent_volume_claim'):
                pvc_names[volume['name']] = volume['persistent_volume_claim']['claim_name']
        return pvc_names

    def wait_test(self, filepath, log_path=settings.ARTIFACTS_DIR,
                  timeout=600, interval=3, container=None):
        """Method that waits till test finishes.

        :param filepath: file that indicates test has finished
        :param timeout: time to wait before fail
        :param interval: interval between checks
        :param container: container to execute command
        """
        try:
            helpers.wait(lambda: self.check_file_exists(filepath, container=container),
                         timeout=timeout, interval=interval,
                         timeout_msg='Timeout waiting, pod {0} does not have '
                                     'file "{1}"'.format(self.name, filepath))
            res = self.exec(
                ['/bin/sh', '-c', 'ls -la {0} || true'.format(filepath)],
                container=container).strip()
            LOG.info("In the pod {0}, expected the file {1}, found:\n{2}"
                     .format(self.name, filepath, res))
        finally:
            pod_name = self.read().metadata.name
            logs = self.get_logs()
            timestamp = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
            logfile = '{path}pod-{pod_name}-{timestamp}.log'.format(
                path=log_path,
                pod_name=pod_name,
                timestamp=timestamp
            )
            with open(logfile, 'w') as f:
                f.write(logs)
            LOG.info(logs)
        return self

    def get_owner_reference(self):
        if self.exists():
            pod_meta = self.read().to_dict()['metadata']
        else:
            raise Exception("Cannot find pod {}".format(self.name))
        if pod_meta['owner_references']:
            kind = pod_meta['owner_references'][0]['kind']
            name = pod_meta['owner_references'][0]['name']
            return "{kind}/{name}".format(kind=kind, name=name)
        else:
            return "Node/no_owner"

    def eviction(self, wait_timeout=None):
        """
        API-initiated eviction is the process by which you use the Eviction API to create an Eviction object
        that triggers graceful pod termination.
        https://kubernetes.io/docs/concepts/scheduling-eviction/api-eviction/

        Args:
            wait_timeout: wait timeout in seconds or None - without wait pod eviction

        Returns: V1Eviction

        """
        body = V1Eviction(metadata=V1ObjectMeta(name=self.name, namespace=self.namespace))
        response = self._manager.api.create_namespaced_pod_eviction_with_http_info(
            name=self.name, namespace=self.namespace, body=body)
        if wait_timeout:
            helpers.wait(lambda: not self.exists(),
                         timeout=wait_timeout,
                         timeout_msg='{0}/{1} pod eviction timeout '
                         .format(self.namespace, self.name))
        return response

    @property
    def node_name(self):
        return self.data["spec"]["node_name"]


class K8sPodManager(K8sBaseManager):
    resource_class = K8sPod

    @property
    def api(self):
        return self._cluster.api_core

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_pod(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_pod_for_all_namespaces(**kwargs)

    def get_pods_statuses(self, pods):
        pods_info = {}
        for pod in pods:
            pod_name = (f"{pod['metadata']['namespace']}/"
                        f"{pod['metadata']['name']}")
            pod_phase = pod['status']['phase']
            pod_message = pod['status']['message']
            pod_owner_str = 'Node/no_owner'
            if pod['metadata']['owner_references']:
                pod_owner = pod['metadata']['owner_references'][0]
                pod_owner_str = f"{pod_owner['kind']}/{pod_owner['uid']}"
            if 'container_statuses' not in pod['status'] or \
                    not pod['status']['container_statuses']:
                pod_ready = {"no_containers_found": False}
                restarts = {}
            else:
                pod_ready = {x['name']: x['ready']
                             for x in pod['status']['container_statuses']}
                restarts = {x['name']: x['restart_count']
                            for x in pod['status']['container_statuses']}
            pods_info[pod_name] = {
                'phase': pod_phase,
                'message': pod_message,
                'containers': pod_ready,
                'containers_restarts': restarts,
                'owner': pod_owner_str,
            }
        return pods_info

    def check_pods_statuses(self,
                            target_namespaces=None,
                            phases=('Running', 'Succeeded'),
                            excluded_pods=[],
                            pods_prefix='',
                            replicasets=None):
        if not replicasets:
            replicasets = {}
        failed_pods, pods_info = self.list_not_in_phases(
            target_namespaces=target_namespaces,
            phases=phases)

        ready_text = " and/or Ready=False for " \
                     "some containers"

        if pods_prefix:
            failed_pods = \
                {k: v for k, v in failed_pods.items()
                 if k.startswith(pods_prefix)}

        for excl_pod in excluded_pods:
            failed_pods = \
                {k: v for k, v in failed_pods.items()
                 if excl_pod not in k}

        # In case the owner of the failed pod is a replicaset, it might be
        # fine to have some failed pods, if required replicas amount is ready.
        # For example, it can be caused by a node that went down for some
        # short period of time, and one pod failed to be scheduled. Another
        # replica will be brought up anyway and replicaset will become active
        # when the node comes back up. Such failed pods will be cleaned up by
        # Kubernetes itself.
        exclude_failed_rs_replicas = {}
        for k, v in failed_pods.items():
            # pod_namespace is extracted from pod_name (see get_pods_statuses() above)
            pod_namespace = k.split('/')[0]
            namespace_rs = replicasets.get(pod_namespace, {})
            pod_owner = v['owner']
            if pod_owner in namespace_rs:
                # or 0 bit below is needed in case ready_replicas is None
                if (namespace_rs[pod_owner].spec.replicas ==
                        (namespace_rs[pod_owner].ready_replicas or 0)):
                    LOG.warning(f"Pod '{k}' not in phase '{phases}', "
                                f"but it is ignored because all it's owner replicas are ready")
                    continue
            exclude_failed_rs_replicas[k] = v

        if len(exclude_failed_rs_replicas) > 0:
            LOG.warning("Next pods are not in one of {} phase {}: \n{}".format(
                phases, ready_text,
                yaml.dump(exclude_failed_rs_replicas,
                          default_flow_style=False)))
            return False
        # Return non-empty object with pods info
        return pods_info

    def list_not_in_phases(self, target_namespaces=None,
                           phases=('Running', 'Succeeded'),
                           check_jobs=False,
                           excluded_nodes=None):
        pods = self.list_raw().to_dict()['items']
        if target_namespaces:
            if isinstance(target_namespaces, str):
                target_namespaces = [target_namespaces]
            pods = [pod for pod in pods
                    if pod['metadata']['namespace'] in target_namespaces]
        assert len(pods) > 0, "List of pods is empty"

        if not check_jobs:
            LOG.debug("Filter out Jobs from pod list")
            pods = \
                [x for x in pods if not (
                    x['metadata']['owner_references'] and
                    x['metadata']['owner_references'][0]['kind'] == 'Job')]
            LOG.debug(pods)
        if excluded_nodes:
            LOG.debug("Return pods from all, except {} nodes".format(
                excluded_nodes))
            # if None is in excluded_nodes list
            # pods w/o scheduled node will be filtered out
            pods = [x for x in pods
                    if x['spec']['node_name'] not in excluded_nodes]
            LOG.debug(pods)

        pods_info = self.get_pods_statuses(pods)
        failed_pods = {pod: phase
                       for pod, phase in pods_info.items()
                       if phase['phase'] not in phases or
                       (False in phase['containers'].values() and
                        phase['phase'] not in 'Succeeded')}
        return failed_pods, pods_info

    def list_pods_except_jobs(self, target_namespaces=None):
        pods = self.list_raw().to_dict()['items']
        if target_namespaces:
            if isinstance(target_namespaces, str):
                target_namespaces = [target_namespaces]
            pods = [pod for pod in pods
                    if pod['metadata']['namespace'] in target_namespaces]

        LOG.debug("Filter out Jobs from pod list")
        pods = \
            [x for x in pods if not (
                x['metadata']['owner_references'] and
                x['metadata']['owner_references'][0]['kind'] == 'Job')]
        LOG.debug(pods)
        return pods

    @skip_rc_retry((ApiException, MaxRetryError, WebSocketBadStatusException), delay=20, tries=8)
    @relogin_401
    def status(self, name, namespace):
        """Get status of the specified pod in the namespace"""
        return self.api.read_namespaced_pod_status(name, namespace)

    def present(self, name, namespace):
        """Check that specified pod name exists in the namespace"""
        return any([pod.name for pod in self.list(namespace=namespace)
                    if pod.name == name])

    def wait_pod_present(self, name_prefix, namespace, field_selector=None,
                         timeout=1800, interval=15, timeout_msg=None):
        """Wait until any pod with the specified parameters is appeared in the pods list"""
        helpers.wait(lambda: self.list(name_prefix=name_prefix, namespace=namespace, field_selector=field_selector),
                     timeout=timeout, interval=interval,
                     timeout_msg=timeout_msg)

    def extract_registry_url(self,
                             pod_name="helm-controller-",
                             pod_ns="kube-system",
                             default="mirantis.azurecr.io"):
        """Extract address of images registry from the specified pod"""
        pods = self.list(
            namespace=pod_ns,
            name_prefix=pod_name)
        if len(pods) == 0:
            LOG.warning(f"Pod {pod_ns}/{pod_name} not found. "
                        f"Use the default registry for images: {default}")
            return default

        pod = pods[0].read()
        containers = pod.spec.containers
        image = containers[0].image
        patterns = [i for i in image.split("/") if i and 'http' not in i]
        return patterns[0]

    # TODO(va4st): Uncomment and update after machine interface (for different providers) will be ported
    # def exec_pod_cmd(self, cmd, registry,
    #                  pod_name=None, pod_namespace=None, node_name=None,
    #                  pod_template_path=settings.MACHINE_PRIVELEGED_POD_YAML, render_options=None,
    #                  verbose=True, get_events=True, timeout=600, delete_pod=True):
    #     """Run a single command in a priveleged container on a specified node"""
    #
    #     pod_namespace = pod_namespace or 'default'
    #     pod_name = pod_name or "exec-pod-cmd-{}".format(utils.gen_random_string(6))
    #     render_options = render_options or {}
    #
    #     render_options.update({
    #         'POD_NAME': pod_name.lower(),
    #         'POD_CMD': cmd,
    #         'NODE_NAME': str(node_name).lower(),
    #         'MCP_DOCKER_REGISTRY': registry,
    #     })
    #
    #     pod_template = yaml.safe_load(template_utils.render_template(
    #         pod_template_path, options=render_options,
    #         log_env_vars=False, log_template=False))
    #
    #     pod = self.create(name=pod_name,
    #                       namespace=pod_namespace,
    #                       body=pod_template,
    #                       log_body=False)
    #     if verbose:
    #         _LOG = LOG.info
    #         _ERRLOG = LOG.error
    #     else:
    #         _LOG = LOG.debug
    #         _ERRLOG = LOG.debug
    #     try:
    #         pod.wait_phase(['Succeeded', 'Failed'],
    #                        timeout=timeout, interval=1)
    #         pod_status = pod.read().status
    #         container_state = pod_status.container_statuses[0].state
    #         if container_state.terminated is not None:
    #             exit_code = container_state.terminated.exit_code
    #         else:
    #             exit_code = -1
    #             LOG.debug(f"Pod {pod.name} has wrong container status:\n"
    #                       f"{pod_status}")
    #
    #         logs = pod.get_logs()
    #         if get_events:
    #             events = self._cluster.events.get_events_by_uid_str(
    #                 pod.namespace, pod.uid,
    #                 event_prefix=pod.name, sort=False)
    #         else:
    #             events = None
    #
    #         _LOG(f"[{node_name}] Output of the command \"{cmd}\":\n{logs}")
    #         _LOG(f"Events after command execution:\n{events}")
    #         _LOG(f"Exit code: {exit_code}")
    #         return {'logs': logs,
    #                 'events': events,
    #                 'exit_code': exit_code}
    #     except Exception as e:
    #         _ERRLOG(f"Error while running pod {pod_name}:\n{e}")
    #         raise e
    #     finally:
    #         if delete_pod:
    #             pod.delete()
