#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
from si_tests.clients.k8s import base


class K8sClusterRole(base.K8sClusterScopedResource):
    resource_type = 'clusterrole'

    def _read(self, **kwargs):
        return self._manager.api.read_cluster_role(self.name, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_cluster_role(body, **kwargs)

    def _patch(self, body, **kwargs):
        """Patch a K8sClusterRole object

        Input should be in a following scheme (as received from k8s by kubernetes python client):
        body = {
                  'rules': [
                      {'api_groups': [''], 'resources': ['secrets'], 'verbs': ['get', 'list', 'watch']},
                      {'api_groups': ['*'], 'resources': ['*'], 'verbs': ['get', 'list', 'watch']},
                ]
            }

        Input body will be transformed into
        body = {
                  'rules': [
                      {'apiGroups': [''], 'resources': ['secrets'], 'verbs': ['get', 'list', 'watch']},
                      {'apiGroups': ['*'], 'resources': ['*'], 'verbs': ['get', 'list', 'watch']},
                ]
            }

        Fields 'non_resource_ur_ls' and 'resource_names' will be removed since these fields not available for patching

        :param body:
        :param kwargs:
        :return:
        """
        proc_body = self.__prepare_patch_body(body)
        return self._manager.api.patch_cluster_role(
            self.name, proc_body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_cluster_role(
            self.name, body, **kwargs)

    def _delete(self, **kwargs):
        self._manager.api.delete_cluster_role(
            self.name, **kwargs)

    @staticmethod
    def __prepare_patch_body(body):
        rule_template = {
            'apiGroups': [],
            'resources': [],
            'verbs': []
        }
        processed_rules = []
        rules = body.get('rules', [])
        for rule in rules:
            obj = rule_template.copy()
            obj['apiGroups'] = rule['api_groups']
            obj['resources'] = rule['resources']
            obj['verbs'] = rule['verbs']
            processed_rules.append(obj)
        body['rules'] = processed_rules
        return body


class K8sClusterRoleManager(base.K8sBaseManager):
    resource_class = K8sClusterRole

    @property
    def api(self):
        return self._cluster.api_rbac_auth

    def _list(self, namespace, **kwargs):
        return self.api.list_cluster_role(**kwargs)

    def _list_all(self, **kwargs):
        return self._list(None, **kwargs)
