from si_tests import logger
from si_tests import settings
import requests
import json
import os
import re

LOG = logger.logger

NUM_JOBS = settings.FIO_JOBS_TO_COMPARE
DEVIATION = settings.FIO_NIGHTLY_DEVIATION_PERCENT
ARTIFACTS_DIR = settings.ARTIFACTS_DIR
WORKFLOW_NAME = settings.FIO_NIGHTLY_WORKFLOW_NAME
FIO_JOB_SCENARIO_NAME = settings.FIO_JOB_SCENARIO_NAME

JENKINS_URL = 'https://ci.mcp.mirantis.net/job'
ARTIFACTORY_URL = ''.join(['https://artifactory.mcp.mirantis.net:443/',
                           'artifactory/si-local/jenkins-job-artifacts'])
FIO_JOB = 'test-k8s-fio'
RALLY_ARTIFACT_PATH = 'artifacts/list-fio-results.json'
IO_OPERATION_TYPES = ['write', 'read', 'mixed']
METRICS = ['throughput', 'iops', 'latency']
SCENARIO = {'rw': 'randwrite'}
EPS = 10 ** -7


def get_fio_averages(reports):
    result = {}
    for io_operation, dict_metrics in reports.items():
        result[io_operation] = {}
        len_metrics = len(dict_metrics['metrics'])
        result[io_operation]['length'] = reports[io_operation]['length']
        if len_metrics > 0:
            for metric in METRICS:
                metric_sum = 0
                for metrics in dict_metrics['metrics']:
                    metric_sum += metrics[metric]
                if metric_sum != 0:
                    result[io_operation][metric] = float(metric_sum / len_metrics)
    return result if result else None


def get_json(url):
    file = requests.get(url=url)
    if file.status_code == 404:
        LOG.error(f'URL does not exist! {file.url}')
        return None
    file.raise_for_status()
    return file.json()


def get_fio_data(file, scenario):
    data = {operation: [] for operation in IO_OPERATION_TYPES}
    try:
        for client in file['client_stats']:
            if (client['jobname'] == FIO_JOB_SCENARIO_NAME and
                    all(client['job options'][key] == value for key, value in scenario.items())):
                for operation in IO_OPERATION_TYPES:
                    if operation in client:
                        sorted_metrics = {}
                        metrics = {}
                        metrics['throughput'] = client[operation]['bw_mean']
                        metrics['iops'] = client[operation]['iops_mean']
                        metrics['latency'] = client[operation]['lat_ns']['mean']
                        for key, value in metrics.items():
                            if value != 0:
                                sorted_metrics[key] = value
                        if sorted_metrics:
                            data[operation].append(sorted_metrics)
    except KeyError as k:
        LOG.error(f'JSON parsing error: {k}')
    except Exception as e:
        LOG.error(e)
    return data if any(data[operation] for operation in IO_OPERATION_TYPES) else None


def get_fio_reports(artifacts_path, scenario):
    results = {operation: {'length': 0, 'metrics': []} for operation in IO_OPERATION_TYPES}
    reports = get_json(url=artifacts_path)
    if reports:
        for report in reports:
            fio_report = get_json(url=report)
            if fio_report:
                fio_data = get_fio_data(fio_report, scenario)
                if fio_data:
                    for io_operation, metrics in fio_data.items():
                        if metrics:
                            results[io_operation]['length'] = 1
                            for metric in metrics:
                                results[io_operation]['metrics'].append(metric)
                    LOG.info(f'Got fio results from {report}')
            else:
                LOG.error('Could not fetch fio results!')
    return results if any(results[operation]['metrics'] for operation in IO_OPERATION_TYPES) else None


def timer_trigger(url):
    causes = 'causes'
    class_cause = 'hudson.triggers.TimerTrigger$TimerTriggerCause'
    build = get_json(url=url)
    if build:
        try:
            for action in build['actions']:
                if causes in action and len(action['causes']) == 1:
                    return action['causes'][0]['_class'] == class_cause
        except KeyError as k:
            LOG.error(f'JSON parsing error: {k}')
        except Exception as e:
            LOG.error(e)
    return False


def get_upstream_wf(job):
    wf_number = 0
    description = 'shortDescription'
    upstream = 'upstreamProject'
    causes = 'causes'
    for action in job['actions']:
        if causes not in action:
            break
        for cause in action['causes']:
            if description in cause and cause[description].startswith('Rebuilds'):
                return 0
            if upstream in cause and cause[upstream] == WORKFLOW_NAME:
                upstream_build_number = cause['upstreamBuild']
                upstream_build_url = f'{JENKINS_URL}/{WORKFLOW_NAME}/{upstream_build_number}/api/json'
                if timer_trigger(upstream_build_url):
                    wf_number = upstream_build_number
    return wf_number


def test_compare_fio_results():
    """This test retrieves fio results from a running scale nightly
    performance workflow. It then compares these results with the fio
    results from the desired number of previous nightly.
    """
    LOG.info('Fetching current fio results')
    fio_artifacts_path = settings.get_var('FIO_ARTIFACTS_PATH', None)
    assert fio_artifacts_path, 'fio artifacts are not defined'

    job_number = re.search(fr'(?<={FIO_JOB}\/)\d+(?=\/{RALLY_ARTIFACT_PATH})', fio_artifacts_path).group()
    url = f'{JENKINS_URL}/{FIO_JOB}/{job_number}/api/json'

    fio_current = get_fio_reports(fio_artifacts_path, SCENARIO)
    results = {operation: {'length': 0, 'metrics': []} for operation in IO_OPERATION_TYPES}
    while (len(results) < NUM_JOBS):
        try:
            # was the job triggered by workflow?
            url = f'{JENKINS_URL}/{FIO_JOB}/{job_number}/api/json'
            job = get_json(url=url)
            if not job:
                break
            wf_number = get_upstream_wf(job=job)
            if job['result'] != 'SUCCESS' or wf_number == 0:
                continue

            # get fio report
            url = f'{ARTIFACTORY_URL}/{FIO_JOB}/{job_number}/{RALLY_ARTIFACT_PATH}'
            if url == fio_artifacts_path:
                continue
            LOG.info(f'Fetching data from {url}')
            fio_report = get_fio_reports(url, SCENARIO)
            if fio_report:
                for io_operation, metrics in fio_report.items():
                    if metrics['metrics']:
                        results[io_operation]['length'] += 1
                        for metric in metrics['metrics']:
                            results[io_operation]['metrics'].append(metric)
                LOG.info(f'Got data from workflow {WORKFLOW_NAME} number {wf_number}')
            else:
                LOG.warning(f'Could not retrieve data from {FIO_JOB} number {job["number"]}')
                LOG.debug(f'URL: {url}')
        except Exception as e:
            LOG.error(e)
        finally:
            # previous workflow
            try:
                job_number = job['previousBuild']['number']
            except TypeError:
                LOG.error('No more jobs left')
                break

    current_average = get_fio_averages(fio_current)
    previous_results = get_fio_averages(results)

    # dump results
    with open(os.path.join(ARTIFACTS_DIR, 'previous_results.json'), 'w') as f:
        json.dump(previous_results, f)
    with open(os.path.join(ARTIFACTS_DIR, 'current_average.json'), 'w') as f:
        json.dump(current_average, f)

    # get statistics
    stat = {}
    for operation in IO_OPERATION_TYPES:
        stat[operation] = {}
        LOG.info(f'Comparing {operation}')
        for metric in METRICS:
            stat[operation][metric] = {}
            current = average = 0.0
            if metric in current_average[operation]:
                current = current_average[operation][metric]
            if metric in previous_results[operation]:
                average = previous_results[operation][metric]

            if current > EPS and average > EPS:
                diff = 100.0 * current / average - 100
                if diff > DEVIATION:
                    LOG.warning(f'Current {metric} results are {abs(diff):.1f}% slower than average!')
                if diff < -DEVIATION:
                    LOG.warning(f'Current {metric} results are {abs(diff):.1f}% faster than average!')
                if abs(diff) < DEVIATION:
                    LOG.info(f'Current {metric} results are within the deviation range')
            else:
                diff = 0.0
                LOG.warning(f'There is nothing to compare {metric} with!')
            LOG.debug(f'Current {metric} result: {current:.3f}\nAverage result: {average:.3f}')

            # dump
            stat[operation][metric]['current'] = current
            stat[operation][metric]['average'] = average
            stat[operation][metric]['diff'] = diff
        stat[operation]['length'] = previous_results[operation]['length']

    with open(os.path.join(ARTIFACTS_DIR, 'results.json'), 'w') as f:
        json.dump(stat, f)
