from datetime import datetime
import os
import pytest
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import utils

LOG = logger.logger


# Expected downtimes provide data per different test cases
# for statistic collected from different endpoints.
# (single_downtime, max_downtimes) = expected_downtimes[<test_name>][<provider_name>][<endpoint_name>]
# - single_downtime : maximum period of a single downtime, seconds
# - max_downtimes : maximum summary downtimes during the test, seconds
#
# Downtime expectations for RefApp may be based on various processes in the cloud, for example:
# - Floating IP downtime (while neutron gateway is migrated)
# - Loadbalancer downtime (while Amphora VMs are migrated)
# - Loadbalancer backends downtime (while RefApp web servers are migrated all at once)
# - Application downtime (while RefApp databases are migrated/not in quorum)
#
# Downtime expectations for kube_api:
# (TBD)

__openstack_icmp_downtimes = {
    # first level keys of dictionary are versions of Openstack Controller
    # if not found 'default' is used
    'default': {
        'queens': (110, 210),
        'rocky': (110, 210),
        'stein': (110, 210),
        'train': (110, 210),
        'ussuri': (110, 210),
        'victoria': (110, 210),
        'wallaby': (110, 210),
        'xena': (110, 210),
        'yoga': (110, 210),
        'antelope': (110, 210),
        'caracal': (110, 210),
        'epoxy': (110, 210),
    }
}

__openstack_icmp_downtimes_upgrade = {
    'default': {
        'victoria': (300, 480),
        'wallaby': (110, 210),
        'xena': (110, 210),
        'yoga': (300, 480),
        'antelope': (300, 480),
        'caracal': (110, 210),
        'epoxy': (110, 210),
    }
}

__openstack_icmp_downtimes_ovs_restart = {
    # first level keys of dictionary are versions of Openstack Controller
    # if not found 'default' is used
    'default': {
        'victoria': (300, 480),
        'wallaby': (300, 480),
        'xena': (300, 480),
        'yoga': (300, 480),
        'antelope': (300, 480),
        'caracal': (300, 480),
        'epoxy': (110, 210),
    }
}

__openstack_icmp_downtimes_ovn_migration = {
    'default': {
        'caracal': (30, 60),
        'epoxy': (30, 60),
    }
}

__openstack_icmp_downtimes_patch_updates = {
    # first level keys of dictionary are versions of Openstack Controller
    # if not found 'default' is used
    # By default during patch updates no ovs image changes expected
    'default': __openstack_icmp_downtimes['default'],
    # in 0.13.12 (mosk-15-0-4-23-2-3) ovs image is changed
    '0.13.12': __openstack_icmp_downtimes_ovs_restart['default'],
    # in 0.14.12 (mosk-17-0-1-23-3-1) ovs helm chart is changed
    '0.14.12': __openstack_icmp_downtimes_ovs_restart['default'],
    # in 0.14.17 (mosk-17-0-1-23-3-3) ovs helm chart is changed due to k8s entrypoint changes
    '0.14.17': __openstack_icmp_downtimes_ovs_restart['default']
}

__recovery_after_mcc_lcm = {
    'baremetal': {
        'load_manager': {
            "refapp": {"records_get": (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 400)},
            'prometheus': {'get_targets': (200, 400)},
            'grafana': {'records_get_all_dashboards': (200, 400)},
            'kibana': {'get_dashboard_data': (200, 400)},
            'alertmanager': {'get_status':  (200, 400)},
            'keycloak': {'get_token':  (200, 300)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
        # 'kube_api': (5, 30),
    },
}

# TODO update after iam fixes
__recovery_after_mcc_lcm_with_iam_downtime = {
    'baremetal': {
        'load_manager': {
            "refapp": {"records_get": (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 400)},
            'prometheus': {'get_targets':  (200, 400)},
            'grafana': {'records_get_all_dashboards':  (200, 400)},
            'kibana': {'get_dashboard_data':  (200, 400)},
            'alertmanager': {'get_status':  (200, 400)},
            'keycloak': {'get_token':  (200, 400)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
        # 'kube_api': (5, 30),
    },
}

__recovery_after_mcc_child_major_update = {
    'baremetal': {
        'load_manager': {
            "refapp": {"records_get": (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 400)},
            'prometheus': {'get_targets': (200, 400)},
            'grafana': {'records_get_all_dashboards': (200, 400)},
            'kibana': {'get_dashboard_data':  (200, 400)},
            'alertmanager': {'get_status':  (200, 400)},
            'keycloak': {'get_token':  (200, 400)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_ovs_restart},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_ovs_restart},
        # 'kube_api': (5, 30),
    },
}

__recovery_after_mcc_child_patch_update = {
    'baremetal': {
        'load_manager': {
            "refapp": {"records_get": (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 400)},
            'prometheus': {'get_targets': (200, 400)},
            'grafana': {'records_get_all_dashboards': (200, 400)},
            'kibana': {'get_dashboard_data':  (200, 400)},
            'alertmanager': {'get_status':  (200, 400)},
            'keycloak': {'get_token':  (200, 400)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_patch_updates},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_patch_updates},
    },
}

__recovery_after_mcc_disaster = {
    'baremetal': {
        'load_manager': {
            "refapp": {"records_get": (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 400)},
            'prometheus': {'get_targets': (200, 400)},
            'grafana': {'records_get_all_dashboards': (200, 400)},
            'kibana': {'get_dashboard_data':  (200, 400)},
            'alertmanager': {'get_status':  (200, 400)},
            'keycloak': {'get_token':  (200, 400)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
        # 'kube_api': (5, 30),
    },
}

# TODO(vsaienko): this is too generic, we need to be super specific.
__recovery_after_mos_lcm = {
    '': {
        "load_manager": {
            'refapp': {'records_get': (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (75, 250)},
            'prometheus': {'get_targets': (75, 250)},
            'grafana': {'records_get_all_dashboards': (75, 250)},
            'kibana': {'get_dashboard_data':  (75, 250)},
            'alertmanager': {'get_status':  (75, 250)},
            'keycloak': {'get_token':  (200, 300)},
         },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
    },
    'baremetal': {
        'load_manager': {
            'refapp': {'records_get': (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (75, 250)},
            'prometheus': {'get_targets': (75, 250)},
            'grafana': {'records_get_all_dashboards': (75, 250)},
            'kibana': {'get_dashboard_data':  (75, 250)},
            'alertmanager': {'get_status':  (75, 250)},
            'keycloak': {'get_token':  (25, 75)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
    },
}

__recovery_after_os_upgrade = {
    '': {
        "load_manager": {
            'refapp': {'records_get': (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 300)},
            'prometheus': {'get_targets': (200, 300)},
            'grafana': {'records_get_all_dashboards': (200, 300)},
            'kibana': {'get_dashboard_data': (200, 300)},
            'alertmanager': {'get_status':  (200, 300)},
            'keycloak': {'get_token':  (200, 300)},
         },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_upgrade},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_upgrade},
    },
    'baremetal': {
        'load_manager': {
            'refapp': {'records_get': (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
            'alerta': {'get_alerts': (200, 300)},
            'prometheus': {'get_targets': (200, 300)},
            'grafana': {'records_get_all_dashboards': (200, 300)},
            'kibana': {'get_dashboard_data':  (200, 300)},
            'alertmanager': {'get_status':  (200, 300)},
            'keycloak': {'get_token':  (200, 300)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_upgrade},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_upgrade},
    },
}

__recovery_after_credentials_rotation_mos = {
    '': {
        "load_manager": {
            # during l3 agent restart couple of packets may lost we register it as downtime
            'refapp': {'records_get': (5, 15),
                       "records_create": (5, 15),
                       "records_delete": (5, 15)},
            # We allow small downtimes during mdns/mariadb pods restarts, no longer than 5 seconds
            # to switch metallb ip to new node.
            "dnsquery": {"query_a": (10, 20)},
            # Higher downtimes for getting token are expected due to RabbitMQ restarts
            "keystone": {"post_token": (30, 90)},
         },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
    },
    'baremetal': {
        'load_manager': {
            # during l3 agent restart couple of packets may lost we register it as downtime
            'refapp': {'records_get': (5, 15),
                       "records_create": (5, 15),
                       "records_delete": (5, 15)},
            # We allow small downtimes during mdns pods restarts, no longer than 5 seconds
            # to switch metallb ip to new node.
            "dnsquery": {"query_a": (10, 20)},
            # Higher downtimes for getting token are expected due to RabbitMQ restarts
            "keystone": {"post_token": (30, 90)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes},
    },
}

__recovery_after_mos_ovn_migration = {
    '': {
        "load_manager": {
            'refapp': {'records_get': (15, 60),
                       "records_create": (15, 60),
                       "records_delete": (15, 60)},
            'dnsquery': {'query_a': (0, 0)},
         },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_ovn_migration},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_ovn_migration},
    },
    'baremetal': {
        'load_manager': {
            'refapp': {'records_get': (15, 60),
                       "records_create": (15, 60),
                       "records_delete": (15, 60)},
            'dnsquery': {'query_a': (0, 0)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_ovn_migration},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_ovn_migration},
    },
}

__recovery_after_mos_update = {
    '': {
        'load_manager': {
            'refapp': {'records_get': (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_ovs_restart},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_ovs_restart},
    },
    'baremetal': {
        'load_manager': {
            'refapp': {'records_get': (200, 300),
                       "records_create": (200, 300),
                       "records_delete": (200, 300)},
            "dnsquery": {"query_a": (25, 75)},
            "keystone": {"post_token": (25, 75)},
        },
        'cloudprober': {'openstack_instances_icmp': __openstack_icmp_downtimes_ovs_restart},
        'portprober': {'openstack_ports_arping': __openstack_icmp_downtimes_ovs_restart},
    },
}

__no_downtime = {
    '': {
        'load_manager': {
            'refapp': {'records_get': (0, 0),
                       'records_create': (0, 0),
                       'records_delete': (0, 0)},
            "dnsquery": {"query_a": (0, 0)},
            "keystone": {"post_token": (0, 0)},
            "tungstenfabric": {"virtual-networks": (0, 0)},
            'alerta': {'get_alerts': (0, 0)},
            'prometheus': {'get_targets': (0, 0)},
            'grafana': {'records_get_all_dashboards': (0, 0)},
            'kibana': {'get_dashboard_data':  (0, 0)},
            'alertmanager': {'get_status':  (0, 0)},
            'keycloak': {'get_token':  (0, 0)},
        },
        'cloudprober': {'openstack_instances_icmp': (0, 0)},
        'portprober': {'openstack_ports_arping': (0, 0)},
    },
    'baremetal': {
        'load_manager': {
            'refapp': {'records_get': (0, 0),
                       'records_create': (0, 0),
                       'records_delete': (0, 0)},
            "dnsquery": {"query_a": (0, 0)},
            "keystone": {"post_token": (0, 0)},
            "tungstenfabric": {"virtual-networks": (0, 0)},
            'alerta': {'get_alerts': (0, 0)},
            'prometheus': {'get_targets': (0, 0)},
            'grafana': {'records_get_all_dashboards': (0, 0)},
            'kibana': {'get_dashboard_data':  (0, 0)},
            'alertmanager': {'get_status':  (0, 0)},
            'keycloak': {'get_token':  (0, 0)},
        },
        'cloudprober': {'openstack_instances_icmp': (0, 0)},
        'portprober': {'openstack_ports_arping': (0, 0)},
    },
}


__no_downtime_live_migration = {
    '': {
        'load_manager': {
            'refapp': {'records_get': (15, 100),
                       'records_create': (15, 100),
                       'records_delete': (15, 100)},
            "dnsquery": {"query_a": (0, 0)},
            "keystone": {"post_token": (0, 0)},
        },
        'cloudprober': {'openstack_instances_icmp': (60, 300)},
        'portprober': {'openstack_ports_arping': (60, 100)},
    },
    'baremetal': {
        'load_manager': {
            'refapp': {'records_get': (15, 100),
                       'records_create': (15, 100),
                       'records_delete': (15, 100)},
            "dnsquery": {"query_a": (0, 0)},
            "keystone": {"post_token": (0, 0)},
        },
        'cloudprober': {'openstack_instances_icmp': (60, 300)},
        'portprober': {'openstack_ports_arping': (60, 100)},
    },
}


expected_downtimes = {
    # Downtimes expected in major releases with heavy component upgrades
    'major_release': {
        'test_update_child_clusterrelease': __recovery_after_mcc_child_major_update,
        'test_update_child_via_update_plan_steps_sequentally': __recovery_after_mcc_child_major_update,
        'test_update_child_via_update_plan_steps_bulk': __recovery_after_mcc_child_major_update,
        'test_replace_master_node': __recovery_after_mcc_lcm,
        'test_add_remove_ssh_keys': __recovery_after_mcc_lcm,
        'test_reboot_machine': __recovery_after_mcc_lcm,
        'test_replace_broken_drive_master_node': __recovery_after_mcc_lcm,
        'test_replace_broken_network_master_node': __recovery_after_mcc_lcm,
        'test_replace_broken_ipmi_master_node': __recovery_after_mcc_lcm,
        'test_ovn_controller_restart_after_maintenance': __recovery_after_mcc_lcm,
        'test_update_license': __recovery_after_mcc_lcm,
        'test_upgrade_tf': __recovery_after_mcc_lcm,
        'test_positive_maintenance_mod': __recovery_after_mcc_lcm,
        'test_add_bm_control_node': __recovery_after_mcc_lcm,
        'test_add_delete_bm_worker_node': __recovery_after_mcc_lcm,
        'test_update_child_stacklight': __recovery_after_mcc_lcm,
        'test_add_bm_compute': __recovery_after_mcc_lcm,
        'test_ceph_filesystem': __recovery_after_mcc_lcm,
        'test_delete_bm_compute': __recovery_after_mcc_lcm,
        'test_downscale_child_cluster_slave_nodes': __recovery_after_mcc_lcm,
        'test_prepare_ceph_cluster': __recovery_after_mcc_lcm,
        'test_add_bm_ceph_node': __recovery_after_mcc_lcm,
        'test_replace_failed_osd_disk': __recovery_after_mcc_lcm,
        'test_add_remove_additional_osds_by_path_by_id_by_device': __recovery_after_mcc_lcm,
        'test_remove_ceph_osd_node': __recovery_after_mcc_lcm,
        'test_graceful_reboot': __recovery_after_mcc_lcm,
        'test_replace_failed_os_control': __recovery_after_mcc_lcm,
        'test_replace_failed_tf_control': __recovery_after_mcc_lcm,
        'test_inplace_distribution_upgrade': __recovery_after_mcc_lcm,

        'test_ha_kill_system_proc_in_cont': __recovery_after_mcc_disaster,
        'test_update_mgmt_cluster': __recovery_after_mcc_lcm,
        'test_ha_kill_ucp_cluster_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_proc_pods': __recovery_after_mcc_disaster,
        'test_ha_delete_helm_controller_leader_pod': __recovery_after_mcc_disaster,
        'test_ha_delete_pods': __recovery_after_mcc_disaster,
        'test_ha_kill_containerd_shim': __recovery_after_mcc_disaster,
        'test_ha_restart_docker_service': __recovery_after_mcc_disaster,
        'test_ha_haproxy': __recovery_after_mcc_disaster,
        'test_ha_sl_svc': __recovery_after_mcc_disaster,
        'test_ha_shutdown_mgmt_vip': __recovery_after_mcc_disaster,
        'test_ha_child_reboot_mosk': __recovery_after_mcc_disaster,
        'test_ha_mgmt_sequential_reboot': __recovery_after_mcc_disaster,
        'test_ha_child_sequential_reboot': __recovery_after_mcc_disaster,
        'test_ha_kill_mariadb': __recovery_after_mcc_disaster,
        'test_ha_kill_memcached': __recovery_after_mcc_disaster,
        'test_ha_kill_etcd': __recovery_after_mcc_disaster,
        'test_ha_kill_rabbitmq': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_rabbitmq': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_ovs_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_l3_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_dhcp_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_control': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_kafka': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_zookeeper': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_zookeeper_nal': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_rabbitmq': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_redis': __recovery_after_mcc_disaster,
        'test_ha_kill_libvirt': __recovery_after_mcc_disaster,
        'test_ha_kill_patroni': __recovery_after_mcc_disaster,
        'test_ha_fluentd_logs': __recovery_after_mcc_disaster,
        'test_ha_cadvisor': __recovery_after_mcc_disaster,
        'test_ha_alerta': __recovery_after_mcc_disaster,
        'test_ha_opensearch_master': __recovery_after_mcc_disaster,
        'test_ha_machine_exporter': __recovery_after_mcc_disaster,
        'test_ha_telegraf_ds_smart': __recovery_after_mcc_disaster,
        'test_ha_telemeter_server': __recovery_after_mcc_disaster,
        'test_ha_telegraf_docker_swarm': __recovery_after_mcc_disaster,
        'test_ha_prometheus_relay': __recovery_after_mcc_disaster,
        'test_ha_prometheus_kube_state_metrics': __recovery_after_mcc_disaster,
        'test_ha_prometheus_es_exporter': __recovery_after_mcc_disaster,
        'test_ha_prometheus_blackbox_exporter': __recovery_after_mcc_disaster,
        'test_ha_elasticsearch_exporter': __recovery_after_mcc_disaster,
        'test_ha_opensearch_dashboards': __recovery_after_mcc_disaster,
        'test_ha_metricbeat': __recovery_after_mcc_disaster,
        'test_ha_metric_collector': __recovery_after_mcc_disaster,
        'test_ha_grafana': __recovery_after_mcc_disaster,
        'test_ha_prometheus_alertmanager': __recovery_after_mcc_disaster,
        'test_ha_prometheus_server': __recovery_after_mcc_disaster,
        'test_ha_stop_containerd': __no_downtime,
        'test_ha_freeze_mariadb_server': __recovery_after_mcc_disaster,
        'test_ha_freeze_nova_scheduler': __recovery_after_mcc_disaster,
        'test_ha_freeze_nova_conductor': __recovery_after_mcc_disaster,
        'test_ha_freeze_neutron_server': __recovery_after_mcc_disaster,
        'test_ha_freeze_keystone_api': __recovery_after_mcc_disaster,
        'test_ha_freeze_cinder_volume': __recovery_after_mcc_disaster,
        'test_ha_child_shutdown_mosk': __recovery_after_mcc_disaster,
        'test_ha_mgmt_sequential_shutdown': __recovery_after_mcc_disaster,
        'test_runtime_migrate_partial_child': __recovery_after_mcc_child_major_update,
        'test_runtime_migrate_quick': __recovery_after_mcc_child_major_update,
        'test_inplace_distribution_upgrade_and_migrate_runtime': __recovery_after_mcc_child_major_update,
    },

    # Downtimes expected in patch releases with minor/security component updates
    'patch_release': {
        'test_update_child_clusterrelease': __recovery_after_mcc_child_patch_update,
        'test_update_child_via_update_plan_steps_sequentally': __recovery_after_mcc_child_patch_update,
        'test_update_child_via_update_plan_steps_bulk': __recovery_after_mcc_child_patch_update,
        'test_replace_master_node': __recovery_after_mcc_lcm,
        'test_add_remove_ssh_keys': __recovery_after_mcc_lcm,
        'test_reboot_machine': __recovery_after_mcc_lcm,
        'test_replace_broken_drive_master_node': __recovery_after_mcc_lcm,
        'test_replace_broken_network_master_node': __recovery_after_mcc_lcm,
        'test_replace_broken_ipmi_master_node': __recovery_after_mcc_lcm,
        'test_ovn_controller_restart_after_maintenance': __recovery_after_mcc_lcm,
        'test_update_license': __recovery_after_mcc_lcm,
        'test_upgrade_tf': __recovery_after_mcc_lcm,
        'test_positive_maintenance_mod': __recovery_after_mcc_lcm,
        'test_add_bm_control_node': __recovery_after_mcc_lcm,
        'test_add_delete_bm_worker_node': __recovery_after_mcc_lcm,
        'test_update_child_stacklight': __recovery_after_mcc_lcm,
        'test_add_bm_compute': __recovery_after_mcc_lcm,
        'test_ceph_filesystem': __recovery_after_mcc_lcm,
        'test_delete_bm_compute': __recovery_after_mcc_lcm,
        'test_downscale_child_cluster_slave_nodes': __recovery_after_mcc_lcm,
        'test_prepare_ceph_cluster': __recovery_after_mcc_lcm,
        'test_add_bm_ceph_node': __recovery_after_mcc_lcm,
        'test_replace_failed_osd_disk': __recovery_after_mcc_lcm,
        'test_add_remove_additional_osds_by_path_by_id_by_device': __recovery_after_mcc_lcm,
        'test_remove_ceph_osd_node': __recovery_after_mcc_lcm,
        'test_tf_backup_resotre': __recovery_after_mcc_lcm,
        'test_graceful_reboot': __recovery_after_mcc_lcm,
        'test_replace_failed_os_control': __recovery_after_mcc_lcm,
        'test_replace_failed_tf_control': __recovery_after_mcc_lcm,
        'test_inplace_distribution_upgrade': __recovery_after_mcc_lcm,
        'test_update_mgmt_cluster': __recovery_after_mcc_lcm_with_iam_downtime,
        'test_ha_kill_system_proc_in_cont': __recovery_after_mcc_disaster,
        'test_ha_kill_ucp_cluster_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_proc_pods': __recovery_after_mcc_disaster,
        'test_ha_delete_helm_controller_leader_pod': __recovery_after_mcc_disaster,
        'test_ha_delete_pods': __recovery_after_mcc_disaster,
        'test_ha_kill_containerd_shim': __recovery_after_mcc_disaster,
        'test_ha_restart_docker_service': __recovery_after_mcc_disaster,
        'test_ha_haproxy': __recovery_after_mcc_disaster,
        'test_ha_sl_svc': __recovery_after_mcc_disaster,
        'test_ha_shutdown_mgmt_vip': __recovery_after_mcc_disaster,
        'test_ha_child_reboot_mosk': __recovery_after_mcc_disaster,
        'test_ha_mgmt_sequential_reboot': __recovery_after_mcc_disaster,
        'test_ha_child_sequential_reboot': __recovery_after_mcc_disaster,
        'test_ha_kill_mariadb': __recovery_after_mcc_disaster,
        'test_ha_kill_memcached': __recovery_after_mcc_disaster,
        'test_ha_kill_etcd': __recovery_after_mcc_disaster,
        'test_ha_kill_rabbitmq': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_rabbitmq': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_ovs_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_l3_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_neutron_dhcp_agent': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_control': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_kafka': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_zookeeper': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_zookeeper_nal': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_rabbitmq': __recovery_after_mcc_disaster,
        'test_ha_kill_tf_redis': __recovery_after_mcc_disaster,
        'test_ha_kill_libvirt': __recovery_after_mcc_disaster,
        'test_ha_stop_containerd': __no_downtime,
        'test_ha_freeze_mariadb_server': __recovery_after_mcc_disaster,
        'test_ha_freeze_nova_scheduler': __recovery_after_mcc_disaster,
        'test_ha_freeze_nova_conductor': __recovery_after_mcc_disaster,
        'test_ha_freeze_neutron_server': __recovery_after_mcc_disaster,
        'test_ha_freeze_keystone_api': __recovery_after_mcc_disaster,
        'test_ha_freeze_cinder_volume': __recovery_after_mcc_disaster,
        'test_ha_child_shutdown_mosk': __recovery_after_mcc_disaster,
        'test_ha_mgmt_sequential_shutdown': __recovery_after_mcc_disaster,
        'test_ha_kill_patroni': __recovery_after_mcc_disaster,
        'test_ha_fluentd_logs': __recovery_after_mcc_disaster,
        'test_ha_cadvisor': __recovery_after_mcc_disaster,
        'test_ha_alerta': __recovery_after_mcc_disaster,
        'test_ha_opensearch_master': __recovery_after_mcc_disaster,
        'test_ha_machine_exporter': __recovery_after_mcc_disaster,
        'test_ha_telegraf_ds_smart': __recovery_after_mcc_disaster,
        'test_ha_telemeter_server': __recovery_after_mcc_disaster,
        'test_ha_telegraf_docker_swarm': __recovery_after_mcc_disaster,
        'test_ha_prometheus_relay': __recovery_after_mcc_disaster,
        'test_ha_prometheus_kube_state_metrics': __recovery_after_mcc_disaster,
        'test_ha_prometheus_es_exporter': __recovery_after_mcc_disaster,
        'test_ha_prometheus_blackbox_exporter': __recovery_after_mcc_disaster,
        'test_ha_elasticsearch_exporter': __recovery_after_mcc_disaster,
        'test_ha_opensearch_dashboards': __recovery_after_mcc_disaster,
        'test_ha_metricbeat': __recovery_after_mcc_disaster,
        'test_ha_metric_collector': __recovery_after_mcc_disaster,
        'test_ha_grafana': __recovery_after_mcc_disaster,
        'test_ha_prometheus_alertmanager': __recovery_after_mcc_disaster,
        'test_ha_prometheus_server': __recovery_after_mcc_disaster,
        'test_runtime_migrate_partial_child': __recovery_after_mcc_child_major_update,
        'test_runtime_migrate_quick': __recovery_after_mcc_child_major_update,
        'test_inplace_distribution_upgrade_and_migrate_runtime': __recovery_after_mcc_child_major_update,
    },

    'mos': {
        'test_openstack_upgrade_simple': __recovery_after_os_upgrade,
        'test_openstack_upgrade_with_refapp': __recovery_after_os_upgrade,
        'test_credentials_rotation': __recovery_after_credentials_rotation_mos,
        'test_update_os_controller_simple': __recovery_after_mos_update,
        'test_update_os_controller_with_refapp': __recovery_after_mos_update,
        'test_node_controller_parallel': __no_downtime_live_migration,
        'test_tf_node_controller_parallel': __no_downtime,
        'test_node_controller_computes_az': __no_downtime_live_migration,
        'test_node_controller_evacuate': __no_downtime_live_migration,
        'test_tf_amphora_evacuate': __recovery_after_mos_lcm,
        'test_restart_containerd_tf_compute_node': __no_downtime,
        'test_autocleanup_db.py': __no_downtime,
        'test_change_mariadb_replicas_count_and_wait_for_restoration': __no_downtime,
        'test_delete_mariadb_pod_and_wait_for_restoration': __no_downtime,
        'test_force_kill_mariadb_pod_and_wait_for_restoration': __no_downtime,
        'test_force_sst_mariadb_and_wait_for_restoration': __no_downtime,
        'test_backup_basic': __no_downtime,
        'test_backup_with_remote_fix_corrupted_full': __no_downtime,
        'test_restore_full': __no_downtime,
        'test_restore_incremental': __no_downtime,
        'test_restore_from_remote_local_storage_corrupted_full': __no_downtime,
        'test_restore_from_remote_local_storage_corrupted_incremental': __no_downtime,
        'test_update_mgmt_cluster': __no_downtime,
        'test_graceful_mosk_pod_restarts': __recovery_after_mos_update,
        'test_ovn_migration_basic': __recovery_after_mos_ovn_migration,
        'test_delete_cassandra_db_pod': __no_downtime,
        'test_drop_data_cassandra_db_pod': __no_downtime,
        'test_ha_tf_control': __no_downtime,
        'test_ha_tf_config': __no_downtime,
        'test_ha_tf_rabbitmq': __no_downtime,
        'test_ha_tf_redis': __no_downtime,
        'test_ha_tf_zookeeper': __no_downtime,
    },
}


def calculate_load_manager_item_downtime(data: list, chunk_size=3, skip_not_sent=False) -> list:
    """ Calcuate downtime windows for specific load_manager item

    :param data: List of items with statistics
    :param chunk_size: Size of chunk during calculations. int
    :param skip_not_sent: Whether to exclude from calculations time when requests were not sent
                          or not. True/False
    :returns List with dowtime windows
             [{"start": "time_start", "stop": "time_stop", "duration": "time_stop - time_start"}]
    """
    res = []
    window = {}

    def aggregate_statistics(data, chunk_size):
        stats = []
        for index in range(0, len(data), chunk_size):
            samples_chunk = data[index:index + chunk_size]
            chunk = {}
            sample = samples_chunk[-1]
            total_keys = ["Name", "Timestamp", "Total Failure Count", "Total Request Count"]
            for key in total_keys:
                chunk[key] = sample[key]
            stats.append(chunk)
        return stats

    def _get_short_stats(item):
        timestamp = datetime.fromtimestamp(int(item["Timestamp"]))
        total_failure_count = int(item["Total Failure Count"])
        total_request_count = int(item["Total Request Count"])
        return (timestamp, total_failure_count, total_request_count)

    # NOTE(vsaienko): Locust provides statistic each second, when request is
    # timed out neither failure or requests count is increased. Devide data
    # by chunks the length of the chunk should be a bit longer that request timeout
    # in our case we set it to 1 sec everywhere, so use 3 chunks.
    aggregated_data = aggregate_statistics(data, chunk_size=chunk_size)
    data_len = len(aggregated_data)
    for index, item in enumerate(aggregated_data):
        timestamp, total_failure_count, total_request_count = _get_short_stats(item)
        prev_timestamp, prev_total_failure_count, prev_total_request_count = (timestamp, 0, 0)
        if index > 0:
            prev_timestamp, prev_total_failure_count, prev_total_request_count = _get_short_stats(
                aggregated_data[index - 1])

        # Handle no requests in time series as a downtime, test tool should send at least one request in series
        if total_failure_count > prev_total_failure_count or (prev_total_request_count == total_request_count
                                                              and not skip_not_sent):
            if not window:
                window["start"] = prev_timestamp
                window["stop"] = prev_timestamp
                window["total_failure_count_start"] = prev_total_failure_count
                window["total_request_count_start"] = prev_total_request_count
            else:
                window["stop"] = prev_timestamp
            if index == data_len - 1:
                window["duration"] = prev_timestamp - window["start"]
                window["failure_count"] = prev_total_failure_count - window.pop("total_failure_count_start")
                window["request_count"] = prev_total_request_count - window.pop("total_request_count_start")
                window["start_timestamp"] = int(window["start"].timestamp())
                window["stop_timestamp"] = int(window["stop"].timestamp())
                res.append(window)
        else:
            if not window:
                continue
            else:
                window["stop"] = prev_timestamp
                window["duration"] = prev_timestamp - window["start"]
                window["failure_count"] = prev_total_failure_count - window.pop("total_failure_count_start")
                window["request_count"] = prev_total_request_count - window.pop("total_request_count_start")
                window["start_timestamp"] = int(window["start"].timestamp())
                window["stop_timestamp"] = int(window["stop"].timestamp())
                res.append(window)
                window = {}
    return res


def calculate_load_manager_downtime(data: dict, start_ts: int, end_ts: int) -> list:
    """
    Calculate potential downtime period

    Args:
        data: load_managerManager.get_full_history object

    Returns: result list of downtimes

    """
    split_statistics = {}
    skip_names = ["Aggregated"]

    for item in data:
        if item["Name"] in skip_names:
            continue
        item_ts = int(item["Timestamp"])
        if end_ts < item_ts or item_ts < start_ts:
            continue

        ident = (item["Type"], item["Name"])
        split_statistics.setdefault(ident, [])
        split_statistics[ident].append(item)

    downtimes = {}
    chunk_size = 3
    skip_not_sent = False
    use_custom_conf = settings.DOWNTIME_STATISTIC_USE_CUSTOM_CONFIG
    if use_custom_conf:
        LOG.info("Custom config for downtime calculations will be used")
        with open(settings.DOWNTIME_STATISTIC_CUSTOM_CONFIG, 'r') as conf:
            custom_conf = yaml.safe_load(conf)
        LOG.info(f"Config data:\n{yaml.dump(custom_conf)}")
    for ident, items in split_statistics.items():
        if use_custom_conf:
            check_name = ident[1].split(':')[0]
            chunk_size = custom_conf.get(check_name, {}).get('chunk_size', chunk_size)
            skip_not_sent = custom_conf.get(check_name, {}).get('skip_not_sent', skip_not_sent)
        downtimes[ident] = calculate_load_manager_item_downtime(
            items, chunk_size=chunk_size, skip_not_sent=skip_not_sent)
    return downtimes


def get_statistic_data():
    if os.path.isfile(settings.DOWNTIME_STATISTIC_FILE_PATH):
        with open(settings.DOWNTIME_STATISTIC_FILE_PATH) as f:
            collected_data = yaml.safe_load(f.read())
        return collected_data.get('collected_data', [])
    msg = f"File with statistic data '{settings.DOWNTIME_STATISTIC_FILE_PATH}' not found, skipping the test"
    LOG.error(msg)
    pytest.skip(msg, allow_module_level=True)


def get_test_ids(stat_data):
    ids = []
    for x in stat_data:
        cluster_data = x['cluster_namespace'] + "/" + x['cluster_name']
        test_id = '-'.join([x['test_name'], cluster_data, x['statistic_endpoint_name']])
        ids.append(f"STATISTIC={test_id}")
    return ids


statistic_data_list = get_statistic_data()


@pytest.fixture(scope='function', params=statistic_data_list,
                ids=get_test_ids(statistic_data_list))
def statistic_data(request):
    """Generates tests for each test record in the file <settings.DOWNTIME_STATISTIC_FILE_PATH>"""
    yield request.param


def test_check_downtime_statistic(statistic_data):
    """MCC cluster updates downtime analyzer"""
    errors = {}
    statistics = statistic_data['statistic_data']
    cluster_name = statistic_data.get('cluster_name')
    namespace_name = statistic_data.get('cluster_namespace')

    test_name = statistic_data.get('test_name')
    cluster_provider_name = statistic_data.get('cluster_provider_name')
    statistic_endpoint_name = statistic_data.get('statistic_endpoint_name')
    statistic_data_source = statistic_data.get('statistic_data_source')
    version_after_test = statistic_data.get('version_after_test')
    openstack_version_after_test = statistic_data.get('openstack_version_after_test')
    release_key = statistic_data.get('release_key')
    assert release_key, (
        f"Unset 'release_key' after {test_name} for the cluster '{namespace_name}/{cluster_name}' "
        f"with release '{version_after_test} {openstack_version_after_test}'")

    test_expected_downtimes = expected_downtimes[release_key].get(test_name)
    LOG.info(f"Expected downtime: {test_expected_downtimes}")
    assert type(test_expected_downtimes) is dict, (
        f"Test name '{test_name}' is missing in '{release_key}', please add")
    provider_expected_downtimes = test_expected_downtimes.get(cluster_provider_name)
    assert type(provider_expected_downtimes) is dict, (
        f"Cluster provider '{cluster_provider_name}' is missing in '{release_key}' "
        f"for the test '{test_name}', please add")

    data_source_downtimes = provider_expected_downtimes.get(statistic_data_source, {})
    LOG.info(f"Checking [{cluster_provider_name}] cluster '{namespace_name}/{cluster_name}' "
             f"({release_key} '{version_after_test} {openstack_version_after_test}') downtime statistic "
             f"collected from '{statistic_endpoint_name}' endpoint during '{test_name}' "
             f"with single/total maximum endpoint downtimes = {data_source_downtimes} seconds")

    if statistic_data_source in ['cloudprober', 'portprober']:
        endpoint_expected_downtime = data_source_downtimes.get(statistic_endpoint_name)
        if type(endpoint_expected_downtime) is dict:
            versioned_downtimes = endpoint_expected_downtime.get(
                version_after_test, endpoint_expected_downtime.get('default', {}))
            endpoint_expected_downtime = versioned_downtimes.get(openstack_version_after_test)
        # mosk statistics is calculated already
        single_downtime, max_downtimes = endpoint_expected_downtime
        utils.check_mosk_workloads_downtime(statistics, statistic_endpoint_name, single_downtime, max_downtimes)

    elif statistic_data_source == 'load_manager':
        start_ts = statistic_data["start_time_timestamp"]
        end_ts = statistic_data["end_time_timestamp"]
        downtimes = calculate_load_manager_downtime(statistics, start_ts, end_ts)
        for ident, downtime in downtimes.items():
            if not downtime:
                LOG.info("No Downtime detected for %s", ident)
                continue
            try:
                application, method = ident[1].split(':')
                application_expected_downtime = data_source_downtimes.get(application)
                assert method is not None, f"Unable to parse method for application {application}"
                assert application_expected_downtime is not None, \
                    f"Unable to get expected downtime for application {application}"
                assert method in application_expected_downtime, (
                    f"Expected dowtimes are missing for statistic_data_source: "
                    f"{statistic_data_source}, application: {application}, method: {method}")
                single_downtime, max_downtimes = application_expected_downtime.get(method)
                LOG.info("Downtime for %s %s", application, method)
                errors[ident] = utils.check_downtime(downtime=downtime, expected_downtime=single_downtime,
                                                     max_downtimes=max_downtimes, raise_on_error=False)
            except ValueError as e:
                LOG.debug(f"Got unpack error for intent {ident} , error {e}")
    else:
        raise Exception(f"Unsupported statistic_data_source {statistic_data_source}")
    failed_checks = [{check[1]: check[0]} for check, is_fail in errors.items() if is_fail]
    assert not failed_checks, f"Next items have unexpected downtimes:\n{yaml.dump(failed_checks)}"
