#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import pytest
import os

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import utils
from si_tests.utils import waiters

LOG = logger.logger


class TestCreateBootstrapv2Em2Cluster():
    def test_bootstrapv2_em2_create(self, show_step) -> None:
        """Bootstrap V2 create KinD cluster for EquinixMetalV2

         Scenario:
             1. Verify seed node accessibility
             2. Prepare seed node to run MCC bootstrap for EquinixMetalV2
             3. Get network setup
             4. Get kubeconfig for regional deployment
             5. Setup EquinixMetalV2 variables
             6. Run bootstrap V2 cluster
         """

        # step 001: Verify seed node accessibility
        show_step(1)
        assert waiters.icmp_ping(settings.SEED_STANDALONE_EXTERNAL_IP), (
            f"EM2 seed node for management cluster is not accessible by IP {settings.SEED_STANDALONE_EXTERNAL_IP}")
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=settings.SEED_STANDALONE_EXTERNAL_IP)

        # step 002: Setup seed node
        show_step(2)
        bootstrap.create_seed_bash_completion()
        run_envs = bootstrap.step_003_prepare_standalone_seed_node(render_templates=False)

        # step 003: Network setup
        show_step(3)
        is_mgmt_deployment = False if os.environ.get("KUBECONFIG_ARTIFACT", None) else True
        network_config = {}
        mgmt_kubeconfig = ''
        if is_mgmt_deployment:
            management_network_config, region_network_configs, child_network_configs = \
                bootstrap.equinixmetalv2_configure_networking()
            cluster_key = "{}/{}".format(settings.CLUSTER_NAMESPACE, settings.CLUSTER_NAME)
            network_config = management_network_config[cluster_key]
        else:
            # step 004: Get kubeconfig for regional deployment
            show_step(4)
            if not settings.SI_BINARIES_DIR:
                raise Exception("WORKSPACE environment variable is not set, unable to find 'kubeconfig'")
            mgmt_kubeconfig = f"/home/{settings.SEED_SSH_LOGIN}/{settings.KAAS_BOOTSTRAP_TARGET_DIR}/kubeconfig"
            bootstrap.copy_user_file(
                os.path.join(settings.SI_BINARIES_DIR, "artifacts/management_kubeconfig"),
                settings.SEED_SSH_LOGIN,
                mgmt_kubeconfig)

            kaas_manager = Manager(kubeconfig=mgmt_kubeconfig)
            region_config = kaas_manager.si_config.data
            assert 'network_config' in region_config, "No 'network_config' key specified in the SI_CONFIG"
            region_key = "{}/{}".format(settings.REGION_NAMESPACE, settings.REGIONAL_CLUSTER_NAME)
            network_config = region_config['network_config'][region_key]

        # steps 005: Prepare env variables
        show_step(5)
        if not is_mgmt_deployment:
            run_envs['MGMT_KUBECONFIG'] = mgmt_kubeconfig
        run_envs['KAAS_BOOTSTRAP_LOG_LVL'] = settings.KAAS_BOOTSTRAP_LOG_LVL
        run_envs['KAAS_EQUINIXMETALV2_ENABLED'] = True
        run_envs['KAAS_BM_PXE_BRIDGE'] = settings.KAAS_EQUINIX_PRIVATE_PXE_BRIDGE
        run_envs['KAAS_BM_PXE_IP'] = network_config['bootstrap']['bm_pxe_ip']
        run_envs['KAAS_BM_PXE_MASK'] = network_config['bootstrap']['bm_pxe_mask']

        # EQUINIXMETALV2 needs proxy for bootstrap
        proxy_url = "http://{}:{}".format(network_config['networkSpec']['gateway'],
                                          settings.KAAS_EQUINIX_PRIVATE_PROXY_PORT)
        run_envs['HTTP_PROXY'] = proxy_url
        run_envs['HTTPS_PROXY'] = proxy_url
        run_envs['http_proxy'] = proxy_url
        run_envs['https_proxy'] = proxy_url
        LOG.info(f"HTTP_PROXY and HTTPS_PROXY is set to {proxy_url}")

        # step 006: Run bootstrap V2 cluster
        show_step(6)
        envs_string = utils.make_export_env_strting(run_envs)
        LOG.info(f"Current envs_string {envs_string}")
        if not bootstrap.setup_bootstrapv2_cluster(envs_string=envs_string):
            pytest.fail("Failed to setup bootstrapv2 cluster")
