from si_tests.utils import packaging_version as version
from si_tests import settings

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.kaas_manager import Cluster


class WorkaroundCheckManagerBase(object):
    """Common methods useful for workarounds check"""

    def __init__(self, cluster: "Cluster"):
        self._cluster: "Cluster" = cluster

    @property
    def cluster(self) -> "Cluster":
        return self._cluster

    @property
    def k8sclient(self):
        return self._cluster.k8sclient

    @property
    def mcc_version(self):
        """Active MCC release version"""
        mgmt_cluster = self._cluster._manager.get_mgmt_cluster()
        return mgmt_cluster.get_kaasrelease_version()

    @property
    def clusterrelease_version(self):
        """Current Cluster version"""
        return self._cluster.clusterrelease_version


class WorkaroundCheckManager(WorkaroundCheckManagerBase):
    """Workaround check manager

    Purpose of this class is to keep all the checks for workarounds in a single place.

    - Each method should return 'True' if the related workaround should be applied.
      Please don't add any waiters/actions here. Only checks for versions/providers/cluster objects/etc...

    - Methods should be named using PRODX bugs/tasks, for example:

          def prodx_12345(self):

    - Please clear the code from outdated workarounds after the related tasks are resolved
    """

    def field_6283(self):
        # resolve https://mirantis.jira.com/browse/FIELD-6283 planned for 3.6.7 MKE release
        if version.parse(self.clusterrelease_version) < version.parse('mke-16-0-0-3-6-7'):
            return True
        else:
            return False

    def field_5850(self):
        # https://mirantis.jira.com/browse/FIELD-5850
        # current status is pending investigation
        return True

    def prodx_27814(self):
        # workaround regarding https://mirantis.jira.com/browse/PRODX-27814
        if version.parse(self.mcc_version) >= version.parse("kaas-2-25-0-rc"):
            if (self.clusterrelease_version.startswith("mke-14")
                    and version.parse(self.clusterrelease_version) >= version.parse('mke-14-1-0-3-6-6')):
                return True
            elif (self.clusterrelease_version.startswith("mke-16")
                  and version.parse(self.clusterrelease_version) >= version.parse('mke-16-0-0-3-7-1')):
                return True
            else:
                return False
        else:
            return False

    def prodx_36209(self):
        # https://mirantis.jira.com/browse/PRODX-36209
        if version.parse(self.clusterrelease_version) >= version.parse('mosk-17-0-0-23-3'):
            return True
        return False

    def prodx_37306(self):
        # https://mirantis.jira.com/browse/PRODX-37306
        if (('mke-16' in self.clusterrelease_version and
             version.parse(self.clusterrelease_version) >= version.parse('mke-16-0-1-3-7-2')) or
                ('mosk' in self.clusterrelease_version and
                 version.parse(self.clusterrelease_version) >= version.parse('mosk-17-0-1-23-3-1'))):
            return True
        return False

    def prodx_51513(self):
        # https://mirantis.jira.com/browse/PRODX-51513
        if (('mke-20' in self.clusterrelease_version and
             version.parse(self.clusterrelease_version) >= version.parse('mke-20-0-0-3-8-3')) or
                ('mosk' in self.clusterrelease_version and
                 version.parse(self.clusterrelease_version) >= version.parse('mosk-21-0-0-25-1-1'))):
            return True
        return False

    def prodx_40036(self):
        """Node deletion wokraround is be enabled until mosk-17-1-1-24-1-1 in 2.26.1.

        https://mirantis.jira.com/browse/PRODX-40036

        Please update the version if the bug not fixed for next versions"""
        if ('mosk' in self.clusterrelease_version and
                version.parse(self.clusterrelease_version) < version.parse('mosk-17-1-1')):
            return True
        return False

    def prodx_40790(self):
        """LCM stuck flag workaround will be enabled until 2.27"""
        # https://mirantis.jira.com/browse/PRODX-40790
        if version.parse(self.mcc_version) < version.parse("kaas-2-27"):
            return True
        return False

    def prodx_42484(self):
        """LCM stuck flag cleanup will be fixed in 2.28"""
        # https://mirantis.jira.com/browse/PRODX-42484
        if version.parse(self.mcc_version) < version.parse("kaas-2-28"):
            return True
        return False

    def prodx_42401(self):
        """Remove GracefulRebootRequest manually before moving SL/Ceph labels to other Machines"""
        # https://mirantis.jira.com/browse/PRODX-42401
        # Always apply the workaround, as described in PRODX-42401
        return True

    def prodx_43331(self):
        """Do not execute phase check in case if codebase from 2.26 used"""
        # https://mirantis.jira.com/browse/PRODX-43331
        if ((version.parse(self.clusterrelease_version) < version.parse('mke-16-2-0-rc-3-7-8')) or
                (self.clusterrelease_version.startswith(settings.MOSK_RELEASE_PREFIX)
                 and version.parse(self.clusterrelease_version) < version.parse('mosk-17-2-0-rc-24-2'))):
            return True
        return False

    def prodx_44228(self):
        """Check that ceph cluster is affected because of OSD replace procedure and
           update ceph keyrings if needed
        """
        return True

    def prodx_47277(self):
        # https://mirantis.jira.com/browse/PRODX-47277
        """ Autoupgrade BlockUntil functionality from 2.28.0 moved to a new approach AutoDelay"""
        if version.parse(self.mcc_version) >= version.parse("kaas-2-28-0-rc"):
            return True
        return False

    def prodx_47907(self):
        # https://mirantis.jira.com/browse/PRODX-47907
        return ['Kubelet', 'AgentTokenRotation']

    def skip_kaascephcluster_usage(self):
        # https://mirantis.jira.com/browse/PRODX-48011
        """ Do not use KaasCephCluster and use child resources directly """
        # Part of epic for KaasCephCluster remove
        return version.parse(self.mcc_version) >= version.parse("kaas-2-31-0-rc") or settings.NO_KAASCEPHCLUSTER

    def prodx_52237(self):
        """
        Workaround for PRODX-52237:
        Ignore any pods in the node-feature-discovery namespace.
        https://mirantis.jira.com/browse/PRODX-52237
        """
        return (version.parse(self.clusterrelease_version) >= version.parse('mke-16-3-9-3-7-23') or
                version.parse(self.clusterrelease_version) >= version.parse('mosk-17-3-9-24-3-6')
                )


def wa_phys429_hack_mgmt_deploy_on_vsphere():
    """
    PRODX-42152  Hack-owner: bm-team.
    Special hack for quick deployment mgmt on standalone vsphere seed.
    Hardcoded everything
    :return:
    """

    from si_tests.managers import si_config_manager
    from si_tests import settings
    from si_tests.managers import bootstrap_manager
    from si_tests import logger

    LOG = logger.logger

    if not settings.BM_DEPLOY_MGMT_ON_VSPHERE:
        raise Exception('Wrong test invocation')
    vs_seed_ip = '172.16.38.123'
    LOG.warning(f'Switching to work with hardcoded vs-seed:{vs_seed_ip}')
    si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
    si_config.store_seed_ip(vs_seed_ip)
    bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
    remote = bootstrap.remote_seed()
    # perform ugly hard-coded cleanup
    remote.execute('~/bootstrap/dev/bin/kind delete cluster --name clusterapi', verbose=True)
    remote.execute('rm -rf ~/kaas_releases ~/bootstrap '
                   '/tmp/step_003_prepare_bm_seed_node '
                   '/var/lib/si-tests/workspace_ptr* ', verbose=True)
    env_config_name = bootstrap.bm_determine_env_config_name()
    bootstrap.step_003_prepare_bm_seed_node()
    # need to init for sync kaas-bm-env and some default pkgs
    bootstrap.ansible_init()
    bootstrap.prepare_bm_seed_node_templates(env_config_name)

    envs = ['HOME=/home/ubuntu/bootstrap/dev/',
            'KAAS_BOOTSTRAP_DEBUG=True',
            'KAAS_BM_ENABLED=True',
            'KAAS_BM_PXE_BRIDGE=br0',
            'KAAS_BM_PXE_IP="10.0.25.20"',
            'KAAS_BM_PXE_MASK=24']
    for i in envs:
        remote.execute(f"echo 'export {i}' >> /home/ubuntu/bootstrap/dev/bootstrap.env", verbose=True)
    return bootstrap
