import pytest

from si_tests import logger
from si_tests import settings

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
def test_cleanup_kcm(kcm_bootstrap_manager, show_step):
    """Cleanup KCM chart out of k8s cluster

    Scenario:
        1. Verify kcm installed inside provided cluster
        2. Initialize kcm manager
        3. Check clusterdeployments presence and remove if present (optional)
        4. Check credential objects and remove if present (optional)
        5. Check and remove any static identities (optional)
        6. Check and remove auth configmaps if exists (optional) and secrets (if exists)
        7. Check and remove kof (if exists)
        8. Remove management object
        9. Uninstall helm kcm release
        10. Remove kcm-system and projectsveltos namespaces
        11. Remove all clusterroles and clusterrolebindings related to sveltos and CAPI
        12. Remove CRDs
    """

    show_step(1)
    assert kcm_bootstrap_manager.is_kcm_installed(), 'KCM chart is not installed in cluster'

    show_step(2)
    kcm_manager = kcm_bootstrap_manager.check.kcm_mgr

    if settings.REMOVE_RESOURCES_ON_KCM_CLEANUP:
        show_step(3)
        clds = kcm_manager.list_all_clusterdeployments()
        if clds:
            LOG.info(f"Found clusterdeployments {clds}.")
            LOG.warning("Async removal of all clusterdeployments")

            for cl in clds:
                LOG.info(f"Initialising resource chain for cluster {cl.namespace}/{cl.name}")
                # (va4st): initialize resource chain before removal start for each cluster to check leftovers
                _ = cl.clusterobject.infracluster

            for cl in clds:
                cl.delete()
                LOG.info(f"Request for removal {cl.namespace}/{cl.name} was sent.")

            LOG.info('Additionally checking for leftovers after clusterdeployments removal')
            for cl in clds:
                cl.check.check_clusterdeployment_deleted()
            # (va4st): Async removal of clds make us able not to wait for removal of each cluster separately. But it's
            # still need to check anyway that everything was cleared up including capi objects.
        show_step(4)
        creds = kcm_manager.list_all_credentials()
        if creds:
            LOG.info(f"Credentials found: {creds}")
            for cred in creds:
                LOG.info(f"Removing Credential {cred.namespace}/{cred.name}")
                cred.delete()

        show_step(5)
        awssis = kcm_manager.api.awsclusterstaticidentities.list_all()
        if awssis:
            LOG.info(f"AWS Static Identities found: {awssis}")
            for awssi in awssis:
                awssi.delete()
                LOG.info(f"AWSSI {awssi.name} removed")

        azcids = kcm_manager.api.azureclusteridentities.list_all()
        if azcids:
            LOG.info(f"Azure Cluster IDs found: {azcids}")
            for azcid in azcids:
                azcid.delete()
                LOG.info(f"Azure cID {azcid.namespace}/{azcid.name} removed")

        vspcids = kcm_manager.api.vsphereclusteridentities.list_all()
        if vspcids:
            LOG.info(f"vSphere Cluster IDs found: {vspcids}")
            for vspcid in vspcids:
                vspcid.delete()
                LOG.info(f"vSphere cID {vspcid.name} removed")

        show_step(6)
        # TODO(va4st): Temporal cleanup by name. When separated auth manager will be introduced - need to change
        # TODO: that behavior to remove auth configs not by name filtering but by labels
        auth_cms = kcm_manager.api.configmaps.list_all(name_prefix='ksi')
        if auth_cms:
            LOG.info(f"ConfigMaps contains ksi auth creds found: {auth_cms}")
            for cm in auth_cms:
                cm.delete()
                LOG.info(f"ConfigMap {cm.namespace}/{cm.name} removed")

        auth_secrets = kcm_manager.api.secrets.list_all(name_prefix='ksi')
        if auth_secrets:
            LOG.info(f"Secrets contains ksi auth creds found: {auth_secrets}")
            for secr in auth_secrets:
                secr.delete()
                LOG.info(f"Secret {secr.namespace}/{secr.name} removed")

    show_step(7)
    if kcm_manager.kof.is_kof_mothership_present():
        kcm_bootstrap_manager.check.cleanup_grafana_entities()

    if settings.KSI_KOF_ISTIO_NAMESPACE not in [ns.name for ns in kcm_manager.get_namespaces()]:
        pass
    else:
        LOG.info(f"{settings.KSI_KOF_ISTIO_NAMESPACE} found. Double check that kof also will be removed")
        kof_istio_ns = kcm_manager.get_namespace(settings.KSI_KOF_ISTIO_NAMESPACE)
        if kcm_manager.kof.is_kof_istio_present():
            kcm_bootstrap_manager.check.uninstall_kof_istio_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        kof_istio_ns.delete(a_sync=False)

    if settings.KSI_KOF_NAMESPACE not in [ns.name for ns in kcm_manager.get_namespaces()]:
        pass
    else:
        # full copy of test_cleanup_kof_mothership
        LOG.info(f"{settings.KSI_KOF_NAMESPACE} found. Double check that kof also will be removed")
        kof_ns = kcm_manager.get_namespace(settings.KSI_KOF_NAMESPACE)
        if kcm_manager.kof.is_kof_child_present():
            kcm_bootstrap_manager.check.uninstall_kof_child_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_regional_present():
            kcm_bootstrap_manager.check.uninstall_kof_regional_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_collectors_present():
            kcm_bootstrap_manager.check.uninstall_kof_collectors_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_mothership_present():
            kcm_bootstrap_manager.check.uninstall_kof_mothership_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_operators_present():
            kcm_bootstrap_manager.check.uninstall_kof_op_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        kof_ns.delete(a_sync=False)

    show_step(8)
    kcm_bootstrap_manager.check.delete_mgmt_object_and_wait()

    show_step(9)
    kcm_bootstrap_manager.check.uninstall_kcm_and_wait()

    show_step(10)
    kcm_bootstrap_manager.check.kcm_namespace.delete()
    nss = kcm_manager.get_namespaces()
    if settings.SVELTOS_NAMESPACE in [ns.name for ns in nss]:
        LOG.info(f"{settings.SVELTOS_NAMESPACE} present")
        sveltos_ns = kcm_manager.get_namespace(settings.SVELTOS_NAMESPACE)
        sveltos_ns.delete()
        sveltos_ns.wait_for_deletion()
    if settings.KSI_SVELTOS_MGMT_NAMESPACE in [ns.name for ns in nss]:
        LOG.info(f"{settings.KSI_SVELTOS_MGMT_NAMESPACE} present")
        sveltos_ns = kcm_manager.get_namespace(settings.KSI_SVELTOS_MGMT_NAMESPACE)
        sveltos_ns.delete()
        sveltos_ns.wait_for_deletion()

    show_step(11)
    capi_label = 'clusterctl.cluster.x-k8s.io'
    capi_clusterroles = kcm_manager.api.clusterrole.list_all(label_selector=capi_label)
    if capi_clusterroles:
        for role in capi_clusterroles:
            LOG.info(f"Removing clusterrole {role.name}")
            role.delete()
    capi_clusterrolebindings = kcm_manager.api.clusterrolebindings.list_all(label_selector=capi_label)
    if capi_clusterrolebindings:
        for rolebind in capi_clusterrolebindings:
            LOG.info(f"Removing clusterrolebinding {rolebind.name}")
            rolebind.delete()
    # NOTE(va4st): sveltos resources have no labels, so it can be identified only by name
    sv_name_pattern = 'sveltos'
    all_clusterroles = kcm_manager.api.clusterrole.list_all()
    sv_clusterroles = [role for role in all_clusterroles if sv_name_pattern in role.name]
    if sv_clusterroles:
        for role in sv_clusterroles:
            LOG.info(f"Removing clusterrole {role.name}")
            role.delete()
    all_clusterrolebindings = kcm_manager.api.clusterrolebindings.list_all()
    sv_clusterrolesbindings = [binding for binding in all_clusterrolebindings if sv_name_pattern in binding.name]
    if sv_clusterrolesbindings:
        for rolebind in sv_clusterrolesbindings:
            LOG.info(f"Removing clusterrolebinding {rolebind.name}")
            rolebind.delete()

    show_step(12)
    kcm_bootstrap_manager.check.cleanup_crds()
