#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from _datetime import datetime, timezone

import si_tests.utils.waiters as helpers

from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
from si_tests import logger

LOG = logger.logger


class K8sDeployment(K8sNamespacedResource):
    resource_type = 'deployment'

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_deployment(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_deployment(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_deployment(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_deployment(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, **kwargs):
        self._manager.api.delete_namespaced_deployment(
            self.name, self.namespace, **kwargs)

    @property
    def ready(self):
        obj = self.data
        if obj["status"]["observed_generation"] >= obj["metadata"]["generation"]:
            if (obj["status"].get("updated_replicas") or 0) == obj['spec']['replicas']:
                if (obj["status"].get("ready_replicas") or 0) == obj['spec']['replicas']:
                    return True

        LOG.debug("Deployment {}/{} \nstatus.observed_generation: '{}', metadata.generation: '{}'\n"
                  "status.updated_replicas: '{}', expected: '{}' \nstatus.ready_replicas: '{}', expected: '{}'"
                  "".format(self.namespace, self.name,
                            obj["status"]["observed_generation"], obj["metadata"]["generation"],
                            obj["status"].get("updated_replicas"), obj['spec']['replicas'],
                            obj["status"].get("ready_replicas"), obj['spec']['replicas']))
        return False

    def wait_ready(self, timeout=120, interval=5):
        helpers.wait(lambda: self.ready,
                     timeout=timeout, interval=interval)
        return self

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind, metadata,
        spec, status or items
        """
        return self.read().to_dict()

    @property
    def desired_replicas(self):
        return self.read().spec.replicas

    @property
    def ready_replicas(self):
        status = self.read().status
        if status:
            return status.ready_replicas
        else:
            LOG.info("Status of deployment {0}/{1} not populated yet".format(self.namespace, self.name))
            # If status does not present - no ready replicas expected
            return 0

    def set_replicas(self, replicas):
        return self.patch(body={"spec": {"replicas": replicas}})

    def rollout_restart(self):
        restart_time = datetime.now(timezone.utc).isoformat()
        LOG.info(f"Rolling restart of deployment {self.namespace}/{self.name}")

        patch = {
            "spec": {
                "template": {
                    "metadata": {
                        "annotations": {
                            "ksi.kubernetes.io/restartedAt": restart_time
                        }
                    }
                }
            }
        }
        self.patch(body=patch)


class K8sDeploymentManager(K8sBaseManager):
    resource_class = K8sDeployment

    @property
    def api(self):
        return self._cluster.api_apps

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_deployment(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_deployment_for_all_namespaces(**kwargs)
