#!/usr/bin/env python3
"""Extract package dependencies from pyproject.toml and write to requirements.txt."""

import tomllib
from pathlib import Path

# Assume script is run from the repo root (where pyproject.toml and tox.ini live)
# PROJECT_ROOT = Path.cwd()
# PYPROJECT = PROJECT_ROOT / "pyproject.toml"
# REQUIREMENTS = PROJECT_ROOT / "si_tests" / "requirements.txt"

BASE_DIR = Path.cwd()
PYPROJECT = BASE_DIR / "pyproject.toml"
REQUIREMENTS = BASE_DIR / "si_tests" / "requirements.txt"
DO_NOT_EDIT = """# -> DO NOT EDIT DIRECTLY <-
# Edit dependencies in 'pyproject.toml' in repo root then run:
#
#    make sync-dependencies
#
"""

# Load dependencies from pyproject.toml
pyproject = tomllib.loads(Path(PYPROJECT).read_text())
deps = pyproject.get("project", {}).get("dependencies", [])

# Generate the new requirements content
new_content = DO_NOT_EDIT + "\n".join(deps) + "\n"

# Read existing requirements.txt (if any)
old_content = REQUIREMENTS.read_text() if REQUIREMENTS.exists() else ""

# Write only if the contents are different
if new_content != old_content:
    REQUIREMENTS.write_text(new_content)
    print(f"{REQUIREMENTS} updated from {PYPROJECT}")
    print(f"Be sure to include the updated {REQUIREMENTS} in your commit.")
else:
    print(f"{REQUIREMENTS} is already up to date.")
