#    Copyright 2016 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from __future__ import division

import datetime
import time
from si_tests import settings
from si_tests.utils import templates
import pytest

from si_tests import logger
from si_tests.utils import log_step
from si_tests.utils.utils import LogTime

LOG = logger.logger

# Required env vars for tests using provider authentication
PROVIDER_ENV_VARS = {
    "aws": ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
    "eks": ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
    "azure": [
        "AZURE_SUBSCRIPTION_ID",
        "AZURE_CLIENT_ID",
        "AZURE_TENANT_ID",
        "AZURE_CLIENT_SECRET",
    ],
    "os": ["OS_APPLICATION_CREDENTIAL_ID", "OS_APPLICATION_CREDENTIAL_SECRET"],
    "vsphere": [
        "VSPHERE_SERVER_ADDR",
        "VSPHERE_DATACENTER_NAME",
        "VSPHERE_DATASTORE_PATH",
        "VSPHERE_USER",
        "VSPHERE_PASSWORD",
        "VSPHERE_NETWORK_PATH",
        "VSPHERE_VM_TEMPLATE_PATH",
        "VSPHERE_RESOURCE_POOL_PATH",
        "VSPHERE_FOLDER_PATH",
    ],
    "gcp": [
        "GCP_PROJECT_ID",
        "GCP_PRIVATE_KEY_ID",
        "GCP_PRIVATE_KEY",
        "GCP_CLIENT_EMAIL",
        "GCP_CLIENT_ID",
        "GCP_CLIENT_X509_CERT_URI",
        "GCP_NETWORK_NAME",
    ],
}


@pytest.hookimpl(tryfirst=True, hookwrapper=True)
def pytest_runtest_makereport(item, call):
    outcome = yield
    rep = outcome.get_result()
    setattr(item, "rep_" + rep.when, rep)


def pytest_runtest_setup(item):
    if item.cls is not None:
        item.cls._current_test = item.function
    item._start_time = time.time()
    item._step_time = {"start": datetime.datetime.now()}
    head = "#" * 30 + "[ {} ]" + "#" * 30
    head = head.format(item.function.__name__)
    docstring = item.function.__doc__
    info = "\n{head}".format(head=head)
    if docstring and type(docstring) is str:
        docstring = "\n".join([f"### {line}" for line in docstring.splitlines()])
        info = f"{info}\n{docstring}"
    LOG.info(info)
    check_auth_env_vars(item)


def pytest_runtest_teardown(item):
    step_name = item.function.__name__
    if hasattr(item, "_step_time"):
        item._step_time.update({"stop": datetime.datetime.now()})
    if hasattr(item, "_start_time"):
        spent_time = time.time() - item._start_time
    else:
        spent_time = 0
    minutes = spent_time // 60
    seconds = int(round(spent_time)) % 60
    finish_step = "FINISH {} TEST. TOOK {} min {} sec".format(
        step_name, minutes, seconds
    )
    print("\n\n")
    foot = "\n" + "<" * 5 + "#" * 30 + "[ {} ]" + "#" * 30 + ">" * 5
    foot = foot.format(finish_step)
    LOG.info(foot)


@pytest.fixture(scope="function", autouse=True)
def show_step(request):
    def _show_step(step_number):
        request.node._step_time.update({step_number: datetime.datetime.now()})
        return log_step.log_step(request.function, step_number)

    return _show_step


@pytest.fixture(scope="function")
def steps(request):
    steps_mark = request.keywords.get("steps", None)
    steps = steps_mark.args[0]
    return steps


@pytest.fixture(scope="function", autouse=True)
def func_name(request):
    """Name of the current test function"""
    return getattr(request.node.function, "_name", request.node.function.__name__)


@pytest.fixture(scope="function")
def log_method_time(func_name):
    with LogTime(func_name, file_path=settings.ARTIFACTS_DIR + "time_spent.yaml"):
        yield


@pytest.fixture(
    scope="function", autouse=False, params=["ENV_NAME={0}".format(settings.ENV_NAME)]
)
def modify_case_name(request, func_name):
    """Make unique test cases name by adding ENV_NAME to it as a parameter"""
    LOG.debug(
        "Extend the test case name '{0}' with the suffix '{1}'".format(
            func_name, request.param
        )
    )


@pytest.hookimpl(tryfirst=True)
def store_testcase_doc_to_junit(item):
    """Attach the test function's docstring as a property in JUnit XML"""
    docstring = getattr(item.function, "__doc__", None)
    if docstring:
        text = "**Scenario:**\\n" + "\\n".join(docstring.strip().splitlines()) + "\\n"
        item.user_properties.append(("__doc__", text))


@pytest.fixture(scope="function")
def log_step_time(request):
    yield
    original = request.node._step_time
    keys = list(original)
    step_names = log_step.parse_test_doc(request.function.__doc__)["steps"]
    max_step_len = max(len(x) for x in step_names)
    result = f"\n\n{'Step': ^{max_step_len}} || {'Time': ^10}\n"
    yaml_result = {}
    yaml_step_data = {}
    for idx, val in enumerate(keys[1:-1], start=1):
        step_name = step_names[idx - 1]
        step_time = original[keys[idx + 1]] - original[val]
        hours, minutes = step_time.seconds // 3600, step_time.seconds % 3600 // 60
        seconds = step_time.seconds - hours * 3600 - minutes * 60
        time_string = f"{hours} h {minutes} min {seconds} sec"
        result += f"{step_name: <{max_step_len}} || {str(time_string): <10}\n"
        formatted_step_name = "_".join(
            i[0].lower() + i[1:] for i in step_name.split(" ")
        )
        yaml_step_data.update(
            {
                formatted_step_name: {
                    "environment": settings.ENV_NAME,
                    "raw_duration": step_time.total_seconds(),  # raw seconds
                    "duration": time_string,  # human readable format
                }
            }
        )

    LOG.info(result)
    yaml_result[request.node.originalname] = yaml_step_data
    with templates.YamlEditor(
        file_path=settings.ARTIFACTS_DIR + "steps_time_spent.yaml"
    ) as editor:
        editor.content = yaml_result


@pytest.fixture(scope="function")
def log_start_end_timestamps(func_name):
    start_time = str(int(time.time()))
    LOG.info("%s function/method start Unix timestamp is %s", func_name, start_time)
    with open(f"{settings.ARTIFACTS_DIR}/timestamp_start", mode="w") as f:
        f.write(start_time)
    yield
    end_time = str(int(time.time()))
    LOG.info("%s function/method end Unix timestamp is %s", func_name, end_time)
    with open(f"{settings.ARTIFACTS_DIR}/timestamp_end", mode="w") as f:
        f.write(end_time)


def check_auth_env_vars(item):
    """This is designed to fail if any of the env vars for
    the given provider do not have a value in `settings`
    """
    provider_marker = None

    # look for a provider marker
    for marker in item.iter_markers():
        if marker.name in PROVIDER_ENV_VARS:
            provider_marker = marker.name
            break

    if provider_marker:
        missing_vars = []
        for env_var in PROVIDER_ENV_VARS.get(provider_marker, []):
            value = getattr(settings, env_var, None)
            if value is None or value == "":
                missing_vars.append(env_var)

        if missing_vars:
            pytest.fail(
                f"Missing required env variables for {provider_marker}: {', '.join(missing_vars)}"
            )
