import yaml

from kubernetes.client.exceptions import ApiException
from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, utils

LOG = logger.logger


def get_openstack_metadata(ostcm, node_name):
    metadata = {}
    for resource in ["compute_service", "network_agent", "volume_service"]:
        metadata[resource] = getattr(ostcm, resource).list(["--host", node_name])

    return metadata


def is_pv_bound_to_node(pv, node_name):
    pv_obj = pv.read().to_dict()
    node_affinity = pv_obj["spec"].get("node_affinity", {})
    if not node_affinity:
        return False
    if pv_obj["status"].get("phase") != "Bound":
        return False
    for node_selector in node_affinity.get("required", {}).get("node_selector_terms", []):
        for expression in node_selector.get("match_expressions", []):
            if (
                expression.get("key") == "kubernetes.io/hostname"
                and expression.get("operator") == "In"
                and node_name in expression.get("values", [])
            ):
                return True
    return False


def get_pvs(ostcm, node_name, namespaces=None):
    namespaces = namespaces or []
    k8scluster = ostcm.os_manager.api
    pvs = k8scluster.pvolumes.list()
    res = []
    for pv in pvs:
        pv_obj = pv.read().to_dict()
        if not pv_obj["spec"].get("claim_ref"):
            continue
        if not is_pv_bound_to_node(pv, node_name):
            continue

        if namespaces:
            if pv_obj["spec"]["claim_ref"]["namespace"] in namespaces:
                res.append(pv)
        else:
            res.append(pv)
    return res


def _test_node_deletion_request(ostcm, node_name, role):
    k8scluster = ostcm.os_manager.api
    node = k8scluster.nodes.get(node_name)

    assert any(get_openstack_metadata(ostcm, node_name).values()), f"There is no metadata on the {node_name} node"

    if role == "controller":
        namespaces = [ostcm.os_manager.openstack_namespace, ostcm.os_manager.redis_namespace]
        node_pvs = get_pvs(ostcm, node_name, namespaces=namespaces)

        assert any(node_pvs), f"No PVs found bound to node {node_name}"

    LOG.info("Creating node deletion request")
    body = {
        "apiVersion": "lcm.mirantis.com/v1alpha1",
        "kind": "NodeDeletionRequest",
        "metadata": {"name": node_name},
        "spec": {"nodeName": node_name},
    }
    ndr = k8scluster.nodedeletionrequests.create(body=body)
    LOG.info("Waiting for inactive node workload lock")
    waiters.wait(
        lambda: k8scluster.nodeworkloadlocks.get(f"openstack-{node_name}").state == "inactive",
        timeout=300,
        interval=10,
    )

    LOG.info("Blocking up the node (mark it as unschedulable)")
    node.cordon()

    LOG.info("Draining the node (move out pods except DaemonSets)")
    pods = []
    for pod in k8scluster.pods.list_all(
        field_selector=f"spec.nodeName={node_name}"
    ):
        try:
            owner_references = pod.data["metadata"].get("owner_references")
            owner_kind = owner_references[0]["kind"] if owner_references else "Undefined"
            if owner_kind != "DaemonSet":
                LOG.info(f"Moving out the {pod.namespace}/{pod.name}")
                pod.eviction()
                pods.append(pod)
        except ApiException as e:
            if e.status == 404 and e.reason == "Not Found":
                LOG.info(f"Cannot find pod {pod.namespace}/{pod.name}. Skip.")
                continue
            raise e
    if pods:
        LOG.info("Waiting for moved pods out")

        def _status_msg_pods():
            remained_pods = [f"{pod.namespace}/{pod.name}" for pod in pods if pod.exists()]
            return f"\n{yaml.dump(remained_pods)}"

        waiters.wait(
            lambda: all([not pod.exists() for pod in pods]),
            timeout=900,
            interval=30,
            status_msg_function=_status_msg_pods,
        )

    LOG.info("Deleting the node and node deletion request")
    node.delete()
    ndr.delete()

    LOG.info("Waiting for node workload lock to be removed")
    waiters.wait(
        lambda: not [
            nwl.uid for nwl in k8scluster.nodeworkloadlocks.list_all() if nwl.name == f"openstack-{node_name}"
        ],
        timeout=420,
        interval=20,
    )

    LOG.info("Checking out all the metadata is cleaned up")
    metadata = get_openstack_metadata(ostcm, node_name).values()
    assert not any(metadata), f"\n{yaml.dump(metadata)}"
    if role == "controller":
        namespaces = [ostcm.os_manager.openstack_namespace, ostcm.os_manager.redis_namespace]
        node_pvs = get_pvs(ostcm, node_name, namespaces=namespaces)
        assert not any(node_pvs), f"Some PVs are still bound to the node {node_name}"
    LOG.info("All the metadata has been cleaned up")

    ostcm.os_manager.wait_openstackdeployment_health_status()
    LOG.info("Check that OpenStack pods are active.")
    ostcm.os_manager.wait_os_resources(timeout=settings.OPENSTACK_LCM_OPERATIONS_TIMEOUT, interval=20)


def test_node_deletion_request_compute(openstack_client_manager):
    """Test node deletion process of compute node

    - check out original OpenStack metadata
    - create NodeDeletionRequest
    - wait for inactive NodeWorkloadLock
    - cordon/drain Node
    - wait for Pods are evicted
    - delete Node
    - delete NodeDeletionRequest
    - check out OpenStack metadata is cleaned up

    Parameters required:
      - KUBECONFIG
    """
    node_name = sorted(
        openstack_client_manager.os_manager.api.nodes.list(label_selector=utils.NodeLabel.os_compute.value),
        key=lambda n: n.data["metadata"]["creation_timestamp"],
        reverse=True,
    )[0].name

    _test_node_deletion_request(
        openstack_client_manager,
        node_name=node_name,
        role="compute",
    )


def test_node_deletion_request_controller(openstack_client_manager):
    """Test node deletion process of controller node

    - check out original OpenStack metadata
    - create NodeDeletionRequest
    - wait for inactive NodeWorkloadLock
    - cordon/drain Node
    - wait for Pods are evicted
    - delete Node
    - delete NodeDeletionRequest
    - check out OpenStack metadata is cleaned up

    Parameters required:
      - KUBECONFIG
    """
    MARIADB_POD_NAME = "mariadb-server-2"
    node_name = openstack_client_manager.os_manager.api.pods.get(
        MARIADB_POD_NAME, openstack_client_manager.os_manager.openstack_namespace
    ).node_name
    _test_node_deletion_request(openstack_client_manager, node_name=node_name, role="controller")
