import cachetools.func as cachetools_func
from si_tests import logger
from si_tests.utils import waiters, exceptions
from si_tests import settings

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.kaas_manager import Cluster


LOG = logger.logger


class RuntimeManager(object):
    """Runtime manager"""

    def __init__(self, cluster: "Cluster"):
        self._cluster: "Cluster" = cluster

    @property
    def cluster(self) -> "Cluster":
        return self._cluster

    @property
    def runtime(self):
        return self.cluster.get_cluster_runtime()

    @property
    @cachetools_func.ttl_cache(ttl=300)
    def k8sclient(self):
        return self._cluster.k8sclient

    def partial_migrate(self, machines, runtime, timeout=settings.WAIT_RUNTIME_MIGRATION_STARTED_TIMEOUT):
        """Partial change runtime of cluster nodes

        :param timeout: Timeout for migration start
        :param machines: list of Machine objects to migrate
        :param runtime: target runtime
        """
        for machine in machines:
            machine.set_annotations({'kaas.mirantis.com/preferred-container-runtime': runtime})

        LOG.info(f"Wait {timeout} seconds to ensure that cluster conditions are Ready")
        waiters.wait(lambda: not self.cluster.are_conditions_ready(),
                     timeout=timeout,
                     interval=10,
                     timeout_msg=f"Cluster {self.cluster.namespace}/{self.cluster.name} does not reach NOT READY "
                                 f"state after runtime migration started")

        LOG.info(f"Wait {timeout} seconds to ensure that any machine is moved to Deploy state")
        try:
            self.cluster.check.check_any_machine_exp_status(expected_status='Deploy', timeout=timeout)
        except exceptions.TimeoutError as e:
            LOG.error("Tired waiting for any machine in Deploy state,"
                      " check if cluster has active clusterworkloadlocks and raise error")
            clusterworkloadlocks = self.cluster.get_clusterworkloadlocks()
            active_cwl = []
            if clusterworkloadlocks:
                active_cwl = [cwl.name for cwl in clusterworkloadlocks if
                              self.cluster.check.get_clusterworkloadlock_state(name=cwl.name) == "active"]
            if active_cwl:
                LOG.error(f"List of active clusterworkloadlock: {active_cwl}")
            raise e

        LOG.info('Migration started')

    def upgrade_distrib_and_migrate(self, machines, runtime, distrib, timeout=300):
        """Set distribution (if possible) and runtime and wait for migration start

        :param distrib: Target distribution
        :param machines: list of Machine objects to migrate
        :param runtime: target runtime
        :param timeout: Timeout for migration start
        """
        for machine in machines:
            machine.set_distribution_and_migrate(distrib, runtime)

        LOG.info(f"Wait {timeout} seconds to ensure that migration and distrib upgrade started")
        waiters.wait(lambda: not self.cluster.are_conditions_ready(),
                     timeout=timeout,
                     interval=10,
                     timeout_msg=f"Cluster {self.cluster.namespace}/{self.cluster.name} does not reach NOT READY "
                                 f"state after runtime migration started")
        LOG.info('Migration and distrib upgrade started')
