#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import json
import pytest
from si_tests.utils import packaging_version as version

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager
from si_tests.utils.utils import Provider
from si_tests.utils import waiters
from si_tests.managers.kaas_manager import Manager

LOG = logger.logger
mcc_upgrade_object_name = settings.MCC_UPGRADE_OBJECT_NAME


def dump_nodes_info(nwll):
    nodes_data_map = {}
    for node in nwll:
        node_name = node["spec"]["nodeName"]
        nwl_name = node["metadata"]["name"]
        nwl_state = node["status"]["state"]
        nwl_resource_version = node["metadata"]["resource_version"]
        nodes_data_map[node_name] = {"workloadlock_name": nwl_name, "workloadlock_state": nwl_state,
                                     "resource_version": nwl_resource_version}
    return nodes_data_map


def save_management_version_artifact(kaas_manager):
    kaas_release = kaas_manager.get_active_kaasrelease()
    kaas_version = kaas_release.data.get('spec', {}).get('version')
    if not kaas_version:
        raise Exception("Cannot get release version from the active KaaSRelease object")
    LOG.info(f"Save artifact with MCC version: {kaas_version}")
    with open(f"{settings.ARTIFACTS_DIR}/management_version", "w") as f:
        f.write(kaas_version)


class TestUpdateMgmCluster():

    @pytest.mark.usefixtures("store_updated_mgmt_cluster_description")
    @pytest.mark.usefixtures("introspect_management_upgrade_objects")
    @pytest.mark.usefixtures("introspect_PRODX_9696_mgmt")
    @pytest.mark.usefixtures('introspect_mgmt_lcm_operation_stuck')
    @pytest.mark.usefixtures('introspect_machines_stages_mgmt')
    @pytest.mark.usefixtures('log_start_end_timestamps')
    @pytest.mark.usefixtures('mcc_loadtest_prometheus')
    @pytest.mark.usefixtures('mcc_loadtest_grafana')
    @pytest.mark.usefixtures('mcc_loadtest_alerta')
    @pytest.mark.usefixtures('mcc_loadtest_keystone')
    @pytest.mark.usefixtures('mcc_loadtest_kibana')
    @pytest.mark.usefixtures('mcc_loadtest_alertmanager')
    @pytest.mark.usefixtures('mcc_loadtest_keycloak')
    @pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
    def test_update_mgmt_cluster(self, kaas_manager: Manager, mgmt_k8s_ip: str):
        try:
            self.update_mgmt_cluster(kaas_manager)
        except Exception as e:
            LOG.error("Management upgrade failed: {}".format(e))
            raise e
        finally:
            save_management_version_artifact(kaas_manager)

            LOG.info('Updating bootstrap version')
            bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
            bootstrap.update_bootstrap_version(mgmt_k8s_ip)

    def update_mgmt_cluster(self, kaas_manager: Manager):
        """Check mgmt is updated

        Scenario:
            1. Check release
        """

        def sanitize_lcmmachines(lcmmachines):
            sanitized_lcmmachines = []
            for lcm in lcmmachines:
                lcm_status = lcm.get('status') or {}
                sanitized_lcmmachines.append({
                    'status': {
                        'addresses': lcm_status.get('addresses'),
                        'state': lcm_status.get('state')
                    }
                })
            return sanitized_lcmmachines

        cluster = kaas_manager.get_mgmt_cluster()
        # Get the current mgmt kaasrelease version until update is not started
        kr_before = kaas_manager.get_mgmt_cluster().get_kaasrelease_version()
        # Get the current mgmt clusterrelease version until update is not started
        cr_before = kaas_manager.get_mgmt_cluster().clusterrelease_version
        LOG.info(f"Management cluster ClusterRelease version is "
                 f"'{cr_before}' , KaaSRelease is '{kr_before}'")

        ucp_tag_in_cr_before = set(
            [x['params']['ucp_tag'] for x in kaas_manager.get_clusterrelease(
                cr_before).data['spec']['machineTypes']['control']
             if 'ucp_tag' in x['params'].keys()])
        LOG.info("Current management cluster ucp version is "
                 "{}".format(ucp_tag_in_cr_before))
        # START WA:PRODX-34279
        # vbmc internal WA, related to migration which we dont have, for vmbc
        # BM-CI only!
        if settings.KAAS_BM_PRODX_34279 and version.parse(kr_before) < version.parse("kaas-2-25-0-rc"):
            LOG.info('KAAS_BM_PRODX_34279: processing')
            _cm_name = 'vbmc-config'

            def _msg34279():
                return f"KAAS_BM_PRODX_34279: waiting till {_cm_name} disappear"

            waiters.wait(lambda: not cluster.k8sclient.configmaps.present(namespace='kaas', name=_cm_name),
                         timeout=300, interval=10, status_msg_function=_msg34279)
            remote = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager().remote_seed()
            # ultra-hardcoded WA
            remote.check_call(
                "KUBECONFIG=/home/ubuntu/bootstrap/dev/kubeconfig /home/ubuntu/bootstrap/dev/bin/kubectl"
                " apply -f /home/ubuntu/bootstrap/dev/templates/bm/bm_prodx_34279.yaml.template || true",
                verbose=True)
            LOG.info('KAAS_BM_PRODX_34279: finished')
        # END WA: PRODX-34279

        # Collect reboots count for mgmt and region clusters before update
        uptimes_mgmt = {}
        uptimes_mgmt_after = {}
        uptimes_regions = {}
        uptimes_regions_after = {}
        uptimes_mgmt = cluster.get_machines_uptime()

        lcmms = kaas_manager.api.kaas_lcmmachines.list_raw().to_dict()
        all_managed_lcmm = [lm for lm in lcmms['items']
                            if lm['metadata']['namespace'] != "default"]
        r_clusters = kaas_manager.get_regional_clusters()
        for r_cluster in r_clusters:
            lcmms = r_cluster.k8sclient.kaas_lcmmachines.list_raw().to_dict()
            all_managed_lcmm.extend([lm for lm in lcmms['items']
                                     if lm['metadata']['namespace'] != r_cluster.namespace])
            uptimes_regions[r_cluster.name] = r_cluster.get_machines_uptime()

        # Collect Machines distributions for Management and Region clusters before upgrade
        machines_distributions_before_update = {}
        if cluster.provider is Provider.baremetal:
            machines_distributions_before_update.update(cluster.get_machines_distributions_from_nodes())
        for r_cluster in r_clusters:
            if r_cluster.provider is Provider.baremetal:
                machines_distributions_before_update.update(r_cluster.get_machines_distributions_from_nodes())

        mcc_upgrade_obj = cluster._manager.get_kaas_mcc_upgrade(name=mcc_upgrade_object_name)
        is_auto_delay_enabled = mcc_upgrade_obj.data.get('spec', {}).get('autoDelay', False)
        if is_auto_delay_enabled:
            msg = "MCC upgrade auto-delay enabled as expected. Going to disable it to start update"
            LOG.info(msg)
            LOG.info("Disabling mcc upgrade auto-delay to continue upgrade")
            body = {
                "spec": {
                    "autoDelay": False,
                }
            }
            kaas_manager.api.kaas_mccupgrades.update(name=mcc_upgrade_object_name, body=body)

        ###############################
        #  Start MCC release upgrade  #
        ###############################

        # Wait until KaaS downloads the second kaasrelease
        LOG.info("Waiting for downloading of new  kaasrelease")
        kaas_manager.wait_kaasrelease(
            releases_count=2, timeout=settings.KAAS_UPDATE_MGMT_WAIT_TIMEOUT)

        avail_kaas_releases = kaas_manager.get_kaasrelease_names()
        LOG.info('Available KaaS releases are {}'.format(avail_kaas_releases))

        assert len(avail_kaas_releases) > 1, (
            "There are not new kaas releases, current list: {}".format(
                avail_kaas_releases))

        kaas_r_sorted = str(max(
            [version.parse(x) for x in avail_kaas_releases]))
        LOG.info('Sorted release {}'.format(kaas_r_sorted))
        expected_cr = kaas_manager.get_kaasrelease(kaas_r_sorted).data['spec']['clusterRelease']
        is_maintenance_skip = cluster.is_skip_maintenance_set(cr_before=cr_before, target_clusterrelease=expected_cr,
                                                              kaasrelease_version=kaas_r_sorted)

        reboot_required = cluster.update_requires_reboot_mgmt(
            cr_before=cr_before, kaasrelease_version=kaas_r_sorted)
        bm_regions = []
        for r_cluster in r_clusters:
            if r_cluster.provider is Provider.baremetal:
                bm_regions.append(r_cluster)

        # let's pass target kaasrelease_version here to fetch
        # target clusterrelease version for mgmt/region clusters
        if reboot_required:
            LOG.info("Restart of nodes will be required during update")

        all_managed_lcmm = sanitize_lcmmachines(all_managed_lcmm)

        kr_data = kaas_manager.get_kaasrelease(kaas_r_sorted).data
        LOG.debug("Release_details {}".format(kr_data))
        kaas_manager.wait_kaasrelease_label(kaas_r_sorted, timeout=settings.KAAS_UPDATE_MGMT_ACTIVE_WAIT_TIMEOUT)

        formated_kr = kr_data['spec']['version'].replace('.', '-')
        assert formated_kr == kaas_r_sorted.replace('kaas-', '')
        mgmt_cluster = kaas_manager.get_mgmt_cluster().spec['providerSpec']
        assert kaas_r_sorted == mgmt_cluster['value']['kaas']['release']
        LOG.info('MGMT spec {}'.format(mgmt_cluster['value']['release']))

        cluster_releases = kaas_manager.get_clusterrelease_names()
        spec_cr = kr_data['spec']['clusterRelease']
        LOG.info('Cluster version in new KaaSRelease: {}'.format(spec_cr))
        assert spec_cr in cluster_releases
        assert spec_cr == mgmt_cluster['value']['release'], (
            "Cluster version in KaasRelease and Mgmt "
            "Cluster differs, that's wrong")

        cluster.check.check_cluster_release(spec_cr)
        # TODO: we need to check regional child cluster updates too
        if cr_before != expected_cr:
            cluster.check.check_update_finished(check_managed_clusters=True)

        # TODO: since 2.31 it will be done automatically, required only for 2.30
        if cluster.workaround.skip_kaascephcluster_usage():
            child_clusters = kaas_manager.get_child_clusters()
            for child_cluster in child_clusters:
                child_cluster.remove_kaascephcluster()
                child_cluster.check.wait_kaascephcluster_removed()

        ######################################
        #  MCC release upgrade is completed  #
        ######################################

        if cluster.provider is Provider.baremetal and cr_before != expected_cr:
            kaascephclusters = cluster.get_kaascephclusters()
            num_clusters = len(kaascephclusters)
            assert num_clusters == 0, (f"Unexpectedly {num_clusters} kaascephclusters found "
                                       f"with spec.k8sCluster '{self.namespace}/{self.name}' : "
                                       f"{kaascephclusters}")

        ucp_tag_after = set(
            [x['params']['ucp_tag']
             for x in kaas_manager.get_clusterrelease(
                spec_cr).data['spec']['machineTypes']['control']
             if 'ucp_tag' in x['params'].keys()])
        if ucp_tag_in_cr_before != ucp_tag_after:
            LOG.info("Management cluster new ucp version will be "
                     "{}".format(ucp_tag_after))

        cluster.check.check_cluster_nodes()
        cluster.check.check_cluster_readiness(timeout=3600)
        cluster.check.check_helmbundles()
        cluster.check.check_k8s_nodes()
        cluster.check.check_persistent_volumes_mounts()
        cluster.check.check_k8s_pods()
        cluster.check.check_no_leftovers_after_upgrade()
        cluster.check.check_upgrade_stage_success()
        if cluster.provider is Provider.baremetal:
            LOG.info("Check repository url for management cluster")
            cluster.check.check_repository_url()
            cluster.check.check_actual_expected_kernel_versions()
        if bm_regions:
            for r_cluster in bm_regions:
                LOG.info("Check repository url for regional cluster")
                r_cluster.check.check_repository_url()
                r_cluster.check.check_actual_expected_kernel_versions()

        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_actual_expected_pods(timeout=600)
        # TODO enable after verification on all providers
        # Related blocker PRODX-19064
        # cluster.check.check_actual_expected_rolebindings()

        # Collect Machines distributions for Management and Region clusters after upgrade
        machines_distributions_after_update = {}
        if cluster.provider is Provider.baremetal:
            machines_distributions_after_update.update(cluster.get_machines_distributions_from_nodes())
        for r_cluster in r_clusters:
            if r_cluster.provider is Provider.baremetal:
                machines_distributions_after_update.update(r_cluster.get_machines_distributions_from_nodes())
        # Dictionary with machine names in keys and boolean in values, to reflect distribution changes
        # If distribution has been changed during machines upgrade, then reboot is expected for this machine
        # even if "reboot_required" flag is not set.
        reboot_expected = {machine_name: (machines_distributions_after_update[machine_name] !=
                                          machines_distributions_before_update[machine_name]) or reboot_required
                           for machine_name in machines_distributions_after_update.keys()}
        # Set reboot_required flag for non-BM machines
        for machine in cluster.get_machines():
            if machine.name not in reboot_expected:
                reboot_expected[machine.name] = reboot_required
        for r_cluster in r_clusters:
            for machine in r_cluster.get_machines():
                if machine.name not in reboot_expected:
                    reboot_expected[machine.name] = reboot_required

        # Check the changes in reboots count on mgmt cluster after upgrade
        uptimes_mgmt_after = cluster.get_machines_uptime(dump_reboot_list=True)
        cluster.check.check_machines_reboot(uptimes_mgmt, uptimes_mgmt_after, reboot_expected)

        # Check that no 'nodeworkloadlock' objects are present on Management cluster
        if cluster.provider is Provider.baremetal and cr_before != expected_cr:
            # Refresh nodeworkloadlock dict
            nwll = [nwl.data for nwl in cluster.get_nodeworkloadlocks()]
            # NodeWorkloadLocks should be empty on management cluster as there is no Ceph cluster
            assert len(nwll) == 0, f"NodeWorkLoadLocks list is not empty in mgm cluster: {cluster.name}"

        lcmms = kaas_manager.api.kaas_lcmmachines.list_raw().to_dict()
        all_managed_lcmm_new = [lm for lm in lcmms['items']
                                if lm['metadata']['namespace'] != "default"]

        cluster.check.check_upgraded_machines_cordon_drain_stages(
            skip_maintenance=is_maintenance_skip, reboot_expected=reboot_expected)

        for r_cluster in r_clusters:
            # get regional clusters and check CR there
            r_cr = r_cluster.spec['providerSpec']['value']['release']
            LOG.info("Current regional cr is {}".format(r_cr))
            assert spec_cr in r_cr, ("Cluster version in regional cluster "
                                     "differs from latest kaas release, "
                                     "that's wrong")
            r_cluster.check.check_cluster_release(spec_cr)

            r_cluster.check.check_cluster_nodes()
            r_cluster.check.check_cluster_readiness()
            r_cluster.check.check_helmbundles()
            r_cluster.check.check_k8s_nodes()
            r_cluster.check.check_persistent_volumes_mounts()
            r_cluster.check.check_k8s_pods()
            r_cluster.check.check_no_leftovers_after_upgrade()
            r_cluster.check.check_upgrade_stage_success()

            # Check/wait for correct docker service replicas in cluster
            r_cluster.check.check_actual_expected_docker_services()
            r_cluster.check.check_actual_expected_pods(timeout=600)
            # TODO enable after verification on all providers
            # Related blocker PRODX-19064
            # r_cluster.check.check_actual_expected_rolebindings()

            if r_cluster.provider is Provider.baremetal:
                LOG.info("Check repository url for regional cluster")
                r_cluster.check.check_repository_url()

            lcmms = r_cluster.k8sclient.kaas_lcmmachines.list_raw().to_dict()
            all_managed_lcmm_new.extend(
                [lm for lm in lcmms['items']
                 if lm['metadata']['namespace'] != r_cluster.namespace]
            )
            uptimes_regions_after[r_cluster.name] = r_cluster.get_machines_uptime(dump_reboot_list=True)

            cluster.check.check_upgraded_machines_cordon_drain_stages(
                skip_maintenance=is_maintenance_skip, reboot_expected=reboot_expected)

        LOG.info("Check child clusters")
        child_clusters = kaas_manager.get_child_clusters()
        for child in child_clusters:
            LOG.info(f" - check {child.name} cluster")
            child.check.check_cluster_readiness()
            if child.provider is Provider.baremetal:
                LOG.info("Check repository url for child cluster")
                child.check.check_repository_url()

        all_managed_lcmm_new = sanitize_lcmmachines(all_managed_lcmm_new)

        LOG.debug(f"All managed lcmmachines before update: "
                  f"{json.dumps(all_managed_lcmm, indent=4, default=str)}")
        LOG.debug(f"All managed lcmmachines after update: "
                  f"{json.dumps(all_managed_lcmm_new, indent=4, default=str)}")

        # Check the changes in reboots count on region clusters after upgrade
        for r_cluster in r_clusters:
            r_cluster.check.check_machines_reboot(uptimes_regions[r_cluster.name],
                                                  uptimes_regions_after[r_cluster.name],
                                                  reboot_expected)

        assert all_managed_lcmm_new == all_managed_lcmm, \
            "Error: lcmmachines of managed cluster have changed"

        LOG.info("Check clusterRelease objects are existed for every release "
                 "announced as supported in kaasRelease object")
        cluster.check.compare_supported_and_existed_clusterreleases()
        if cluster.provider is Provider.baremetal:
            LOG.info("Check actual/expected reboot-required status for all machines")
            cluster.check.check_expected_actual_reboot_required_status()

        # Check that Management cluster has been upgraded to the latest available distribution
        cr_version = cluster.clusterrelease_version
        latest_distro = kaas_manager.get_latest_available_clusterrelease_distro(cluster, cr_version)

        if cluster.provider is Provider.baremetal:
            # Check that Management and Region clusters distributions were upgraded to the latest available version
            cluster.check.check_inplace_distribution_upgrade_completed(latest_distro)
            for r_cluster in r_clusters:
                r_cluster.check.check_inplace_distribution_upgrade_completed(latest_distro)
            cluster.check.check_actual_expected_distribution()

        # Check correct runtime
        if settings.DESIRED_RUNTIME:
            cluster.check.compare_cluster_runtime_with_desired()
        if cluster.provider is Provider.baremetal:
            cluster.check.check_diagnostic_cluster_status()
