# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
import pytest
import yaml

from si_tests import logger, settings
from si_tests.deployments.utils import file_utils
from si_tests.managers.clustercheck_mos_manager import ClusterCheckMosManager
from si_tests.utils import waiters
from si_tests.utils.ha_helpers import (collect_pods_restart_count,
                                       compare_restarts_number)

LOG = logger.logger


@pytest.mark.usefixtures('tf_workload')
class TestTFOperatorUpgrade(object):

    @staticmethod
    def _get_image_info(url):
        img_data = {}
        parts = url.split("/")
        img_data["registry"] = parts[0]
        img_data["path"] = "/".join(parts[1:-1])
        img_data["name"] = parts[-1].split(":")[0]
        img_data["tag"] = parts[-1].split(":")[-1]
        return img_data

    @staticmethod
    def _check_ds(daemonsets):
        status = True
        for daemonset in daemonsets:
            desired_num = daemonset.data['status'].get('desired_number_scheduled')
            ready_num = daemonset.data['status'].get('number_ready')
            if ready_num != desired_num:
                status = False
        return status

    @pytest.mark.dependency(name="test_tf_operator_upgrade")
    def test_tf_operator_upgrade(self, show_step, tf_manager, openstack_client_manager):
        """ Upgrade TFOperator

            Scenario:
                1. Make TF DB backup
                2. Collect data for upgrade
                3. Upgrade operator through helmbundle
                4. Wait TFOperator health status
                5. Check vRouters
                6. Update vRouters (optional)
        """
        show_step(1)
        tf_manager.db_backup(state=True)
        tf_manager.trigger_tfdbbackup_cronjob()

        show_step(2)
        # Collect current TFOperator data
        tf_helmbundle = tf_manager.get_tf_helmbundle()
        helmbundle_data = tf_helmbundle.data
        helm_ver = helmbundle_data["spec"]["releases"][0]["version"]
        LOG.info("Initial TF operator version:")
        LOG.info(f"   Helmbundle: {helm_ver}")
        LOG.info(f"   Operator: {tf_manager.tfoperator_version()}")
        LOG.info(f"   TF: {tf_manager.get_tf_version()}")

        # Get TF operator version for upgrade (from artifact-metadata)
        path = "metadata"
        metadata_namespace = settings.ARTIFACT_METADATA_NAMESPACE
        if metadata_namespace:
            path += f"/{metadata_namespace}"
        tf_chart = file_utils.join(settings.ARTIFACT_METADATA_DIR,
                                   f"{path}/charts/tungsten/tungstenfabric-operator/data.yml")
        with open(tf_chart, 'r') as f:
            tf_chart_data = yaml.load(f, Loader=yaml.SafeLoader)
        LOG.info(f"TF operator chart data (artifact-metadata):\n{tf_chart_data}")
        tf_image = file_utils.join(settings.ARTIFACT_METADATA_DIR,
                                   f"{path}/images/tungsten/tungstenfabric-operator/data.yml")
        with open(tf_image, 'r') as f:
            tf_image_data = yaml.load(f, Loader=yaml.SafeLoader)
        LOG.info(f"TF operator image data (artifact-metadata):\n{tf_image_data}")

        url = tf_image_data["url"]
        img_data = self._get_image_info(url)

        repo = tf_chart_data["tungstenfabric-operator"]["repo"]
        new_version = tf_chart_data["tungstenfabric-operator"]["version"]

        LOG.info(f"Upgrade to TF operator {new_version} from artifact-metadata/{path}")
        assert helm_ver != new_version, "Initial and destination versions are the same"

        restart_numbers = collect_pods_restart_count(tf_manager.get_vrouter_pods())

        show_step(3)
        spec = helmbundle_data["spec"]
        spec["repositories"][0]["url"] = repo
        spec["releases"][0]["version"] = new_version
        spec["releases"][0]["values"] = \
            {
                "global": {"dockerBaseUrl": img_data["registry"]},
                "image": {
                    "repository": img_data["path"],
                    "name": img_data["name"],
                    "tag": img_data["tag"],
                },
            }

        tf_deployment = tf_manager.get_tfoperator_deployment()
        dpl_gen = tf_deployment.read().status.observed_generation

        tf_helmbundle.patch({'spec': spec})

        # Wait for TF to start redeployment
        waiters.wait(lambda gen: gen != tf_deployment.read().status.observed_generation,
                     predicate_args=[dpl_gen], timeout=180)
        waiters.wait(lambda: "Ready" != tf_manager.tfoperator(read=True).status.get("health"), timeout=120)

        show_step(4)
        # Wait pre-caching daemon sets
        daemonsets = tf_manager.get_precaching_daemonsets()
        waiters.wait(self._check_ds, predicate_args=[daemonsets], timeout=600, interval=10)
        LOG.info("Precaching DaemonSets are ready")

        tf_manager.wait_for_hold_healthy(timeout=1500)

        show_step(5)
        compare_restarts_number(tf_manager.get_vrouter_pods(), restart_numbers)

        show_step(6)
        tf_manager.update_tfvrouter_pods()
        tf_manager.wait_tf_controllers_healthy(timeout=300)
        tf_manager.wait_tfoperator_healthy(timeout=210)

        LOG.info("Current TF operator version:")
        LOG.info(f"   Helmbundle: {tf_manager.get_tf_helmbundle(read=True).spec['releases'][0]['version']}")
        LOG.info(f"   Operator: {tf_manager.tfoperator_version()}")
        LOG.info(f"   TF: {tf_manager.get_tf_version()}")

    @pytest.mark.dependency(depends=["test_tf_operator_upgrade"])
    def test_tf_db_restore(self, show_step, request, tf_manager, openstack_client_manager):
        """ Restore TF DB after successful upgrade

            Scenario:
                1. Remove tfdbrestore before test if needed
                2. Enable DB restore specifying DB backup file
                3. Wait for TF Operator health status and check contrail api readiness
                4. Verify that the network exists and the load balancer is functioning correctly.
        """
        show_step(1)
        # Check that we don't have tfdbrestore objects before test,
        # https://docs.mirantis.com/mosk/latest/single/#restore-tf-data
        if tf_manager.is_tfdbrestore_present():
            LOG.info('Removing tfdbrestore object before test')
            tf_manager.tfdbrestore().delete()

        show_step(2)
        tf_manager.db_restore()

        show_step(3)
        tf_manager.wait_for_hold_healthy()
        waiters.wait(ClusterCheckMosManager.check_contrail_api_readiness,
                     predicate_args=[openstack_client_manager],
                     timeout=300, interval=15,
                     timeout_msg="Timeout: Contrail api is not Ready")

        show_step(4)
        workload_data = request.getfixturevalue("tf_workload")
        assert ClusterCheckMosManager.is_network_present(openstack_client_manager, workload_data['network_name']), \
            f"Can't find test network {workload_data['network_name']} after DB Restore"
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, workload_data['lb_url'])
