import yaml

from si_tests.deployments.utils.file_utils import save_to_yaml
from si_tests import settings
from si_tests import logger
from si_tests.utils import utils
from si_tests.utils import waiters
from si_tests.utils import exceptions
from si_tests.managers import bootstrap_manager
from si_tests.utils.utils import Provider
LOG = logger.logger


def test_delete_regional_or_mgmt_cluster(kaas_manager):
    """ Test delete regional or mgmt cluster """
    bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    cluster_name = cluster.name
    cluster_provider = cluster.provider
    cluster_is_regional = cluster.is_regional
    cluster_is_management = cluster.is_management
    region_name = cluster.region_name
    cluster_resources = {}
    cluster_cred_before = []
    remote = bootstrap.remote_seed()
    keyfile_path = bootstrap.get_bootstrap_ssh_keyfile_path(remote)
    management_kubeconfig = "kubeconfig"

    target_kubeconfig_path = "kubeconfig"\
        if cluster_is_management else f"kubeconfig-{cluster_name}"

    # Optional: BM provider specific case
    if remote.isdir('./bootstrap/dev'):
        bootstrap_dir = 'bootstrap/dev'
        # bm mgmt - kubeconfig in '~/bootstrap/dev'
        # bm region, mgmt-other - kubeconfig in '~/'
        # bm region, mgmt-bm - kubeconfig in '~/bootstrap/dev'
        # bm child - kubeconfig in '~/bootstrap/dev' or we'll copy it there
        # we rely on either of these 2 folders and kubeconfig format:
        # f"kubeconfig-{cluster_name}" for non-mgmt cluster
        # kubeconfig - for mgmt
        target_kubeconfig_path = target_kubeconfig_path \
            if remote.isfile(target_kubeconfig_path) else \
            f'{bootstrap_dir}/{target_kubeconfig_path}'
        management_kubeconfig = management_kubeconfig \
            if remote.isfile(management_kubeconfig) else \
            f'{bootstrap_dir}/{management_kubeconfig}'
    else:
        bootstrap_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR

    # Optional: Clean seed regional bootstrap specific case
    artifacts_home = settings.SEED_NODE_MCC_ARTIFACTS_BASE_DIR
    if cluster_is_regional and \
            remote.isdir(f"{artifacts_home}/default/{cluster_name}"):
        regional_artifacts = bootstrap.get_regional_artifacts(cluster_name)
        keyfile_path = regional_artifacts["keyfile_path"]
        target_kubeconfig_path = regional_artifacts["kubeconfig_path"]

    assert remote.isfile(f"./{bootstrap_dir}/bootstrap.sh"), (
        f"File bootstrap.sh not found in {bootstrap_dir}")

    volumes_cleanup_check = False
    if settings.CHECK_RESOURCES_DELETION:
        cluster_resources = cluster.provider_resources.resources
        cluster_cred_before = ns.get_provider_credentials(
            cluster_provider, region_name)
        if cluster_provider is Provider.openstack:  # TODO PRODX-9728 Check that only OS provider
            cluster.set_volumes_cleanup_enabled(enabled=True)
            volumes_cleanup_check = True

    run_envs = {
        'KAAS_BOOTSTRAP_LOG_LVL': settings.KAAS_BOOTSTRAP_LOG_LVL,
        'KAAS_BOOTSTRAP_INFINITE_TIMEOUT': settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT,
    }
    if cluster_is_regional:
        cleanup_cmd = "destroy_regional"
        run_envs['REGIONAL_CLUSTER_NAME'] = cluster_name
        run_envs['REGIONAL_KUBECONFIG'] = target_kubeconfig_path
        run_envs['KUBECONFIG'] = management_kubeconfig
    elif cluster_is_management:
        cleanup_cmd = "cleanup"
        run_envs['CLUSTER_NAME'] = cluster_name
        run_envs['KUBECONFIG'] = management_kubeconfig
    else:
        raise NotImplementedError(
                "The cluster you are trying to delete is not "
                "management or regional cluster")
    envs_string = utils.make_export_env_strting(run_envs)
    cmd = f"{envs_string}; bash -x " \
          f"./{bootstrap_dir}/bootstrap.sh {cleanup_cmd}"

    ret = remote.execute(cmd, verbose=True)
    if ret.exit_code == 0:
        LOG.info("Cleanup has been finished successfully")
    else:
        try:
            LOG.warning(f"Cleanup failed, trying to collect logs from {ns.name}/{cluster_name}")
            bootstrap.step_collect_logs(
                remote,
                cluster_name=cluster_name,
                cluster_namespace=ns.name,
                kubeconfig_path=target_kubeconfig_path,
                keyfile_path=keyfile_path,
                bootstrap_dir=bootstrap_dir
            )
        except Exception as e:
            LOG.warning(f"Unable to collect logs from Target cluster: "
                        f"{e}")
        # If cleanup fails KinD cluster may contain useful logs
        try:
            bootstrap.step_collect_logs(
                remote,
                cluster_name=cluster_name,
                cluster_namespace=ns.name,
                management_kubeconfig_path=settings.KUBECONFIG_KIND_PATH,
                kubeconfig_path="",
                keyfile_path=keyfile_path,
                bootstrap_dir=bootstrap_dir,
                cluster_type="bootstrap")
        except Exception as e:
            LOG.warning("Unable to collect KinD logs: {}".format(e))
        if not cluster_is_management and settings.BM_COLLECT_LOGS:
            try:
                LOG.info("Collecting logs from mgmt cluster as well")
                mgmt = kaas_manager.get_mgmt_cluster()
                mgmt_key_path = bootstrap.get_bootstrap_ssh_keyfile_path(
                        remote)
                if mgmt.provider == utils.Provider.baremetal:
                    mgmt_bootstrap_dir = 'bootstrap/dev'
                else:
                    mgmt_bootstrap_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR

                bootstrap.step_collect_logs(
                    remote,
                    cluster_name=mgmt.name,
                    cluster_namespace=mgmt.namespace,
                    kubeconfig_path=management_kubeconfig,
                    keyfile_path=mgmt_key_path,
                    bootstrap_dir=mgmt_bootstrap_dir
                )
            except Exception as e:
                LOG.warning(f"Unable to collect logs from "
                            f"management cluster: {e}")
        LOG.error("Cleanup hasn't been finished successfully")
        raise Exception(f"Cleanup procedure failed. {ret.stdout}")

    if settings.CHECK_RESOURCES_DELETION:
        LOG.info(f"Will check resources deletion "
                 f"for {cluster_provider} provider and "
                 f" {cluster_name} cluster")
        try:
            waiters.wait(
                lambda:
                not cluster.provider_resources.check_resources_deletion(
                    cluster_resources, volumes_cleanup_check), timeout=1200, interval=120)
            save_to_yaml(cluster.provider_resources.not_deleted_resources,
                         settings.ARTIFACTS_DIR + f"/not_deleted_resources_{cluster_name}.yaml")
        except exceptions.TimeoutError:
            save_to_yaml(
                    cluster.provider_resources.not_deleted_resources,
                    settings.ARTIFACTS_DIR +
                    f"/not_deleted_resources_{cluster_name}.yaml")
            raise Exception(
                    "Some resources were not deleted: {}".format(
                        yaml.dump(
                            cluster.provider_resources.not_deleted_resources))) # noqa

        LOG.info(f"Will check credential resource deletion "
                 f"for {cluster_provider} provider and "
                 f"{cluster_name} cluster")
        if cluster_is_regional:
            cluster_credentials = ns.get_provider_credentials(
                cluster_provider, region_name) or None
            if cluster_credentials:
                LOG.info("Regional cluster credentials were not deleted. "
                         "Starting cleanup")
                for cl_cred in cluster_cred_before:
                    if cl_cred in cluster_credentials:
                        cl_cred.delete()
                        LOG.info(f"Credential: {cl_cred.name} "
                                 f"in {cluster_name} "
                                 f"(provider: {cluster_provider}) cluster "
                                 f"has been deleted")
            else:
                LOG.info("Regional cluster credentials were deleted "
                         "during cluster deletion")
        elif cluster_is_management:
            LOG.info("Management cluster credentials were deleted "
                     "during cluster deletion")
