# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import pytest
import os
from si_tests import logger
from si_tests import settings
from si_tests.utils import templates, waiters

LOG = logger.logger


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_delete_hoc_after_lcm_bm(kaas_manager, _, show_step):
    """ This is a first part of LCM operations verification scenario
    when HOC objects are present (see below)
    This test re-verifies a hoc object after LCM tests (upgrades, reboots etc.)
    and deletes it.

    Scenario:
        1. Create and verify a dummy HostOSConfiguration object
        via si_tests/tests/lcm/day2operations/test_create_hoc_before_lcm_bm.py.
        2. Run any LCM tests for the clusters (upgrades, reboot, replacement etc.)
        3. Verify HOC object again and delete it via
        si_tests/tests/lcm/day2operations/test_delete_hoc_after_lcm_bm.py
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)
    """Get HOC name and machine label from SI config"""
    assert settings.SI_CONFIG_PATH and os.path.isfile(settings.SI_CONFIG_PATH), (
        f"Environment variable 'SI_CONFIG_PATH' contains wrong path to YAML file: {settings.SI_CONFIG_PATH}")
    with templates.YamlEditor(settings.SI_CONFIG_PATH) as si_config:
        assert 'test_create_hoc_before_lcm_bm' in si_config.content, (
            f"{settings.SI_CONFIG_PATH} does not contain key 'test_create_hoc_before_lcm_bm'")
        assert f'{namespace_name}_{cluster_name}' in si_config.content['test_create_hoc_before_lcm_bm'], (
            f"{settings.SI_CONFIG_PATH} does not contain key "
            f"'test_create_hoc_before_lcm_bm.{namespace_name}_{cluster_name}'")
        expected_si_cluster_keys = ['hoc_name', 'hoc_machine_labels', 'lcmmachines_timestamps_before',
                                    'lcmmachines_timestamps_after']
        for k in expected_si_cluster_keys:
            assert k in \
                si_config.content['test_create_hoc_before_lcm_bm'][f'{namespace_name}_{cluster_name}'], (
                    f"{settings.SI_CONFIG_PATH} does not contain key "
                    f"'test_create_hoc_before_lcm_bm.{namespace_name}_{cluster_name}.{k}'")
        hoc_name = (si_config.content['test_create_hoc_before_lcm_bm']
                    [f'{namespace_name}_{cluster_name}']
                    ['hoc_name'])
        day2_label = (si_config.content['test_create_hoc_before_lcm_bm']
                      [f'{namespace_name}_{cluster_name}']
                      ['hoc_machine_labels'])
        lcmmachines_timestamps_before = (si_config.content['test_create_hoc_before_lcm_bm']
                                         [f'{namespace_name}_{cluster_name}']
                                         ['lcmmachines_timestamps_before'])
        lcmmachines_timestamps_after = (si_config.content['test_create_hoc_before_lcm_bm']
                                        [f'{namespace_name}_{cluster_name}']
                                        ['lcmmachines_timestamps_after'])
    machines = cluster.get_machines()
    assert machines, f"No machines in cluster: {cluster_name}"
    hostoscfg = ns.get_hostosconfiguration(name=hoc_name)

    show_step(2)
    """Re-verify HOC object"""
    # Wait for the selected Machines in the hostosconfiguration status
    LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
    cluster.check.check_hostosconfig_machine_selector(hostoscfg)
    LOG.info("Check that new items added into machineTypes in LCMCluster")
    cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)
    LOG.info("Check that new items added into stateItems in LCMMachine")
    cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg,
                                                  lcmmachines_timestamps_before=lcmmachines_timestamps_before,
                                                  lcmmachines_timestamps_after=lcmmachines_timestamps_after)
    cluster.check.get_hostosconfig_machines_status(hostoscfg)
    LOG.info(f"Deleting HostOSConfiguration '{hostoscfg.name}'")
    hostoscfg.delete(async_req=True)
    timeout_msg = f"HostOSConfiguration {hoc_name} was not deleted"
    waiters.wait(lambda: not bool(ns.hostosconfiguration_is_present(name=hoc_name)),
                 timeout=1200,
                 interval=10,
                 timeout_msg=timeout_msg)
    for machine in machines:
        machine.remove_machine_labels(list(day2_label.keys()))
