# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import pytest
import os
import random
from si_tests import logger
from si_tests import settings
from si_tests.utils import templates

LOG = logger.logger


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_create_hoc_before_lcm_bm(kaas_manager, _, show_step):
    """ This is a first part of LCM operations verification scenario
    when HOC objects are present (see below)
    This test just creates a hoc object before any LCM operations (upgrades, reboots etc.)
    from other tests.

    Scenario:
        1. Create and verify a dummy HostOSConfiguration object
        via si_tests/tests/lcm/day2operations/test_create_hoc_before_lcm_bm.py.
        2. Run any LCM tests for the clusters (upgrades, reboot, replacement etc.)
        3. Verify HOC object again and delete it via
        si_tests/tests/lcm/day2operations/test_delete_hoc_after_lcm_bm.py
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    hoc_name = "test-create-hoc-before-lcm-bm-" + str(random.randint(10, 99))
    day2_label = {"test-create-hoc-before-lcm-bm-label-" + str(random.randint(10, 99)): "enabled"}

    hostoscfg_config = [
        {
            "module": "package",
            "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
            "values": {"packages": [{"name": "aircrack-ng", "state": "present"}]},
        }
    ]

    # hoc sample
    hostoscfg_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "HostOSConfiguration",
        "metadata": {
            "name": hoc_name,
            "namespace": namespace_name,
        },
        "spec": {
            "configs": hostoscfg_config,
            "machineSelector": {
                "matchLabels": day2_label,
            }
        }
    }

    show_step(1)
    """Label machines for test"""
    machines_to_label = []
    mgmt_machines = cluster.get_machines(machine_type='control')
    assert len(mgmt_machines) > 1, f"We need at least 1 mgmt machine in cluster: {cluster_name}"
    machines_to_label.append(mgmt_machines[0])
    if cluster.is_child:
        worker_machines = cluster.get_machines(machine_type='worker')
        assert len(worker_machines) > 1, f"We need at least 1 worker machine in child cluster: {cluster_name}"
        machines_to_label.append(worker_machines[0])

    for machine in machines_to_label:
        LOG.info(f"Add label {day2_label} to machine: {machine.name} "
                 f"in cluster: {cluster_name}")
        machine.add_machine_labels(day2_label)

    show_step(2)
    """Create HOC object"""
    # Remember LCMMachines timestamps before creating HostOSConfiguration
    lcmmachines_timestamps_before = cluster.get_cluster_lcmmachines_timestamps()

    hostoscfg = ns.create_hostosconfiguration_raw(hostoscfg_data)

    # Wait for the selected Machines in the hostosconfiguration status
    LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
    cluster.check.check_hostosconfig_machine_selector(hostoscfg)
    LOG.info("Check that new items added into machineTypes in LCMCluster")
    cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)
    LOG.info("Check that new items added into stateItems in LCMMachine")
    cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg, lcmmachines_timestamps_before)
    cluster.check.get_hostosconfig_machines_status(hostoscfg)

    lcmmachines_timestamps_after = cluster.get_cluster_lcmmachines_timestamps()
    show_step(3)
    """Store HOC name and machine label in SI config"""
    assert settings.SI_CONFIG_PATH and os.path.isfile(settings.SI_CONFIG_PATH), (
        f"Environment variable 'SI_CONFIG_PATH' contains wrong path to YAML file: {settings.SI_CONFIG_PATH}")
    with templates.YamlEditor(settings.SI_CONFIG_PATH) as editor:
        current_content = editor.content
        if 'test_create_hoc_before_lcm_bm' not in current_content:
            current_content['test_create_hoc_before_lcm_bm'] = {}
        current_content['test_create_hoc_before_lcm_bm'][f'{namespace_name}_{cluster_name}'] = {
                'hoc_name': hoc_name,
                'hoc_machine_labels': day2_label,
                'lcmmachines_timestamps_before': lcmmachines_timestamps_before,
                'lcmmachines_timestamps_after': lcmmachines_timestamps_after
            }
        editor.content = current_content
