import os
import uuid

import pytest
import untangle
import yaml

from si_tests import logger, settings
from si_tests.deployments.utils import file_utils
from si_tests.utils import utils

LOG = logger.logger


@pytest.mark.usefixtures('restore_osdpl_cr')
def test_tf_sriov_spoof_check(show_step, openstack_client_manager, os_manager, request):
    """Check spoof checking option with SRIOV interface.

            Scenario:
                1. Enable pass_hwveb_ports_to_os_vif_plugin option (OSDPL)
                2. Deploy heat stack with sriov instances
                3. Check spoof checking option is disabled on appropriate VF interface
                4. Enable port security
                5. Check spoof checking option is enabled
        """
    show_step(1)
    osdpl = os_manager.get_osdpl_deployment()
    if not os_manager.is_pass_hwveb_ports_to_os_vif_plugin():
        patch = {
            "spec": {
                "services": {
                    "compute": {
                        "nova": {
                            "values": {
                                "conf": {
                                    "nova": {
                                        "workarounds": {
                                            "pass_hwveb_ports_to_os_vif_plugin": True
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        osdpl.patch(patch)
        os_manager.wait_os_deployment_status(timeout=90, status="APPLYING")
        os_manager.wait_os_deployment_status(timeout=600, status="APPLIED")
        os_manager.wait_openstackdeployment_health_status(timeout=180)

    show_step(2)
    stack_id = uuid.uuid4()
    stack_name = f"si-test-tf-sriov-{stack_id}"
    template_path = file_utils.join(os.path.dirname(os.path.abspath(__file__)),
                                    "templates/sriov_tf_scenario.yaml")
    environment_path = file_utils.join(os.path.dirname(os.path.abspath(__file__)),
                                       f"templates/envs/{settings.ENV_NAME}/sriov_tf_scenario_env.yaml")
    with open(environment_path, 'r') as f:
        data = yaml.full_load(f)
    parameters = data['parameters']
    ssh_keys = utils.generate_keys()
    stack_params = {
        "physnet_vlan": parameters['physnet_vlan'],
        "physnet_addr": parameters['physnet_addr'],
        "physnet_prefix": parameters['physnet_prefix'],
        "physnet_gw": parameters['physnet_gw'],
        "ssh_public_key": f"ssh-rsa {ssh_keys['public']}",
        "ssh_private_key": ssh_keys["private"]
    }
    openstack_client_manager.create_stack(request, stack_name, template_path,
                                          stack_params=stack_params)

    stack = openstack_client_manager.stack.show([stack_name])
    server_1_name = None
    sriov_port_1_name = None
    sriov_port_1_mac = None
    for output in stack['outputs']:
        if output['output_key'] == 'server_1_name':
            server_1_name = output['output_value']
        if output['output_key'] == 'sriov_port_1_name':
            sriov_port_1_name = output['output_value']
        if output['output_key'] == 'sriov_port_1_mac':
            sriov_port_1_mac = output['output_value']

    show_step(3)
    server = openstack_client_manager.server.show([server_1_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split('.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    found = False
    for interface in domain_obj.domain.devices.interface:
        if sriov_port_1_mac == interface.mac.get_attribute("address"):
            interface_type = interface.get_attribute("type")
            assert interface_type in ['hostdev', 'direct']
            found = True
            break
    mac = sriov_port_1_mac

    assert found, "Can't detect interface for instance"

    port = openstack_client_manager.port.show([sriov_port_1_name])
    assert not port["port_security_enabled"], \
        "Check port security option for SRIOV interface"

    cmd = ["ip", "link"]
    output = openstack_client_manager.exec_libvirt(hypervisor_hostname, cmd)
    assert mac in output, "Can't find interface mac address"
    for line in output.splitlines():
        if mac in line:
            LOG.info(f"Interface info:\n{line}")
            assert 'spoof checking off' in line, "Spoof checking isn't disabled"

    show_step(4)
    openstack_client_manager.port.set(["--enable-port-security", sriov_port_1_name])
    port = openstack_client_manager.port.show([sriov_port_1_name])
    assert port["port_security_enabled"], \
        "Check port security option for SRIOV interface"

    show_step(5)
    output = openstack_client_manager.exec_libvirt(hypervisor_hostname, cmd)
    assert mac in output, "Can't find interface mac address"
    for line in output.splitlines():
        if mac in line:
            LOG.info(f"Interface info:\n{line}")
            assert 'spoof checking on' in line, "Spoof checking isn't enabled"
