import pytest
import re
import time

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Machine
from si_tests.utils import utils

LOG = logger.logger


def patch_daemonset(cluster, pattername):
    ds = cluster.k8sclient.daemonsets.get(
        name=pattername, namespace="openstack")
    annotations = ds.read().to_dict()['spec']['template']['metadata']['annotations']
    LOG.info(f"Annotations before: {annotations}")
    ds.patch(body={"spec": {"template": {"metadata": {
        "annotations": {"si-team/restart-policy-checker" + utils.gen_random_string(8): ""}}}}})
    time.sleep(30)
    annotations = ds.read().to_dict()['spec']['template']['metadata']['annotations']
    LOG.info(f"Annotations after: {annotations}")


@pytest.mark.usefixtures('post_action_stop_mm_mode')
@pytest.mark.usefixtures('introspect_no_PRODX_51933_after_lcm')
def test_openvswitch_restart_after_maintenance(kaas_manager, show_step):
    """Check that openstack controller works correctly with openvswitch pods restart policy.
       It should restart openvswitchpods after removing node maintenance
       Short description of openvswitch restarts logic (Why? When?):
        Restart of openvswitch may happen in the following scenarios:
         -config option change - will stay as it is now (i.e. not deterministic)
         -update of MOSK child cluster
         -upgrade of OpenStack - will stay as it is now (i.e. not deterministic)
        Always restart openvswitch pods only when:
         -clustermaintenancerequest exists
         -nodemainteanancerequest is removed for specific node. Which basically means
          everything is done with the node from LCM standpoint.
    Scenario:
            1. Check cluster readiness, get one openvswitchpod, node and machine name
            2. Initiate cluster maintenance and selected machine maintenance
            3. Patch annotations in openvswitch daemonset(to change generation) because this is a reboot criteria
            4. Stop machine maintenance and check that pod is restarted by openstack-controller
            5. Stop cluster maintenance and check that we have same pod name
            6. Exit from mm mode (post action)
    """

    managed_ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)

    show_step(1)
    cluster.check.check_k8s_pods()
    cluster.check.check_k8s_pods(target_namespaces="openstack")
    cluster.check.check_cluster_readiness()
    ovs_pods = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                if re.match(settings.OPENVSWITCHD_PATTERN_NAME, pod.name)]
    LOG.info(f"List of openvswitchpods pods: {ovs_pods}")

    neutron_l3_pods = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                       if re.match(settings.NEUTRON_L3_PATTERN_NAME, pod.name)]
    LOG.info(f"List of neutron-l3-agent pods: {neutron_l3_pods}")

    if not ovs_pods:
        raise Exception("Looks like we don't have openvswitch pods on env. "
                        "Please check that Openstack is installed correctly")
    if not neutron_l3_pods:
        raise Exception("Looks like we don't have neutron-l3 pods on env. "
                        "Please check that Openstack is installed correctly")

    # Record initial openvswitch machines, we need all of them
    machines: list[Machine] = cluster.get_machines()

    show_step(2)
    cluster.cluster_maintenance_start()

    # Check all openvswitch pods
    for selected_pod in ovs_pods:
        target_machine = None
        for machine in machines:
            k8s_node_name = machine.get_k8s_node_name()
            pod_node_name = selected_pod.node_name
            if k8s_node_name == pod_node_name:
                LOG.info(f"Pod '{selected_pod.name}' found on K8s node name : {k8s_node_name}")
                target_machine = machine
                break
        if not target_machine:
            raise Exception(f"Selected pod: {selected_pod} wasn't scaled to any machine in cluster, "
                            f"it's NOT OK for current test suite, node_name field: {pod_node_name}")
        # Check do we have neutron l3 pods on this machine
        neutron_l3_pod = [pod for pod in neutron_l3_pods if pod.node_name == target_machine.get_k8s_node_name()]

        target_machine.machine_maintenance_start()

        show_step(3)
        patch_daemonset(cluster, settings.OPENVSWITCHD_PATTERN_NAME)

        if neutron_l3_pod:
            patch_daemonset(cluster, settings.NEUTRON_L3_PATTERN_NAME)

        # Check that selected pod still exists
        assert selected_pod in cluster.k8sclient.pods.list(namespace='openstack'), \
            f"Pod {selected_pod.name} was restarted before we remove nodemaintenancerequest"
        LOG.info(f"Selected pod: {selected_pod} exists as expected")
        if neutron_l3_pod:
            assert neutron_l3_pod[0] in cluster.k8sclient.pods.list(namespace='openstack'), \
                f"Pod {neutron_l3_pod[0].name} was restarted before we remove nodemaintenancerequest"
            LOG.info(f"Selected pod: {neutron_l3_pod[0]} exists as expected")

        show_step(4)
        time.sleep(60)
        target_machine.machine_maintenance_stop()
        cluster.check.check_k8s_pods()
        cluster.check.check_k8s_pods(target_namespaces="openstack")

        # Get list of new ovs pods
        new_ovs_pods = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                        if re.match(settings.OPENVSWITCHD_PATTERN_NAME, pod.name)]
        LOG.info(f"List of openvswitchpods pods: {new_ovs_pods}")

        # Get list of new neutron-l3 pods
        if neutron_l3_pod:
            new_neutron_l3_pods = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                                   if re.match(settings.NEUTRON_L3_PATTERN_NAME, pod.name)]
            LOG.info(f"List of neutron-l3 pods: {new_neutron_l3_pods}")

        assert selected_pod not in new_ovs_pods, (f"Looks like that initial pod with name: {selected_pod.name} still "
                                                  f"exists in cluster after nodemaintenancerequest was deleted")
        if neutron_l3_pod:
            assert neutron_l3_pod[0] not in new_neutron_l3_pods, \
                (f"Looks like that initial pod with name: {neutron_l3_pod[0].name} "
                 f"still exists in cluster after nodemaintenancerequest was deleted")
            # Need to remove l3 neutron pod from list because we will try to process unexisted pod
            neutron_l3_pods.remove(neutron_l3_pod[0])

    show_step(5)
    # Get list of new ovs and neutron-l3 pods before clustermaintenancerequest deletion
    pods_before_clmr = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                        if re.match(settings.OPENVSWITCHD_PATTERN_NAME, pod.name)
                        or re.match(settings.NEUTRON_L3_PATTERN_NAME, pod.name)]
    cluster.cluster_maintenance_stop()
    cluster.check.check_k8s_pods()
    cluster.check.check_k8s_pods(target_namespaces="openstack")
    cluster.check.check_cluster_readiness()

    # Check that we have same pods after clustermaintenancerequest deletion
    pods_after_clmr = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                       if re.match(settings.OPENVSWITCHD_PATTERN_NAME, pod.name)
                       or re.match(settings.NEUTRON_L3_PATTERN_NAME, pod.name)]
    assert pods_before_clmr == pods_after_clmr, \
        (f"Pods before: {pods_before_clmr} are not same as after "
         f"clustermaintenancerequest deletion: {pods_after_clmr}")
