import functools
import os
import pytest
import tempfile
import uuid
import untangle

from si_tests.deployments.utils import (
    commons,
    kubectl_utils,
    file_utils,
    namespace,
    wait_utils,
)
from si_tests import settings


def test_nodes_hugepages(openstack_client_manager):
    kubectl = kubectl_utils.Kubectl()

    nodes = kubectl.get_node_name_by_label(
        settings.OPENSTACK_NODES_WITH_HUGEPAGES_LABEL)
    assert (
        len(nodes) != 0
    ), f"There are no nodes with {settings.OPENSTACK_NODES_WITH_HUGEPAGES_LABEL} label"  # noqa: E501
    hugepages_enabled = []
    for node in nodes:
        hp = int(
            openstack_client_manager.exec_libvirt(
                node, ["/sbin/sysctl", "-b", "vm.nr_hugepages"]
            )
        )
        commons.LOG.info(f"{node} => {hp} hugepages are available")
        hugepages_enabled.append(hp > 0)

    if not all(hugepages_enabled):
        pytest.fail("Not all nodes have hugepages enabled")


def test_nova_compute_instance_hugepages(openstack_client_manager, request):
    stack_name = f"hugepages-{uuid.uuid4()}"
    source = file_utils.join(
        os.path.dirname(os.path.abspath(__file__)), "templates/hugepages.yaml"
    )
    target_template = f"/tmp/{stack_name}.yaml"
    target_parameters = f"/tmp/{stack_name}-parameters.yaml"
    client_name = openstack_client_manager.client.name
    kubectl = kubectl_utils.Kubectl()
    destination = (
        f"{namespace.NAMESPACE.openstack}/{client_name}:{target_template}"
    )
    parameters_dest = (
        f"{namespace.NAMESPACE.openstack}/{client_name}:{target_parameters}"
    )
    with tempfile.NamedTemporaryFile(mode="w") as tmp:
        az_name = settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME
        content = f"""
        parameters:
          server_availability_zone: {az_name}
        """
        tmp.write(content)
        tmp.flush()
        kubectl.cp(tmp.name,  parameters_dest)

    commons.LOG.info("Copy heat template")
    kubectl.cp(source, destination)

    commons.LOG.info("Create heat stack")
    openstack_client_manager.stack.create(
        [stack_name, "-t", target_template, "-e", target_parameters]
    )

    request.addfinalizer(
        functools.partial(
            openstack_client_manager.stack.delete, [stack_name, "-y", "--wait"]
        )
    )

    commons.LOG.info("Wait for heat stack to create")
    wait = wait_utils.Waiter(openstack_client_manager.os_manager, 3600)

    def _stack():
        status = openstack_client_manager.stack.show([stack_name])[
            "stack_status"
        ]
        if status == "CREATE_FAILED":
            failed_reason = {'resource_name': '', 'message': ''}
            resource_list = openstack_client_manager.stack.resource_list(
                ["-n", "10", stack_name]
            )
            failed_resources = openstack_client_manager.stack.resource_list(
                ["--filter", "status=CREATE_FAILED", stack_name],
                yaml_output=True
            )
            for r in failed_resources:
                resource_events = openstack_client_manager.stack.event_list(
                    ["--resource", r.get('resource_name'), stack_name],
                    yaml_output=True
                )
                failed_event = [
                    event for event in resource_events if event.get(
                        'resource_status') == 'CREATE_FAILED']
                for failed in failed_event:
                    failed_reason['resource_name'] = failed['resource_name']
                    failed_reason['message'] = failed['resource_status_reason']

            commons.LOG.error(
                "Resource info: \n%s",
                resource_list,
            )
            raise Exception("Failed to create stack: {}".format(failed_reason))
        return status == "CREATE_COMPLETE"

    wait.wait(_stack)
    commons.LOG.info("Heat stack created successfully")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server[
        "OS-EXT-SRV-ATTR:hypervisor_hostname"].split('.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    try:
        for hp in domain_obj.domain.memoryBacking.hugepages:
            hp_size = hp.page.get_attribute("size")
            commons.LOG.info(
                f"Compute instance has hugepages with size: {hp_size}"
            )
    except AttributeError as e:
        commons.LOG.error("Compute instance has no hugepages on board")
        raise e
