#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.utils import waiters
from si_tests.managers.kaas_manager import Manager


LOG = logger.logger


class TestProvisionAwsMgmCluster():

    @pytest.mark.usefixtures("introspect_standalone_seed_management_deploy_objects")
    def test_bootstrap_kaas_on_aws_macos_seed_node(self, show_step):
        """Bootstrap KaaS using standalone permanent MacOS seed node

        Scenario:
            1. Verify seed node accessibility
            2. Cleanup seed node docker workspace
               and possible bootstrap leftovers (optional)
            3. Prepare seed node to run KaaS bootstrap on AWS
            4. Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
            5. Check aws bootstrap templates for management cluster (optional)
            6. Bootstrap KaaS with AWS provider
            7. Create default MCC users, if requested (optional)
            8. Check versions
            9. Wait for all pods to be Running and Ready
            10. Check that all expected pods exist
            11. Download bootstrap artifacts
            12. Store seed IP in si_config
            13. Check LDAP integration (optional)
            14. Cleanup seed node docker workspace
                and possible bootstrap leftovers (optional)
        """

        show_step(1)
        assert waiters.icmp_ping(settings.SEED_STANDALONE_EXTERNAL_IP)
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=settings.SEED_STANDALONE_EXTERNAL_IP)

        if settings.KEEP_ENV_BEFORE:
            LOG.info("Skip erase: KEEP_ENV_BEFORE flag is set")
        else:
            show_step(2)
            bootstrap.step_001_cleanup_seed_workspace()

        show_step(3)
        bootstrap.step_003_prepare_standalone_seed_node()

        # step 004
        show_step(4)
        bootstrap.add_credentials_data_to_si_config()

        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check: "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
        else:
            show_step(5)
            bootstrap.check_templates()

        show_step(6)
        bootstrap.step_004_deploy_kaas_cluster()

        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(7)
            bootstrap.step_004_create_default_mcc_users()

        if settings.MCC_SCHEDULED_UPGRADES:
            LOG.info("Update mcc-upgrade crd for feature upgrade scheduling")
            bootstrap.step_006_update_mcc_upgrade_crd()

        show_step(8)
        bootstrap.step_005_check_versions()

        show_step(9)
        bootstrap.step_006_wait_for_pods()

        show_step(10)
        bootstrap.step_006_postdeployment_checks()

        show_step(11)
        bootstrap.step_007_download_bootstrap_artifacts()

        show_step(12)
        kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig"
        if not os.path.isfile(kubeconfig_path):
            raise Exception(f"KUBECONFIG {kubeconfig_path} not found in"
                            f" artifacts!")
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        kaas_manager.si_config.store_seed_ip(settings.SEED_STANDALONE_EXTERNAL_IP)

        # need to double check it for standalone seed node
        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(13)
            bootstrap.check_keystone_ldap_integration()

        if settings.KEEP_ENV_AFTER:
            LOG.warning("Skip erase: KEEP_ENV_AFTER flag is set")
        else:
            # Erase kaas environment after bootstrap is passed
            show_step(14)
            bootstrap.step_007_erase_env_after()
            bootstrap.step_001_cleanup_seed_workspace()
