#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import base64
import re
import time

import pytest
import yaml
from playwright.sync_api import expect

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import waiters, exceptions

LOG = logger.logger

hotfix_labels = ["hotfix/hugepages", "hotfix/openstack-compute-node-dpdk"]


@pytest.mark.parametrize("traced_page", ["test_kaas_create_namespace"], indirect=True)
def test_kaas_create_namespace(kaas_manager, traced_page):
    """Create new namespace

    Scenario:
        1. Collecting env data
        2. Creating namespace
    """
    page = traced_page

    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.get_by_role("button", name="Close").click()
    page.get_by_role("link", name="Projects").click()
    page.locator("[data-test-id=\"btn-create-namespace\"]").click()
    page.get_by_role("textbox", name="Project Name").fill(settings.TARGET_NAMESPACE)
    page.get_by_role("dialog", name="Create new project").get_by_label("Create").click()
    expect(page.get_by_text(settings.TARGET_NAMESPACE)).to_be_visible()
    # Wait Clusters tab it shows that Namespace was created successfully
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.get_by_role("link", name=settings.TARGET_NAMESPACE).click()
    page.get_by_role("button", name="Select").click()

    def _wait_cluster_button(page):
        page.reload()
        # Wait Clusters tab it shows that Namespace was created successfully
        page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
        page.get_by_role("link", name=settings.TARGET_NAMESPACE).click()
        page.get_by_role("button", name="Select").click()
        try:
            page.get_by_role("link", name="Clusters").click()
            return True
        except Exception as e:
            LOG.info(f'Cluster button is not presented in cluster: \n{e}')

    waiters.wait(lambda: _wait_cluster_button(page), timeout=420, interval=60)


@pytest.mark.parametrize("traced_page", ["test_kaas_create_ssh_public_key"], indirect=True)
def test_kaas_create_ssh_public_key(kaas_manager, traced_page):
    """Create ssh public key

    Scenario:
        1. Collecting env data
        2. Creating ssh public key
    """
    page = traced_page

    # Get pub_key from job
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()

    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="SSH Keys").click()
    page.reload()

    # Wait button create button
    def _wait_ssh_create_button(page):
        page.reload()
        page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
        page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
        page.get_by_role("button", name="Select").click()
        try:
            expect(page.get_by_role("link", name="SSH Keys")).to_be_visible()
            expect(page.locator("[data-test-id=\"btn-create-keypair\"]")).to_be_visible()
            return True
        except Exception as e:
            LOG.info(f'"Create ssh" button is not presented in cluster: \n{e}')

    waiters.wait(lambda: _wait_ssh_create_button(page), timeout=420, interval=60)

    page.locator("[data-test-id=\"btn-create-keypair\"]").click()
    page.locator("[data-test-id=\"input-name\"]").fill(settings.TARGET_CLUSTER + "-key")
    page.locator("[data-test-id=\"input-key\"]").click()
    page.locator("[data-test-id=\"input-key\"]").fill(pub_key_content)
    # Check that name is free
    expect(page.get_by_role("button", name="This name is already in use")).not_to_be_visible()
    page.get_by_role("button", name="Create").click()
    # Check that key was added
    expect(page.get_by_role("dialog", name="Add new SSH key")).not_to_be_visible()

    page.reload()
    page.get_by_role("link", name=settings.TARGET_CLUSTER).click()
    expect(page.get_by_text("ssh-rsa")).to_be_visible()
    page.locator("button").filter(has_text="Close").click()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_bm_credentials"], indirect=True)
def test_kaas_create_bm_credentials(kaas_manager, traced_page, ui_child_data):
    """Create bm credentials

    Scenario:
        1. Collecting env data
        2. Creating bm credentials
    """
    page = traced_page

    bm_hosts_data = ui_child_data.get('nodes', [])

    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Baremetal").click()
    page.get_by_text("Credentials").click()
    for node in bm_hosts_data:
        if not any(item in node.get('si_roles', []) for item in ['nodeforscale', 'child-scale-controller']):
            ipmi_user = base64.b64decode(
                node['ipmi']['username']).decode("utf-8")
            ipmi_password = base64.b64decode(
                node['ipmi']['password']).decode("utf-8")
            page.get_by_role("button", name="Create Credential").click()
            page.get_by_role("textbox", name="Name", exact=True).click()
            page.get_by_role("textbox", name="Name", exact=True).fill(node['name'] + "-cred")
            page.get_by_role("textbox", name="Username").click()
            page.get_by_role("textbox", name="Username").fill(ipmi_user)
            page.get_by_role("textbox", name="Username").press("Tab")
            page.get_by_role("textbox", name="Password").fill(ipmi_password)
            page.get_by_role("button", name="Create", exact=True).click()
            page.get_by_role("button", name="Refresh table").click()
            time.sleep(2)
            expect(page.get_by_text(node['name'] + "-cred")).to_be_visible()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_bmhprofile"], indirect=True)
def test_kaas_create_bmhprofile(kaas_manager, traced_page, ui_child_data):
    """Create bmhprofile

    Scenario:
        1. Collecting env data
        2. Creating bmhprofile
    """
    page = traced_page

    bmhp_data = ui_child_data.get('bmh_profiles', {})

    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Baremetal").click()
    page.get_by_text("Host Profiles").click()

    for n, bmhp in bmhp_data.items():
        body = {**bmhp['spec'], }
        LOG.info(f'Custom bmhp: {yaml.dump(body)}')
        page.get_by_role("button", name="Create Host Profile").click()
        page.get_by_role("textbox", name="Name").click()
        page.get_by_role("textbox", name="Name").fill(n)
        page.get_by_role("button", name="Edit").click()
        page.get_by_role("textbox", name="YAML").fill(yaml.dump(body))
        page.locator("button").filter(has_text="Close").click()
        # Add labels
        labels_flag = True
        if bmhp['labels']:
            for label, value in bmhp['labels'].items():
                if labels_flag:
                    page.get_by_role("button", name="Add a label").click()
                    page.get_by_role("textbox", name="Enter a label").click()
                    page.get_by_role("textbox", name="Enter a label").fill(label)
                    page.get_by_role("textbox", name="Enter a value").click()
                    page.get_by_role("textbox", name="Enter a value").fill(value)
                    labels_flag = False
                else:
                    page.get_by_role("button", name="Add").click()
                    page.get_by_placeholder("Enter a label").last.click()
                    page.get_by_placeholder("Enter a label").last.fill(label)
                    page.get_by_placeholder("Enter a value").last.click()
                    page.get_by_placeholder("Enter a value").last.fill(value)

        page.get_by_role("button", name="Create", exact=True).click()
        expect(page.get_by_role("dialog", name="Create host profile")).not_to_be_visible()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_bmh"], indirect=True)
def test_kaas_create_bmh(kaas_manager, traced_page, ui_child_data):
    """Create bmh

    Scenario:
        1. Collecting env data
        2. Creating bmh
    """
    page = traced_page

    bm_hosts_data = ui_child_data.get('nodes', [])
    hosts_deploy_data = [node for node in bm_hosts_data if not any(item in node.get('si_roles', [])
                                                                   for item in ['nodeforscale',
                                                                                'child-scale-controller'])]
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Baremetal").click()
    page.get_by_text("Host Inventories").click()
    for node in hosts_deploy_data:
        page.get_by_role("button", name="Create Host Inventory").click()
        page.get_by_role("textbox", name="Name", exact=True).click()
        page.get_by_role("textbox", name="Name", exact=True).fill(node["name"])
        page.get_by_role("combobox").filter(has_text="Boot Mode*").get_by_role("img").click()
        page.get_by_role("option", name="UEFI", exact=True).click()
        page.get_by_role("textbox", name="MAC Address").click()
        page.get_by_role("textbox", name="MAC Address").fill(node['networks'][0]['mac'])
        page.get_by_role("textbox", name="IP Address").click()
        page.get_by_role("textbox", name="IP Address").fill(node['ipmi']['ipmi_ip'])
        page.get_by_role("textbox", name="Credential Name").click()
        page.get_by_role("option", name=node["name"] + "-cred").click()
        page.get_by_role("textbox", name="Cert Validation").click()
        page.get_by_role("option", name="Disabled").click()
        page.get_by_role("dialog", name="Create baremetal host").click()
        page.get_by_role("button", name="Create", exact=True).click()
        expect(page.get_by_role("dialog", name="Create baremetal host")).not_to_be_visible()
        page.reload()

    # Wait button create button
    def _wait_bmh_state(page, bm_hosts_data, expected_state=None):
        page.reload()
        page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
        page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
        page.get_by_role("button", name="Select").click()
        page.get_by_role("link", name="Baremetal").click()
        page.get_by_text("Host Inventories").click()
        try:
            for node in bm_hosts_data:
                page.get_by_role("button", name="Refresh table").click()
                expect(page.locator(f"[data-test-id=\"{node['name']}\"]").
                       get_by_text(expected_state)).to_be_visible()
            return True
        except Exception as e:
            LOG.info(f'BMH state is not in Expected state: {expected_state} '
                     f'and failed with error: \n{e}')

    # Wait bmh inspecting finished
    waiters.wait(lambda: _wait_bmh_state(page, hosts_deploy_data,
                                         expected_state="Inspecting Finished"),
                 timeout=9000,
                 interval=180)


@pytest.mark.parametrize("traced_page", ["test_kaas_create_proxy_object"], indirect=True)
def test_kaas_create_proxy_object(kaas_manager, traced_page, ui_child_data):
    """Create proxy object

    Scenario:
        1. Collecting env data
        2. Creating cluster proxy object
    """
    page = traced_page

    if not settings.KAAS_OFFLINE_DEPLOYMENT:
        pytest.skip("Skip because we don't need to create proxy object for online env")

    if (settings.KAAS_EXTERNAL_PROXY_ACCESS_STR and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    LOG.info("Found KAAS_EXTERNAL_PROXY_ACCESS_STR. Creating "
             f"proxyobject with proxy {settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}")
    if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
        with open(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE, 'r') as f:
            proxy_cert_content = base64.b64encode(f.read().encode("utf-8")).decode()
            LOG.info("Using proxy with custom certificate")
    else:
        proxy_cert_content = None
    proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR

    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.locator("[data-test-id=\"nav-proxies\"]").click()
    page.locator("[data-test-id=\"btn-create-proxy\"]").click()
    page.get_by_role("textbox", name="Proxy Name").click()
    page.get_by_role("textbox", name="Proxy Name").fill(f"{settings.TARGET_CLUSTER}-{settings.KAAS_PROXYOBJECT_NAME}")
    page.get_by_role("textbox", name="HTTP Proxy").click()
    page.get_by_role("textbox", name="HTTP Proxy").fill(proxy_access_str)
    page.get_by_role("textbox", name="HTTPS Proxy").click()
    page.get_by_role("textbox", name="HTTPS Proxy").fill(proxy_access_str)
    page.get_by_role("textbox", name="No Proxy").click()
    page.get_by_role("textbox", name="No Proxy").fill(settings.KAAS_PROXYOBJECT_NO_PROXY)
    if proxy_cert_content:
        page.get_by_role("checkbox", name="CA Certificate").click()
        page.get_by_role("textbox", name="CA Certificate").click()
        page.get_by_role("textbox", name="CA Certificate").fill(proxy_cert_content)
    page.get_by_role("button", name="Create").click()
    expect(page.get_by_role("dialog", name="Add new Proxy")).not_to_be_visible()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_cluster"], indirect=True)
def test_kaas_create_cluster(kaas_manager, traced_page, ui_child_data):
    """Create cluster

    Scenario:
        1. Collecting env data
        2. Creating cluster
    """
    page = traced_page

    load_balancer_host = [subnet for subnet in ui_child_data['ipam_subnets']
                          if subnet["name"] == 'child-k8s-api-lb'][0]["cidr"].split('/')[0]
    services_cidr = "10.232.0.0/18"
    pods_cidr = "192.168.0.0/16"
    LOG.info(f"Load balancer host: {load_balancer_host}")
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Clusters").click()
    page.locator("[data-test-id=\"btn-create-cluster\"]").click()
    page.get_by_role("button", name="Close Tour").click()
    page.get_by_role("textbox", name="Cluster Name").click()
    page.get_by_role("textbox", name="Cluster Name").fill(settings.TARGET_CLUSTER)
    page.get_by_role("textbox", name="Release Version").click()
    page.get_by_role("option", name="openstack (current major)").click()
    if settings.KAAS_OFFLINE_DEPLOYMENT:
        page.get_by_role("textbox", name="Proxy").click()
        page.get_by_role("option", name=f"{settings.TARGET_CLUSTER}-{settings.KAAS_PROXYOBJECT_NAME}").click()
    page.get_by_role("textbox", name="SSH Keys").click()
    page.locator("#ssh-keys-area").get_by_role("button", name="Select").click()
    page.get_by_role('checkbox').last.set_checked(True)
    page.get_by_role("button", name="OK").click()
    page.locator("a").filter(has_text="Kubernetes").click()
    page.get_by_role("textbox", name="Services CIDR blocks").click()
    page.get_by_role("textbox", name="Services CIDR blocks").fill(services_cidr)
    page.get_by_role("textbox", name="Pods CIDR blocks").click()
    page.get_by_role("textbox", name="Pods CIDR blocks").fill(pods_cidr)
    page.locator("form").click()
    page.locator("a").filter(has_text="StackLight").click()
    page.get_by_role("checkbox", name="HA Mode").click()
    page.get_by_role("checkbox", name="Enable Logging").click()
    page.get_by_role("button", name="Create", exact=True).click()
    page.get_by_role("button", name="Close Tour").click()
    expect(page.get_by_role("link", name="child-cl")).to_be_visible()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_subnetworks"], indirect=True)
def test_kaas_create_subnetworks(kaas_manager, traced_page, ui_child_data):
    """Create subnetworks

    Scenario:
        1. Collecting env data
        2. Create subnetworks
    """
    page = traced_page

    ipam_subnets = ui_child_data['ipam_subnets']
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Networks").click()
    for ipam_subnet in ipam_subnets:
        page.get_by_role("link", name="Networks").click()
        page.get_by_text("Subnets").click()
        page.get_by_role("button", name="Create Subnet").click()
        page.get_by_role("textbox", name="Name", exact=True).click()
        page.get_by_role("textbox", name="Name", exact=True).fill(ipam_subnet["name"])
        # Only DHCP subnetwork doesn't need a cluster
        if ipam_subnet["name"] != "subnet-child-pxe":
            page.get_by_role("textbox", name="Cluster").click()
            page.get_by_role("option", name=settings.TARGET_CLUSTER).click()

        # Add subnetwork type
        subnet_type_mapping = {
            "subnet-child-pxe": "DHCP",
            "child-k8s-api-lb": "LB",
            "lcm-nw": "LCM",
            "storage-frontend": "Storage access",
            "storage-backend": "Storage replication"
        }

        subnet_name = ipam_subnet["name"]
        subnet_type = subnet_type_mapping.get(subnet_name, "Custom")

        page.get_by_role("textbox", name="Subnet Type").click()
        page.get_by_role("option", name=subnet_type).click()

        page.get_by_role("textbox", name="CIDR", exact=True).click()
        page.get_by_role("textbox", name="CIDR", exact=True).fill(ipam_subnet["cidr"])
        if ipam_subnet.get("gateway", None):
            page.get_by_role("textbox", name="Gateway").click()
            page.get_by_role("textbox", name="Gateway").fill(ipam_subnet["gateway"])
        if ipam_subnet.get("nameservers", None):
            page.get_by_role("textbox", name="Nameservers").click()
            page.get_by_role("textbox", name="Nameservers").fill(ipam_subnet['nameservers'][0])
        if ipam_subnet.get('includeRanges', None):
            page.get_by_role("textbox", name="Include CIDR IP Ranges").click()
            page.get_by_role("textbox", name="Include CIDR IP Ranges").fill(ipam_subnet['includeRanges'][0])
        if ipam_subnet.get('excludeRanges', None):
            page.get_by_role("textbox", name="Exclude CIDR IP Ranges").click()
            page.get_by_role("textbox", name="Exclude CIDR IP Ranges").fill(ipam_subnet['excludeRanges'][0])
        page.get_by_role("button", name="Create", exact=True).click()
        expect(page.get_by_role("dialog", name="Create subnet")).not_to_be_visible()
        # Refresh table and check network
        page.get_by_role("button", name="Refresh table").click()
        # Check that created network exist
        expect(page.get_by_text(ipam_subnet["name"])).to_be_visible()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_l2templates"], indirect=True)
def test_kaas_create_l2templates(kaas_manager, traced_page, ui_child_data):
    """Create l2templates

    Scenario:
        1. Collecting env data
        2. Create l2templates
    """
    page = traced_page

    l2templates = ui_child_data['l2templates']
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Networks").click()
    page.get_by_text("L2 Templates").click()
    for l2template in l2templates:
        page.get_by_role("link", name="Networks").click()
        page.get_by_text("L2 Templates").click()
        page.get_by_role("button", name="Create L2 Template").click()
        page.get_by_role("textbox", name="Name").click()
        page.get_by_role("textbox", name="Name").fill(l2template["name"])
        page.get_by_role("textbox", name="Cluster").click()
        page.get_by_role("option", name=settings.TARGET_CLUSTER).click()
        # Generate body:

        page.get_by_role("button", name="Edit").click()
        page.get_by_role("textbox", name="YAML").fill(yaml.dump(l2template["spec"]))
        page.locator("button").filter(has_text="Close").click()
        expect(page.get_by_role("dialog", name="L2 template specification YAML")).not_to_be_visible()
        first_label = True
        for label, label_value in l2template["labels"].items():
            if first_label:
                page.get_by_role("button", name="Add a label").click()
                page.get_by_role("textbox", name="Enter a label").click()
                page.get_by_role("textbox", name="Enter a label").fill(label)
                page.get_by_role("textbox", name="Enter a value").click()
                page.get_by_role("textbox", name="Enter a value").fill(label_value)
                first_label = False
            else:
                page.get_by_role("button", name="Add").click()
                page.get_by_placeholder("Enter a label").last.click()
                page.get_by_placeholder("Enter a label").last.fill(label)
                page.get_by_placeholder("Enter a value").last.click()
                page.get_by_placeholder("Enter a value").last.fill(label_value)

        page.get_by_role("button", name="Create", exact=True).click()
        expect(page.get_by_role("dialog", name="Create L2 template")).not_to_be_visible()
        # Check l2template exists
        expect(page.get_by_text(l2template["name"])).to_be_visible()


@pytest.mark.parametrize("traced_page", ["test_kaas_create_metallb"], indirect=True)
def test_kaas_create_metallb(kaas_manager, traced_page, ui_child_data):
    """Create metallb

    Scenario:
        1. Collecting env data
        2. Create metallb
    """
    page = traced_page

    metallb_config = ui_child_data['metallb_config']
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name="Networks").click()
    page.get_by_text("MetalLB Configs", exact=True).click()

    page.get_by_role("button", name="Create MetalLB Config").click()
    page.get_by_role("textbox", name="Name", exact=True).click()
    page.get_by_role("textbox", name="Name", exact=True).fill(metallb_config["name"])
    page.get_by_role("textbox", name="Cluster").click()
    page.get_by_role("option", name=settings.TARGET_CLUSTER).click()
    page.get_by_role("textbox", name="Enter addresses range").click()
    (page.get_by_role("textbox", name="Enter addresses range").
     fill(metallb_config['spec']['ipAddressPools'][0]['spec']['addresses'][0]))
    if metallb_config['spec']['ipAddressPools'][0]['spec']['autoAssign']:
        LOG.info("Auto Assign")
    page.get_by_placeholder("Enter a name").last.fill(metallb_config['spec']['l2Advertisements'][0]['name'])
    page.get_by_role("button", name="Select", exact=True).click()
    page.get_by_role("checkbox").last.click()
    page.get_by_role("button", name="OK").click()
    expect(page.get_by_role("dialog", name="Select")).not_to_be_visible()
    page.get_by_role("button", name="Create", exact=True).click()
    expect(page.get_by_role("dialog", name="Create MetalLB Config")).not_to_be_visible()
    # Check metallb existance
    page.get_by_role("button", name="Refresh table").click()
    expect(page.get_by_text(settings.TARGET_CLUSTER).first).to_be_visible()
    # We don't check state at this phase, because it will be
    # in Ready state only after finished Deploy


@pytest.mark.parametrize("traced_page", ["test_kaas_create_cluster_machines"], indirect=True)
def test_kaas_create_cluster_machines(kaas_manager, traced_page, ui_child_data):
    """Create cluster machines

    Scenario:
        1. Collecting env data
        2. Create cluster machines
    """
    page = traced_page

    bm_hosts_data = ui_child_data.get('nodes', [])
    hosts_deploy_data = [node for node in bm_hosts_data if not any(item in node.get('si_roles', [])
                                                                   for item in ['nodeforscale',
                                                                                'child-scale-controller'])]
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    bm_update_group_data = ui_child_data.get('groups', [])
    LOG.info(f"Update group data {bm_update_group_data}")

    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.get_by_role("link", name=f"{settings.TARGET_NAMESPACE}").click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name=f"{settings.TARGET_CLUSTER}").click()
    machine_names_list = []
    for machine in hosts_deploy_data:
        page.locator("[data-test-id=\"btn-create-machine\"]").click()
        machine_name = machine['name'] + f'-{settings.TARGET_CLUSTER}'
        if machine.get('bmh_labels', {}).get('hostlabel.bm.kaas.mirantis.com/worker', ''):
            machine_name = machine_name + "-worker"
            page.get_by_text("Worker", exact=True).click()
        else:
            machine_name = machine_name + "-control"
            page.get_by_text("Manager", exact=True).click()

        machine_names_list.append(machine_name)
        page.get_by_role("textbox", name="Name").click()
        page.get_by_role("textbox", name="Name").fill(machine_name)
        page.get_by_role("textbox", name="L2 Template").click()
        page.get_by_role("option", name=machine['l2TemplateSelector']['name'], exact=True).click()
        page.get_by_role("textbox", name="Distribution").click()
        page.get_by_role("option", name=settings.UI_OS_DISTRO).click()
        page.get_by_role("textbox", name="Host Inventory", exact=True).click()
        page.get_by_role("option", name=machine['name']).click()
        page.get_by_role("checkbox", name="Auto-commence provisioning").click()
        page.get_by_role("checkbox", name="Auto-commence deployment").click()
        page.get_by_role("textbox", name="Host Profile").click()
        # We should cover case with empty field and choose default one
        bmh_profile = machine.get('bmh_profile', {})
        if bmh_profile:
            page.get_by_role("option", name=machine['bmh_profile']).click()
        else:
            default_bmhp = [p.name for p in ns.get_baremetalhostprofiles() if "default" in p.name]
            if default_bmhp:
                page.get_by_role("option", name=default_bmhp[0]).click()
            else:
                raise Exception(f"Machine {machine_name} doesn't have bmh_profile field in spec."
                                f"And also there are no default bmhp in cluster.")
        # Create labels for worker
        if machine.get('bmh_labels', {}).get('hostlabel.bm.kaas.mirantis.com/worker', ''):
            first_label = True
            for label in machine["node_labels"].keys():
                label_name = ""
                # Add label
                if label == "openstack-compute-node":
                    label_name = "OpenStack compute"
                if label == "openstack-gateway":
                    label_name = "OpenStack gateway"
                if label == "openstack-control-plane":
                    label_name = "OpenStack control plane"
                if label == "openvswitch":
                    label_name = "Open vSwitch"
                if label == "stacklight":
                    label_name = "Stacklight"
                if label == "tfvrouter":
                    label_name = "Tungsten Fabric vRouter"
                if label == "tfanalytics":
                    label_name = "Tungsten Fabric Analytics"
                if label == "tfanalyticsdb":
                    label_name = "Tungsten Fabric Analytics database"
                if label == "tfconfig":
                    label_name = "Tungsten Fabric Config"
                if label == "tfconfigdb":
                    label_name = "Tungsten Fabric Config database"
                if label == "tfwebui":
                    label_name = "Tungsten Fabric web UI"
                if label == "tfcontrol":
                    label_name = "Tungsten Fabric Control"

                if not label_name:
                    if label in hotfix_labels:
                        continue
                    else:
                        raise Exception(f"Unknown label in spec: {label}, need to align UI label")
                if first_label:
                    page.get_by_role("button", name="Add node label").click()
                    first_label = False
                else:
                    page.get_by_role("button", name="Add").click()
                # Fill label
                page.get_by_role("combobox").filter(has_text=re.compile(r"^$")).get_by_role(
                    "img").last.click()
                page.get_by_role("option", name=label_name, exact=True).click()
        # Create machine
        page.get_by_role("button", name="Create", exact=True).click()
        expect(page.get_by_role("dialog", name="Create new machine")).not_to_be_visible()
        # Check machine appeared in Cluster page
        expect(page.get_by_text(machine_name)).to_be_visible()

    # WA
    # Create labels for control nodes. Currently in UI it is not possible to create labels for control nodes
    # but we need to support compact labs so we need to make it through API
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    child_config = ui_child_data["child_config"]
    if not child_config['dedicated_controlplane']:
        LOG.info("Set manually False to dedicated_controlplane in Cluster for cover compact labs")
        dedicated_ctrlplane = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "dedicatedControlPlane": False
                    }
                }
            }
        }
        cluster.patch(dedicated_ctrlplane)

    # Create update groups based on labels
    upd_groups_name = []
    for node in bm_hosts_data:
        if node['machine_labels'].get("kaas.mirantis.com/update-group"):
            upd_groups_name.append(node['machine_labels'].get("kaas.mirantis.com/update-group"))
        LOG.info(f"Update group names are found and be created: {upd_groups_name}")
    for group in bm_update_group_data:
        if group.get('name') in upd_groups_name:
            cluster.create_update_group(
                cluster_name=settings.TARGET_CLUSTER,
                namespace=settings.TARGET_NAMESPACE,
                name=group.get('name'),
                index=group.get('index'),
                concurrent_updates=group.get('concurrentUpdates'))

    # Create update groups, _raw case
    for upd_g in ui_child_data.get('updategroups_raw', []):
        upd_g_name = upd_g['metadata']['name']
        if upd_g_name in [x.name for x in cluster.get_update_groups()]:
            LOG.info(f"updategroup obj '{upd_g_name}' already exists.")
        else:
            cluster.create_update_group_raw(upd_g)
            LOG.info(f"updategroup obj '{upd_g_name}' has been created")

    nodes = cluster.get_machines(machine_type="control")
    for node in nodes:
        machine = [machine for machine in hosts_deploy_data if machine['name'] in node.name]
        if machine:
            control_labels = machine[0].get('node_labels', {})
            if control_labels:
                list = []
                for key, value in control_labels.items():
                    list.append({"key": key, "value": value})
                    LOG.info(f"list of dicts = {list}")
                node.add_labels(list)

    # WA for worker machines
    # Add machine labels to worker machines
    nodes = cluster.get_machines(machine_type="worker")
    for node in nodes:
        machine = [machine for machine in hosts_deploy_data if machine['name'] in node.name]
        if machine:
            worker_labels = machine[0].get('machine_labels', {})
            if worker_labels:
                list = []
                for key, value in worker_labels.items():
                    list.append({key: value})
                    LOG.info(f"list of dicts = {list}")
                for label in list:
                    node.add_machine_labels(label)

    # Wait all machines will get status ready
    def _wait_machines_state(page, machine_name_list, expected_state=None):
        page.reload()
        page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
        page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
        page.get_by_role("button", name="Select").click()
        page.get_by_role("link", name="Clusters").first.click()
        page.get_by_role("link", name=settings.TARGET_CLUSTER).click()
        try:
            for mname in machine_name_list:
                page.get_by_role("button", name="Refresh table").click()
                expect(page.locator(f"[data-test-id=\"{mname}\"]").get_by_text(expected_state)).to_be_visible()
            return True
        except Exception as e:
            LOG.info(f'Machine state is not in Expected state: {expected_state} '
                     f'and failed with error: \n{e}')

    # Wait machines are Ready
    waiters.wait(lambda: _wait_machines_state(page, machine_names_list,
                                              expected_state="ready"),
                 timeout=7200,
                 interval=300)


@pytest.mark.parametrize("traced_page", ["test_kaas_wait_cluster_ready_state"], indirect=True)
def test_kaas_wait_cluster_ready_state(kaas_manager, traced_page, ui_child_data):
    """Wait cluster Ready state

    Scenario:
        1. Collecting env data
        2. Wait cluster state
    """
    page = traced_page
    bm_hosts_data = ui_child_data.get('nodes', [])
    osdpl_node_overrides = dict()
    hosts_deploy_data = [node for node in bm_hosts_data if not any(item in node.get('si_roles', [])
                                                                   for item in ['nodeforscale',
                                                                                'child-scale-controller'])]
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
    page.get_by_role("link", name=f"{settings.TARGET_NAMESPACE}").click()
    page.get_by_role("button", name="Select").click()
    page.get_by_role("link", name=f"{settings.TARGET_CLUSTER}").click()

    # Wait cluster will get status ready
    def _wait_cluster_state_ready(page):
        page.reload()
        page.locator("[data-test-id=\"nav-switch-namespace\"]").click()
        page.locator(f"[data-test-id=\"{settings.TARGET_NAMESPACE}\"] div").nth(1).click()
        page.get_by_role("button", name="Select").click()
        page.get_by_role("link", name="Clusters").first.click()
        try:
            page.get_by_role("button", name="Refresh table").click()
            page.locator("[data-test-id=\"open-dropdown-icon\"]").click()
            page.get_by_text("Cluster info").click()
            expect(page.get_by_text("ready: false")).not_to_be_visible()
            expect(page.get_by_text("ready: true")).to_be_visible()
            page.locator("button").filter(has_text="Close").click()
            return True
        except Exception as e:
            page.locator("button").filter(has_text="Close").click()
            LOG.info(f'Cluster state is not in Expected state: Ready '
                     f'and failed with error: \n{e}')

    # Wait machines are Ready
    waiters.wait(lambda: _wait_cluster_state_ready(page),
                 timeout=3600,
                 interval=120)

    # WA for 2 known labels: SRIOV and hugepages
    cluster.apply_hotfix_labels(bm_nodes=hosts_deploy_data)
    cluster.apply_nodes_annotations(bm_nodes=hosts_deploy_data)
    # WA for osdpl overrides
    # openstack-controller requires ceph cluster
    for node in hosts_deploy_data:
        node_override = node.get('osdpl_node_override')
        if node_override:
            osdpl_node_overrides[node['name']] = node_override
    cluster.add_osdpl_overrides(osdpl_node_overrides)
    # Collecting artifacts
    ns = Manager(settings.KUBECONFIG_PATH).get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    cluster.store_k8s_artifacts()


def test_kaas_create_ceph_cluster(kaas_manager, ui_child_data):
    """Deploy Ceph cluster through API because it is not supported through UI
    """
    # Collecting artifacts
    ns = Manager(settings.KUBECONFIG_PATH).get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    bm_hosts_data = ui_child_data.get('nodes', [])
    child_config = ui_child_data["child_config"]

    if settings.MOSK_CHILD_DEPLOY_CEPH:
        nodes_data = cluster.check.get_ceph_nodes(child_config, ns, bm_hosts_data)
        ceph_cluster_pools = child_config.get('ceph_pools_config')

        ceph_rgw_config = None
        if settings.CEPH_DEPLOY_RGW:
            ceph_rgw_config = child_config.get('ceph_rgw_config')
            LOG.info(f"Adding Ceph rgw metadata {ceph_rgw_config}")

        cephfs_config = None
        if settings.CEPH_DEPLOY_MDS:
            cephfs_config = child_config.get('cephfs_config')
            LOG.info(f"Adding CephFS metadata {cephfs_config}")
            # Add mds roles only on nodes with mon roles
            for name, node_spec in nodes_data.items():
                if "mon" in node_spec.get('roles', []):
                    nodes_data[name]['roles'].append('mds')
                    LOG.info(f"Adding mds role to Ceph {name} node spec")

        ceph_rook_config = ui_child_data.get('ceph_rook_config')

        LOG.info('Create Ceph cluster')
        if not ui_child_data.get('kaascephcluster_raw'):
            ns.create_ceph_cluster(
                name=settings.CEPH_CLUSTER_NAME,
                cluster_name=settings.TARGET_CLUSTER,
                hdd_type=ui_child_data.get('cephClusterDeviceClass', 'hdd'),
                nodes_data=nodes_data,
                pools=ceph_cluster_pools,
                rgw=ceph_rgw_config,
                rook_config=ceph_rook_config,
                network=ui_child_data.get('ceph_network_config', None),
                tolerations=ui_child_data.get('ceph_tolerations', None),
                cephfs=cephfs_config)
        else:
            LOG.info("Create kaasCephCluster using raw methods")
            ns.create_ceph_cluster_raw(child_config['kaascephcluster_raw'])
        try:
            waiters.wait(lambda: cluster.check.check_child_ceph_cluster(check_health=True),
                         timeout=90 * 60, interval=60)
        except exceptions.TimeoutError as e:
            LOG.error("Tired waiting for ceph cluster correct status,"
                      "events from management cluster:")
            ns.print_events()
            raise e
    else:
        LOG.info('Ceph cluster create skipped due to there is no ceph-controller on cluster')
