#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os

import pytest

from si_tests import logger
from si_tests import settings
from si_tests.clients.openstack import OpenstackClient
from si_tests.managers import bootstrap_manager
from si_tests.managers.bootstrapcheck_manager import BoostrapCheckMgr
from si_tests.managers.kaas_manager import Manager
from si_tests.managers import si_config_manager
from si_tests.utils import utils

LOG = logger.logger


class TestCreateBootstrapv2Cluster():
    def test_bootstrapv2_create(self, openstack_client: OpenstackClient, show_step) -> None:
        """Bootstrap V2 create KinD cluster

         Scenario:
             1. Erase previous environment and seed node if management deployment
             2. Create a seed VM using heat template if required
             3. Prepare seed node to run bootstrap V2
             4. Run bootstrap V2 cluster
             5. Expose bv2 cluster for further use
             6. Check expected pods
         """
        # step 001: Erase previous environment and seed node if management deployment
        is_mgmt_deployment = False if os.environ.get("KUBECONFIG_ARTIFACT", None) else True
        if not settings.SEED_STANDALONE_EXTERNAL_IP:
            # TODO remove this if after propagate to all SI and Core CI jobs
            if is_mgmt_deployment:
                show_step(1)
                if openstack_client._current_stack:
                    seed_ip = openstack_client.get_seed_ip()
                    old_bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
                    old_bootstrap.step_001_erase_env_before()
                    LOG.info(f"Remove heat stack {settings.ENV_NAME}")
                    openstack_client.delete_stack()
                else:
                    LOG.warning("Can't erase environment due "
                                "seed node stack is absent")

            # step 002: Create a seed VM using heat template if required
            if is_mgmt_deployment and not openstack_client._current_stack:
                show_step(2)
                openstack_client.create_seed_node()
            else:
                LOG.warning("Won't create seed node due the node already exists")
                openstack_client.save_seed_node_info()

            seed_ip = openstack_client.get_seed_ip()

            # TODO <-- end
        else:
            seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP

        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        bootstrap.wait_for_seed_node_ssh()
        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        si_config.store_seed_ip(seed_ip)

        # step 003: Prepare seed node to run bootstrap V2
        if not settings.MCC_AIRGAP:
            show_step(3)
            if not is_mgmt_deployment:
                # this will erase everything and allow to fetch data again
                bootstrap.step_003_cleanup_bootstrapv2_seed_workspace()

            run_envs = bootstrap.step_003_prepare_seed_node(render_templates=False)
        else:
            run_envs = os.environ.copy()

        # step 004: Run bootstrap V2 cluster
        show_step(4)
        run_envs['KAAS_BOOTSTRAP_LOG_LVL'] = settings.KAAS_BOOTSTRAP_LOG_LVL
        if not is_mgmt_deployment:
            bootstrap.step_004_restore_preserved_bootstrapv2_metadata()
            run_envs['MGMT_KUBECONFIG'] = "kubeconfig"

        envs_string = utils.make_export_env_strting(run_envs)
        LOG.info(f"Current envs_string {envs_string}")
        if not bootstrap.setup_bootstrapv2_cluster(envs_string=envs_string):
            pytest.fail("Failed to setup bootstrapv2 cluster")

        # Access to KinD cluster using expose
        show_step(5)
        kind_kubeconfig_remote_path = bootstrap.get_remote_kubeconfig_path(settings.KUBECONFIG_KIND_PATH)

        is_kind_running = bootstrap.is_kind_running(kind_kubeconfig_remote_path)
        if is_kind_running:
            si_config_data = si_config.data
            run_on_remote = settings.RUN_ON_REMOTE or si_config_data.get(
                'run_on_remote', {}).get('RUN_ON_REMOTE', False)
            kind_kubeconfig_path = bootstrap.expose_kind_cluster(
                kind_kubeconfig_remote_path=kind_kubeconfig_remote_path,
                run_on_remote=run_on_remote)
            kind_manager = Manager(kubeconfig=kind_kubeconfig_path)
        else:
            raise Exception("KinD is not running")

        show_step(6)
        bvcheckmgr = BoostrapCheckMgr(kind_manager)
        bvcheckmgr.check_actual_expected_pods_in_boostrap()
