import yaml
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import templates
from si_tests.utils import utils
from si_tests.clients.openstack import OpenstackClient
from si_tests.managers.openstack_client_manager import OpenStackClientManager

LOG = logger.logger


@pytest.fixture(scope='session')
def openstack_client():
    cloud_name = settings.CLOUD_NAME
    clouds_yaml = yaml.load(
        templates.render_template(settings.CLOUDS_YAML_PATH),
        Loader=yaml.Loader)
    auth = clouds_yaml["clouds"][cloud_name]

    client = OpenstackClient(auth)
    return client


@pytest.fixture(scope='module')
def extra_offline_sg(openstack_client, squid_ip, mgmt_k8s_ip, keycloak_ip):
    # Creating additional SG
    sg_name = "kaas-sg-offline-" + utils.gen_random_string(8) + \
              settings.ENV_NAME
    sg = openstack_client.add_sg(sg_name, settings.ENV_NAME)
    for sg_rule in sg['security_group']['security_group_rules']:
        openstack_client.neutron.delete_security_group_rule(
            sg_rule['id']
        )
    # Adding main portion of rules
    for sg_rule in settings.OFFLINE_SG_RULES:
        openstack_client.add_sg_rule(id=sg['security_group']['id'],
                                     **sg_rule)
    # Adding squid, mgmt_k8s_ip and keycloak ips
    for ip in [squid_ip, mgmt_k8s_ip, keycloak_ip]:
        openstack_client.add_sg_rule(id=sg['security_group']['id'],
                                     direction="egress",
                                     port="443", proto="tcp",
                                     cidr=ip+"/32")
    return sg


@pytest.fixture(scope='module')
def check_sg(openstack_client):
    enabled_offline = settings.KAAS_OFFLINE_DEPLOYMENT
    if not enabled_offline:
        LOG.info("Offline deployment for cluster is disabled. "
                 "Nothing to check.")
        return
    elif enabled_offline and not settings.OFFLINE_SG_NAME:
        LOG.error("Offline deployment is enabled, "
                  "but corresponding security group is not set. "
                  "Please set OFFLINE_SG_NAME variable")
        raise Exception("OFFLINE_SG_NAME is unfilled but "
                        "KAAS_OFFLINE_DEPLOYMENT enabled")
    elif enabled_offline and settings.OFFLINE_SG_NAME:
        LOG.info("Offline mode is enabled. "
                 "Security group name: {0}".format(settings.OFFLINE_SG_NAME))
        LOG.info("Checking if security group is present in parent cloud.")
        sg = openstack_client.get_sg_by_name(settings.OFFLINE_SG_NAME)
        if not sg:
            raise Exception("Security group with name {0} "
                            "does not exist in cloud".format(
                             settings.OFFLINE_SG_NAME))
        return sg


@pytest.fixture(scope='session')
def openstack_client_manager():
    return OpenStackClientManager()
