#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import si_tests.utils.waiters as helpers

from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
from si_tests import logger

LOG = logger.logger


class K8sDaemonSet(K8sNamespacedResource):
    resource_type = 'daemonset'

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_daemon_set(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_daemon_set(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_daemon_set(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_daemon_set(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, **kwargs):
        self._manager.api.delete_namespaced_daemon_set(
            self.name, self.namespace, **kwargs)

    @property
    def desired_replicas(self):
        status = self.read().status
        if status:
            return status.desired_number_scheduled
        else:
            LOG.info("Status of daemonset {0}/{1} not populated yet".format(self.namespace, self.name))
            # Using None because it's incorrect to use 0 here like in sts, ds and rs. Daemonsets self-regulated object
            # in case of desired replicas count
            return None

    @property
    def ready_replicas(self):
        status = self.read().status
        if status:
            return status.number_ready
        else:
            LOG.info("Status of daemonset {0}/{1} not populated yet".format(self.namespace, self.name))
            # If status does not present - no ready replicas expected
            return 0

    @property
    def ready(self):
        obj = self.read().to_dict()
        if obj['status'].get('observed_generation', 0) or 0 >= obj['metadata']['generation']:
            if (obj['status'].get('desired_number_scheduled', 0) or 0) == 0:
                # NOTE(vsaienko): when do not have replicas (nodes where to schedule) treat as ready
                # in this case updated_number_scheduled is set to None explicitly
                return True
            if (obj['status'].get('updated_number_scheduled', 0) or 0) == obj['status'].get('number_ready', 0):
                return True

        LOG.debug("DaemonSet {}/{} \nstatus.observed_generation: '{}', metadata.generation: '{}'"
                  "\nstatus.updated_number_scheduled: '{}', expected: '{}'"
                  "".format(self.namespace, self.name,
                            obj['status']['observed_generation'], obj['metadata']['generation'],
                            obj['status'].get('updated_number_scheduled'), obj['status'].get('number_ready')))

        return False

    @property
    def pods(self):
        def _is_owned_by_uid(uid, owner_refs=None):
            owner_refs = owner_refs or []
            for owner in owner_refs:
                if owner.get("uid") == uid:
                    return True
            return False

        pod_labels = self.data["spec"]["selector"].get("match_labels", {})
        selector = ",".join([f"{k}={v}" for k, v in pod_labels.items()])
        pods_items = self._manager._cluster.pods.list(
            namespace=self.namespace, label_selector=selector
        )
        pods = [
            x
            for x in pods_items
            if _is_owned_by_uid(self.uid, x.data["metadata"]["owner_references"])
        ]
        return pods

    def restart(self):
        self._rollout_restart()
        if self.data["spec"]["update_strategy"]["type"] == "OnDelete":
            LOG.info(
                "DaemonSet {0}/{1} has update strategy 'OnDelete'. Deleting pods one by one.".format(
                    self.namespace, self.name
                )
            )
            while not self.ready:
                for pod in self.pods:
                    if self.generation != int(
                        pod.data["metadata"]["labels"].get(
                            "pod-template-generation", "0"
                        )
                    ):
                        LOG.info("Delete pod {0}".format(pod.name))
                        pod.delete()
                        helpers.wait(
                            lambda: self.ready_replicas == self.desired_replicas,
                            timeout=300,
                            interval=20,
                        )


class K8sDaemonSetManager(K8sBaseManager):
    resource_class = K8sDaemonSet

    @property
    def api(self):
        return self._cluster.api_apps

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_daemon_set(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_daemon_set_for_all_namespaces(**kwargs)
