import pytest

from si_tests import logger
from si_tests import settings
from si_tests.utils import update_child_clusterrelease_actions
from si_tests.utils import update_release_names
from si_tests.utils.utils import check_test_result
from si_tests.utils.workload_report import BMWorkloadDowntimeReport

LOG = logger.logger


update_release_names = list(update_release_names.generate_update_release_names())
is_update_test_failed = False


@pytest.fixture(scope='function', params=update_release_names,
                ids=[f"RELEASE={x}" for x in update_release_names])
def update_release_name(request):
    global is_update_test_failed
    # Check if the previous update steps failed
    if is_update_test_failed:
        msg = (f"Skip updating clusterrelease to {request.param} because "
               f"previous update step failed")
        LOG.info(msg)
        pytest.skip(msg)

    yield request.param

    # Check the result of the current step
    test_passed = (hasattr(request.node, 'rep_call') and
                   request.node.rep_call.passed)
    if not test_passed:
        is_update_test_failed = True


@pytest.fixture(scope='function')
def bm_mcc_workload_downtime_report(func_name, kaas_manager, update_release_name, request):
    if check_test_result(request, ['skipped']):
        LOG.warning("Test was skipped, so skip workload downtime report")
        yield
        return
    if not settings.MOSK_WORKLOAD_DOWNTIME_REPORT:
        LOG.info("MOSK_WORKLOAD_DOWNTIME_REPORT is disabled, skipping workload downtime report")
        yield
        return

    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    cr_before = child_cluster.clusterrelease_version

    if not child_cluster.k8sclient.openstackdeployment.available:
        LOG.info("Openstack deployment is not present, skipping workload downtime report")
        yield
        return

    if child_cluster.is_patchrelease_upgrade(clusterrelease_version_before=cr_before,
                                             clusterrelease_version_after=update_release_name):
        release_key = 'patch_release'
    else:
        release_key = 'major_release'

    report = BMWorkloadDowntimeReport(func_name, release_key, child_cluster)
    report.set_up()
    yield
    if check_test_result(request, ['skipped', 'failed']):
        LOG.warning("Test wasn't successful, so skip downtime report saving after test")
        return
    report.save()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_updated_child_cluster_description")
@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.usefixtures("introspect_PRODX_9696_target_cluster")
@pytest.mark.usefixtures("introspect_ceph_child")
@pytest.mark.usefixtures("bm_mcc_workload_downtime_report")
@pytest.mark.usefixtures('mcc_per_node_workload_check_after_test')
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('collect_machines_timestamps')
@pytest.mark.usefixtures('introspect_child_lcm_operation_stuck')
@pytest.mark.usefixtures('introspect_machines_stages')
@pytest.mark.usefixtures('log_start_end_timestamps')
@pytest.mark.usefixtures("check_ceph_keyrings")
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures('mcc_loadtest_kibana')
@pytest.mark.usefixtures('mcc_loadtest_alertmanager')
@pytest.mark.usefixtures("runtime_restart_checker")
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
@pytest.mark.usefixtures('introspect_no_PRODX_51933_after_lcm')
@pytest.mark.usefixtures('post_action_update_coredns')
def test_update_child_via_update_plan_steps_sequentally(kaas_manager, update_release_name, _):
    """Update child cluster via update plan, step by step"""

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    till_step_name = ''
    if update_release_name == update_release_names[-1]:
        # Use KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME only for the latest update test during current test run
        till_step_name = settings.KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME

    update_actions = update_child_clusterrelease_actions.UpdateChildClusterreleaseActions(child_cluster)
    update_actions.pre_update(update_release_name)
    steps_to_update = update_actions.update_plan_get_steps_names_to_process(
        till_step_name)

    assert steps_to_update, (f"No steps found in UpdatePlan for Cluster {namespace_name}/{cluster_name} "
                             f"target version {update_actions.update_info.target_cr_version}")

    steps_estimated_list = update_actions.create_estimated_time_list_from_update_plan(
        update_actions.update_info.target_cr_version)

    if till_step_name:
        # Check that the specified step name is actually exists in the update plan
        assert (any(up_step.get('id') == till_step_name or
                    up_step.get('name') == till_step_name
                    for up_step in steps_to_update)
                ), (f"KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME contains step that is missing in the "
                    f"UpdatePlan for Cluster {namespace_name}/{cluster_name} target version "
                    f"{update_actions.update_info.target_cr_version}")

    step_key = "name"  # <2.28
    if all("id" in up_step for up_step in steps_to_update):
        step_key = "id"   # >= 2.28

    # TODO: since 2.31 it will be done automatically, required only for 2.30
    # Delete KaaSCephCluster if required
    if child_cluster.workaround.skip_kaascephcluster_usage():
        child_cluster.remove_kaascephcluster()
        child_cluster.check.wait_kaascephcluster_removed()

    #######################################
    #  Start step by step cluster update  #
    #######################################
    for up_step in steps_to_update:

        step_id = up_step[step_key]
        LOG.banner(f"Process update plan step: {step_id}")

        child_cluster.update_plan_set_steps_commence(
            update_actions.update_info.target_cr_version, steps_id=[step_id], commence=True, step_key=step_key)

        # check that release is changed right after update plan is triggered
        child_cluster.check.check_cluster_release_in_spec(update_actions.update_info.update_release_name)

        child_cluster.update_plan_steps_wait_for_completion(
            update_actions.update_info.target_cr_version, steps_id=[step_id], step_key=step_key)

        if till_step_name and (up_step != steps_to_update[-1]):
            if (up_step.get('id') == till_step_name
                    or up_step.get('name') == till_step_name):
                LOG.banner(f"Reached the target update plan step "
                           f"'{settings.KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME}', interrupt the update test")
                LOG.info(f"Step {settings.KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME} is not the latest step, "
                         f"skipping post-update checks. "
                         f"Ensure that the test will be run one more time to complete the update.")
                update_actions.base_readiness_check(os_transitional_replicas=True)
                return

    LOG.banner("********* All Cluster update plan steps completed *************")

    child_cluster.check.check_lcmmachine_releases("control", update_actions.update_info.target_cr_version)
    child_cluster.check.check_lcmmachine_releases("worker", update_actions.update_info.target_cr_version)

    update_actions.compare_estimated_time_list_with_duration_time(
        steps_estimated_list=steps_estimated_list,
        target_cr_version=update_actions.update_info.target_cr_version)

    child_cluster.check.check_cluster_release(update_actions.update_info.update_release_name)
    child_cluster.check.check_update_finished(timeout=settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT, interval=120)

    #################################
    #  Cluster update is completed  #
    #################################
    update_actions.base_readiness_check()
    update_actions.post_update()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_updated_child_cluster_description")
@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.usefixtures("introspect_PRODX_9696_target_cluster")
@pytest.mark.usefixtures("introspect_ceph_child")
@pytest.mark.usefixtures("bm_mcc_workload_downtime_report")
@pytest.mark.usefixtures('mcc_per_node_workload_check_after_test')
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('collect_machines_timestamps')
@pytest.mark.usefixtures('introspect_child_lcm_operation_stuck')
@pytest.mark.usefixtures('introspect_machines_stages')
@pytest.mark.usefixtures('log_start_end_timestamps')
@pytest.mark.usefixtures("check_ceph_keyrings")
@pytest.mark.usefixtures("runtime_restart_checker")
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
@pytest.mark.usefixtures('post_action_update_coredns')
def test_update_child_via_update_plan_steps_bulk(kaas_manager, update_release_name, _):
    """Update child cluster via update plan, enable all steps at once"""

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    till_step_name = ''
    if update_release_name == update_release_names[-1]:
        # Use KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME only for the latest update test during current test run
        till_step_name = settings.KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME

    update_actions = update_child_clusterrelease_actions.UpdateChildClusterreleaseActions(child_cluster)
    update_actions.pre_update(update_release_name)
    steps_to_update = update_actions.update_plan_get_steps_names_to_process(till_step_name)

    # In case when all steps are completed (as example in previous test run), then
    # we should perform post actions and checks
    if not steps_to_update:
        LOG.banner(f"No steps found in UpdatePlan for Cluster {namespace_name}/{cluster_name} "
                   f"target version {update_actions.update_info.target_cr_version}. "
                   f"Looks like cluster has been updated, continue to check cluster and perform post actions")
        child_cluster.check.check_cluster_release_in_spec(update_actions.update_info.update_release_name)
        child_cluster.check.check_lcmmachine_releases("control", update_actions.update_info.target_cr_version)
        child_cluster.check.check_lcmmachine_releases("worker", update_actions.update_info.target_cr_version)

        child_cluster.check.check_cluster_release(update_actions.update_info.update_release_name)
        child_cluster.check.check_update_finished(timeout=settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT, interval=120)

        update_actions.base_readiness_check()
        update_actions.post_update()
        return

    steps_estimated_list = update_actions.create_estimated_time_list_from_update_plan(
        update_actions.update_info.target_cr_version)

    if till_step_name:
        # Check that the specified step name is actually exists in the update plan
        assert (any(up_step.get('id') == till_step_name or
                    up_step.get('name') == till_step_name
                    for up_step in steps_to_update)
                ), (f"KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME contains step that is missing in the "
                    f"UpdatePlan for Cluster {namespace_name}/{cluster_name} target version "
                    f"{update_actions.update_info.target_cr_version}")

    step_key = "name"  # <2.28
    if all("id" in up_step for up_step in steps_to_update):
        step_key = "id"   # >= 2.28

    # TODO: since 2.31 it will be done automatically, required only for 2.30
    # Delete KaaSCephCluster if required
    if child_cluster.workaround.skip_kaascephcluster_usage():
        child_cluster.remove_kaascephcluster()
        child_cluster.check.wait_kaascephcluster_removed()

    #####################################
    #  Start bulk steps cluster update  #
    #####################################
    steps_id = []
    for up_step in steps_to_update:
        steps_id.append(up_step[step_key])

        if till_step_name:
            if (up_step.get('id') == till_step_name
                    or up_step.get('name') == till_step_name):
                break

    LOG.banner(f"Process update plan bulk steps: {steps_id}")
    child_cluster.update_plan_set_steps_commence(
        update_actions.update_info.target_cr_version, steps_id=steps_id, commence=True, step_key=step_key)

    # check that release is changed right after update plan is triggered
    child_cluster.check.check_cluster_release_in_spec(update_actions.update_info.update_release_name)

    child_cluster.update_plan_steps_wait_for_completion(
        update_actions.update_info.target_cr_version, steps_id=steps_id, step_key=step_key)

    if till_step_name and len(steps_to_update) != len(steps_id):
        LOG.banner(f"Reached the target update plan step "
                   f"'{settings.KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME}', interrupt the update test")
        LOG.info(f"Step {settings.KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME} is not the latest step, "
                 f"skipping post-update checks. "
                 f"Ensure that the test will be run one more time to complete the update.")
        update_actions.base_readiness_check(os_transitional_replicas=True)
        return

    LOG.banner("********* All Cluster update plan steps completed *************")

    child_cluster.check.check_lcmmachine_releases("control", update_actions.update_info.target_cr_version)
    child_cluster.check.check_lcmmachine_releases("worker", update_actions.update_info.target_cr_version)

    update_actions.compare_estimated_time_list_with_duration_time(
        steps_estimated_list=steps_estimated_list,
        target_cr_version=update_actions.update_info.target_cr_version)

    child_cluster.check.check_cluster_release(update_actions.update_info.update_release_name)
    child_cluster.check.check_update_finished(timeout=settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT, interval=120)

    #################################
    #  Cluster update is completed  #
    #################################
    update_actions.base_readiness_check()
    update_actions.post_update()
