import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils


LOG = logger.logger


@pytest.mark.usefixtures("mariadb_backup_restore_cleanup_helper")
def test_mcc_mariadb_backup_restore(kaas_manager, show_step):
    """Test MCC MariaDB (Part of Keycloak stack) ability to backup and restore data

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Create new MCC user and verify it
        3. Configure backup schedule of MCC MariaDB
        4. Wait for settings applied
        5. Wait for backup creation started and backup job finished
        6. Remove user and check it
        7. Restore backup of MCC MariaDB
        8. Check that user restored
    """
    # constant variables
    backup_cronjob_name = "mariadb-phy-backup"
    restore_job_prefix = "mariadb-phy-restore"
    mcc_maria_namespace = 'kaas'

    show_step(1)
    cluster = kaas_manager.get_mgmt_cluster()
    LOG.info(f"Target cluster: {cluster.namespace}/{cluster.name}")
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    LOG.info('Preflight check passed')

    show_step(2)
    username = utils.gen_random_string(6)
    password = utils.gen_random_string(10)
    email = f"{utils.gen_random_string(6)}@{utils.gen_random_string(4)}.org"
    LOG.info(f"Creating user {username} with email {email}")
    _, iamusername = kaas_manager.keycloak.create_user_and_wait(username=username, password=password, email=email)
    LOG.info(f"User {username} created and appeared in IAM")

    show_step(3)
    backup_cronjob = cluster.k8sclient.cronjobs.get(name=backup_cronjob_name, namespace=mcc_maria_namespace)
    schedule = utils.get_schedule_for_cronjob(settings.IAM_APPROX_RECONFIGURE_TIME)
    kaas_manager.keycloak.configure_keyclock_mariadb_backup(schedule_time=schedule)

    show_step(4)
    LOG.info('Waiting for backup cronjob to be updated')
    backup_cronjob.wait_for_new_generation()
    # iam-keycloak pods will be re-created, so we need to make sure that cluster is ready since it will lost Ready flag
    cluster.check.check_cluster_readiness()
    assert backup_cronjob.suspended is False, (f"Cronjob {backup_cronjob.namespace}/{backup_cronjob.name} "
                                               f"should not be suspended")
    assert backup_cronjob.schedule == schedule, (f"Cronjob {backup_cronjob.namespace}/{backup_cronjob.name} "
                                                 f"schedule haven't been changed but should")

    show_step(5)
    LOG.info(f"Waiting for job with prefix {backup_cronjob_name} to appear")
    cluster.wait_for_job_to_appear(name_prefix=backup_cronjob_name,
                                   namespace=mcc_maria_namespace,
                                   timeout=settings.IAM_APPROX_RECONFIGURE_TIME)
    backupjob = cluster.k8sclient.jobs.list(name_prefix=backup_cronjob_name, namespace=mcc_maria_namespace)[0]
    LOG.info(f"Job {backupjob.namespace}/{backupjob.name} found. Waiting job to finish")
    backupjob.wait_finished()
    assert backupjob.succeeded, f"Job {backupjob.namespace}/{backupjob.name} failed"
    LOG.info("MariaDB Keycloak backup created")

    show_step(6)
    LOG.info(f"Deleting user {username}")
    kaas_manager.keycloak.delete_user_and_wait(username=username)
    LOG.info(f"User {username} deleted and removed from IAM")

    show_step(7)
    LOG.info('Spawn a pod to mount backup folder and get a backup name')
    # Backup job have only one container, so get needed image from there
    maria_image = backup_cronjob.data['spec']['job_template']['spec']['template']['spec']['containers'][0]['image']
    # Assuming only one backup exists in test so default cmd should not be changed
    backup_name = kaas_manager.keycloak.mariadb_volume_mgmt(
        namespace=mcc_maria_namespace,
        image=maria_image,
        cmd='ls /var/backup/base').strip()
    LOG.info(f"Will restore backup with name {backup_name}")
    kaas_manager.keycloak.restore_mariadb_keycloak_backup(backup_name=backup_name)
    LOG.info('Restore configured. Waiting for restore job to appear')
    cluster.wait_for_job_to_appear(name_prefix=restore_job_prefix, namespace=mcc_maria_namespace)
    restorejob = cluster.k8sclient.jobs.list(name_prefix=restore_job_prefix, namespace=mcc_maria_namespace)[0]
    LOG.info(f"Job {restorejob.namespace}/{restorejob.name} found. Waiting for finish")
    restorejob.wait_finished(timeout=settings.MCC_MARIADB_RESTORE_JOB_TIMEOUT)
    assert restorejob.succeeded, f"Job {restorejob.namespace}/{restorejob.name} failed"
    cluster.check.check_cluster_readiness()
    LOG.info("MariaDB backup restoration finished")

    show_step(8)
    LOG.info(f"Waiting for iam username {iamusername} appear in k8s resources")
    kaas_manager.keycloak.wait_iamuser_created(iamusername)
    LOG.info(f"Iamuser {iamusername} found. Backup restoration successful.")
