import random
import string
import pytest

from si_tests import logger
from si_tests.lcm.openstack_lcm.policies import Policy
from si_tests.utils.utils import verify

LOG = logger.logger

letters = string.ascii_lowercase
# creating a unique part of a server name to delete it after the test
name_postfix = ''.join(random.choice(letters) for i in range(5))


@pytest.fixture(scope='function')
def delete_resources(openstack_client_manager):
    """Function to delete the servers that were created during the test"""

    # executing test_crud_operations_server_policy
    yield

    # deletion of servers that have been used for the test
    server_list = openstack_client_manager.server.list([])
    for server in server_list:
        if name_postfix in server.get('Name'):
            openstack_client_manager.server.delete([server.get('ID')])


@pytest.mark.usefixtures('delete_resources')
def test_crud_operations_server_policy(openstack_client_manager, policy_user):
    """Test CRUD operations for server policy
       https://mirantis.testrail.com/index.php?/cases/view/4963696

       Parameters required for test execution:
         - KUBECONFIG

       Setup environment:
       Create user with role member
    """
    user_name, password = policy_user
    os_service = 'nova'
    os_pod = 'nova-api-osapi'
    policy = Policy(os_service, os_pod)
    check_policy_file = ['/bin/sh', '-c', ' cat /etc/nova/policy.d/02-custom.yaml']
    cirros_image_name = openstack_client_manager.cirros_image_name

    def contains_http_exceptions(text, does_not_contain=False):
        http_exceptions = ['HTTP 403',
                           'Exception: 403']
        occurance = any([http_exception in text for http_exception in http_exceptions])
        if does_not_contain:
            return not occurance
        return occurance

    try:
        # checking than the policy defining file is not changed before the test
        get_policy_file_before_update = ['/bin/sh', '-c',
                                         r' grep -irn "os_compute_api:servers:index:\|os_compute_api:servers:create:"'
                                         ' /etc/nova/policy.d/02-custom.yaml']
        policy.check_policy_absent_before_update(get_policy_file_before_update)

        LOG.info('Step 1: Checking that policies are absent. Admin and user can create a server and see server'
                 ' details')
        step = 1

        # creating a server with the admin credentials
        server = openstack_client_manager.server.create([f'test_server_no_policy_admin_{name_postfix}',
                                                         '--os-compute-api-version', '2.37', '--image',
                                                         cirros_image_name, '--flavor',
                                                         'm1.tiny', '--nic', 'none'], combined_output=True)
        verify(contains_http_exceptions(server["stderr"], does_not_contain=True),
               f'[Step {step}]: Server was not created with admin credentials without policies',
               f'[Step {step}]: Policies are absent. Admin can create a new server')
        assert (
            server["stdout"]["name"] == f'test_server_no_policy_admin_{name_postfix}'
            and server["stdout"]["status"] == 'BUILD'
        )

        # checking the possibility to get the server details with the admin credentials
        server_list = openstack_client_manager.server.list([])
        verify(f'test_server_no_policy_admin_{name_postfix}' in str(server_list),
               f'[Step {step}]: It was impossible to get server list with admin credentials without policies',
               f'[Step {step}]: Policies are absent. Admin can obtain the server list')

        # creating a server with the user credentials
        server = openstack_client_manager.server.create([f'test_server_no_policy_user_{name_postfix}',
                                                         '--os-compute-api-version', '2.37', '--image',
                                                         cirros_image_name, '--flavor',
                                                         'm1.tiny', '--nic', 'none', f'--os-username={user_name}',
                                                         f'--os-password={password}'], combined_output=True)
        verify(contains_http_exceptions(server["stderr"], does_not_contain=True),
               f'[Step {step}]: Server was not created with user credentials without policies',
               f'[Step {step}]: Policies are absent. User can create a new server')
        assert (
            server["stdout"]["name"] == f'test_server_no_policy_user_{name_postfix}'
            and server["stdout"]["status"] == 'BUILD'
        )

        # checking the possibility to get the server details with the user credentials
        server_list = openstack_client_manager.server.list(
            [f'--os-username={user_name}', f'--os-password={password}'])
        verify(f'test_server_no_policy_user_{name_postfix}' in str(server_list),
               f'[Step {step}]: It was impossible to get server list with user credentials without policies',
               f'[Step {step}]: Policies are absent. User can get the server list')

        LOG.info('Step 2: Adding new rules to OSDPL file')
        step = 2

        # changing the policy defining file, no one will be able to create a server, no one will be able to watch
        # server details
        policy_add_role = {'os_compute_api:servers:detail': '!', 'os_compute_api:servers:create': '!'}
        policy.update_policy_osdpl(policy_add_role, step)

        LOG.info('Step 3: Checking that the policies were added to Nova API-OSAPI pod')
        step = 3

        # checking that the policies were updated
        policy.wait_policy_updated_on_pod(check_policy_file, policy_add_role, step)

        LOG.info('Step 4: The policies were added. Checking that admin and user cannot create a new server and cannot'
                 ' obtain the server list')
        step = 4

        # checking the impossibility to create a server with the admin credentials
        error_output = openstack_client_manager.server.create(
            [f'test_server_policy_not_admin_{name_postfix}', '--os-compute-api-version', '2.37', '--image',
             cirros_image_name, '--flavor', 'm1.tiny', '--nic', 'none'], combined_output=True)
        verify(contains_http_exceptions(error_output["stderr"]),
               f'[Step {step}]: Server was created with admin credentials and with "!" policies',
               f'[Step {step}]: Policies disallow everyone to create a new server. Admin cannot create a new server as'
               f' well')

        # checking the impossibility to get server details with the admin credentials
        error_output = openstack_client_manager.server.list([], combined_output=True)
        verify(contains_http_exceptions(error_output["stderr"]),
               f'[Step {step}]: It was possible to get server list with admin credentials and with "!" policies',
               f'[Step {step}]: Policies disallow everyone to get the server list. Admin cannot obtain the server list'
               f'as well')

        # checking the impossibility of server creating with the user credentials
        error_output = openstack_client_manager.server.create(
            [f'test_server_policy_not_user_{name_postfix}', '--os-compute-api-version', '2.37', '--image',
             cirros_image_name, '--flavor', 'm1.tiny', '--nic', 'none',
             f'--os-username={user_name}', f'--os-password={password}'], combined_output=True)
        verify(contains_http_exceptions(error_output["stderr"]),
               f'[Step {step}]: Server was created with user credentials and with "!" policies',
               f'[Step {step}]: Policies disallow everyone to create a new server. User cannot create a new server as'
               ' well')

        # checking the impossibility to get server details with the user credentials
        error_output = openstack_client_manager.server.list(
            [f'--os-username={user_name}', f'--os-password={password}'], combined_output=True)
        verify(contains_http_exceptions(error_output["stderr"]),
               f'[Step {step}]: It was possible to get server list with user credentials and with "!" policies',
               f'[Step {step}]: Policies disallow everyone to get the server list. User cannot obtain the server list'
               'as well')

        LOG.info('Step 5: Updating policies. Checking that the policies were added to Nova API-OSAPI pod')
        step = 5

        # changing the policy defining file, only admin will be able to create a server and see the server list
        policy_update_role = {'os_compute_api:servers:detail': 'rule:admin_api',
                              'os_compute_api:servers:create': 'rule:admin_api'}
        policy.update_policy_osdpl(policy_update_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, policy_update_role, step)

        LOG.info('Step 6: The policies were updated. Checking that only admin can create a new server and obtain the'
                 ' server list')
        step = 6

        # creating a server with the admin credentials
        server = openstack_client_manager.server.create([f'test_server_policy_admin_only_admin_{name_postfix}',
                                                         '--os-compute-api-version', '2.37', '--image',
                                                         cirros_image_name, '--flavor',
                                                         'm1.tiny', '--nic', 'none'], combined_output=True)
        verify(contains_http_exceptions(server["stderr"], does_not_contain=True),
               f'[Step {step}]: Server was not created with admin credentials and with "rule:admin_api" policies',
               f'[Step {step}]: Policies allow admin to create a new server, admin can create a new server')
        assert (
            server["stdout"]["name"] == f'test_server_policy_admin_only_admin_{name_postfix}'
            and server["stdout"]["status"] == 'BUILD'
        )

        # checking the possibility to get server details with the admin credentials
        server_list = openstack_client_manager.server.list([])
        verify(f'test_server_policy_admin_only_admin_{name_postfix}' in str(server_list),
               f'[Step {step}]: It was impossible to get server list with admin credentials and with'
               ' "rule:admin_api" policies',
               f'[Step {step}]: Policies allow admin to get the server list, admin can obtain the server list')

        # checking the impossibility of server creating with the user credentials
        error_output = openstack_client_manager.server.create(
            [f'test_server_policy_admin_only_user_{name_postfix}', '--os-compute-api-version', '2.37',
             '--image', cirros_image_name, '--flavor', 'm1.tiny', '--nic', 'none',
             f'--os-username={user_name}', f'--os-password={password}'], combined_output=True)
        verify(contains_http_exceptions(error_output["stderr"]),
               f'[Step {step}]: Server was created with user credentials and with "rule:admin_api" policies',
               f'[Step {step}]: Policies allow only admin to create a new server, user cannot create a new server')

        # checking the impossibility to get server details with the user credentials
        error_output = openstack_client_manager.server.list(
            [f'--os-username={user_name}', f'--os-password={password}'], combined_output=True)
        verify(contains_http_exceptions(error_output["stderr"]),
               f'[Step {step}]: It was possible to get server list with user credentials and with'
               ' "rule:admin_api" policies',
               f'[Step {step}]: Policies allow only admin to get the server list, user cannot obtain the server list')

    finally:
        try:
            LOG.info('Step 7: Deleting policies. Checking that the policies were deleted and'
                     ' the result is the same as in the first step')
            step = 7

            # changing the policy defining file to the pre-test condition
            policy_not_role = None
            policy.update_policy_osdpl(policy_not_role, step)
            policy.wait_policy_updated_on_pod(check_policy_file, {}, step)

            # creating a server with the admin credentials
            server = openstack_client_manager.server.create(
                [f'test_server_policy_delete_admin_{name_postfix}', '--os-compute-api-version', '2.37',
                 '--image', cirros_image_name, '--flavor', 'm1.tiny', '--nic', 'none'],
                combined_output=True)
            verify(contains_http_exceptions(server["stderr"], does_not_contain=True),
                   f'[Step {step}]: Server was not created with admin credentials without policies',
                   f'[Step {step}]: Policies are absent. Admin can create a new server')
            assert (
                server["stdout"]["name"] == f'test_server_policy_delete_admin_{name_postfix}'
                and server["stdout"]["status"] == 'BUILD'
            )

            # checking the possibility to get server details with the admin credentials
            server_list = openstack_client_manager.server.list([])
            verify(f'test_server_policy_delete_admin_{name_postfix}' in str(server_list),
                   f'[Step {step}]: It was impossible to get server list with admin credentials without policies',
                   f'[Step {step}]: Policies are absent. Admin can obtain the server list')

            # creating a server with the user credentials
            server = openstack_client_manager.server.create(
                [f'test_server_policy_delete_user_{name_postfix}', '--os-compute-api-version', '2.37',
                 '--image', cirros_image_name, '--flavor', 'm1.tiny', '--nic', 'none',
                 f'--os-username={user_name}', f'--os-password={password}'], combined_output=True)
            verify(contains_http_exceptions(server["stderr"], does_not_contain=True),
                   f'[Step {step}]: Server was not created with user credentials without policies',
                   f'[Step {step}]: Policies are absent. User can create a new server')
            assert (
                server["stdout"]["name"] == f'test_server_policy_delete_user_{name_postfix}'
                and server["stdout"]["status"] == 'BUILD'
            )

            # checking the possibility to get server details with the user credentials
            server_list = openstack_client_manager.server.list(
                [f'--os-username={user_name}', f'--os-password={password}'], yaml_output=False)
            verify(f'test_server_policy_delete_user_{name_postfix}' in str(server_list),
                   f'[Step {step}]: It was impossible to get server list with user credentials without policies',
                   f'[Step {step}]: Policies are absent. User can get the server list')

        finally:
            LOG.info('Deleting created resources')
