#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import settings
from si_tests import logger

from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager

LOG = logger.logger


class TestProvisionMgmCluster():

    @pytest.mark.usefixtures("introspect_MKE_7914",
                             "introspect_PRODX_9238",
                             "introspect_openstack_seed_management_deploy_objects")
    def test_bootstrap_kaas_on_openstack_vm(self, openstack_client, check_sg,
                                            show_step):
        """Bootstrap KaaS on OpenStack VMs

        Scenario:
            1. Erase previous KaaS environment and seed node, if exists
            2. Create a seed VM using heat template
            3. Prepare seed node to run KaaS bootstrap
            4. Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
            5. Check bootstrap templates for management cluster
            6. Bootstrap KaaS seed and managed clusters
            7. Create default MCC users, if requested
            8. Check versions
            9. Wait for all pods to be Running and Ready
            10. Check that all expected pods exists
            11. Download bootstrap artifacts
            12. Store seed IP in si_config
            13. Check bootFromVolume if enable
            14. Check LDAP integration if enabled
            15. Erase created KaaS environment
        """

        endpoints = openstack_client.get_os_cloud_endpoints()

        # step 001
        if settings.KEEP_ENV_BEFORE:
            LOG.info("Skip erase due KEEP_ENV_BEFORE flag is setted")
        else:
            if openstack_client._current_stack:
                # Erase the stack with the seed node
                show_step(1)
                seed_ip = openstack_client.get_seed_ip()
                old_bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
                old_bootstrap.step_001_erase_env_before()
                LOG.info(f"Remove heat stack {settings.ENV_NAME}")
                openstack_client.delete_stack()
            else:
                LOG.warning("Can't erase environment due "
                            "seed node stack is absent")

        # Create a stack with a seed node
        show_step(2)
        openstack_client.create_seed_node()
        seed_ip = openstack_client.get_seed_ip()
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        bootstrap.wait_for_seed_node_ssh()

        # step 003
        show_step(3)
        bootstrap.step_003_prepare_seed_node(endpoints=endpoints)

        # step 004
        show_step(4)
        bootstrap.add_credentials_data_to_si_config()

        # step 005
        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
            if not settings.KAAS_CUSTOM_LMA_LOGGING_ENABLED:
                bootstrap.disable_lma_logging()
            if settings.CORE_KEYCLOAK_LDAP_ENABLED:
                bootstrap.enable_keycloak_ldap()
            if settings.CORE_KAAS_NTP_ENABLED:
                bootstrap.set_ntp_servers()
        else:
            show_step(5)
            bootstrap.check_templates()

        # step 006
        show_step(6)
        seed_instance = openstack_client.get_seed_instance()
        bootstrap.step_004_deploy_kaas_cluster(endpoints=endpoints, seed_instance=seed_instance)

        # step 007
        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(7)
            bootstrap.step_004_create_default_mcc_users()

        if settings.MCC_SCHEDULED_UPGRADES:
            LOG.info("Update mcc-upgrade crd for feature upgrade scheduling")
            bootstrap.step_006_update_mcc_upgrade_crd()

        # step 008
        show_step(8)
        bootstrap.step_005_check_versions()

        # step 009
        show_step(9)
        bootstrap.step_006_wait_for_pods()

        # step 010
        show_step(10)
        bootstrap.step_006_postdeployment_checks()

        # step 011
        show_step(11)
        bootstrap.step_007_download_bootstrap_artifacts()

        # step 012
        show_step(12)
        kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig"
        if not os.path.isfile(kubeconfig_path):
            raise Exception(f"KUBECONFIG {kubeconfig_path} not found in"
                            f" artifacts!")
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        kaas_manager.si_config.store_seed_ip(seed_ip)

        # step 013
        if settings.OPENSTACK_BOOT_FROM_VOLUME:
            show_step(13)
            bootstrap.check_boot_from_volume(openstack_client,
                                             boot_volume_size=settings.OPENSTACK_BOOT_VOLUME_SIZE)
        # step 014
        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(14)
            bootstrap.check_keystone_ldap_integration()

        # step 015
        if settings.KEEP_ENV_AFTER:
            LOG.warning("Skip erase due KEEP_ENV_AFTER flag is set")
        else:
            # Erase kaas environment after bootstrap is passed
            show_step(15)
            bootstrap.step_007_erase_env_after()
