#    Copyright 2020 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager

LOG = logger.logger


class TestNegativeProvisionAwsMgmCluster():

    def test_bootstrap_kaas_on_aws_negative(self, openstack_client, show_step):
        """Bootstrap KaaS on OpenStack VMs

        Scenario:
            1. Erase previous KaaS environment and seed node, if exists
            2. Create a seed VM using heat template
            3. Prepare seed node to run KaaS bootstrap on aws.
               Set wrong KAAS_MANAGEMENT_CLUSTER_AWS_IMAGE
            4. Check aws bootstrap templates for management cluster
            5. Try to bootstrap KaaS
            6. Revert KAAS_MANAGEMENT_CLUSTER_AWS_IMAGE,
               set wrong KAAS_MANAGEMENT_CLUSTER_AWS_IMAGE.
            7. Rerun deployment.
            8. Erase created KaaS environment and seed node
        """

        init_image = os.environ['KAAS_MANAGEMENT_CLUSTER_AWS_IMAGE']

        # step 001
        if settings.KEEP_ENV_BEFORE:
            LOG.info("Skip erase due KEEP_ENV_BEFORE flag is setted")
        else:
            if openstack_client._current_stack:
                # Erase the stack with the seed node
                show_step(1)
                seed_ip = openstack_client.get_seed_ip()
                old_bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
                old_bootstrap.step_001_erase_env_before()
                LOG.info(f"Remove heat stack {settings.ENV_NAME}")
                openstack_client.delete_stack()
            else:
                LOG.warning("Can't erase environment due "
                            "seed node stack is absent")

        # Create a stack with a seed node
        show_step(2)
        openstack_client.create_seed_node()
        seed_ip = openstack_client.get_seed_ip()
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        bootstrap.wait_for_seed_node_ssh()

        # step 003
        show_step(3)
        os.environ['KAAS_MANAGEMENT_CLUSTER_AWS_IMAGE'] = \
            'nonexisting_image'
        bootstrap.step_003_prepare_seed_node()

        remote = bootstrap.remote_seed()

        # step 004
        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
        else:
            show_step(4)
            bootstrap.check_templates()

        # step 005
        show_step(5)
        try:
            seed_instance = openstack_client.get_seed_instance()
            bootstrap.step_004_deploy_kaas_cluster(seed_instance=seed_instance)
        except Exception as e:
            LOG.error(e)
            if 'no return code!' in str(e):
                pytest.fail("Bootstrap script returned unexpected success")
            # Check error when it is implemented
            # assert 'wrong image' in e.stderr, \
            #    "Unknown error occurred"
        else:
            pytest.fail("Bootstrap script returned unexpected success")

        # step 006
        show_step(6)
        os.environ['KAAS_MANAGEMENT_CLUSTER_AWS_IMAGE'] = init_image
        os.environ['KAAS_MANAGEMENT_CLUSTER_AWS_FLAVOR'] = 'nonexisting_flavor'
        bootstrap.step_003_prepare_seed_node_templates()

        remote.execute('docker rm -f clusterapi-control-plane')
        remote.execute('rm -fr ./kaas-bootstrap/tls/kaas.local')

        # step 007
        show_step(7)
        try:
            seed_instance = openstack_client.get_seed_instance()
            bootstrap.step_004_deploy_kaas_cluster(seed_instance=seed_instance)
        except Exception as e:
            LOG.error(e)
            if 'no return code!' in str(e):
                pytest.fail("Bootstrap script returned unexpected success")
            # Check error when it is implemented
            # assert 'wrong flavor' in e.stderr, \
            #    "Unknown error occurred"
        else:
            pytest.fail("Bootstrap script returned unexpected success")

        # step 008
        show_step(8)
        bootstrap.step_001_erase_env_before()
        LOG.info(f"Remove heat stack {settings.ENV_NAME}")
        openstack_client.delete_stack()
