import datetime
import functools
import re
from kubernetes.client.rest import ApiException
from retry import retry
from paramiko import ssh_exception
import tempfile
import yaml
import os

from si_tests.settings import KUBECONFIG_PATH
from si_tests import logger
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.tungstenfafric_manager import TFManager
from si_tests.deployments.utils import (
    commons,
    kubectl_utils,
    namespace,
    wait_utils,
)
from si_tests.clients.openstack_client.resources import OpenStackResource
from si_tests.clients.openstack_client.resource_providers import VMProvider

LOG = logger.logger


class OpenStackClientManager:
    pattern = re.compile(r'(?<!^)(?=[A-Z])')

    def __init__(self, kubeconfig=KUBECONFIG_PATH, os_cloud="admin"):
        self._os_cloud = os_cloud
        self.os_manager = OpenStackManager(kubeconfig=kubeconfig)
        self.tf_manager = TFManager(kubeconfig=kubeconfig)
        for pod in self.os_manager.api.pods.list_starts_with(
                'keystone-client', self.os_manager.openstack_namespace
        ):
            pod_obj = pod.read()
            if pod_obj.status.phase == "Running":
                self.client = pod
                break
        else:
            msg = "There are no keystone-client pod in Running state on env"
            LOG.error(msg)
            raise Exception(msg)

        self.kubeconfig = kubeconfig

        for sub_cls in OpenStackResource.__subclasses__():
            name = self.pattern.sub('_', sub_cls.__name__).lower()
            setattr(self, name, sub_cls(self, self._os_cloud))

    def reload_client(self):
        for pod in self.os_manager.api.pods.list_starts_with(
            'keystone-client', self.os_manager.openstack_namespace
        ):
            pod_obj = pod.read()
            if pod_obj.status.phase == "Running":
                self.client = pod
                break
        else:
            msg = "There are no keystone-client pod in Running state on env"
            LOG.error(msg)
            raise Exception(msg)

        for sub_cls in OpenStackResource.__subclasses__():
            name = self.pattern.sub('_', sub_cls.__name__).lower()
            setattr(self, name, sub_cls(self, self._os_cloud))

    def _pod_cmd_client(self, label_selector, hostname):
        pod_client = self.os_manager.api.pods.list_starts_with(
            '', self.os_manager.openstack_namespace,
            label_selector=label_selector,
            field_selector=f"spec.nodeName={hostname}"
        )[0]
        return pod_client

    def exec_neutron_ovs_agent(self, hostname, request):
        label_selector = "application=neutron,component=neutron-ovs-agent"
        return self._pod_cmd_client(label_selector, hostname).exec(request)

    def exec_vrouter_agent(self, hostname, request):
        label_selector = "app=tf-vrouter-agent-dpdk"
        pod_client = self.os_manager.api.pods.list_starts_with(
            '', self.os_manager.tf_namespace,
            label_selector=label_selector,
            field_selector=f"spec.nodeName={hostname}"
        )
        return pod_client[0].exec(request, container='agent')

    def exec_libvirt(self, hostname, request):
        label_selector = "application=libvirt,component=libvirt"
        return self._pod_cmd_client(label_selector, hostname).exec(
            request, container='libvirt'
        )

    def virsh_dumpXML(self, hypervisor_hostname, domain):
        return self.exec_libvirt(
            hypervisor_hostname, ["virsh", "dumpxml", domain])

    def virsh_capabilities(self, hypervisor_hostname):
        return self.exec_libvirt(
            hypervisor_hostname, ["virsh", "capabilities"])

    def provide_vm(self, vm_name):
        return VMProvider(self, vm_name)

    def exec(self, request, *args, **kwargs):
        try:
            response = self.client.exec(request, *args, **kwargs)
        except ApiException as e:
            if 'Not Found' in e.reason:
                self.reload_client()
                return self.client.exec(request, *args, **kwargs)
            raise e
        return response

    def exec_os_cli(self, cli_cmd, deserialize=False, *args, **kwargs):
        if not cli_cmd.startswith("openstack"):
            cli_cmd = f"openstack {cli_cmd}"
        if deserialize and "-f yaml" not in cli_cmd:
            cli_cmd += " -f yaml"
            # WA for openstack cli output with warnings in std_error
            cli_cmd = f"PYTHONWARNINGS=ignore::UserWarning {cli_cmd}"
        cmd = ['/bin/sh', '-c', cli_cmd]
        output = self.exec(cmd, *args, **kwargs)
        LOG.debug(f"Output:\n{output}")
        if deserialize:
            output = yaml.safe_load(output)
        return output

    def _stats_request(self, request):
        """Return a list of prometheus counters or an empty list in case of any error"""
        try:
            response = self.exec(request)
            return response.splitlines()
        except Exception as e:
            commons.LOG.error(f"Request {request} failed with: {e}")
            return list()

    def get_prometheus_counters(self, service_ip):
        # Get all prometheus counters using service internal IP
        url = "http://" + service_ip + ":9102"
        request = ["curl", "-s", url]
        response = self._stats_request(request)
        return {response[i].split()[0]: response[i].split()[1] for i in range(0, len(response)) if 'osh-dev' in
                response[i]}

    def ensure_stack(self, request, stack_name, template_path, kubeconfig=None, stack_params=None, finalizer=True):
        if self.stack.exists(stack_name):
            LOG.info(f"Stack with name {stack_name} already exists. Skipping creation.")
            return
        self.create_stack(request, stack_name, template_path, kubeconfig=kubeconfig, stack_params=stack_params,
                          finalizer=finalizer)

    def create_stack(self, request, stack_name, template_path, kubeconfig=None, stack_params=None, finalizer=True):
        client_name = self.client.name
        # TODO:(mkarpin) kubeconfig parameter from this method should be removed
        # later - for now leave it for compatibility.
        kubeconfig = kubeconfig or self.kubeconfig
        kubectl = kubectl_utils.Kubectl(kubeconfig=kubeconfig)

        target_template = f"/tmp/{stack_name}.yaml"
        destination = "{}/{}:{}".format(
            namespace.NAMESPACE.openstack, client_name, target_template)
        stack_cmd = [stack_name, "-t", target_template]

        if stack_params:
            target_parameters = f"/tmp/{stack_name}-parameters.yaml"
            parameters_dest = "{}/{}:{}".format(
                namespace.NAMESPACE.openstack, client_name, target_parameters)

            with tempfile.NamedTemporaryFile(mode="w") as tmp:
                content = yaml.dump({"parameters": stack_params})
                tmp.write(content)
                tmp.flush()
                kubectl.cp(tmp.name, parameters_dest)
            stack_cmd.extend(["-e", target_parameters])

        commons.LOG.info("Copy heat template")
        kubectl.cp(template_path, destination)

        commons.LOG.info(f"Create heat stack {stack_name}")
        self.stack.create(stack_cmd)

        if finalizer:
            request.addfinalizer(functools.partial(self.stack.delete, [stack_name, "-y", "--wait"]))

        commons.LOG.info(f"Wait for heat stack {stack_name} to create")
        wait = wait_utils.Waiter(self.os_manager, 3600)

        def f():
            target_stack = self.stack.show(
                [stack_name])
            if target_stack:
                status = target_stack["stack_status"]
                if status == "CREATE_FAILED":
                    resource_list = self.stack.resource_list(
                        ['-n', '10', stack_name])
                    event_list = self.stack.event_list(
                        ['--nested-depth', '10', stack_name])
                    commons.LOG.error("Resource info: %s\nHeat stack event list: %s",
                                      resource_list, event_list)
                    raise Exception("Failed to create stack")
                return status == "CREATE_COMPLETE"
            else:
                raise Exception(f'Failed to create stack with name: {stack_name}. '
                                f'Stack was not found on environment')

        wait.wait(f)
        commons.LOG.info("Heat stack created successfully")

    def check_stack(self, stack_name):

        commons.LOG.info(f"Check heat stack {stack_name}")
        self.stack.check([stack_name])

        commons.LOG.info(f"Wait for heat stack {stack_name} to check")
        wait = wait_utils.Waiter(self.os_manager, 3600)

        def f():
            target_stack = self.stack.show(
                [stack_name])
            if target_stack:
                status = target_stack["stack_status"]
                if status == "CHECK_FAILED":
                    resource_list = self.stack.resource_list(
                        ['-n', '10', stack_name])
                    event_list = self.stack.event_list(
                        ['--nested-depth', '10', stack_name])
                    commons.LOG.error("Resource info: %s\nHeat stack event list: %s",
                                      resource_list, event_list)
                    raise Exception("Failed to check stack")
                return status == "CHECK_COMPLETE"
            else:
                raise Exception(f'Stack with name: {stack_name} was not found on environment')

        wait.wait(f)
        commons.LOG.info("Heat stack {} checked successfully".format(stack_name))

    def server_get_floating_ips(self, instance_id):
        ip_list = []
        port_ids = self.port.list(['--device-id', instance_id, '-c', 'ID', '-f', 'value'], False).split()
        for curr_port in port_ids:
            ip_list = self.floatingip.list(['--port', curr_port, '-c', 'Floating IP Address',
                                            '-f', 'value'], False).split()
            if ip_list:
                break
        return ip_list

    def server_collect_console(self, instance_id, arch_path):
        instance_console = self.console.log_show([instance_id])
        with open(os.path.join(arch_path, f"instance_{instance_id}_console.txt"), "w") as f:
            f.write(instance_console)

    @retry(exceptions=(ssh_exception.SSHException, EOFError), tries=5, delay=1.5, jitter=(0, 1.4), logger=logger.logger)
    def server_collect_logs(self, ssh_connect, arch_path, docker_logs=False):
        with ssh_connect.sudo(enforce=True):
            if docker_logs:
                docker_pid = ssh_connect.execute(command="pidof dockerd")
                if docker_pid.stdout:
                    containers_IDs = ssh_connect.execute(command="docker ps -q")
                    for curr_ID in containers_IDs.stdout:
                        curr_ID = curr_ID.decode("utf-8").strip()
                        ssh_connect.execute(
                            command="docker logs " + curr_ID + " > /var/log/docker_" + curr_ID + "_console.log",
                            verbose=False, timeout=120, open_stdout=False, open_stderr=False)
            ssh_connect.execute(command="tar -czf /tmp/var_log.tar.gz -C /var/log .", verbose=False,
                                timeout=120, open_stdout=False, open_stderr=False)
            ssh_connect.download(destination="/tmp/var_log.tar.gz", target=arch_path)

    def server_reboot(self, instance_id):
        start_time = datetime.datetime.now()

        self.server.reboot([instance_id])

        wait = wait_utils.Waiter(self.os_manager, 600)

        def f():
            events = self.server.event_list([instance_id])
            reboot_event = None
            for event in events:
                if (event['Action'] == 'reboot' and
                        datetime.datetime.strptime(event['Start Time'], '%Y-%m-%dT%H:%M:%S.%f') >= start_time):
                    reboot_event = event['Request ID']
                    break
            if reboot_event is None:
                raise Exception(f"Failed to find reboot event for {instance_id}")
            event_output = self.server.event_show([instance_id, reboot_event])
            for item in event_output['events']:
                if ('reboot' in item['event'] and
                        datetime.datetime.strptime(item['start_time'], '%Y-%m-%dT%H:%M:%S.%f') >= start_time):
                    return item['result'] == 'Success'
            raise Exception(f"Failed to find reboot event item in event {reboot_event}")

        wait.wait(f)
        commons.LOG.info("Server {} rebooted successfully".format(instance_id))

    def server_live_migrate(self, instance_id, host=None, wait=True, wait_timeout=600):
        # --os-compute-api-version 2.30 is required for new features like --host
        # and auto shared/block storage migration
        cmd = ["--live-migration", "--os-compute-api-version=2.30"]
        if host:
            cmd.append(f"--host={host}")
        cmd.append(instance_id)
        src_host = self.server.show([instance_id])["OS-EXT-SRV-ATTR:host"]
        commons.LOG.info(f"Started live migration for server {instance_id}")
        res = self.server.migrate(cmd, combined_output=True)
        if res["stderr"]:
            raise Exception(f"Failed to migrate instance {instance_id}. Error: {res['stderr']}")

        def f(src_host):
            srv = self.server.show([instance_id])
            if srv["OS-EXT-SRV-ATTR:host"] == src_host:
                commons.LOG.info(f"Server {instance_id} is still on host {src_host}")
                return False
            if srv["status"] != "ACTIVE":
                commons.LOG.info(f"Server {instance_id} status is {srv['status']}")
                return False
            return True

        if wait:
            wait = wait_utils.Waiter(self.os_manager, wait_timeout)
            wait.wait(f, src_host)
        commons.LOG.info(f"Server {instance_id} live migrated successfully")

    @property
    def cirros_image_name(self):
        """Check or get a Cirros image name

          Try to find a cirros image
          which name starts from "Cirros", except names that end on ".alt".
          Select the image with higher version if possible.
        - raise AssertionError in case if no suitable image names found
        """
        images = self.image.list([])
        image_names = [image['Name'] for image in images]
        cirros_image_names = [image_name
                              for image_name in image_names
                              if image_name.startswith('Cirros')
                              and not image_name.endswith('.alt')]
        assert cirros_image_names, f"No 'cirros' images found in the MOSK cluster, existing images: '{image_names}'"
        cirros_name = sorted(cirros_image_names)[-1]
        commons.LOG.info(f"Found 'cirros' images in the MOSK cluster: '{cirros_image_names}'")
        commons.LOG.info(f"Select 'cirros' image: '{cirros_name}'")
        return cirros_name
