import json
import os
from retry import retry
import shutil

from si_tests import settings
from si_tests.deployments.utils import commons, file_utils, namespace
from si_tests.utils import utils
from . import base


class RefappTerraform(base.RefappBase):
    source = file_utils.join(commons.REFAPP_DIR, "terraform")

    def _exec_terraform(self, cmd_list):
        cmd = " && ".join(["./terraform " + c for c in cmd_list])
        return self.kubectl.exec(
            self.ostcm.client.name,
            f"bash -c 'cd /tmp/terraform; {cmd}'",
        ).result_str

    @retry(AssertionError, delay=5, tries=8, backoff=2, logger=commons.LOG)
    def _exec_terraform_init(self):
        self._exec_terraform(["init"])

    def _ensure_stuff_exists(self):
        try:
            self.kubectl.exec(self.ostcm.client.name, "[ -d /tmp/terraform ]")
        except AssertionError:
            # convert symlink to service library
            service_lib_link = f"{self.source}/templates/service_lib"
            if os.path.islink(service_lib_link):
                service_lib = os.path.realpath(service_lib_link)
                os.unlink(service_lib_link)
                shutil.copy(service_lib, service_lib_link)

            # copy terraform binary
            shutil.copy(utils.get_binary_path("terraform"), self.source)

            # store ssh keys
            key_file = f"{self.source}/templates/.openstack"
            if not all([os.path.exists(k) for k in (key_file, f"{key_file}.pub")]):
                key = utils.load_keyfile(settings.OPENSTACK_DEPLOY_SSH_KEY_PATH)
                utils.dump_keyfile(key_file, key)
                with open(f"{key_file}.pub", "w") as f:
                    f.write("ssh-rsa " + key["public"])

            # variables definition
            tfvars_file = f"{self.source}/terraform.tfvars.json"
            if not os.path.exists(tfvars_file):
                tfvars = {
                    "db_passwords": {
                        db: self.db_passwords[db] for db in ("admin", "app")
                    }
                }
                if settings.OPENSTACK_REFAPP_IMAGE:
                    tfvars.update({"app_docker_image": settings.OPENSTACK_REFAPP_IMAGE})
                with open(tfvars_file, "w") as f:
                    json.dump(tfvars, f)

            commons.LOG.info("Upload terraform stuff to keystone-client pod")
            destination = f"{namespace.NAMESPACE.openstack}/{self.ostcm.client.name}:/tmp"
            self.kubectl.cp(self.source, destination)
            self._exec_terraform_init()

    def _get_loadbalancers_status(self):
        commons.LOG.info("Get LoadBalancers status")
        for lb in self.ostcm.loadbalancer.list([]):
            lb_status = self.ostcm.loadbalancer.status_show([lb["id"]])
            commons.LOG.info(lb_status)

    @property
    def url(self):
        if self._url is None:
            self._url = self._exec_terraform(["output -raw app_url"])
        return self._url

    def deploy(self):
        self._ensure_stuff_exists()
        commons.LOG.info(
            "Create infrastructure according to Terraform configuration"
        )
        terraform_apply_cmd = ["apply",  "-auto-approve"]
        if settings.OPENSTACK_REFAPP_ENVIRONMENT:
            terraform_apply_cmd.extend(["--var-file", f"tfvars/{settings.OPENSTACK_REFAPP_ENVIRONMENT}.tfvars"])
        self._exec_terraform([" ".join(terraform_apply_cmd)])
        # get current state back
        tfstate = "terraform.tfstate"
        destination = f"{namespace.NAMESPACE.openstack}/{self.ostcm.client.name}:/tmp"
        self.kubectl.cp(f"{destination}/terraform/{tfstate}", f"{self.source}/{tfstate}")
        commons.LOG.info("Refapp has been deployed successfully")

    def delete(self):
        self._ensure_stuff_exists()
        self._get_loadbalancers_status()
        commons.LOG.info("Delete RefApp")
        self._exec_terraform(["destroy -auto-approve"])
        commons.LOG.info("Refapp has been deleted successfully")
