import json


class DockerCliClient(object):

    def __init__(self, cluster, private_key=None):

        control_nodes = cluster.get_machines(machine_type='control',
                                             machine_status='Ready')
        if not control_nodes:
            raise Exception("No control nodes in 'Ready' status found "
                            "in the cluster {0}".format(cluster.name))
        self.ctl = control_nodes[0]
        self.private_key = private_key
        self.machines = cluster.get_machines()

    @staticmethod
    def _decode(multistring):
        """Decode docker output in json format

        Docker encodes into json separated lines which are
        many separated json objects instead of a single list.
        """

        data = [
            json.loads(line)
            for line in multistring.splitlines()
        ]
        return data

    def _get_machine(self, machine_name):
        for machine in self.machines:
            if machine.name == machine_name:
                return machine
        raise Exception("Docker node {0} not found in nodes {1}"
                        .format(machine_name, [m.name for m in self.machines]))

    def _run_cmd(self, cmd, verbose=False, machine_name=None, exec_pod_cmd=False):
        if machine_name:
            machine = self._get_machine(machine_name)
        else:
            machine = self.ctl

        if exec_pod_cmd:
            _res = machine.exec_pod_cmd(cmd, verbose=verbose)
            if _res.get('exit_code') != 0:
                raise Exception(f"Command '{cmd}' failed on Machine '{machine.name}' "
                                f"with exit.code='{_res.get('exit_code')}'\n{_res.get('events')}")
            multistring = _res['logs']
        else:
            _res = machine.run_cmd(cmd, verbose=verbose, check_exit_code=True, ssh_key=self.private_key)
            multistring = _res.stdout_str

        return self._decode(multistring)

    def ps(self, machine_name, exec_pod_cmd=False):
        return self._run_cmd("docker ps --format '{{json .}}'",
                             verbose=False, machine_name=machine_name, exec_pod_cmd=exec_pod_cmd)

    def node_ls(self, verbose=False, exec_pod_cmd=False):
        return self._run_cmd("docker node ls --format '{{json .}}'", verbose=verbose, exec_pod_cmd=exec_pod_cmd)

    def node_inspect(self, machine_name, exec_pod_cmd=False):
        return self._run_cmd(f"docker node inspect {machine_name}"
                             " --format '{{json .}}'",
                             verbose=False, exec_pod_cmd=exec_pod_cmd)

    def service_ls(self, exec_pod_cmd=False):
        return self._run_cmd("docker service ls --format '{{json .}}'", verbose=False, exec_pod_cmd=exec_pod_cmd)

    def service_ps(self, service_name, exec_pod_cmd=False):
        return self._run_cmd(f"docker service ps {service_name}"
                             " --format '{{json .}}'",
                             verbose=False, exec_pod_cmd=exec_pod_cmd)

    def service_inspect(self, service_name, exec_pod_cmd=False):
        return self._run_cmd(f"docker service inspect {service_name}"
                             " --format '{{json .}}'",
                             verbose=False, exec_pod_cmd=exec_pod_cmd)

    def service_logs(self, service_name, exec_pod_cmd=False):
        return self._run_cmd(f"docker service logs {service_name}"
                             " --no-trunc",
                             verbose=False, exec_pod_cmd=exec_pod_cmd)

    def docker_info(self, machine_name, verbose=False, exec_pod_cmd=False):
        return self._run_cmd("docker info --format '{{json .}}'",
                             verbose=verbose, machine_name=machine_name, exec_pod_cmd=exec_pod_cmd)

    def container_ls(self, machine_name, exec_pod_cmd=False):
        return self._run_cmd("docker container ls --format '{{json .}}'",
                             verbose=False, machine_name=machine_name, exec_pod_cmd=exec_pod_cmd)

    def container_inspect(self, machine_name, container_name, exec_pod_cmd=False):
        return self._run_cmd(f"docker container inspect {container_name}"
                             " --format '{{json .}}'",
                             verbose=False, machine_name=machine_name, exec_pod_cmd=exec_pod_cmd)

    def container_logs(self, machine_name, container_name, exec_pod_cmd=False):
        return self._run_cmd(f"docker container logs {container_name}",
                             verbose=False, machine_name=machine_name, exec_pod_cmd=exec_pod_cmd)

    def stats(self, exec_pod_cmd=False):
        return self._run_cmd("docker stats --no-stream --no-trunc"
                             " --format '{{json .}}'",
                             verbose=False, exec_pod_cmd=exec_pod_cmd)
