import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.managers import openstack_manager

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
osdpl_name = settings.OSH_DEPLOYMENT_NAME


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('log_method_time')
def test_replace_failed_os_control(kaas_manager, show_step):
    """Replace OpenStack control plane node after the outage.

    Scenario:
        1. Find Machine to replace
        2. Simulate power outage (Hard power off)
        3. Wait for NotReady node state
        4. Replace the powered off OS control Machine and it's BMH/BMHI
        5. Check Machines and Nodes
        6. Recreate octavia resources and check
        7. Identity and rotate StackLight stateful applications
        8. Check child cluster conditions
        9. Check openstack resources
    """

    # Get namespace
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    show_step(1)
    machines = cluster.get_machines(machine_status='Ready')
    # get cluster machine (outage candidate)
    os_control = []
    for w in machines:
        if ("si-role/replaceoscontrol" in w.machinelabels and
                [n for n in w.nodeLabels if 'openstack-control-plane' in n.values()]):
            os_control.append(w)

    assert len(os_control) == 1, (
        f"BM Machine not found on the cluster, "
        f"or Machine not in Ready status. Found {len(os_control)} Machines: {os_control}")
    target_node = os_control[0]

    stacklight = False
    if [n for n in target_node.nodeLabels if 'stacklight' in n.values()]:
        stacklight = True
        LOG.info("StackLight enabled on node")

    show_step(2)
    target_node.set_baremetalhost_power(online=False)

    show_step(3)
    LOG.info("Get k8s node!!!")
    k8s_node = target_node.get_k8s_node()
    LOG.info("Delete, and start waiting k8s node status NotReady!!!")
    old_node_k8s_name = k8s_node.name
    cluster.check.wait_k8s_node_status(old_node_k8s_name, expected_status='NotReady')

    show_step(4)
    new_machine = cluster.day2operations.replace_baremetal_machine_and_bmh(
        target_node, machine_deletion_policy="unsafe")

    show_step(5)
    # Waiting for cluster,pods,hb are Ready after machines were added
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_helmbundles()

    # Check machine runtime
    if settings.DESIRED_RUNTIME:
        machines = cluster.get_machines()
        replaced_machine_for_runtime_check = [machine for machine in machines
                                              if "si-role/replaceoscontrol" in machine.machinelabels][0]
        cluster.check.compare_machines_runtime_with_desired([replaced_machine_for_runtime_check])

    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=[cluster.get_machine(name=new_machine.name)])

    new_node_k8s_name = new_machine.get_k8s_node().name
    LOG.info(f"New node k8s name: {new_node_k8s_name}")

    # Get OS manager
    child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)
    os_manager = openstack_manager.OpenStackManager(kubeconfig='child_conf')

    cluster.day2operations.cleanup_ovn_db_inactive_members(new_node_k8s_name)

    show_step(6)
    cluster.day2operations.recreate_octavia_resources_and_check(os_manager, old_node_k8s_name, new_node_k8s_name)

    # Get keystone client pod
    client_pod = cluster.k8sclient.pods.list(namespace="openstack", name_prefix='keystone-client')
    assert len(client_pod) > 0, "No pods found with prefix keystone-client in namespace openstack"
    client_pod = client_pod[0]

    osdpl = cluster.k8sclient.openstackdeployment.get(osdpl_name, namespace="openstack")
    # If Ironic is enabled, then nova-compute service is existed
    # on control nodes, so we should check that service can be removed
    if osdpl.data.get('spec').get('features', {}).get('ironic', {}):
        service_list_cmd = ['/bin/sh', '-c',
                            'PYTHONWARNINGS=ignore::UserWarning '
                            'openstack compute service list -f yaml']
        s_list = yaml.safe_load(client_pod.exec(service_list_cmd))
        ironic_compute_service = [
            s for s in s_list if s.get('Host') == old_node_k8s_name]

        if ironic_compute_service:
            LOG.info("Check Ironic compute-service can be deleted.")
            # After PRODX-15809 is fixed, an exception must be thrown
            # if the service can not be deleted.
            ironic_compute_service = ironic_compute_service[0]
            service_state = ironic_compute_service.get('State')
            if service_state == 'down':
                LOG.info('Serivce is down. Trying to delete service')
                s_id = ironic_compute_service.get('ID')
                delete_cmd = ['/bin/sh', '-c',
                              f'PYTHONWARNINGS=ignore::UserWarning'
                              f' openstack compute service delete {s_id}']
                stdout = client_pod.exec(delete_cmd)
                if stdout:
                    raise Exception("Something went wrong during "
                                    f"service deletion: {stdout}")
                else:
                    LOG.info("Service deleted succesfully")
            else:
                raise Exception(f"Wrong service status: {service_state}")
        else:
            LOG.info(f"nova-compute service for host {old_node_k8s_name} "
                     "is not existed anymore")

    os_controller_version = os_manager.os_controller_version()
    LOG.info(f"OpenStack controller version: {os_controller_version}")

    if stacklight:
        show_step(7)  # Identity and rotate StackLight stateful applications
        cluster.day2operations.cleanup_pods_and_pvc_from_k8s_node(old_node_k8s_name)

    show_step(8)
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()
    cluster.check.check_k8s_pods()
    cluster.check.check_deploy_stage_success()

    show_step(9)
    LOG.info("Check Openstack resources statuses")
    os_manager.wait_all_osdpl_children_status()
    os_manager.wait_openstackdeployment_health_status()
    os_manager.wait_all_os_pods(timeout=900, interval=90)

    cluster.check.check_bmh_inventory_presense()
