import pytest
from si_tests import logger
from si_tests import settings


LOG = logger.logger


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_change_ntp_servers(kaas_manager, show_step, _):
    """Test add/change NTP servers

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Update NTP servers (Add NTP servers in case if does not set before)
        3. Wait for cluster nodes' config became updated
        4. Check that chrony on nodes have proper config and time synced
        5. Check no cordon/drain occured for nodes
        6. (Situational) Check if child clusters updated NTP settings and only Reconfigure phase executed

    """
    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    machines = cluster.get_machines()
    # Gather chrony status and save it to verify that NTP config canged after update
    prev_ntp_reference = {}
    for m in machines:
        chr_status = m.get_chrony_status()
        prev_ntp_reference[m.name] = chr_status['Reference ID']
        LOG.info(f"Reference ID for machine {m.namespace}/{m.name}: {chr_status['Reference ID']}")
    # Gather lcmmachines' timestamps to verify that only Reconfigure phase was executed
    ts_before = cluster.get_cluster_lcmmachines_timestamps()

    chi_lcm_ts_before = {}
    chi_ntp_ref_before = {}
    managed_clusters = kaas_manager.get_child_clusters()
    if managed_clusters:
        LOG.info(f"Also found child clusters: {[(x.namespace, x.name) for x in managed_clusters]}")
        for clu in managed_clusters:
            chi_lcm_ts_before[f"{clu.namespace}-{clu.name}"] = clu.get_cluster_lcmmachines_timestamps()
            chi_ntp_ref_before[f"{clu.namespace}-{clu.name}"] = {}
            for m in clu.get_machines():
                chr_status = m.get_chrony_status()
                chi_ntp_ref_before[f"{clu.namespace}-{clu.name}"][m.name] = chr_status['Reference ID']
                LOG.info(f"Reference ID for machine {m.namespace}/{m.name}: {chr_status['Reference ID']}")

    show_step(2)
    ntp_servers = [i.strip() for i in settings.KAAS_NTP_LIST.split(',')]
    cluster.set_or_update_ntp_servers(ntp_servers)
    LOG.info('NTP servers were set. Waiting for cluster moved to Updating state')
    cluster.check.check_cluster_status(exp_cluster_status='Updating', interval=10)
    LOG.info('Cluster is Updating')

    show_step(3)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()

    show_step(4)
    cluster.check.check_ntp_status(prev_ntp_reference)

    # TODO (vastakhov): Remove if condition when CR 16-2-0+ will be used as a base
# https://gerrit.mcp.mirantis.com/plugins/gitiles/kaas/core/+/refs/heads/release/2.27/hack/backend/kaasLibrary.groovy
    show_step(5)
    if cluster.workaround.prodx_43331():
        LOG.warning(f"LCM phases check disabled for cluster {cluster.namespace}/{cluster.name} "
                    f"due codebase from 2.26 used.")
    else:
        cluster.check.check_no_unexpected_lcm_phases_reexecuted(ts_before, allowed_phases=['reconfigure'])

    if managed_clusters:
        show_step(6)
        for clu in managed_clusters:
            LOG.info(f"Checking cluster {clu.namespace}/{clu.name}")
            clu.check.check_machines_status()
            clu.check.check_cluster_readiness()
            clu.check.check_k8s_nodes()
            clu.check.check_ntp_status(chi_ntp_ref_before[f"{clu.namespace}-{clu.name}"])
            # TODO (vastakhov): Remove if condition when CR 16-2-0+ will be used as a base
            if clu.workaround.prodx_43331():
                LOG.warning(f"LCM phases check disabled for cluster {clu.namespace}/{clu.name} "
                            f"due codebase from 2.26 used.")
            else:
                clu.check.check_no_unexpected_lcm_phases_reexecuted(
                    chi_lcm_ts_before[f"{clu.namespace}-{clu.name}"],
                    allowed_phases=['reconfigure'])
