import yaml
import json
import os
import os.path
import time

import kubernetes

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils

import exec_helpers

import pytest

LOG = logger.logger


def wait_pods_status(pods, status, timeout=300, pause=5):
    start = time.time()
    pods = list(pods)
    while pods:
        for p in pods:
            try:
                p_data = p.read()
            except kubernetes.client.rest.ApiException as e:
                if e.status == 404:
                    LOG.info('Pod %s removed from namespace')
                LOG.warning("Got HTTP error %s duging fetch data about pod %s",
                            e.status, p.name)
            LOG.info("Pod %s has status %s",
                     p_data.metadata.name, p_data.status.phase)
            if p_data.status.phase == status:
                pods.remove(p)
        time.sleep(pause)
        if time.time() > start + timeout:
            return False, pods
    return True, None


def wait_deployments(deployments, timeout=300, pause=5):
    start = time.time()
    deployments = list(deployments)
    while deployments:
        for d in deployments:
            try:
                d_data = d.read()
            except kubernetes.client.rest.ApiException as e:
                if e.status == 404:
                    LOG.info('Deployment %s removed from namespace', d)
                    deployments.remove(d)
                LOG.warning("Got HTTP error %s during fetch data about "
                            "deployment %s", e.status, d.name)
            LOG.info("Deployment %s has ready replicas %s from %s",
                     d_data.metadata.name,
                     d_data.status.ready_replicas,
                     d_data.status.replicas)
            if d_data.status.ready_replicas == d_data.spec.replicas:
                deployments.remove(d)
        time.sleep(pause)
        if time.time() > start + timeout:
            return False, deployments
    return True, None


def wait_all_pods(kaas_manager, namespace, timeout=300, pause=5):
    time.sleep(pause)
    start = time.time()
    while time.time() < start + timeout:
        pods_ = kaas_manager.api.pods.list(namespace=namespace)
        pods = []
        for p in pods_:
            try:
                p_data = p.read()
                pods.append(p_data)
            except kubernetes.client.rest.ApiException:
                LOG.info('Pod %s removed from namespace', p.name)
                pods_.remove(p)
                continue
        inactive = [p for p in pods if p.status.phase.lower()
                    not in ('running', 'succeeded')]
        LOG.info('Namespace has incative %s pods',
                 [(p.metadata.name, p.status.phase) for p in inactive])
        if not inactive:
            return True, None
        time.sleep(pause)
    return False, inactive


def wait_all_jobs(kaas_manager, namespace, timeout=300, pause=5):
    time.sleep(pause)
    start = time.time()
    while time.time() < start + timeout:
        jobs = kaas_manager.api.jobs.list(namespace=namespace)
        if not jobs:
            return True, None
        jobs = [j.read() for j in jobs]
        inprogress = [j for j in jobs if j.status.succeeded != 1]
        LOG.info('Namespace has %s jobs in progress',
                 [j.metadata.name for j in inprogress])
        if not inprogress:
            return True, None
        time.sleep(pause)
    return False, inprogress


def test_deploy_openstack_1_plus_3(kaas_manager):
    crds_path = os.path.join(settings.OSH_OPERATOR_PATH, 'crds')
    assert os.path.isdir(settings.OSH_OPERATOR_PATH)
    assert os.path.isdir(crds_path)
    assert os.path.isfile(os.path.join(
        settings.OSH_OPERATOR_PATH,
        'examples/stein/core-ceph.yaml'))

    assert os.path.isdir(settings.MCP_PIPELINE_PATH)
    assert os.path.isfile(os.path.join(
        settings.MCP_PIPELINE_PATH,
        'tools/openstack-local-volume-provisoner.yaml'))

    child_cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    child_ns = kaas_manager.get_namespace(settings.KAAS_NAMESPACE)
    child_cluster = child_ns.get_cluster(child_cluster_name)
    kubectl_client_child = child_cluster.k8sclient

    # Collecting artifacts
    child_cluster.store_k8s_artifacts()
    child_cluster.provider_resources.save_artifact()

    nodes = kubectl_client_child.nodes.list_all()
    if len(nodes) != 4:
        pytest.fail("Child cluster doesn't have 3 nodes")

    child_nodes = [
        n for n in nodes if
        'node-role.kubernetes.io/master' not in n.read().metadata.labels]

    new_labels = {
        'openstack-control-plane': 'enabled',
        'openstack-compute-node': 'enabled',
        'openvswitch': 'enabled',
        'role': 'ceph-osd-node'
    }
    nodes_names = [n.name for n in nodes]
    LOG.info("Add labels %s to child nodes %s", new_labels, nodes_names)
    for n in child_nodes:
        n.patch({'metadata': {'labels': new_labels}})

    LOG.info("Add 'openstack-ceph-shared' namespace")
    kubectl_client_child.namespaces.create(
        name='openstack-ceph-shared',
        namespace='default',
        body={'metadata': {'name': 'openstack-ceph-shared'}})

    crds_files = []
    for file in os.listdir(crds_path):
        if file.endswith('.yaml'):
            crds_files.append(os.path.join(crds_path, file))
    crds_files.sort()

    master_node = next(
        n for n in nodes if 'node-role.kubernetes.io/master'
        in n.read().metadata.labels)
    assert master_node, "Cluster doesn't have the master node!"

    external_ip = next(a.address for a
                       in master_node.read().status.addresses if
                       a.type == 'ExternalIP')
    assert external_ip, "Master node doesn't have an external IP!"

    username = 'mcc-user'
    keys = utils.load_keyfile(settings.KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE)
    pkey = utils.get_rsa_key(keys['private'])
    auth = exec_helpers.SSHAuth(username=username, password='', key=pkey)
    master_remote = exec_helpers.SSHClient(
        host=external_ip,
        port=22,
        auth=auth,
        )

    home_path = '/home/{username}'.format(username=username)
    LOG.info("Copy kubeconfig from from root user to mcc-user")
    master_remote.check_call(
        'sudo cp /root/.kube/config {home_path}/kubeconfig;'
        'sudo chown {user}:{user} {home_path}/kubeconfig'.format(
            home_path=home_path, user=username))

    LOG.info("Apply crd files")
    master_remote.check_call('mkdir {home}/crds'.format(home=home_path))
    for crd in crds_files:
        crd_file = os.path.basename(crd)
        master_remote.upload(crd, os.path.join(home_path, 'crds', crd_file))
        master_remote.check_call(
            'kubectl --kubeconfig=kubeconfig apply -f {do}'.format(
                do=os.path.join(home_path, 'crds', crd_file)))
        LOG.info('%s applyed successfully', crd_file)

    LOG.info('Deploy Rook')
    rook_yml = ('https://raw.githubusercontent.com/jumpojoy/os-k8s/'
                '80e851bf377c18f1dd4aa20b63585c9536377581/crds/helmbundle'
                '/ceph/rook.yaml')
    master_remote.check_call(
        'kubectl --kubeconfig=kubeconfig apply -f {do}'.format(do=rook_yml))
    time.sleep(30)

    LOG.info("Wait when jobs done")
    status, jobs = wait_all_jobs(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some jobs {jobs} wasn't finished in "
                    "proper time {timeout}").format(jobs=jobs,
                                                    timeout=300)
    LOG.info("Wait when pods are active")
    status, pods = wait_all_pods(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some pods {pods} wasn't active/finished in "
                    "proper time {timeout}").format(pods=pods,
                                                    timeout=300)

    LOG.info('Deploy ceph cluster')
    ceph_cluster_yml = ('https://raw.githubusercontent.com/jumpojoy/os-k8s/'
                        '80e851bf377c18f1dd4aa20b63585c9536377581/crds'
                        '/ceph/cluster.yaml')
    master_remote.check_call(
        'kubectl --kubeconfig=kubeconfig apply -f {do}'.format(
            do=ceph_cluster_yml))
    time.sleep(30)

    LOG.info("Wait when jobs done")
    status, jobs = wait_all_jobs(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some jobs {jobs} wasn't finished in "
                    "proper time {timeout}").format(jobs=jobs,
                                                    timeout=300)
    LOG.info("Wait when pods are active")
    status, pods = wait_all_pods(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some pods {pods} wasn't active/finished in "
                    "proper time {timeout}").format(pods=pods,
                                                    timeout=300)

    LOG.info('Create Ceph storageclass')
    storageclass_yml = ('https://raw.githubusercontent.com/jumpojoy/os-k8s/'
                        '80e851bf377c18f1dd4aa20b63585c9536377581/crds'
                        '/ceph/storageclass.yaml')
    master_remote.check_call(
        'kubectl --kubeconfig=kubeconfig apply -f {do}'.format(
            do=storageclass_yml))
    time.sleep(60)

    LOG.info("Wait when jobs done")
    status, jobs = wait_all_jobs(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some jobs {jobs} wasn't finished in "
                    "proper time {timeout}").format(jobs=jobs,
                                                    timeout=300)
    LOG.info("Wait when pods are active")
    status, pods = wait_all_pods(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some pods {pods} wasn't active/finished in "
                    "proper time {timeout}").format(pods=pods,
                                                    timeout=300)

    LOG.info('Share metadata with openstack')
    # Waiter should be here.
    # Need wait when secret and configmap will be available
    master_remote.check_call(
        'kubectl --kubeconfig=kubeconfig -n rook-ceph '
        'get secret rook-ceph-admin-keyring -o yaml --export | '
        'kubectl --kubeconfig=kubeconfig apply -n openstack-ceph-shared -f-')
    master_remote.check_call(
        'kubectl --kubeconfig=kubeconfig -n rook-ceph '
        'get configmaps rook-ceph-mon-endpoints -o yaml --export | '
        'kubectl --kubeconfig=kubeconfig apply -n openstack-ceph-shared -f-')

    child_nodes_external_ip = [next(
        filter(lambda x: x.type == 'ExternalIP',
               n.read().status.addresses)).address for n in child_nodes]

    for node_ip in child_nodes_external_ip:
        auth = exec_helpers.SSHAuth(username=username, password='', key=pkey)
        child_remote = exec_helpers.SSHClient(
            host=node_ip,
            port=22,
            auth=auth,
            )
        for num in range(3):
            vol = "vol-{num}".format(num=num)
            child_remote.check_call(
                'sudo  mkdir -p /mnt/local-provisioner/{vol} && '
                'sudo mkdir -p /mnt/{vol}'.format(vol=vol))
            child_remote.check_call(
                'sudo mount --bind /mnt/{vol} '
                '/mnt/local-provisioner/{vol}'.format(vol=vol))
            child_remote.check_call(
                "echo '/mnt/{vol} "
                "/mnt/local-provisioner/{vol} none defaults,bind 0 0' | "
                "sudo tee -a /etc/fstab".format(vol=vol))

    LOG.info('Run openstack local volume provisoner')
    os_local_volume_provisioner = os.path.join(
        settings.MCP_PIPELINE_PATH,
        'tools/openstack-local-volume-provisoner.yaml')
    master_remote.upload(os_local_volume_provisioner,
                         os.path.join(home_path,
                                      os.path.basename(os_local_volume_provisioner)))  # noqa
    master_remote.check_call('kubectl --kubeconfig=kubeconfig apply -f '
                             '{do}'.format(do=os.path.join(
                                 home_path,
                                 os.path.basename(os_local_volume_provisioner))))  # noqa

    cert_path = os.path.join(home_path, 'cert')
    master_remote.mkdir(cert_path)
    master_remote.check_call(
        'cd {cert_path};'
        'curl -L https://pkg.cfssl.org/R1.2/cfssl_linux-amd64 -o cfssl;'
        'chmod +x cfssl'.format(cert_path=cert_path))
    # Add sha256 check sum verification
    master_remote.check_call(
        'cd {cert_path}; curl -L '
        'https://pkg.cfssl.org/R1.2/cfssljson_linux-amd64 -o cfssljson;'
        'chmod +x cfssljson'.format(cert_path=cert_path))
    # Add sha256 check sum verification

    ca_csr_json = {
        "CN": "kubernetes",
        "key": {
            "algo": "rsa",
            "size": 2048
        },
        "names": [{
            "C": "<country>",
            "ST": "<state>",
            "L": "<city>",
            "O": "<organization>",
            "OU": "<organization unit>"
        }]
    }

    child_cluster_uuid = child_cluster.data['metadata']['annotations'][
        'kaas.mirantis.com/uid']

    ca_csr_content = json.dumps(ca_csr_json)
    with master_remote.open(os.path.join(cert_path, 'ca-csr.json'), 'w') as f:
        f.write(ca_csr_content)
    master_remote.check_call(
        'cd {cert_path};'
        './cfssl gencert -initca ca-csr.json | '
        './cfssljson -bare ca'.format(cert_path=cert_path))

    server_csr_json = {
        "CN": "*.openstack.svc.kaas-kubernetes-{short_uuid}".format(
            short_uuid=child_cluster_uuid.replace('-', '')),
        "hosts": [
            "keystone",
            "keystone.openstack",
            "glance",
            "glance.openstack",
            "cinder",
            "cinder.openstack",
            "cloudformation",
            "cloudformation.openstack",
            "glance-reg",
            "glance-reg.openstack",
            "heat",
            "heat.openstack",
            "horizon",
            "horizon.openstack",
            "metadata",
            "metadata.openstack",
            "neutron",
            "neutron.openstack",
            "nova",
            "nova.openstack",
            "novncproxy",
            "novncproxy.openstack",
            "placement",
            "placement.openstack",
            "*.openstack.svc.kaas-kubernetes-{short_uuid}".format(
                short_uuid=child_cluster_uuid.replace('-', ''))
        ],
        "key": {
            "algo": "rsa",
            "size": 2048
        },
        "names": [{
            "C": "US",
            "L": "CA",
            "ST": "San Francisco"
        }]
    }

    server_csr_content = json.dumps(server_csr_json)
    with master_remote.open(os.path.join(cert_path,
                                         'server-csr.json'), 'w') as f:
        f.write(server_csr_content)

    ca_config_json = {
        "signing": {
            "default": {
                "expiry": "8760h"
            },
            "profiles": {
                "kubernetes": {
                    "usages": [
                        "signing",
                        "key encipherment",
                        "server auth",
                        "client auth"
                    ],
                    "expiry": "8760h"
                }
            }
        }
    }

    ca_config_content = json.dumps(ca_config_json)
    with master_remote.open(os.path.join(cert_path,
                                         'ca-config.json'), 'w') as f:
        f.write(ca_config_content)

    master_remote.check_call(
        'cd {cert_path}; '
        './cfssl gencert -ca=ca.pem -ca-key=ca-key.pem '
        '--config=ca-config.json -profile=kubernetes server-csr.json | '
        './cfssljson -bare server'.format(cert_path=cert_path))

    with master_remote.open(os.path.join(cert_path, 'ca.pem')) as f:
        ca_cert_pem_content = f.read()

    with master_remote.open(os.path.join(cert_path, 'server.pem')) as f:
        api_cert_pem_content = f.read()

    with master_remote.open(os.path.join(cert_path, 'server-key.pem')) as f:
        api_key_pem_content = f.read()

    with open(os.path.join(
            settings.OSH_OPERATOR_PATH,
            'examples/stein/core-ceph.yaml'), 'r') as f:
        core_ceph_data = f.read()
        core_ceph_data = yaml.load(core_ceph_data, Loader=yaml.SafeLoader)
        # find current domain inside cluster
        core_ceph_data['spec']['domain_name'] = \
            'kaas-kubernetes-{short_uuid}'.format(
                short_uuid=child_cluster_uuid.replace('-', ''))
        core_ceph_data['spec']['features']['ssl'][
            'public_endpoints']['ca_cert'] = ca_cert_pem_content
        core_ceph_data['spec']['features']['ssl'][
            'public_endpoints']['api_cert'] = api_cert_pem_content
        core_ceph_data['spec']['features']['ssl'][
            'public_endpoints']['api_key'] = api_key_pem_content
        core_ceph_data['spec']['features']['services'].remove('messaging')
        core_ceph_data['spec']['features']['services'].append('rabbitmq')

    core_ceph_yaml_content = yaml.dump(core_ceph_data)
    with master_remote.open(
            os.path.join(home_path, 'core-ceph.yaml'), 'w') as f:
        f.write(core_ceph_yaml_content)
    master_remote.check_call(
        'kubectl --kubeconfig=kubeconfig apply -f {do}'.format(
            do='core-ceph.yaml'))

    time.sleep(60)
    LOG.info("Wait when jobs done")
    status, jobs = wait_all_jobs(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some jobs {jobs} wasn't finished in "
                    "proper time {timeout}").format(jobs=jobs,
                                                    timeout=300)
    LOG.info("Wait when pods are active")
    status, pods = wait_all_pods(kubectl_client_child, 'rook-ceph',
                                 timeout=300)
    assert status, ("Some pods {pods} wasn't active/finished in "
                    "proper time {timeout}").format(pods=pods,
                                                    timeout=300)
