#    Copyright 2023 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest
import yaml

from collections import namedtuple

from si_tests import settings
from si_tests import logger
from si_tests.deployments.utils import kubectl_utils
from si_tests.managers import bootstrap_manager
from si_tests.managers import si_config_manager
from si_tests.clients.openstack import OpenstackClient
from si_tests.utils import templates


LOG = logger.logger
templates_path = f'{settings.KAAS_BOOTSTRAP_TARGET_DIR}/templates/'


@pytest.mark.usefixtures("introspect_MKE_7914",
                         "introspect_PRODX_9238",
                         "introspect_openstack_seed_management_deploy_objects")
@pytest.fixture(scope="module")
def prep_managers(openstack_client: OpenstackClient):
    """
    Initialize managers before test
    """
    LOG.info("Initialize bootstrap manager")
    seed_ip = openstack_client.get_seed_ip()
    bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
    LOG.info("Add data to si config")
    si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
    si_config.store_seed_ip(seed_ip)
    bootstrap.add_credentials_data_to_si_config()
    LOG.info("Prepare kubectl")
    remote = bootstrap.remote_seed()
    kubectl = kubectl_utils.RemoteKubectl(
        remote=remote,
        binary_path=os.path.join(".", settings.KAAS_BOOTSTRAP_TARGET_DIR, "bin", "kubectl"),
        kubeconfig=settings.KUBECONFIG_KIND_PATH)
    LOG.info("Prepare templater")
    region = bootstrap.get_region_from_bootstrap_region(
        provider=settings.OPENSTACK_PROVIDER_NAME, path='templates.orig')
    templater = templates.Bootstrapv2Applier(kubectl, templates_dir=templates_path, region=region,
                                             provider=settings.OPENSTACK_PROVIDER_NAME)

    return namedtuple('Managers', 'bootstrap kubectl templater remote openstack_client')(bootstrap, kubectl,
                                                                                         templater, remote,
                                                                                         openstack_client)


def test_wrong_bootstrapregion(prep_managers, show_step):
    """Negative scenario for bootstrap region with wrong provider value

    Scenario:
         1. Prepare templates to apply on bootstrap cluster and check them
         2. Modify provider to wrong value in bootstrapregion.yaml.template
         3. Try to apply template with error
    """
    # step 001: Prepare templates to apply on bootstrap cluster and check them
    show_step(1)
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    # step 002: Modify bootstrapregion.yaml.template with wrong provider value
    show_step(2)
    with prep_managers.remote.open(templates_path + 'bootstrapregion.yaml.template',
                                   mode="r+") as f:
        template = yaml.load(f.read(), Loader=yaml.SafeLoader)
        template['spec']['provider'] = 'unknown_provider'
        f.seek(0)
        f.write(yaml.dump(template))

    # step 003: Try to apply template with error
    show_step(3)
    try:
        prep_managers.templater.apply_bootstrap_region_and_wait()
    except Exception as e:
        LOG.error(e)
        if 'denied the request: unknown provider' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with wrong provider value")
    else:
        pytest.fail("Bootstrap script returned unexpected success with wrong provider value")


def test_wrong_openstackconfig(prep_managers, show_step):
    """Negative scenario for openstack-config.yaml.template with wrong URL

    Scenario:
         1. Prepare templates to apply on bootstrap cluster and check them again
         2. Modify openstack-config.yaml.template with wrong URL
         3. Try to apply openstack-config.yaml.template with error
    """
    # step 001: Prepare templates to apply on bootstrap cluster and check them again
    show_step(1)
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    # step 002: Modify openstack-config.yaml.template with wrong URL
    show_step(2)
    with prep_managers.remote.open(templates_path + 'openstack-config.yaml.template',
                                   mode="r+") as f:
        template = yaml.load(f.read(), Loader=yaml.SafeLoader)
        template['spec']['auth']['authURL'] = 'wrong_url'
    with prep_managers.remote.open(templates_path + 'openstack-config.yaml.template',
                                   mode="w") as f:
        f.write(yaml.dump(template))

    # step 003: Try to apply openstack-config.yaml.template with error
    show_step(3)
    try:
        prep_managers.templater.apply_provider_credentials_and_wait()
    except Exception as e:
        LOG.error(e)
        if 'Timeout for waiting openstackcredentials' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with incorrect URL")
    else:
        pytest.fail("Bootstrap script returned unexpected success with incorrect URL")


def test_wrong_serviceusers(prep_managers, show_step):
    """Negative scenario for serviceusers.yaml.template with wrong field

    Scenario:
         1. Prepare templates to apply on bootstrap cluster and check them again
         2. Modify serviceusers.yaml.template add wrong field
         3. Try to apply serviceusers.yaml.template with wrong data
    """
    # step 001: Prepare templates to apply on bootstrap cluster and check them again
    show_step(1)
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    # step 002: Modify serviceusers.yaml.template by adding wrong field
    show_step(2)
    with prep_managers.remote.open(templates_path + 'serviceusers.yaml.template',
                                   mode="r+") as f:
        template = yaml.load(f.read(), Loader=yaml.SafeLoader)
        template['items'][0]['spec']['wrong_extra_field'] = 'wrong_value'
        f.seek(0)
        f.write(yaml.dump(template))

    # step 003: Try to apply serviceusers.yaml.template with wrong data
    show_step(3)
    try:
        prep_managers.templater.apply_service_user()
    except Exception as e:
        LOG.error(e)
        if 'unknown field "wrong_extra_field"' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with wrong field in serviceusers")
    else:
        pytest.fail("Bootstrap script returned unexpected success with wrong field in serviceusers")


@pytest.mark.xfail(reason="PRODX-32224")
def test_wrong_machine_flavor(prep_managers, show_step):
    """Negative scenario for machine.yaml with wrong flavor

    Scenario:
        1. Prepare templates to apply on bootstrap cluster and check them again
            with wrong value in variable KAAS_MANAGEMENT_CLUSTER_FLAVOR
        2. Try to apply machines.yaml.template with wrong FLAVOR
    """
    # step 001: Prepare templates to apply on bootstrap cluster and check them again
    # with wrong value in variable KAAS_MANAGEMENT_CLUSTER_FLAVOR
    show_step(1)
    os.environ['KAAS_MANAGEMENT_CLUSTER_FLAVOR'] = 'small_unreal_flavor'
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    # step 002: Try to deploy cluster with wrong FLAVOR, need to apply cluster with correct values before
    show_step(2)
    prep_managers.templater.apply_cluster()
    try:
        prep_managers.templater.apply_machines()
    except Exception as e:
        LOG.error(e)
        # TODO update error sample after PRODX-32224
        if 'small_unreal_flavor' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with wrong machine flavor")
    else:
        pytest.fail("Bootstrap script returned unexpected success with wrong machine flavor")


@pytest.mark.xfail(reason="PRODX-32294")
def test_wrong_machine_image(prep_managers, show_step):
    """Negative scenario for machine.yaml with wrong image

    Scenario:
        1. Prepare templates to apply on bootstrap cluster and check them again
            with wrong value in variable KAAS_MANAGEMENT_CLUSTER_IMAGE
        2. Try to apply machines.yaml.template with wrong Image
    """
    # step 001: Prepare templates to apply on bootstrap cluster and check them again
    # with wrong value in variable KAAS_MANAGEMENT_CLUSTER_IMAGE
    show_step(1)
    os.environ['KAAS_MANAGEMENT_CLUSTER_IMAGE'] = 'unreal_image'
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    # step 002: Try to deploy cluster with wrong IMAGE
    show_step(2)
    try:
        prep_managers.templater.apply_machines()
    except Exception as e:
        LOG.error(e)
        # TODO update error sample after PRODX-32294
        if 'unreal_image' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with wrong machine image")
    else:
        pytest.fail("Bootstrap script returned unexpected success with wrong machine image")


@pytest.mark.xfail(reason="PRODX-32302")
def test_wrong_bootfromvolume_size(prep_managers, show_step):
    """Negative scenario for machine.yaml with too small BFV size

    Scenario:
        1. Prepare machine templates with active BFV and wrong volume size
        2. Try to apply machines with active BFV and wrong size
    """
    # step 001: Prepare machine templates with active BFV and wrong volume size, 10 instead 120
    show_step(1)
    os.environ['OPENSTACK_BOOT_FROM_VOLUME'] = 'True'
    os.environ['OPENSTACK_BOOT_VOLUME_SIZE'] = '10'
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()
    del os.environ['OPENSTACK_BOOT_FROM_VOLUME']
    del os.environ['OPENSTACK_BOOT_VOLUME_SIZE']

    # step 002: Try to apply machines with wrong BOOT_FROM_VOLUME size
    show_step(2)
    try:
        prep_managers.templater.apply_machines()
    except Exception as e:
        LOG.error(e)
        # TODO update error sample after PRODX-32302
        if '10' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with wrong bfv size")
    else:
        pytest.fail("Bootstrap script returned unexpected success with wrong bfv size")

# TODO: Need to add subtest with wrong parameters in flavor


def test_more_machines_than_allowed(prep_managers, show_step):
    """Negative scenario for machine.yaml with greater number of machines, than allowed -> 4 instead of 3

    Scenario:
    1. Prepare machine templates with bigger number of machines
    2. Try to apply machines with wrong machine numbers 4 instead of 3
    """
    # step 001: Prepare machine templates with greater machines
    show_step(1)
    os.environ['KAAS_MANAGEMENT_CLUSTER_NODES'] = '4'
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    # step 002: Try to apply machines with wrong machine numbers 4 instead of 3
    # templater.apply_cluster()
    show_step(2)
    try:
        prep_managers.templater.apply_machines()
    except Exception as e:
        LOG.error(e)
        if 'admission webhook "machines.kaas.mirantis.com" denied the request' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with more machines than allowed")
    else:
        pytest.fail("Bootstrap script returned unexpected success with more machines than allowed")


def test_less_machines_than_allowed(prep_managers, show_step):
    """Negative scenario for machine.yaml with smaller number of machines, than allowed -> 2 instead of 3

    Scenario:
    1. Prepare machine templates with smaller number of machines
    2. Delete machines from cluster
    3. Try to apply machines with wrong machine numbers 2 instead of 3
    """
    # step 001: Prepare machine templates with smaller number of machines
    show_step(1)
    os.environ['KAAS_MANAGEMENT_CLUSTER_NODES'] = '2'
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()

    show_step(2)
    # Need to delete machine with wrong template anf then we will recreate only 2
    machines = prep_managers.kubectl.get('machines', '-o yaml', settings.CLUSTER_NAMESPACE).result_yaml
    machines = machines['items']
    for machine in machines:
        name = machine['metadata']['name']
        LOG.info(f'machine {name}')
        prep_managers.kubectl.delete("machine", name, settings.CLUSTER_NAMESPACE)

    # step 003: Try to apply 2 machines and then try to launch cluster deploy
    show_step(3)
    try:
        prep_managers.templater.apply_machines()
        prep_managers.templater.start_cluster_deployment(
                 settings.CLUSTER_NAME,
                 True,
                 settings.OPENSTACK_PROVIDER_NAME,
                 prep_managers.bootstrap.bootstrapv2_print_controller_logs)
    except Exception as e:
        LOG.error(e)
        if 'admission webhook "machines.kaas.mirantis.com" denied the request' not in str(e):
            pytest.fail("Bootstrap script returned unexpected success with less number of machines")
    else:
        pytest.fail("Bootstrap script returned unexpected success with less number of machines")


# TODO Turn-on this part after PRODX-32224, PRODX-32302, PRODX-32294
@pytest.mark.skip(reason="Skip for test purposes")
def test_deploy_after_mistakes(prep_managers, show_step):
    """Check that after all mistakes we can deploy with correct templates

    Scenario:
        1. Prepare correct templates
        2. Check that after all procedures cluster will bootstrap correctly
        3. Wait for all pods to be Running and Ready
        4. Check that all expected pods exist
        5. Erase created KaaS environment
    """
    # step 001: Prepare correct templates
    show_step(1)
    os.environ['KAAS_MANAGEMENT_CLUSTER_NODES'] = '3'
    LOG.info("Prepare seed node templates")
    prep_managers.bootstrap.step_003_prepare_seed_node_templates()
    LOG.info("Patch release refs")
    prep_managers.bootstrap.step_003_patch_bootstrapv2_release_refs()
    prep_managers.bootstrap.check_templates()
    endpoints = None
    if settings.KAAS_OPENSTACK_ENABLED:
        endpoints = prep_managers.openstack_client.get_os_cloud_endpoints()

    # step 002: Check that after all procedures cluster will bootstrap correctly
    show_step(2)
    prep_managers.bootstrap.step_004_deploy_kaas_cluster(endpoints=endpoints,
                                                         seed_instance=prep_managers.openstack_client.
                                                         get_seed_instance())

    # step 003: Wait for all pods to be Running and Ready
    show_step(3)
    prep_managers.bootstrap.step_006_wait_for_pods()

    # step 004: Check that all expected pods exist
    show_step(4)
    prep_managers.bootstrap.step_006_postdeployment_checks()

    # step 005: Erase created KaaS environment
    if settings.KEEP_ENV_AFTER:
        LOG.warning("Skip erase due KEEP_ENV_AFTER flag is set")
    else:
        # Erase kaas environment after bootstrap is passed
        show_step(5)
        prep_managers.bootstrap.step_007_erase_env_after()
