import logging
import os

import pytest
import time

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.utils.utils import check_downtime, generate_list_pods

LOG = logger.logger
init_parent_handler = LOG.parent.handlers[0]

LOG.info("=========GENERATING POD LIST=========")
generated_pod_list = generate_list_pods(
    kaas_manager=Manager(kubeconfig=settings.KUBECONFIG_PATH),
    target_nss=settings.HA_TEST_NAMESPACE_LIST
)


_env_error = ""


@pytest.fixture(scope='module')
def cluster_condition_check_after_tests(target_cluster):
    global _env_error

    yield

    if _env_error:
        raise Exception(f"Cluster {target_cluster.name} is not ready after HA tests:\n{_env_error}")

    LOG.info("Check that all expected pods and their replicas are present after HA test")
    try:
        check_namespaces = list(target_cluster.expected_pods.keys())
        if target_cluster.is_mosk:
            check_namespaces.append("openstack")
        target_cluster.check.check_machines_status()
        target_cluster.check.check_cluster_nodes()
        target_cluster.check.check_k8s_nodes()
        target_cluster.check.check_k8s_pods(target_namespaces=check_namespaces)
        target_cluster.check.check_actual_expected_pods()
    except Exception as e:
        LOG.error(f"Cluster {target_cluster.name} is not ready after HA tests")
        LOG.debug(e)
        raise


@pytest.fixture(scope='function')
def cluster_condition_check(target_cluster, request, cluster_condition_check_after_tests):
    global _env_error
    if _env_error:
        raise Exception(f"Cluster {target_cluster.name} is not ready, cannot run HA test:\n{_env_error}")

    LOG.info("Ensure that all expected pods and their replicas are present before HA test")
    try:
        check_namespaces = list(target_cluster.expected_pods.keys())
        if target_cluster.is_mosk:
            check_namespaces.append("openstack")
            child_kubeconfig_name, child_kubeconfig = target_cluster.get_kubeconfig_from_secret()
            with open('child_conf', 'w') as f:
                f.write(child_kubeconfig)
            os_manager = OpenStackManager(kubeconfig='child_conf')
            if target_cluster.tf_enabled():
                target_cluster.mos_check.check_cassandra_nodes_config(
                    os_manager=os_manager,
                    actualize_nodes_config=settings.TF_CASSANDRA_NODES_CLENAUP)
        target_cluster.check.check_machines_status()
        target_cluster.check.check_cluster_nodes()
        target_cluster.check.check_k8s_nodes()
        target_cluster.check.check_k8s_pods(target_namespaces=check_namespaces)
        target_cluster.check.check_actual_expected_pods()
    except Exception as e:
        LOG.error(f"Cluster {target_cluster.name} is not ready for HA test")
        LOG.debug(e)
        _env_error = e
        raise

    yield


@pytest.fixture()
def helmbundles_check(target_cluster):
    LOG.info("Check that helmbundles ready before HA test")
    try:
        target_cluster.check.check_helmbundles()
        target_cluster.check.check_cluster_readiness()
    except Exception as e:
        LOG.error(f"Cluster {target_cluster.name} is not ready for HA test")
        LOG.debug(e)
        raise

    yield
    LOG.info("Check that helmbundles ready after HA test")
    try:
        target_cluster.check.check_helmbundles()
        target_cluster.check.check_cluster_readiness()
    except Exception as e:
        LOG.error(f"Cluster {target_cluster.name} is not ready after HA test")
        LOG.debug(e)
        raise


@pytest.fixture(scope='function', params=generated_pod_list.values(),
                ids=generated_pod_list.keys())
def pod_group(request, target_cluster):
    """This fixture parametrize tests by group of pods
    that will be used in each test scenario.
    After scenario is finished, fixture will check phases
    for all pods in cluster (should be running)
    """
    # fetching variables
    failed_num = request.session.testsfailed
    pod_name, pod_num, pods = request.param

    try:
        # replace LOG handler for the test
        LOG.parent.handlers.clear()
        log_name = request.node.name.split('[')[0] + '-' + pod_name
        log_file = os.path.join(settings.LOGS_DIR,
                                log_name + ".log")
        fh = logging.FileHandler(log_file)
        fh.setFormatter(LOG.handlers[0].formatter)
        LOG.parent.addHandler(fh)

        # Compare actual and expected pods
        assert len(pods) != 0, "Cannot find {} running pods".format(pod_name)
        if not pod_num:
            # new pods are discovered that not specified in expected pod list
            # we do not skip them for now
            LOG.error("Pod ({0}) is not defined in "
                      "expected list".format(pod_name))
        if len(pods) != int(pod_num):
            # we can fail test here, but for now it is error in logs
            LOG.error("Mismatch between expected ({0}) and "
                      "actual number ({1}) for {2} "
                      "running pods".format(pod_num, len(pods), pod_name))

        nss = target_cluster.expected_pods.keys()
        # Check cluster health
        target_cluster.check.check_k8s_nodes(timeout=20, interval=2)
        target_cluster.check.check_k8s_pods(timeout=170, interval=10,
                                            target_namespaces=nss)
    except Exception as e:
        LOG.parent.handlers.clear()
        LOG.parent.addHandler(init_parent_handler)
        raise e
    yield request.param
    # replace LOG handler back
    LOG.parent.handlers.clear()
    LOG.parent.addHandler(init_parent_handler)
    if failed_num == request.session.testsfailed:
        # remove log file if test passed
        os.remove(log_file)


commands = {'kill_all': 'pkill -u $(whoami)',
            'kill_all_except_init': 'kill -9 -1'}


@pytest.fixture(scope='module', params=commands.values(),
                ids=list(commands.keys()))
def command(request):
    yield request.param


@pytest.fixture(scope='function')
def check_stacklight_svc_downtime(kaas_manager):
    start = time.time()
    LOG.debug(f"start time:{start}")
    namespace = "stacklight"
    services = ["iam-proxy-alerta", "iam-proxy-alertmanager",
                "iam-proxy-grafana", "iam-proxy-kibana",
                "iam-proxy-prometheus"]
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    if cluster.sl_ha_enabled():
        expected_downtime = settings.HA_SL_SVC_HA_MODE_DOWNTIME
        LOG.info("SL HA enabled")
    else:
        LOG.info("SL HA disabled")

    yield

    end = time.time()
    LOG.debug(f"end time:{end}")
    client = cluster.prometheusclient
    err_msgs = []
    for service_name in services:
        LOG.info(f"Check downtime for {service_name}")
        result = client.get_svc_probe_success(namespace=namespace, service_name=service_name, start=start, end=end)
        downtime = cluster.check.calculate_k8s_downtime(result)
        LOG.info(f"Downtime: {downtime}")
        err_msg = check_downtime(downtime, expected_downtime, raise_on_error=False)
        if err_msg:
            err_msgs.append(f"Service [{service_name}]: {err_msg}")
    if err_msgs:
        raise Exception('\n'.join(err_msgs))
