#    Copyright 2024 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import yaml
import json

from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters
from si_tests.utils import templates
from si_tests.managers.kaas_manager import Cluster

LOG = logger.logger


class RestartChecker(object):
    """Control daemonset to check pod restarts during upgrade on containerd runtime"""

    def __init__(self, kubectl_cluster: "Cluster"):
        """kubectl_client: k8s client to KaaS management cluster"""
        self._cluster = kubectl_cluster
        self._kubectl_client = kubectl_cluster.k8sclient
        self.ds = None
        self.timestamps = {}

        LOG.info('Initialized restart checker')

    def _check_spawned_on_every_node(self):
        ds_p_list = self._kubectl_client.pods.list(name_prefix=self.ds.name, namespace=self.ds.namespace)
        nr_ready = int(self.ds.data.get('status', {}).get('number_ready', 0))
        cl_machines = self._cluster.get_machines()
        nr_machines = len(cl_machines)
        LOG.banner(f"READY PODS: {nr_ready}, EXPECTED: {nr_machines}")
        if not nr_ready == nr_machines:
            pod_nodes = {}
            for p in ds_p_list:
                pod_nodes[p.node_name] = p.name
            msg = f"Pods should spawn on every k8s node. Current node:pod map: \n{yaml.dump(pod_nodes)}\n"
            LOG.warning(msg)
            return False
        LOG.info("All pods are ready")
        return True

    def deploy(self, check_spawned_on_every_node=True):
        LOG.info('Deploying simple daemonset to check restarts')
        # there no needs to configure that particular case since it's impossible to interfere with anything
        ds_name = 'restart-checker-ds'
        mcp_docker_registry = self._cluster.determine_mcp_docker_registry()
        opt = {
            'MCC_RESTART_DS_NAME': ds_name,
            'MCP_DOCKER_REGISTRY': mcp_docker_registry
        }
        templates_ds = templates.render_template(
            settings.KAAS_RESTART_DS_YAML, opt)
        json_body = json.dumps(yaml.load(templates_ds, Loader=yaml.SafeLoader))
        self.ds = self._kubectl_client.daemonsets.create(
            name=ds_name,
            namespace='default',
            body=json.loads(json_body))
        LOG.info('Wait for daemonset ready')
        waiters.wait(lambda: self.ds.ready, interval=15, timeout=150,
                     timeout_msg='Restart checker pod were not properly deployed')
        if check_spawned_on_every_node:
            LOG.info("Waiting for all pods are ready")
            waiters.wait(lambda: self._check_spawned_on_every_node(), interval=10, timeout=120)
        LOG.info('Restart DS ready')

    def get_timestamps(self):
        timest = {}
        ds_p_list = self._kubectl_client.pods.list(name_prefix=self.ds.name, namespace=self.ds.namespace)
        for p in ds_p_list:
            ts = p.exec(['/bin/sh', '-c', 'cat /check/checkfile.log']).strip()
            timest[p.node_name] = ts
        return timest

    def save_timestamps(self):
        LOG.info('Saving daemonset start timestamps for each pod')
        self.timestamps = self.get_timestamps()
        LOG.info(self.timestamps)

    def cleanup(self):
        if self.ds:
            LOG.info('Cleaning up restart DS')
            self.ds.delete()
            waiters.wait(lambda: not self._kubectl_client.daemonsets.present(self.ds.name, self.ds.namespace),
                         interval=15, timeout=150,
                         timeout_msg='Restart checker pod were not properly removed within the given time')
            LOG.info('Restart DS cleanup finished')
