#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations


import http
import yaml
import requests
import os
from retry import retry
from pprint import pformat
from six import iteritems
from datetime import datetime, timezone

from si_tests.utils.helpers import skip_rc_retry, relogin_401, retry_auth_500, retry_429
from si_tests import logger
import si_tests.utils.waiters as helpers
from kubernetes.client.rest import ApiException
from urllib3.exceptions import MaxRetryError, ProtocolError

from typing import TYPE_CHECKING
if TYPE_CHECKING:
    from si_tests.clients.k8s.cluster import K8sCluster

LOG = logger.logger


class K8sNamespacedResource(object):
    resource_type = None
    model = None

    def __init__(self, manager, name=None, namespace=None, data=None, log=True):
        self._manager: "K8sBaseManager" = manager
        self._name = name
        self._namespace = namespace
        self._read_cache = None
        if data is not None:
            self._update_cache(data, log=log)

    def __repr__(self):
        uid = 'unknown-uid'
        if self._read_cache is not None:
            uid = self.uid
        return "<{0} name='{1}' namespace='{2}' uuid='{3}'>".format(
            self.__class__.__name__, self.name, self.namespace, uid)

    def _read(self, **kwargs):
        response = self._manager.api.get_namespaced_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            namespace=self.namespace,
            name=self.name,
            plural=self._manager.resource_plural,
            **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _create(self, body, **kwargs):
        response = self._manager.api.create_namespaced_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            namespace=self.namespace,
            plural=self._manager.resource_plural,
            body=body,
            **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _patch(self, body, update_status=False, **kwargs):
        if update_status:
            response = self._manager.api.patch_namespaced_custom_object_status(
                group=self._manager.resource_group,
                version=self._manager.resource_version,
                namespace=self.namespace,
                plural=self._manager.resource_plural,
                name=self.name,
                body=body,
                **kwargs)
        else:
            response = self._manager.api.patch_namespaced_custom_object(
                group=self._manager.resource_group,
                version=self._manager.resource_version,
                namespace=self.namespace,
                plural=self._manager.resource_plural,
                name=self.name,
                body=body,
                **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _replace(self, body, **kwargs):
        response = self._manager.api.replace_namespaced_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            namespace=self.namespace,
            plural=self._manager.resource_plural,
            name=self.name,
            body=body,
            **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _delete(self, body=None, **kwargs):
        if not body:
            body = {}
        self._manager.api.delete_namespaced_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            namespace=self.namespace,
            plural=self._manager.resource_plural,
            name=self.name,
            body=body,
            **kwargs)

    @property
    def data(self) -> dict:
        """Returns dict of k8s object

        Data contains keys like api_version, kind, metadata,
        spec, status or items
        """
        return self.read().to_dict()

    @property
    def name(self) -> str:
        return self._name

    @property
    def namespace(self):
        return self._namespace

    @property
    def uid(self):
        return self.read(cached=True).metadata.uid

    @property
    def generation(self):
        """Read metadata.generation value. Useful to quickly check changes
        :return: Int
        """
        return self.read().metadata.generation

    @property
    def annotations(self):
        """Read metadata.annotations
        :return: dict
        """
        return self.read().metadata.annotations

    def set_annotations(self, annotations):
        """Set annotation for k8s object
        To add or change annotation just use k:v. To remove - set value to None.

        :param annotations: Dict with innotations in k:v format. e.g
        {'annotation.k8s.io':'this_is_annotation', 'other_annotation.k8s.io': None}
        """
        self.patch({'metadata': {'annotations': annotations}})

    def wait_for_new_generation(self, old_generation=None, timeout=300, interval=15):
        """Wait for the new generation of k8s object.
        Can get initial generation by itself (will work properly in case if resources changed not directly. e.g.
        changes should be applied by MCC) OR can get initial generation as separated func param.

        :param old_generation:
        :param timeout:
        :param interval:
        :return:
        """
        if not old_generation:
            current_gen = self.generation
        else:
            current_gen = old_generation

        def new_gen_check():
            if current_gen == self.generation:
                raise Exception(f"{self.resource_type} {self.name} in namespace {self.namespace} does not updated")
            else:
                pass

        helpers.wait_pass(new_gen_check,
                          timeout=timeout,
                          interval=interval,
                          expected=(Exception, ApiException))
        return self

    def _update_cache(self, data, log=True):
        self._read_cache = data
        if log:
            LOG.debug("Read: {}".format(yaml.dump(data, Dumper=yaml.CDumper)))
        self._namespace = data.metadata.namespace
        self._name = data.metadata.name

    @skip_rc_retry((ApiException, MaxRetryError, ProtocolError),
                   delay=10, tries=3)
    @retry_auth_500(delay=40, tries=3)
    @retry_429(delay=40, tries=3)
    @relogin_401
    def read(self, cached=False, _request_timeout=300, **kwargs):
        if not cached or self._read_cache is None:
            self._update_cache(self._read(_request_timeout=_request_timeout,
                                          **kwargs))
        return self._read_cache

    @skip_rc_retry((ApiException, MaxRetryError, ProtocolError),
                   delay=10, tries=3, rcs=[http.HTTPStatus.BAD_REQUEST])
    @relogin_401
    @retry_429(delay=40, tries=3)
    def create(self, body, log_body=True, _request_timeout=300, **kwargs):
        if isinstance(body, dict):
            body = self._manager._prepare_dict(body)
        if log_body:
            LOG.info("K8S API Creating {0} with body:\n{1}".format(
                     self.resource_type, yaml.dump(body)))
        else:
            LOG.debug("K8S API Creating {0} with body:\n{1}".format(
                      self.resource_type, yaml.dump(body)))
        self._update_cache(self._create(body,
                                        _request_timeout=_request_timeout,
                                        **kwargs))
        return self

    @skip_rc_retry(
        (ApiException, MaxRetryError, ProtocolError), delay=10, tries=3,
        rcs=[http.HTTPStatus.BAD_REQUEST, http.HTTPStatus.NOT_FOUND,
             http.HTTPStatus.FORBIDDEN])
    @relogin_401
    @retry_429(delay=40, tries=3)
    def patch(self, body, _request_timeout=300, verbose=True, **kwargs):
        if isinstance(body, dict):
            body = self._manager._prepare_dict(body)
        if verbose:
            LOG.info("K8S API Patching {0} name={1} ns={2} with "
                     "body:\n{3}".format(self.resource_type, self.name,
                                         self.namespace, yaml.dump(body)))
        else:
            LOG.debug("K8S API Patching {0} name={1} ns={2} with "
                      "body:\n{3}".format(self.resource_type, self.name,
                                          self.namespace, yaml.dump(body)))
        self._update_cache(self._patch(body,
                                       _request_timeout=_request_timeout,
                                       **kwargs))
        return self

    @skip_rc_retry(
        (ApiException, MaxRetryError, ProtocolError), delay=10, tries=3,
        rcs=[http.HTTPStatus.BAD_REQUEST, http.HTTPStatus.NOT_FOUND])
    @relogin_401
    @retry_429(delay=40, tries=3)
    def replace(self, body, _request_timeout=300, **kwargs):
        if isinstance(body, dict):
            body = self._manager._prepare_dict(body)
        LOG.debug("K8S API Replacing {0} name={1} ns={2} with "
                  "body:\n{3}".format(self.resource_type, self.name,
                                      self.namespace, yaml.dump(body)))
        self._update_cache(self._replace(body,
                                         _request_timeout=_request_timeout,
                                         **kwargs))
        return self

    @retry((ApiException, MaxRetryError, ProtocolError),
           delay=10, tries=3, logger=LOG)
    @relogin_401
    @retry_429(delay=40, tries=3)
    def delete(self, _request_timeout=300, **kwargs):
        self._delete(_request_timeout=_request_timeout, **kwargs)
        return self

    def __eq__(self, other):
        if not isinstance(other, K8sNamespacedResource):
            return NotImplemented

        if not isinstance(other, self.__class__):
            return False

        return self.uid == other.uid

    def _rollout_restart(self):
        LOG.info(f"Restarting {self.data['kind']} {self.namespace}/{self.name}")
        timestamp = datetime.now(timezone.utc).strftime("%Y-%m-%dT%H:%M:%SZ")
        old_generation = self.generation
        self.patch(
            {
                "spec": {
                    "template": {
                        "metadata": {
                            "annotations": {"kubectl.kubernetes.io/restartedAt": timestamp}
                        }
                    }
                }
            }
        )
        # NOTE(vsaenko): it may take some time to bump generation
        # ensure its updated as later code may rely on it.
        self.wait_for_new_generation(old_generation)


class K8sBaseManager(object):
    """
    This class define both namespaced and cluster scoped methods.
    It is up to resource to decide which ones to use.
    """

    def __init__(self, cluster: "K8sCluster"):
        self._cluster = cluster

    @property
    def api(self):
        return self._cluster.api_custom

    def _list(self, namespace, **kwargs):
        response = self.api.list_namespaced_custom_object(
            group=self.resource_group,
            version=self.resource_version,
            namespace=namespace,
            plural=self.resource_plural,
            **kwargs)
        return self._deserialize_custom_object_list(
            response, self.model, self.resource_class.model)

    def _list_all(self, **kwargs):
        response = self.api.list_cluster_custom_object(
            group=self.resource_group,
            version=self.resource_version,
            plural=self.resource_plural,
            **kwargs)
        return self._deserialize_custom_object_list(
            response, self.model, self.resource_class.model)

    @property
    def resource_type(self):
        return self.resource_class.resource_type

    def get(self, name=None, namespace=None, data=None):
        namespace = namespace or self._cluster.default_namespace
        return self.resource_class(self, name, namespace, data)

    def create(self, name=None, namespace=None, body=None, **kwargs):
        return self.get(name=name, namespace=namespace).create(body, **kwargs)

    def update(self, name=None, namespace=None, body=None, **kwargs):
        return self.get(name=name, namespace=namespace).patch(body, **kwargs)

    def __resource_from_data(self, data, log=True):
        return self.resource_class(self, data=data, log=log)

    def __list_filter(self, items, name_prefix=None):
        if name_prefix is not None:
            items = [item for item in items if
                     item.metadata.name.startswith(name_prefix)]
        return items

    def __list_to_resource(self, items):
        dump_items = [
            {
                'name': item.metadata.name,
                'namespace': item.metadata.namespace
            }
            for item in items]
        LOG.debug(f"Read '{self.resource_type}':\n{yaml.dump(dump_items, Dumper=yaml.CDumper)}")
        return [self.__resource_from_data(item, log=False) for item in items]

    def _deserialize_custom_object(self, data, object_klass):
        return self.api.api_client._ApiClient__deserialize(
            data, object_klass)

    def _deserialize_custom_object_list(self, data, list_klass, object_klass):
        result_list = self.api.api_client._ApiClient__deserialize(data, list_klass)
        result_items = [
            self._deserialize_custom_object(item, object_klass)
            for item in result_list.items]
        result_list.items = result_items
        return result_list

    def _prepare_dict(self, data):
        # After getting k8s objects using .to_dict(), the resulting dict
        # cannot be used in create or patch, because of difference
        # in the attributes from `attribute_map`.
        # Convert attribute name to json key in model definition for request.
        # Keys that are missing in `attribute_map` will be kept as is.
        attribute_map = {
            'api_version': 'apiVersion',
        }
        return {attribute_map.get(attr, attr): value
                for attr, value in data.items()}

    def list(self, namespace=None, name_prefix=None, **kwargs):
        namespace = namespace or self._cluster.default_namespace
        items = self.list_raw(namespace=namespace, **kwargs).items
        items = self.__list_filter(items, name_prefix=name_prefix)
        return self.__list_to_resource(items)

    def list_all(self, name_prefix=None, **kwargs):
        items = self.list_raw(**kwargs).items
        items = self.__list_filter(items, name_prefix=name_prefix)
        return self.__list_to_resource(items)

    @retry((ApiException, MaxRetryError, ProtocolError),
           delay=10, tries=3, logger=LOG)
    @retry_auth_500(delay=40, tries=3)
    @relogin_401
    def list_raw(self, namespace=None, _request_timeout=300, **kwargs):
        """A special method to return raw data -
           full info about listed object
           Returns object like V1PodList (not K8sPod list)
        """
        if namespace:
            return self._list(namespace=namespace,
                              _request_timeout=_request_timeout,
                              **kwargs)
        else:
            return self._list_all(_request_timeout=_request_timeout,
                                  **kwargs)

    def list_starts_with(self, pattern, namespace=None, sort_by_len=False, **kwargs):
        """List items that have pattern at the beginning of their names.
           Items are fetched from all namespaces (by default) or
           from the specified namespace. Method returns all items with the same
           name prefix. sort_by_len sorts resulting list by items name length.
        """
        if not namespace:
            items = [item for item in self.list_all(**kwargs)
                     if item.name.startswith(pattern)]
        else:
            items = [item for item in self.list(namespace=namespace, **kwargs)
                     if item.name.startswith(pattern)]
        if sort_by_len:
            items.sort(key=lambda x: len(x.name))
        return items

    def present(self, name, namespace=None):
        """Check that specified object exists in the namespace"""
        return any([obj.name for obj in self.list(namespace=namespace)
                    if obj.name == name])

    def present_all(self, name):
        """Check that specified object exists in the k8s cluter scope"""
        return any([obj.name for obj in self.list_all()
                    if obj.name == name])

    @retry((ApiException, MaxRetryError, ProtocolError), delay=10, tries=3, logger=LOG)
    @relogin_401
    def get_api_groups(self):
        """Get all available api groups from the current k8s cluster

        :rtype dict: {<api group name>: <api group_version>, ...}
                     Example:  {'metal3.io': 'metal3.io/v1alpha1', ...}
        """
        header_params = {'Accept': 'application/json', 'Content-Type': 'application/json'}
        auth_settings = ['BearerToken']
        result = self.api.api_client.call_api(resource_path='/apis/',
                                              method='GET',
                                              response_type="V1APIGroupList",
                                              header_params=header_params,
                                              auth_settings=auth_settings)

        LOG.debug(f"Response for V1APIGroupList:\n{result}")
        if type(result) is list or type(result) is tuple:
            api_groups = {api_group.name: api_group.preferred_version.group_version
                          for api_group in result[0].groups}
        elif hasattr(result, 'groups'):
            api_groups = {api_group.name: api_group.preferred_version.group_version
                          for api_group in result.groups}
        else:
            raise Exception(f"Invalid response from '/apis/'. "
                            f"Expected V1APIGroupList as an object or list/tuple of objects, got:\n{result}")
        LOG.debug(f"api_groups: {api_groups.keys()}")
        return api_groups

    @retry((ApiException, MaxRetryError, ProtocolError), delay=10, tries=3, logger=LOG)
    @relogin_401
    def get_api_resources(self, api_group_version):
        """Get api resources for the specified api group_version

        :rtype list: plural names of the resources in the group
                     Example: ['dnsmasqs', 'baremetalhosts', ...]
        """
        header_params = {'Accept': 'application/json', 'Content-Type': 'application/json'}
        auth_settings = ['BearerToken']
        result = self.api.api_client.call_api(resource_path=f'/apis/{api_group_version}',
                                              method='GET',
                                              response_type="V1APIResourceList",
                                              header_params=header_params,
                                              auth_settings=auth_settings)
        LOG.debug(f"Response for {api_group_version} V1APIResourceList:\n{result}")
        if type(result) is list or type(result) is tuple:
            api_resources = [api_resource.name for api_resource in result[0].resources]
        elif hasattr(result, 'resources'):
            api_resources = [api_resource.name for api_resource in result.resources]
        else:
            raise Exception(f"Invalid response from '/apis/{api_group_version}'. "
                            f"Expected V1APIResourceList as an object or list/tuple of objects, got:\n{result}")
        LOG.debug(f"api_resources: {api_resources}")
        return api_resources

    @property
    def available(self):
        """Check that the current resource is available in the current k8s cluster

        :rtype bool: True if the resource found in the cluster, else False
        """
        api_groups = self.get_api_groups()
        if self.resource_group in api_groups:
            api_resources = self.get_api_resources(api_groups[self.resource_group])
            if self.resource_plural in api_resources:
                return True
        return False


class K8sClusterScopedResource(K8sNamespacedResource):

    def _read(self, **kwargs):
        response = self._manager.api.get_cluster_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            plural=self._manager.resource_plural,
            name=self.name,
            **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _create(self, body, **kwargs):
        response = self._manager.api.create_cluster_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            plural=self._manager.resource_plural,
            body=body,
            **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _patch(self, body, update_status=False, **kwargs):
        if update_status:
            response = self._manager.api.patch_cluster_custom_object_status(
                group=self._manager.resource_group,
                version=self._manager.resource_version,
                plural=self._manager.resource_plural,
                name=self.name,
                body=body,
                **kwargs)
        else:
            response = self._manager.api.patch_cluster_custom_object(
                group=self._manager.resource_group,
                version=self._manager.resource_version,
                plural=self._manager.resource_plural,
                name=self.name,
                body=body,
                **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _replace(self, body, **kwargs):
        response = self._manager.api.replace_cluster_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            plural=self._manager.resource_plural,
            name=self.name,
            body=body,
            **kwargs)
        return self._manager._deserialize_custom_object(
            response, self.model)

    def _delete(self, body=None, **kwargs):
        if not body:
            body = {}
        self._manager.api.delete_cluster_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            plural=self._manager.resource_plural,
            name=self.name,
            body=body,
            **kwargs)


class BaseK8sModel(object):

    @property
    def api_version(self):
        return self._api_version

    @api_version.setter
    def api_version(self, api_version):
        self._api_version = api_version

    @property
    def kind(self):
        return self._kind

    @kind.setter
    def kind(self, kind):
        self._kind = kind

    @property
    def metadata(self):
        return self._metadata

    @metadata.setter
    def metadata(self, metadata):
        self._metadata = metadata

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return not self == other

    def to_dict(self):
        result = {}
        for attr, _ in iteritems(self.openapi_types):
            value = getattr(self, attr)
            if isinstance(value, list):
                result[attr] = list(map(
                    lambda x: x.to_dict() if hasattr(x, "to_dict") else x,
                    value
                ))
            elif hasattr(value, "to_dict"):
                result[attr] = value.to_dict()
            elif isinstance(value, dict):
                result[attr] = dict(map(
                    lambda item: (item[0], item[1].to_dict())
                    if hasattr(item[1], "to_dict") else item,
                    value.items()
                ))
            else:
                result[attr] = value
        return result

    def to_str(self):
        return pformat(self.to_dict())

    def __repr__(self):
        return self.to_str()


class BaseModel(BaseK8sModel):
    """Base model compatible to kubernetes client"""
    openapi_types = {
        'api_version': 'str',
        # use 'object' type because kubernetes ApiClient
        # cannot deserialize objects using external types
        'kind': 'str',
        'metadata': 'V1ObjectMeta',
        'spec': 'object',
        'status': 'object',

    }

    attribute_map = {
        'api_version': 'apiVersion',
        'kind': 'kind',
        'metadata': 'metadata',
        'spec': 'spec',
        'status': 'status',
    }

    def __init__(self, api_version=None, kind=None, metadata=None, spec=None,
                 status=None):
        self._api_version = None
        self._spec = None
        self._status = None
        self._kind = None
        self._metadata = None
        self.discriminator = None

        if api_version is not None:
            self.api_version = api_version
        if spec is not None:
            self.spec = spec
        if status is not None:
            self.status = status
        if kind is not None:
            self.kind = kind
        if metadata is not None:
            self.metadata = metadata

    @property
    def spec(self):
        return self._spec

    @spec.setter
    def spec(self, spec):
        self._spec = spec

    @property
    def status(self):
        return self._status

    @status.setter
    def status(self, status):
        self._status = status


class BaseModelList(BaseK8sModel):

    openapi_types = {
        'api_version': 'str',
        # use 'object' type because kubernetes ApiClient
        # cannot deserialize objects using external types
        'kind': 'str',
        'metadata': 'V1ListMeta',
        'items': 'list[object]',
    }

    attribute_map = {
        'api_version': 'apiVersion',
        'kind': 'kind',
        'metadata': 'metadata',
        'items': 'items',
    }

    def __init__(self, api_version=None, items=None, kind=None, metadata=None):

        self._api_version = None
        self._items = None
        self._kind = None
        self._metadata = None
        self.discriminator = None

        if api_version is not None:
            self.api_version = api_version
        self.items = items
        if kind is not None:
            self.kind = kind
        if metadata is not None:
            self.metadata = metadata

    @property
    def api_version(self):
        return self._api_version

    @api_version.setter
    def api_version(self, api_version):
        self._api_version = api_version

    @property
    def items(self):
        return self._items

    @items.setter
    def items(self, items):
        if items is None:
            raise ValueError("Invalid value for `items`, must not be `None`")

        self._items = items


def read_yaml_str(yaml_str):
    """ load yaml from string helper """
    return yaml.safe_load(yaml_str)


def read_yaml_file(file_path, *args):
    """ load yaml from joined file_path and *args helper """
    with open(os.path.join(file_path, *args)) as f:
        return yaml.safe_load(f)


def read_yaml_url(yaml_file_url):
    """ load yaml from url helper """
    return yaml.safe_load(requests.get(yaml_file_url).text)


class CrdStatusModel(BaseModel):
    """Base model compatible to kubernetes client"""
    openapi_types = {
        'api_version': 'str',
        # use 'object' type because kubernetes ApiClient
        # cannot deserialize objects using external types
        'kind': 'str',
        'stages': 'list[object]',
        'metadata': 'V1ObjectMeta',
        'spec': 'object',
        'status': 'object',

    }

    attribute_map = {
        'api_version': 'apiVersion',
        'kind': 'kind',
        'stages': 'stages',
        'metadata': 'metadata',
        'spec': 'spec',
        'status': 'status',
    }

    def __init__(self, api_version=None, kind=None, metadata=None, spec=None, status=None, stages=None):
        super().__init__(api_version, kind, metadata, spec, status)
        self.stages = stages

    @property
    def stages(self):
        return self._stages

    @stages.setter
    def stages(self, stages):
        self._stages = stages


class CrdStatusModelList(BaseModelList):

    openapi_types = {
        'api_version': 'str',
        # use 'object' type because kubernetes ApiClient
        # cannot deserialize objects using external types
        'kind': 'str',
        'metadata': 'V1ListMeta',
        'items': 'list[object]',
        'stages': 'list[object]',

    }

    attribute_map = {
        'api_version': 'apiVersion',
        'kind': 'kind',
        'metadata': 'metadata',
        'items': 'items',
        'stages': 'stages',
    }

    def __init__(self, api_version=None, items=None, kind=None, metadata=None, stages=None):

        super().__init__(api_version, items, kind, metadata)
        self.stages = stages

    @property
    def stages(self):
        return self._stages

    @stages.setter
    def stages(self, stages):
        self._stages = stages
