#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import base64
import tempfile
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.kcm_manager import Manager

LOG = logger.logger


class KOFManager(object):
    """KOF installation manager"""

    def __init__(self, kcm_manager: "Manager"):
        self.kcm_manager: "Manager" = kcm_manager
        self._helmmgr = None

    @property
    def helmmgr(self):
        if not self._helmmgr:
            self._helmmgr = self.kcm_manager.helmmgr
        return self._helmmgr

    @property
    def template_path(self):
        """Return kof expected pods template based on helm version
        :return:
        """
        # NOTE(va4st): Assuming that all kof components have same versions. As a base take kof-operators as first
        # kof chart installed into cluster.
        if self.is_kof_operators_present():
            kof_ver = 'kof-' + self.get_kof_operators_chart_metadata()['version'].replace('.', '-')
        elif self.is_kof_istio_present():
            kof_ver = 'kof-' + self.get_kof_istio_chart_metadata()['version'].replace('.', '-')
        else:
            raise RuntimeError('Not possible to gather kof version. kof-operators and kof-istio charts is not present')
        path = settings.EXPECTED_PODS_TEMPLATES_DIR + 'kof/' + kof_ver + '.yaml'
        return path

    def is_kof_operators_installed(self):
        """Verify that required CRD (opentelemetry) present"""
        return True if self.kcm_manager.api.opentelemetry_collectors.available else False

    def is_kof_operators_present(self, verbose=False) -> bool:
        """Verify that kof-operators HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KSI_KOF_OP_CHART_NAME, verbose)

    def is_kof_mothership_present(self, verbose=False) -> bool:
        """Verify that kof-mothership HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KSI_KOF_MS_CHART_NAME, verbose)

    def is_kof_regional_present(self, verbose=False) -> bool:
        """Verify that kof-regional HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KSI_KOF_RE_CHART_NAME, verbose)

    def is_kof_child_present(self, verbose=False) -> bool:
        """Verify that kof-child HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KSI_KOF_CH_CHART_NAME, verbose)

    def is_kof_istio_present(self, verbose=False) -> bool:
        """Verify that kof-istio HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KSI_KOF_IS_CHART_NAME, verbose)

    def is_kof_collectors_present(self, verbose=False) -> bool:
        """Verify that kof-istio HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KSI_KOF_CO_CHART_NAME, verbose)

    def kof_op_deployed(self) -> bool:
        """Verify that kof-operators chart deployed"""
        return True if self.get_kof_operators_chart_status() == 'deployed' else False

    def kof_ms_deployed(self) -> bool:
        """Verify that kof-mothership chart deployed"""
        return True if self.get_kof_mothership_chart_status() == 'deployed' else False

    def kof_re_deployed(self) -> bool:
        """Verify that kof-regional chart deployed"""
        return True if self.get_kof_regional_chart_status() == 'deployed' else False

    def kof_ch_deployed(self) -> bool:
        """Verify that kof-child chart deployed"""
        return True if self.get_kof_child_chart_status() == 'deployed' else False

    def kof_is_deployed(self) -> bool:
        """Verify that kof-istio chart deployed"""
        return True if self.get_kof_istio_chart_status() == 'deployed' else False

    def kof_co_deployed(self) -> bool:
        """Verify that kof-collectors chart deployed"""
        return True if self.get_kof_collectors_chart_status() == 'deployed' else False

    def get_kof_operators_chart_status(self, verbose=False):
        meta = self.get_kof_operators_chart_metadata()
        if not meta:
            return None
        status = meta.get('status', None)
        if verbose:
            LOG.info(f"{settings.KSI_KOF_OP_CHART_NAME} chart status is {status}")
        return status

    def get_kof_operators_chart_metadata(self):
        if self.helmmgr.is_release_present(settings.KSI_KOF_OP_CHART_NAME):
            return self.helmmgr.get_metadata(settings.KSI_KOF_OP_CHART_NAME, settings.KSI_KOF_NAMESPACE)
        else:
            return None

    def get_kof_mothership_chart_status(self, verbose=False):
        meta = self.get_kof_mothership_chart_metadata()
        if not meta:
            return None
        status = meta.get('status', None)
        if verbose:
            LOG.info(f"{settings.KSI_KOF_MS_CHART_NAME} chart status is {status}")
        return status

    def get_kof_mothership_chart_metadata(self):
        if self.helmmgr.is_release_present(settings.KSI_KOF_MS_CHART_NAME):
            return self.helmmgr.get_metadata(settings.KSI_KOF_MS_CHART_NAME, settings.KSI_KOF_NAMESPACE)
        else:
            return {}

    def get_kof_regional_chart_status(self, verbose=False):
        meta = self.get_kof_regional_chart_metadata()
        if not meta:
            return None
        status = meta.get('status', None)
        if verbose:
            LOG.info(f"{settings.KSI_KOF_RE_CHART_NAME} chart status is {status}")
        return status

    def get_kof_regional_chart_metadata(self):
        if self.helmmgr.is_release_present(settings.KSI_KOF_RE_CHART_NAME):
            return self.helmmgr.get_metadata(settings.KSI_KOF_RE_CHART_NAME, settings.KSI_KOF_NAMESPACE)
        else:
            return {}

    def get_kof_istio_chart_status(self, verbose=False):
        meta = self.get_kof_istio_chart_metadata()
        if not meta:
            return None
        status = meta.get('status', None)
        if verbose:
            LOG.info(f"{settings.KSI_KOF_IS_CHART_NAME} chart status is {status}")
        return status

    def get_kof_collectors_chart_metadata(self):
        if self.helmmgr.is_release_present(settings.KSI_KOF_CO_CHART_NAME):
            return self.helmmgr.get_metadata(settings.KSI_KOF_CO_CHART_NAME, settings.KSI_KOF_NAMESPACE)
        else:
            return {}

    def get_kof_collectors_chart_status(self, verbose=False):
        meta = self.get_kof_collectors_chart_metadata()
        if not meta:
            return None
        status = meta.get('status', None)
        if verbose:
            LOG.info(f"{settings.KSI_KOF_CO_CHART_NAME} chart status is {status}")
        return status

    def get_kof_istio_chart_metadata(self):
        if self.helmmgr.is_release_present(settings.KSI_KOF_IS_CHART_NAME):
            return self.helmmgr.get_metadata(settings.KSI_KOF_IS_CHART_NAME, settings.KSI_KOF_ISTIO_NAMESPACE)
        else:
            return {}

    def get_kof_child_chart_status(self, verbose=False):
        meta = self.get_kof_child_chart_metadata()
        if not meta:
            return None
        status = meta.get('status', None)
        if verbose:
            LOG.info(f"{settings.KSI_KOF_CH_CHART_NAME} chart status is {status}")
        return status

    def get_kof_child_chart_metadata(self):
        if self.helmmgr.is_release_present(settings.KSI_KOF_CH_CHART_NAME):
            return self.helmmgr.get_metadata(settings.KSI_KOF_CH_CHART_NAME, settings.KSI_KOF_NAMESPACE)
        else:
            return {}

    @staticmethod
    def get_global_values(registry, reg_ca_secret=None):
        values = {
            'global': {
                'registry': registry,
                'imageRegistry': registry,
                'image': {
                    'registry': registry,
                },
                'hub': f"{registry}/istio",
                'helmChartsRepo': f"oci://{registry}/charts"
            },
            'cert-manager-istio-csr': {
                'image': {
                    'repository': f"{registry}/jetstack/cert-manager-istio-csr"
                }
            },
            'cert-manager-service-template': {
                'verifyJobImage': f"{registry}/alpine/helm:3.14.0",
                'repo': {
                    'url': f"oci://{registry}/charts"
                }
            },
            'ingress-nginx-service-template': {
                'verifyJobImage': f"{registry}/alpine/helm:3.14.0",
                'repo': {
                    'url': f"oci://{registry}/charts"
                }
            },
            'cluster-api-visualizer': {
                'image': {
                    'repository': registry,
                }
            },
            'grafana-operator': {
                'image': {
                    'repository': f"{registry}/grafana/grafana-operator"
                }
            },
            'external-dns': {
                'image': {
                    'repository': f"{registry}/external-dns/external-dns"
                }
            },
            'jaeger-operator': {
                'image': {
                    'repository': f"{registry}/jaegertracing/jaeger-operator"
                }
            },
            'opencost': {
                'opencost': {
                    'exporter': {
                        'image': {
                            'registry': registry
                        }
                    },
                    'ui': {
                        'image': {
                            'registry': registry
                        }
                    }
                }
            },
            'opentelemetry-operator': {
                'manager': {
                    'image': {
                        'repository': f"{registry}/opentelemetry-operator/opentelemetry-operator"
                    },
                    'collectorImage': {
                        'repository': f"{registry}/otel/opentelemetry-collector-contrib"
                    }
                },
                'kubeRBACProxy': {
                    'image': {
                        'repository': f"{registry}/brancz/kube-rbac-proxy"
                    }
                }
            },
            'storage': {
                'cert-manager': {
                    'cluster-issuer': {
                        'provider': 'self-signed'
                    }
                }
            },
            'kcm': {
                'kof': {
                    'operator': {
                        'image': {
                            'repository': 'kof/kof-operator-controller'
                        }
                    },
                    'repo': {
                        'spec': {
                            'url': f"oci://{registry}/charts"
                        }
                    }
                }
            }
        }
        if reg_ca_secret:
            ref = {
                'name': reg_ca_secret
            }
            values['cert-manager-service-template']['repo']['certSecretRef'] = ref
            values['ingress-nginx-service-template']['repo']['certSecretRef'] = ref
            values['kcm']['kof']['repo']['spec']['certSecretRef'] = ref
        return values

    @staticmethod
    def get_mothership_values(install_templates=True,
                              storage_class=None,
                              auto_dns=None,
                              dns_secret_name=None):

        def get_dns_dict(profile, secret):
            dns_dict = {
                'kof': {
                    'clusterProfiles': {
                        profile: {
                            'matchLabels': {
                                f"k0rdent.mirantis.com/{profile}": "true"
                            },
                            'secrets': [
                                secret
                            ]
                        }
                    }
                }
            }
            return dns_dict

        values = {
            'kcm': {
                'installTemplates': install_templates
            },
            'victoria-metrics-operator': {
                'crds': {
                    'cleanup': {
                        'enabled': True
                    }
                }
            }
        }

        if storage_class:
            values['global']['storageClass'] = storage_class

        auto_dns_config = ''
        if auto_dns == 'aws':
            auto_dns_config = get_dns_dict('kof-aws-dns-secrets', dns_secret_name)
        elif auto_dns == 'azure':
            auto_dns_config = get_dns_dict('kof-azure-dns-secrets', dns_secret_name)

        if auto_dns_config:
            values['kcm']['kof'] = auto_dns_config['kof']

        LOG.info(f"kof-mothership values:\n{yaml.dump(values)}")

        return values

    @staticmethod
    def mothership_custom_repo_chunk(custom_repo):
        custom_repo_val = {
            'cert-manager-service-template': {
                'helm': {
                    'repository': {
                        'name': 'cert-manager',
                        'url': f"oci://{custom_repo}/charts",
                        'type': 'oci'
                    },
                    'charts': [
                        {
                            'name': 'cert-manager',
                            'version': 'v1.17.2'
                        }
                    ]
                },
                'namespace': 'kcm-system'
            }
        }
        return custom_repo_val

    @staticmethod
    def get_kof_collectors_values(mesh=None, value=None, insecure=False):
        """Get values for kof-collectors chart deployment
        It's different values for Istio and all other variants

        :param insecure:
        :param mesh: Istio or anything else
        :param value: In case of Istio - this is regional cluster name. In any other case - domain
        :return:
        """

        if mesh == 'istio':
            values = {
                'kcm': {
                    # always enable monitoring for kcm
                    'monitoring': True
                },
                'kof': {
                    'basic_auth': False,
                    'logs': {
                        'endpoint': f"http://{value}-logs-insert:9481/insert/opentelemetry/v1/logs"
                    },
                    'metrics': {
                        'endpoint': f"http://{value}-vminsert:8480/insert/0/prometheus/api/v1/write"
                    },
                    'traces': {
                        'endpoint': f"http://{value}-jaeger-collector:4318"
                    }
                },
                'opencost': {
                    'opencost': {
                        'prometheus': {
                            'existingSecretName': "",
                            'external': {
                                'url': f"http://{value}-vmselect:8481/select/0/prometheus"
                            }
                        }
                    }
                }
            }
        else:
            values = {
                'kcm': {
                    # always enable monitoring for kcm
                    'monitoring': True
                },
                'kof': {
                    'logs': {
                        'endpoint': f"https://vmauth.{value}/vls/insert/opentelemetry/v1/logs"
                    },
                    'metrics': {
                        'endpoint': f"https://vmauth.{value}/vm/insert/0/prometheus/api/v1/write"
                    },
                    'traces': {
                        'endpoint': f"https://jaeger.{value}/collector"
                    }
                },
                'opencost': {
                    'opencost': {
                        'prometheus': {
                            'external': {
                                'url': f"https://vmauth.{value}/vm/select/0/prometheus"
                            }
                        }
                    }
                }
            }

        if insecure:
            tls_opts = {
                'kof': {
                    'logs': {
                        "tls_options": {
                            "insecure_skip_verify": True
                        }
                    },
                    'metrics': {
                        "tls_options": {
                            "insecure_skip_verify": True
                        }
                    },
                    'traces': {
                        "tls_options": {
                            "insecure_skip_verify": True
                        }
                    }
                }
            }
            utils.merge_dicts(values, tls_opts)

        return values

    def install_named_kof_chart(self, chart_name, chart_namespace, values=None, source=None, custom_registry=None):
        ep_values = None
        if not source:
            source = settings.KSI_KOF_SOURCE
        cert_data = None

        if source == 'enterprise':
            registry = settings.KSI_KOF_REPO_DICT['enterprise-repo']
        elif source == 'custom-enterprise':
            kcm_helmrepo = self.kcm_manager.api.helmrepositories.get(
                settings.KSI_KCM_TEMPLATES_REPO_NAME, settings.KCM_NAMESPACE)
            if not custom_registry:
                custom_registry = settings.KSI_KOF_CUSTOM_REGISTRY or kcm_helmrepo.url_raw
            assert custom_registry, ('KSI_KOF_CUSTOM_REGISTRY can not be empty with '
                                     'KSI_KOF_SOURCE=custom-enterprise')
            registry = custom_registry
            cert_secret_name = self.kcm_manager.mgmt.cert_secret_name
            if cert_secret_name:
                LOG.warning(f"Automated gathering cert data from secret {cert_secret_name}")
                cert_secret = self.kcm_manager.api.secrets.get(name=cert_secret_name,
                                                               namespace=settings.KCM_NAMESPACE)
                cert_data = base64.b64decode(cert_secret.read().data['ca.crt']).decode("utf-8")
            else:
                LOG.warning("No cert secret name found in management object. Assuming no certs needed for registry.")
            ep_values = self.get_global_values(registry, reg_ca_secret=cert_secret_name)
        else:
            registry = settings.KSI_KOF_REPO_DICT['oss-repo']

        if values and ep_values:
            utils.merge_dicts(values, ep_values)
        elif ep_values:
            values = ep_values

        if values:
            LOG.info(f"Values for {chart_name} installation: \n{yaml.dump(values)}")
        else:
            LOG.info(f"No values for {chart_name} provided. Using defaults.")

        chart_path = f"oci://{registry}/charts/{chart_name}"

        if values:
            with tempfile.NamedTemporaryFile(mode="w") as values_path:
                yaml.dump(values, values_path)
                self.helmmgr.install_chart(chart_name=chart_name,
                                           chart_path=chart_path,
                                           version=settings.KSI_KOF_CHART_VERSION,
                                           values_path=[values_path.name],
                                           namespace=chart_namespace,
                                           source=source,
                                           cert_text=cert_data)
        else:
            self.helmmgr.install_chart(chart_name=chart_name,
                                       chart_path=chart_path,
                                       version=settings.KSI_KOF_CHART_VERSION,
                                       namespace=chart_namespace,
                                       source=source)

    def install_kof_istio(self, source=None):
        if self.is_kof_istio_present():
            LOG.info(f"{settings.KSI_KOF_IS_CHART_NAME} chart already present in cluster")
            return
        self.install_named_kof_chart(settings.KSI_KOF_IS_CHART_NAME, settings.KSI_KOF_ISTIO_NAMESPACE, source=source)

    def uninstall_kof_istio(self):
        if not self.is_kof_istio_present():
            LOG.info(f"{settings.KSI_KOF_IS_CHART_NAME} chart not exist in cluster")
            return
        self.helmmgr.delete_chart(chart_name=settings.KSI_KOF_IS_CHART_NAME,
                                  namespace=settings.KSI_KOF_ISTIO_NAMESPACE,
                                  cascade='foreground')

    def install_kof_collectors(self, values=None, source=None):
        if self.is_kof_collectors_present():
            LOG.info(f"{settings.KSI_KOF_CO_CHART_NAME} chart already present in cluster")
            return
        self.install_named_kof_chart(settings.KSI_KOF_CO_CHART_NAME, settings.KSI_KOF_NAMESPACE,
                                     values=values, source=source)

    def uninstall_kof_collectors(self):
        if not self.is_kof_collectors_present():
            LOG.info(f"{settings.KSI_KOF_CO_CHART_NAME} chart not exist in cluster")
            return
        self.helmmgr.delete_chart(chart_name=settings.KSI_KOF_CO_CHART_NAME,
                                  namespace=settings.KSI_KOF_NAMESPACE,
                                  cascade='foreground')

    def install_kof_operators(self, source=None):
        if self.is_kof_operators_present():
            LOG.info(f"{settings.KSI_KOF_OP_CHART_NAME} chart already present in cluster")
            return
        self.install_named_kof_chart(settings.KSI_KOF_OP_CHART_NAME, settings.KSI_KOF_NAMESPACE, source=source)

    def uninstall_kof_operators(self):
        if not self.is_kof_operators_present():
            LOG.info(f"{settings.KSI_KOF_OP_CHART_NAME} chart not exist in cluster")
            return
        self.helmmgr.delete_chart(chart_name=settings.KSI_KOF_OP_CHART_NAME,
                                  namespace=settings.KSI_KOF_NAMESPACE,
                                  cascade='foreground')

    def install_kof_mothership(self, values=None, source=None):
        if self.is_kof_mothership_present():
            LOG.info(f"{settings.KSI_KOF_MS_CHART_NAME} chart already present in cluster")
            return
        self.install_named_kof_chart(settings.KSI_KOF_MS_CHART_NAME, settings.KSI_KOF_NAMESPACE,
                                     values=values, source=source)

    def uninstall_kof_mothership(self):
        if not self.is_kof_mothership_present():
            LOG.info(f"{settings.KSI_KOF_MS_CHART_NAME} chart not present in cluster")
            return
        self.helmmgr.delete_chart(chart_name=settings.KSI_KOF_MS_CHART_NAME,
                                  namespace=settings.KSI_KOF_NAMESPACE,
                                  cascade='foreground')

    def install_kof_regional(self, values=None, source=None):
        if self.is_kof_regional_present():
            LOG.info(f"{settings.KSI_KOF_RE_CHART_NAME} chart already present in cluster")
            return
        self.install_named_kof_chart(settings.KSI_KOF_RE_CHART_NAME, settings.KSI_KOF_NAMESPACE,
                                     values=values, source=source)

    def uninstall_kof_regional(self):
        if not self.is_kof_regional_present():
            LOG.info(f"{settings.KSI_KOF_RE_CHART_NAME} chart not present in cluster")
            return
        self.helmmgr.delete_chart(chart_name=settings.KSI_KOF_RE_CHART_NAME,
                                  namespace=settings.KSI_KOF_NAMESPACE,
                                  cascade='foreground')

    def install_kof_child(self, values=None, source=None):
        if self.is_kof_child_present():
            LOG.info(f"{settings.KSI_KOF_CH_CHART_NAME} chart already present in cluster")
            return
        self.install_named_kof_chart(settings.KSI_KOF_CH_CHART_NAME, settings.KSI_KOF_NAMESPACE,
                                     values=values, source=source)

    def uninstall_kof_child(self):
        if not self.is_kof_child_present():
            LOG.info(f"{settings.KSI_KOF_CH_CHART_NAME} chart not present in cluster")
            return
        self.helmmgr.delete_chart(chart_name=settings.KSI_KOF_CH_CHART_NAME,
                                  namespace=settings.KSI_KOF_NAMESPACE,
                                  cascade='foreground')
