#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import pytest
import random
import time

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Cluster, Machine, Manager  # noqa: F401
from si_tests.utils import waiters
from kubernetes.client.rest import ApiException

LOG = logger.logger


"""WARNING! These tests will destroy a control and a worker Machine and BMH in the the provided Child cluster"""


is_child_worker_test_failed = False
is_child_control_test_failed = False


def get_machine_with_bmhinventory(machine_type, ns, cluster, kaas_manager):
    """Select a Machine of the required type, which uses a BMHInventory object"""
    machines = cluster.get_machines(machine_type=machine_type, machine_status="Ready")
    for machine in machines:
        bmh_name = machine.get_bmh_name()
        if (kaas_manager.api.kaas_baremetalhostinventories.present(name=bmh_name, namespace=machine.namespace)):
            bmh = ns.get_baremetalhost(name=bmh_name)
            bmhinventory = ns.get_baremetalhostinventory(name=bmh_name)
            return bmh, bmhinventory, machine
    else:
        raise Exception(f"BMH Inventory objects bound to '{machine_type}' Machines not found in the cluster")


class TestControlBMHInventory(object):
    """Test asynchronous delete of BMHInventory before deleting a Machine

    Scenario steps for a control Machine:
    - Ensure that error is propagated from BMH status to BMHInventory status in case
      if BMC password from Secret doesn't match the BMC password on the hardware node.
      Also check that BMHInventory is updating it's status even after deletion request.
    - Make a deletion API request for BMHInventory object and check that it is still exists
      and it didn't trigger BMH deprovision
    - Delete Machine object using 'spec.delete: true', then ensure that previously deleted
      BMHInventory is now actually removed, as well as BMH object
    """

    @pytest.fixture(autouse=True, scope='function')
    def control_test_result_check(self, request):
        """Skip the rest of tests in the class after any test is failed"""
        global is_child_control_test_failed
        # Check if the previous update steps failed
        if is_child_control_test_failed:
            msg = ("Skip test for a Control node befause a previous test for the node failed")
            LOG.info(msg)
            pytest.skip(msg)

        yield

        # Check the result of the current step
        test_passed = (hasattr(request.node, 'rep_call') and request.node.rep_call.passed)
        if not test_passed:
            is_child_control_test_failed = True

    @pytest.fixture(autouse=True, scope='class')
    def get_control(self, kaas_manager):
        """Get necessary objects to test just once per class"""
        ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        cluster = ns.get_cluster(settings.TARGET_CLUSTER)
        bmh, bmhinventory, machine = get_machine_with_bmhinventory("control", ns, cluster, kaas_manager)
        LOG.banner(f"Selected the control Machine to test: '{machine.namespace}/{machine.name}'")
        yield ns, cluster, bmh, bmhinventory, machine

    @pytest.fixture(autouse=True)
    def setup_control(self, get_control, kaas_manager):
        """Set class instance attributes before each test for tests convience

        For each test case, pytest generates a new class instance with empty attributes.

        Set instance attributes with necessary objects, instead of unpacking the tuple
        with these attributes from the fixture in each test case.
        """
        self.ns, self.cluster, self.bmh, self.bmhinventory, self.machine = get_control
        self.kaas_manager = kaas_manager

    def test_101_check_bmh_bmhinventory_status(self, show_step):
        """Ensure that error status is synced from BMH to BMHInventory, when BMHInventory has a deletionTimestamp

        Scenario:
            1. Set a wrong password for the BMHCredentials secret of the BMH consumed by a control Machine
            2. Set BMHInventory spec.online to False
            3. Wait for "power management error" in BMH status
            4. Error must be synced from BMH to BMHInventory object
        """
        bmhc_name = self.bmhinventory.data['spec']['bmc']['bmhCredentialsName']
        bmhc = self.ns.get_baremetalhostcredential(name=bmhc_name)
        secret_name = bmhc.data['spec']['password']['secret']['name']
        secret = self.ns.get_secret(secret_name)

        show_step(1)
        new_password = "dGVzdAo="  # "test"
        secret.patch({'data': {'password': new_password}})

        show_step(2)
        self.bmhinventory.patch({'spec': {'online': False}})
        waiters.wait(lambda: not self.bmh.data['spec']['online'],
                     interval=5, timeout=60,
                     timeout_msg="spec.online was not synced from BMHInventory to BMH")

        show_step(3)
        waiters.wait(lambda: bool(self.bmh.data['status'].get('errorMessage')),
                     interval=5, timeout=600,
                     timeout_msg=("status.errorMessage contains no errors in BMH, "
                                  "while expected an error because of wrong spec.bmc.address"))

        bmh_status = self.bmh.data['status']
        bmh_error_count = bmh_status.get('errorCount')
        bmh_error_message = bmh_status.get('errorMessage')
        bmh_operational_status = bmh_status.get('operationalStatus')

        LOG.info(f"Got the following BMH status:\n"
                 f"status.errorCount: {bmh_error_count}\n"
                 f"status.operationalStatus: {bmh_operational_status}\n"
                 f"status.errorMessage: {bmh_error_message}\n")

        waiters.wait(lambda: self.bmhinventory.data['status'].get('errorMessage') == bmh_error_message,
                     interval=5, timeout=600,
                     timeout_msg="status.errorMessage was not synced from BMH to BMHInventory")
        bmhinventory_status = self.bmhinventory.data['status']
        assert bmh_error_count == bmhinventory_status.get('errorCount'), (
            "status.errorCount was not synced from BMH to BMHInventory")
        assert bmh_operational_status == bmhinventory_status.get('operationalStatus'), (
            "status.operationalStatus was not synced from BMH to BMHInventory")

    def test_102_ensure_bmhinventory_exists_after_deletion(self, show_step):
        """BMHInventory should not be removed after deletion, if the related Machine is still exists

        Scenario:
            1. make deletion request for BMHInventory object
            2. wait for ~300 sec to ensure that BM provider is processed this request
            3. ensure that related BMH is not deprovisioning and still have "provisioned" state
        """
        show_step(1)
        self.bmhinventory.delete()

        show_step(2)
        pause_sec = 300
        time.sleep(pause_sec)
        assert self.bmhinventory.data['metadata'].get('deletion_timestamp'), (
            f"No 'deletionTimestamp' found in BareMetalHostInventory {self.bmhinventory.name} "
            f"in {pause_sec}sec after deletion request")

        show_step(3)
        # BMH must be 'provisioned' after BMHInventory deletion request, because Machine is not deleted yet.
        bmh_state = self.bmh.data['status']['provisioning']['state']
        assert bmh_state == 'provisioned', (f"BareMetalHost {self.bmh.name} has unexpected "
                                            f"status.provisioning.state='{bmh_state}'")

    def test_103_delete_control_machine(self, show_step):
        """BMHInventory should be removed after the related Machine is deleted

        Scenario:
            1. delete control Machine using spec.delete: True
            2. wait until Machine is deleted
            3. wait until BMHInventory is removed after Machine deletion
            4. wait until BMH is removed after Machine deletion
        """
        bmhinventory_name = self.bmhinventory.name
        bmhinventory_namespace = self.bmhinventory.namespace

        show_step(1)
        machine_name = self.machine.name
        self.machine.delete(new_delete_api=True)

        show_step(2)
        self.cluster.wait_machine_deletion(machine_name, retries=60)

        show_step(3)
        timeout_msg = (f"BareMetalHostInventory {bmhinventory_namespace}/{bmhinventory_name} is still not deleted")
        waiters.wait(lambda: not bool(self.kaas_manager.api.kaas_baremetalhostinventories.present(
                                      name=bmhinventory_name, namespace=bmhinventory_namespace)),
                     timeout=1200,
                     timeout_msg=timeout_msg)

        show_step(4)
        timeout_msg = (f"BareMetalHost {bmhinventory_namespace}/{bmhinventory_name} is still not deleted")
        waiters.wait(lambda: not bool(self.kaas_manager.api.kaas_baremetalhosts.present(
                                      name=bmhinventory_name, namespace=bmhinventory_namespace)),
                     timeout=1200,
                     timeout_msg=timeout_msg)


class TestWorkerBMHInventory(object):
    """Test negative and positive cases while updating BMH/BMHInventory values in different lcm phases

    Scenario steps for a worker Machine:
    - Try to update BMHInventory when it is still consumed by a Machine (negative and positive cases)
    - Delete the Machine (via API request)
    - Check that the restricted values in BMH object are not changed while there is a BMHInventory object
    - Check that the allowed values in BMHinventory object are changed and propagated to BMH object
    - Check that errors are propagated back from BMH status to BMHInventory status
    - Ensure that BMH object cannot be deleted while BMHInventory exists
    - Check that BMHInventory can be deleted while there is an error in BMH, then BMH also should be removed
    """
    @pytest.fixture(autouse=True, scope='function')
    def worker_test_result_check(self, request):
        """Skip the rest of tests in the class after any test is failed"""
        global is_child_worker_test_failed
        # Check if the previous update steps failed
        if is_child_worker_test_failed:
            msg = ("Skip test for a Worker node befause a previous test for the node failed")
            LOG.info(msg)
            pytest.skip(msg)

        yield

        # Check the result of the current step
        test_passed = (hasattr(request.node, 'rep_call') and request.node.rep_call.passed)
        if not test_passed:
            is_child_worker_test_failed = True

    @pytest.fixture(autouse=True, scope='class')
    def get_worker(self, kaas_manager):
        """Get necessary objects to test just once per class"""
        ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        cluster = ns.get_cluster(settings.TARGET_CLUSTER)
        bmh, bmhinventory, machine = get_machine_with_bmhinventory("worker", ns, cluster, kaas_manager)
        LOG.banner(f"Selected the worker Machine to test: '{machine.namespace}/{machine.name}'")
        yield ns, cluster, bmh, bmhinventory, machine

    @pytest.fixture(autouse=True)
    def setup_worker(self, get_worker, kaas_manager):
        """Set class instance attributes before each test for tests convience

        For each test case, pytest generates a new class instance with empty attributes.

        Set instance attributes with necessary objects, instead of unpacking the tuple
        with these attributes from the fixture in each test case.
        """
        self.ns, self.cluster, self.bmh, self.bmhinventory, self.machine = get_worker
        self.kaas_manager = kaas_manager

    def test_001_negative_consumed_bmhinventory_patch_values(self, show_step):
        """When BMHInventory is consumed by a Machine, some BMHInventory values must be refused to change

        Scenario:
            1. [negative]: change BMHInventory spec.bmc.bootMACAddress must be refused while it is consumed by Machine
            2. [negative]: change BMHInventory metadata.annotation.inspect.metal3.io/hardwaredetails-storage-sort-term
               must be refused while it is consumed by Machine
        """

        show_step(1)
        with pytest.raises(ApiException) as exc_info:
            boot_mac_address = self.bmhinventory.data['spec']['bootMACAddress']
            new_boot_mac_address = "f" + str(random.randint(0, 9)) + boot_mac_address[2:]
            LOG.info(f"Old bootMACAddress: {boot_mac_address}")
            LOG.info(f"New bootMACAddress: {new_boot_mac_address}")
            self.bmhinventory.patch({'spec': {'bootMACAddress': new_boot_mac_address}})

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHostInventory "
                        f"'{self.bmhinventory.name}' spec.bootMACAddress was updated without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

        show_step(2)
        with pytest.raises(ApiException) as exc_info:

            sort_term_key_name = 'inspect.metal3.io/hardwaredetails-storage-sort-term'

            ##################################################
            # BMHInventory could be created without annotations.
            # So 'inspect.metal3.io/hardwaredetails-storage-sort-term' can be missing in BMHInventory,
            # and BMH will be created with default one.
            # In this case, we can take a default value for 'inspect.metal3.io/hardwaredetails-storage-sort-term'
            # from BMH object. If BMH annotation also don't have such annotation, than create some empty value.
            ###################################################
            bmhinventory_annotations = self.bmhinventory.data['metadata'].get('annotations') or {}
            sort_term = bmhinventory_annotations.get(sort_term_key_name, '')
            if not sort_term:
                LOG.info(f"BMHInventory '{self.bmhinventory.name}' don't have annotation "
                         f"'{sort_term_key_name}'. Let's use annotation from BMH")
                bmh_annotations = self.bmh.data['metadata'].get('annotations') or {}
                sort_term = bmh_annotations.get(sort_term_key_name, '')

            sort_term = sort_term + ','
            self.bmhinventory.patch({'metadata': {'annotations': {sort_term_key_name: sort_term}}})

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHostInventory "
                        f"'{self.bmhinventory.name}' annotation was updated without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

    def test_002_positive_consumed_bmhinventory_patch_values(self, show_step):
        """When BMHInventory is consumed by a Machine, check BMHInventory values changes take effect

        Scenario:
            1. Check that changes to BMHInventory in 'spec.bmc.address', when it is consumed by a Machine, are allowed
            2. Check that spec.online value is the same in BMHInventory and BMH
            3. Change spec.online value to False in BMHInventory, check that BMH value for spec.online is updated
            4. Wait until the Machine is no longer available
            5. Change spec.online value to True in BMHInventory, check that BMH value for spec.online is updated
            6. Wait until the Machine is available
        """

        show_step(1)
        bmc_address = self.bmhinventory.data['spec']['bmc']['address']
        new_bmc_address = "127.0.0." + str(random.randint(1, 100))
        LOG.info(f"Old spec.bmc.address: {bmc_address}")
        LOG.info(f"New spec.bmc.address: {new_bmc_address}")
        # Change spec.bmc.address to a fake address
        self.bmhinventory.patch({'spec': {'bmc': {'address': new_bmc_address}}})
        waiters.wait(lambda: self.bmh.data['spec']['bmc']['address'] == new_bmc_address,
                     interval=5, timeout=60,
                     timeout_msg="spec.bmc.address was not synced from BMHInventory to BMH")

        # TODO(ddmitriev): consider to add extra checks here to check changes in ironic db for this node

        # Return the correct spec.bmc.address back
        self.bmhinventory.patch({'spec': {'bmc': {'address': bmc_address}}})
        waiters.wait(lambda: self.bmh.data['spec']['bmc']['address'] == bmc_address,
                     interval=5, timeout=60,
                     timeout_msg="spec.bmc.address was not synced from BMHInventory to BMH")

        show_step(2)
        assert self.bmhinventory.data['spec']['online'] is True, (
            f"BMHInventory {self.bmhinventory.name} 'spec.online' is False while expected True")
        assert self.bmhinventory.data['spec']['online'] == self.bmh.data['spec']['online'], (
            f"BMHInventory {self.bmhinventory.name} 'spec.online' value doesn't match the BMH 'spec.online' value")

        show_step(3)
        self.bmhinventory.patch({'spec': {'online': False}})
        waiters.wait(lambda: not self.bmh.data['spec']['online'],
                     interval=5, timeout=60,
                     timeout_msg="spec.online was not synced from BMHInventory to BMH")

        show_step(4)
        machine_ip = self.machine.public_ip or self.machine.internal_ip
        LOG.info("Wait until Machine is no longer available via ICMP")
        waiters.wait(lambda: not waiters.icmp_ping(machine_ip), interval=5, timeout=600)

        show_step(5)
        self.bmhinventory.patch({'spec': {'online': True}})
        waiters.wait(lambda: self.bmh.data['spec']['online'] is True,
                     interval=5, timeout=60,
                     timeout_msg="spec.online was not synced from BMHInventory to BMH")

        show_step(6)
        LOG.info("Wait until Machine is available via SSH")
        waiters.wait_tcp(machine_ip, port=22, timeout=600)

    def test_003_delete_worker_machine(self, show_step):
        """Delete worker Machine using API request and ensure that it is deleted

        Scenario:
            1. Delete the Machine via API request.
               Machine should be deleted while BMHInventory is still exists
            2. Wait until BMH deprovisioning is completed
        """
        show_step(1)
        machine_name = self.machine.name
        self.machine.delete(check_deletion_policy=False, new_delete_api=False)
        self.cluster.wait_machine_deletion(machine_name, retries=60)

        show_step(2)
        waiters.wait(lambda: self.bmh.data['status']['provisioning']['state'] == 'available',
                     interval=5, timeout=1200,
                     timeout_msg=f"BMH {self.bmh.name} is still not get the state 'available' after deleting Machine")

    @pytest.mark.skip(reason="PRODX-50119")
    def test_004_negative_available_bmh_patch_values(self, show_step):
        """Check that patching some BMH values, when it is not consumed by any Machine, is refused

        Scenario:
            1. [negative]: change spec.online directly in BMH when not consumed by a Machine
            2. [negative]: change spec.bmc.bootMACAddress directly in BMH when not consumed by a Machine
            3. [negative]: change spec.bmc.address directly in BMH when not consumed by a Machine
            4. [negative]: change metadata.annotation.inspect.metal3.io/hardwaredetails-storage-sort-term directly
               in BMH when not consumed by a Machine
        """
        show_step(1)
        with pytest.raises(ApiException) as exc_info:
            bmh_online = self.bmh.data['spec']['online']
            self.bmh.patch({'spec': {'online': not bmh_online}})

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHost "
                        f"'{self.bmh.name}' spec.online was updated without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

        show_step(2)
        with pytest.raises(ApiException) as exc_info:
            boot_mac_address = self.bmh.data['spec']['bootMACAddress']
            new_boot_mac_address = "f" + str(random.randint(0, 9)) + boot_mac_address[2:]
            LOG.info(f"Old bootMACAddress: {boot_mac_address}")
            LOG.info(f"New bootMACAddress: {new_boot_mac_address}")
            self.bmh.patch({'spec': {'bootMACAddress': new_boot_mac_address}})

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHost "
                        f"'{self.bmh.name}' spec.bootMACAddress was updated without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

        show_step(3)
        with pytest.raises(ApiException) as exc_info:
            bmc_address = self.bmh.data['spec']['bmc']['address']
            new_bmc_address = "127.0.0." + str(random.randint(1, 100))
            LOG.info(f"Old spec.bmc.address: {bmc_address}")
            LOG.info(f"New spec.bmc.address: {new_bmc_address}")
            # Change spec.bmc.address to a fake address
            self.bmh.patch({'spec': {'bmc': {'address': new_bmc_address}}})

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHost "
                        f"'{self.bmh.name}' spec.bmc.address was updated without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

        show_step(4)
        with pytest.raises(ApiException) as exc_info:

            sort_term_key_name = 'inspect.metal3.io/hardwaredetails-storage-sort-term'

            bmh_annotations = self.bmh.data['metadata'].get('annotations') or {}
            sort_term = bmh_annotations.get(sort_term_key_name, '')

            sort_term = sort_term + ','
            self.bmh.patch({'metadata': {'annotations': {sort_term_key_name: sort_term}}})

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHost "
                        f"'{self.bmh.name}' annotation was updated without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

    def test_005_positive_available_bmhinventory_patch_values(self, show_step):
        """Patch BMHInventory values when it is not consumed by any Machine, ensure that value are changed in BMH

        Scenario:
            1. change BMHInventory spec.bmc.bootMACAddress when not consumed by a Machine
            2. change BMHInventory metadata.annotation.inspect.metal3.io/hardwaredetails-storage-sort-term
               when not consumed by a Machine
        """
        show_step(1)
        boot_mac_address = self.bmhinventory.data['spec']['bootMACAddress']
        new_boot_mac_address = "f" + str(random.randint(0, 9)) + boot_mac_address[2:]
        LOG.info(f"Old bootMACAddress: {boot_mac_address}")
        LOG.info(f"New bootMACAddress: {new_boot_mac_address}")
        self.bmhinventory.patch({'spec': {'bootMACAddress': new_boot_mac_address}})
        waiters.wait(lambda: self.bmh.data['spec']['bootMACAddress'] == new_boot_mac_address,
                     interval=5, timeout=60,
                     timeout_msg="spec.bootMACAddress was not synced from BMHInventory to BMH")

        show_step(2)
        sort_term_key_name = 'inspect.metal3.io/hardwaredetails-storage-sort-term'

        ##################################################
        # BMHInventory could be created without annotations.
        # So 'inspect.metal3.io/hardwaredetails-storage-sort-term' can be missing in BMHInventory,
        # and BMH will be created with default one.
        # In this case, we can take a default value for 'inspect.metal3.io/hardwaredetails-storage-sort-term'
        # from BMH object. If BMH annotation also don't have such annotation, than create some empty value.
        ###################################################
        bmhinventory_annotations = self.bmhinventory.data['metadata'].get('annotations') or {}
        sort_term = bmhinventory_annotations.get(sort_term_key_name, '')
        if not sort_term:
            LOG.info(f"BMHInventory '{self.bmhinventory.name}' don't have annotation "
                     f"'{sort_term_key_name}'. Let's use annotation from BMH")
            bmh_annotations = self.bmh.data['metadata'].get('annotations') or {}
            sort_term = bmh_annotations.get(sort_term_key_name, '')

        # sort_term = sort_term + ','
        sort_term_new = ','.join(x.strip() for x in sort_term.split(',')[::-1])  # reverse the sort order
        assert sort_term_new != sort_term, (
            f"Annotation value for sort_term was not changed, please check the test: '{sort_term_new}'")
        LOG.info(f"Old metadata.annotation: {sort_term}")
        LOG.info(f"New metadata.annotation: {sort_term_new}")
        self.bmhinventory.patch({'metadata': {'annotations': {sort_term_key_name: sort_term_new}}})
        waiters.wait(
            lambda: (self.bmh.data['metadata'].get('annotations') or {}).get(sort_term_key_name) == sort_term_new,
            interval=5, timeout=300,
            timeout_msg=f"metadata.annotations.{sort_term_key_name} was not synced from BMHInventory to BMH")

    def test_006_positive_check_bmh_bmhinventory_status(self, show_step):
        """Ensure that error status is synced from BMH to BMHInventory

        Scenario:
            1. Change BMHInventory spec.bmc.address when not consumed by a Machine. Use a wrong address (127.0.0.dd)
            2. Set BMHInventory spec.online to 'false', to cause a power management error using a wrong bmc address
            3. After setting spec.bmc.address to a wrong address, an error must be expected in BMH status
            4. Check that this error is synced to BMHInventory object
        """
        show_step(1)
        bmc_address = self.bmhinventory.data['spec']['bmc']['address']
        new_bmc_address = "127.0.0." + str(random.randint(1, 100))
        LOG.info(f"Old spec.bmc.address: {bmc_address}")
        LOG.info(f"New spec.bmc.address: {new_bmc_address}")
        # Change spec.bmc.address to a fake address
        self.bmhinventory.patch({'spec': {'bmc': {'address': new_bmc_address}}})
        waiters.wait(lambda: self.bmh.data['spec']['bmc']['address'] == new_bmc_address,
                     interval=5, timeout=60,
                     timeout_msg="spec.bmc.address was not synced from BMHInventory to BMH")

        show_step(2)
        self.bmhinventory.patch({'spec': {'online': False}})
        waiters.wait(lambda: not self.bmh.data['spec']['online'],
                     interval=5, timeout=60,
                     timeout_msg="spec.online was not synced from BMHInventory to BMH")

        show_step(3)
        waiters.wait(lambda: bool(self.bmh.data['status'].get('errorMessage')),
                     interval=5, timeout=600,
                     timeout_msg=("status.errorMessage contains no errors in BMH, "
                                  "while expected an error because of wrong spec.bmc.address"))

        show_step(4)
        bmh_status = self.bmh.data['status']
        bmh_error_count = bmh_status.get('errorCount')
        bmh_error_message = bmh_status.get('errorMessage')
        bmh_operational_status = bmh_status.get('operationalStatus')

        LOG.info(f"Got the following BMH status:\n"
                 f"status.errorCount: {bmh_error_count}\n"
                 f"status.operationalStatus: {bmh_operational_status}\n"
                 f"status.errorMessage: {bmh_error_message}\n")

        waiters.wait(lambda: self.bmhinventory.data['status'].get('errorMessage') == bmh_error_message,
                     interval=5, timeout=600,
                     timeout_msg="status.errorMessage was not synced from BMH to BMHInventory")
        bmhinventory_status = self.bmhinventory.data['status']
        assert bmh_error_count == bmhinventory_status.get('errorCount'), (
            "status.errorCount was not synced from BMH to BMHInventory")
        assert bmh_operational_status == bmhinventory_status.get('operationalStatus'), (
            "status.operationalStatus was not synced from BMH to BMHInventory")

    def test_007_negative_delete_bmh_when_bmhinventory_exists(self, show_step):
        """Ensure that BMH deletion is blocked when the related BMHInventory exists

        Scenario:
            1. [negative]: try to delete BMH when BMHInventory exists, expected error like:
               'admission webhook "validations.kaas.mirantis.com" denied the request ...
               Please, delete BareMetalHostInventory first'
        """
        show_step(1)
        with pytest.raises(ApiException) as exc_info:

            self.bmh.delete()

            pytest.fail(f"Expected result with 'Bad Request', but BareMetalHost "
                        f"'{self.bmh.name}' was accepted delete() request without error")
        LOG.info(f"Got the following response: {exc_info.value.body}")

    def test_008_positive_delete_bmhinventory(self, show_step):
        """Ensure that BMHInventory and BMH are deleted using wrong spec.bmc.address

        Scenario:
            1. delete BMHInventory
            2. Wait BMHInventory is deleted
            2. Wait BMH is deleted
        """
        bmhinventory_name = self.bmhinventory.name
        bmhinventory_namespace = self.bmhinventory.namespace

        show_step(1)
        self.bmhinventory.delete()

        show_step(2)
        timeout_msg = (f"BareMetalHostInventory {bmhinventory_namespace}/{bmhinventory_name} is still not deleted")
        waiters.wait(lambda: not bool(self.kaas_manager.api.kaas_baremetalhostinventories.present(
                                      name=bmhinventory_name, namespace=bmhinventory_namespace)),
                     timeout=1200,
                     timeout_msg=timeout_msg)

        show_step(3)
        timeout_msg = (f"BareMetalHost {bmhinventory_namespace}/{bmhinventory_name} is still not deleted")
        waiters.wait(lambda: not bool(self.kaas_manager.api.kaas_baremetalhosts.present(
                                      name=bmhinventory_name, namespace=bmhinventory_namespace)),
                     timeout=1200,
                     timeout_msg=timeout_msg)
