import os.path

import pytest
from retry import retry
from time import sleep
from cached_property import cached_property_with_ttl
import cachetools.func as cachetools_func

from heatclient import client as heatclient
from heatclient import exc as heat_exceptions
from heatclient.common import template_utils
from keystoneauth1.identity import v3 as keystone_v3
from keystoneauth1 import session as keystone_session

from si_tests import settings
from si_tests.utils import exceptions as si_exceptions
from si_tests import logger
from si_tests.utils import utils

EXPECTED_STACK_STATUS = "CREATE_COMPLETE"
BAD_STACK_STATUSES = ["CREATE_FAILED", "DELETE_FAILED"]
BAD_IRONIC_HW_NODE_STATUSES = ["error", "clean failed"]

LOG = logger.logger


class OpenstackClient:

    def __init__(self, auth):
        super().__init__()
        self.__auth = auth
        self.__token = {}
        self.__heatclient = None

    def stack_create(self):
        # TODO(alexz-kh) looks like need to refactor this func someday.
        stack_name = settings.ENV_NAME
        LOG.info(f'Looking heat configs at {settings.KSI_HEAT_DATA_PATH}')
        heat_env_path = os.path.join(settings.KSI_HEAT_DATA_PATH, 'env.yaml')
        template_path = os.path.join(settings.KSI_HEAT_DATA_PATH, 'heat.hot')
        if not os.path.exists(heat_env_path):
            pytest.fail(f'File not found {heat_env_path}')
        if not os.path.exists(template_path):
            pytest.fail(f'File not found {template_path}')

        LOG.info(f"Creating stack >{stack_name}< with next env file(s):"
                 f"\n{heat_env_path}"
                 f"\nand next template(s):"
                 f"\n{template_path}")
        tpl_files, template = template_utils.get_template_contents(template_path)
        env_files_list = []
        env_files, env = (template_utils.process_multiple_environments_and_files(
            env_paths=[heat_env_path],
            env_list_tracker=env_files_list))

        fields = {
            'stack_name': stack_name,
            'template': template,
            'files': dict(list(tpl_files.items()) + list(env_files.items())),
            'environment': env,
            'parameters': {}
        }

        if settings.KSI_BUILD_URL:
            fields['parameters']['build_url'] = settings.KSI_BUILD_URL

        if env_files_list:
            fields['environment_files'] = env_files_list

        @retry(heat_exceptions.HTTPBadGateway, delay=15, tries=20)
        def _safe_heat_stack_create():
            self.__stacks.create(**fields)

        def _safe_create():
            delay = 15
            tries = 0
            for tnum in range(1, tries + 2):
                try:
                    _safe_heat_stack_create()
                    self.wait_of_stack_status(EXPECTED_STACK_STATUS,
                                              tries=140,
                                              delay=10)
                    LOG.info("Stack '%s' created", stack_name)
                    LOG.info('Double check, that stack has been created w\\o failed resources')
                    fresources = self._verify_resources_status(EXPECTED_STACK_STATUS)
                    if fresources:
                        for r in fresources:
                            LOG.error(
                                'The resource %s has status %s.'
                                'But it should be %s',
                                r.resource_name,
                                r.resource_status,
                                EXPECTED_STACK_STATUS)
                        raise si_exceptions.OsHeatEnvironmentBadStatus(
                            stack_name,
                            EXPECTED_STACK_STATUS,
                            self._current_stack.stack_status,
                            fresources)
                    return
                except si_exceptions.OsHeatEnvironmentBadStatus as e:
                    if tnum > tries:
                        # we need to return wrong_resources here as string,
                        # for proper parsing in testrail.
                        raise si_exceptions.OsHeatEnvironmentBadStatus(
                            stack_name,
                            EXPECTED_STACK_STATUS,
                            e.env_actual_status,
                            self._get_resources_with_wrong_status())
                    LOG.error('Stack create failed.'
                              'Next attempt {}/{}.'
                              'after:{}s'.format(tnum, tries, delay))
                    LOG.error(
                        "Recreating the stack because some resources have "
                        "incorrect status:\n{}\n".format(e.wrong_resources))
                    sleep(delay)
                    self.delete_stack()

        LOG.info(f"Attempt to create heat stack - {stack_name}")
        _safe_create()

    def delete_stack(self):
        stack_name = settings.ENV_NAME
        removed_retry = 10
        wait_retry = 120
        LOG.debug(f"delete_stack: set wait_retry={wait_retry}")

        if list(self.__stacks.list(stack_name=stack_name)):
            LOG.info(f"Delete stack '{stack_name}'")

            @retry(heat_exceptions.HTTPBadGateway, delay=15, tries=20)
            def safe_heat_stack_delete():
                self.__stacks.delete(self._current_stack.id)

            safe_heat_stack_delete()
            self.wait_of_stack_status('DELETE_COMPLETE',
                                      delay=20, tries=wait_retry,
                                      wait_for_delete=True)
            # Status 'REMOVED' doesn't exist, so wait until the stack
            # is actually removed from the stack list with wait_for_delete=True
            self.wait_of_stack_status('REMOVED',
                                      delay=5, tries=removed_retry,
                                      wait_for_delete=True)
        else:
            LOG.warning(f"Can't delete stack {stack_name} due is absent")

    @cached_property_with_ttl(ttl=15 * 60)
    def _keystone_session(self):
        auth_cfg = self.__auth["auth"]
        auth_url = auth_cfg.get("auth_url")
        LOG.debug(f"Creating keystone session to {auth_url}")

        if auth_cfg.get("application_credential_id"):
            # application‐credential auth
            LOG.debug(f"Using ApplicationCredential {auth_cfg['application_credential_id']}")
            keystone_auth = keystone_v3.ApplicationCredential(
                auth_url=auth_url,
                application_credential_id=auth_cfg["application_credential_id"],
                application_credential_secret=auth_cfg["application_credential_secret"],
                project_name=auth_cfg.get("project_name"),
                project_id=auth_cfg.get("project_id"),
                user_domain_name=auth_cfg.get("user_domain_name", "Default"),
                project_domain_name=auth_cfg.get("project_domain_name", "Default"),
            )
        else:
            # fallback to username/password auth
            LOG.debug(f"Using Password auth for user {auth_cfg.get('username')}")
            keystone_auth = keystone_v3.Password(
                auth_url=auth_url,
                username=auth_cfg["username"],
                password=auth_cfg["password"],
                project_name=auth_cfg.get("project_name"),
                project_id=auth_cfg.get("project_id"),
                user_domain_name=auth_cfg.get("user_domain_name", "Default"),
                project_domain_name=auth_cfg.get("project_domain_name", "Default"),
            )

        return keystone_session.Session(auth=keystone_auth, verify=True)

    @cached_property_with_ttl(ttl=15 * 60)
    def token(self):
        return self._keystone_session.get_token()

    @cached_property_with_ttl(ttl=15 * 60)
    def access(self):
        return self._keystone_session.auth.get_access(self._keystone_session)

    @property
    def public_endpoints(self):
        return self.access.service_catalog.get_urls(interface='public')

    @property
    def service_catalog(self):
        return self.access.service_catalog.catalog

    @cached_property_with_ttl(ttl=30 * 60)
    def heat(self):
        self.__init_heatclient()
        return self.__heatclient

    def __init_heatclient(self):
        endpoint_url = self._keystone_session.get_endpoint(
            service_type='orchestration', endpoint_type='publicURL')
        self.__heatclient = heatclient.Client(
            version=1,
            endpoint=endpoint_url,
            token=self.token,
            insecure=False)

    def get_stacks(self, name_prefix=''):
        if name_prefix:
            return [st._info for st in self.__stacks.list()
                    if st.stack_name.startswith(name_prefix)]
        else:
            return [st._info for st in self.__stacks.list()]

    def delete_stack_by_id(self, stack_id):
        self.__stacks.delete(stack_id)

    @property
    def __stacks(self):
        return self.heat.stacks

    @property
    def _current_stack(self):
        try:
            return self.__stacks.get(settings.ENV_NAME)
        except heat_exceptions.HTTPNotFound:
            return None

    def wait_of_stack_status(self, status, delay=30, tries=60,
                             wait_for_delete=False):

        @retry(si_exceptions.OsHeatEnvironmentWrongStatus, delay=delay, tries=tries)
        def wait():
            # try:
            #     st = self._current_stack.stack_status
            # except heat_exceptions.HTTPNotFound as ex:
            #     if wait_for_delete is True:
            #         return
            #     raise ex

            stack = self._current_stack
            if stack is None:
                if wait_for_delete is True:
                    LOG.info(f"Stack '{settings.ENV_NAME}' not exist")
                    return
                raise Exception(f"Stack '{settings.ENV_NAME}' not found")
            st = stack.stack_status
            LOG.info(f"Stack '{settings.ENV_NAME}' status: <{st}>")
            if st == status:
                LOG.info(f"Waiting for stack '{settings.ENV_NAME}' <{status}> "
                         "reached")
                return
            elif st in BAD_STACK_STATUSES:
                raise si_exceptions.OsHeatEnvironmentBadStatus(
                    settings.ENV_NAME,
                    status,
                    st,
                    self._get_resources_with_wrong_status())
            else:
                raise si_exceptions.OsHeatEnvironmentWrongStatus(
                    settings.ENV_NAME,
                    status,
                    st)

        LOG.info(f"Waiting for stack '{settings.ENV_NAME}' status <{status}>")
        wait()

    def _get_resources_with_wrong_status(self):
        res = []
        for item in self.__nested_resources:
            if item.resource_status != EXPECTED_STACK_STATUS:
                res.append({
                    'resource_name': item.resource_name,
                    'resource_status': item.resource_status,
                    'resource_status_reason': item.resource_status_reason,
                    'resource_type': item.resource_type
                })
        wrong_resources = '\n'.join([
            "*** Heat stack resource '{0}' ({1}) has wrong status '{2}': {3}"
            .format(item['resource_name'],
                    item['resource_type'],
                    item['resource_status'],
                    item['resource_status_reason'])
            for item in res
        ])
        return wrong_resources

    @property
    def __nested_resources(self):
        resources = []
        stacks = [s for s in self.__stacks.list(show_nested=True)]
        current_stack_id = self._current_stack.id
        for stack in stacks:
            parent_stack_id = self.__get_stack_parent(
                stack.id, stacks) or stack.id
            if parent_stack_id == current_stack_id:
                # Add resources to list
                LOG.info("Get resources from stack {0}"
                         .format(stack.stack_name))
                resources.extend([
                    res for res in self.__resources.list(stack.id)
                ])
        LOG.info("Found {0} resources".format(len(resources)))
        return resources

    def _verify_resources_status(self, status):
        """Check that all resources have verified `status`

        In case when all resources have expected status return empty list,
            otherwise return a list with resources with incorrect status.
        """
        ret = [r for r in self.__nested_resources if
               r.resource_status != status]
        return ret

    def __get_stack_parent(self, stack_id, stacks):
        """Find the parent ID of the specified stack_id in the 'stacks' list"""
        for stack in stacks:
            if stack_id == stack.id:
                if stack.parent:
                    return self.__get_stack_parent(
                        stack.parent, stacks) or stack.id
                else:
                    return stack.id
        LOG.warning("Stack with ID {} not found!".format(stack_id))
        return None

    @property
    def __resources(self):
        return self.heat.resources

    def _stack_resources_data(self) -> dict:
        """Get list of resources, that could be reused.
        Generally cover: nodes,disks,flavor data.
        This information later could be re-used during product templates logic

        Returns list of dicts.
        """
        nodes = []
        subnets = []
        for heat_node in self._get_resources_by_type_with_ttl('OS::Nova::Server'):
            _node = {'hostname': heat_node.attributes['OS-EXT-SRV-ATTR:hostname'],
                     'name': heat_node.attributes['name'],
                     'cpu': heat_node.attributes['flavor']['vcpus'],
                     'ram': heat_node.attributes['flavor']['ram'],
                     'server_id': heat_node.attributes['id'],
                     'logical_resource_id': heat_node.logical_resource_id or '',
                     'addresses': heat_node.attributes['addresses']}
            nodes.append(_node)
        for heat_subnet in self._get_resources_by_type_with_ttl('OS::Neutron::Subnet'):
            _subnet = {'resource_name': heat_subnet.resource_name or '',
                       'logical_resource_id': heat_subnet.logical_resource_id or '',
                       'cidr': heat_subnet.attributes.get('cidr', ''),
                       'gateway_ip': heat_subnet.attributes.get('gateway_ip', ''),
                       }
            subnets.append(_subnet)
        return {'nodes': nodes,
                'subnets': subnets}

    def _get_resources_by_type(self, resource_type):
        res = []
        for item in self.__nested_resources:
            if item.resource_type == resource_type:
                resource = self.__resources.get(
                    item.stack_name,
                    item.resource_name)
                res.append(resource)
        return res

    @cachetools_func.ttl_cache(ttl=5 * 60)
    def _get_resources_by_type_with_ttl(self, resource_type):
        return self._get_resources_by_type(resource_type)

    def get_seed_ip(self, fip='seed_floating_ip'):
        """

        :param fip: string, heat stack output_key, to be identified like floating ip
        :return:
        """
        LOG.debug("Getting seed ip via openstack client")
        stack_name = settings.ENV_NAME
        stack = self._current_stack
        if stack:
            outputs = stack.outputs
            key = next(
                (o for o in outputs if o["output_key"] == fip),
                {})
            if key:
                return key['output_value']
            else:
                LOG.warning(f"Stack: {stack_name} doesnt contain requested key: {fip}")
                return None
        LOG.warning(f"Stack {stack_name} not found")
        return None

    @utils.log_method_time()
    def create_seed_node(self):
        stack = self._current_stack
        if stack:
            pytest.fail(f'Stack >{settings.ENV_NAME}< already exists!\n{self._current_stack.outputs}')
        self.stack_create()
        return self._current_stack.outputs

    def save_seed_node_info(self):
        seed_ip = self.get_seed_ip()
        LOG.info(f"Seed node ip address is {seed_ip}")
        assert seed_ip
        # TODO( alexz-kh) save to config here!
        # seed_ip
