#    Copyright 2024 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import pytest
import yaml

from datetime import datetime, timedelta
from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters
from si_tests.managers import bootstrap_manager, ntp_manager

from si_tests.managers.kaas_manager import Manager

LOG = logger.logger
mcc_upgrade_object_name = settings.MCC_UPGRADE_OBJECT_NAME


def check_upgrade_auto_delay_schedule_correct(kaas_manager: Manager):

    def wait_next_release_data(kaas_manager, timeout=300):
        def get_next_release_data():
            return kaas_manager.get_kaas_mcc_upgrade_status(name=mcc_upgrade_object_name).get('nextRelease', {})
        waiters.wait(lambda: get_next_release_data().get('version', ''), timeout=timeout, interval=10,
                     timeout_msg=f"Next release was not detected in {timeout} sec")
    LOG.info("Waiting for new release data appears in mccupgrade object")
    wait_next_release_data(kaas_manager)
    next_release_data = kaas_manager.get_kaas_mcc_upgrade_status(name=mcc_upgrade_object_name).get('nextRelease', {})
    LOG.info(f"Next release data:\n{yaml.dump(next_release_data)}")
    next_release_release_date = next_release_data.get('date', '')
    start_time = next_release_data.get('scheduled', {}).get('startTime', '')
    if next_release_release_date.endswith('Z'):
        next_release_release_date = next_release_release_date[:-1] + '+00:00'
    if start_time.endswith('Z'):
        start_time = start_time[:-1] + '+00:00'
    dt_next_release_publish_date = datetime.fromisoformat(next_release_release_date)
    LOG.info(f"Next release publishDate is: {dt_next_release_publish_date.strftime('%a %b %d %H:%M:%S %Y')}")
    dt_start_time = datetime.fromisoformat(start_time)
    LOG.info(f"Next release upgrade start time is: {dt_start_time.strftime('%a %b %d %H:%M:%S %Y')}")
    diff_time = dt_start_time - dt_next_release_publish_date
    assert 27 >= diff_time.days >= 20, (f"Wrong delay for mcc upgrade. Should not be more then 27 and less then 20 days"
                                        f"Current delay is {diff_time.days} days")


def test_disable_auto_delay_check_update_started(kaas_manager: Manager):
    """
    Check mgmt is started to update after disabling autoDelay

    Scenario:
    1. Check cluster readiness before proceed
    2. Check autoDelay enabled and has correct statuses
    3. Disable autoDelay
    4. Check mgmt update started
    """

    cluster = kaas_manager.get_mgmt_cluster()
    cluster.check.check_cluster_readiness()
    mcc_upgrade_obj = cluster._manager.get_kaas_mcc_upgrade(name=mcc_upgrade_object_name)
    is_auto_delay_enabled = mcc_upgrade_obj.data.get('spec', {}).get('autoDelay', False)
    if not is_auto_delay_enabled:
        msg = "MCC upgrade auto-delay disabled. Nothing to check"
        LOG.info(msg)
        pytest.skip(msg)
    check_upgrade_auto_delay_schedule_correct(kaas_manager=kaas_manager)
    LOG.info("Disabling mcc upgrade auto-delay to continue upgrade")
    body = {
        "spec": {
            "autoDelay": False,
        }
    }
    kaas_manager.api.kaas_mccupgrades.update(name=mcc_upgrade_object_name, body=body)

    # Wait until KaaS downloads the second kaasrelease
    LOG.info("Waiting for downloading of new  kaasrelease")
    kaas_manager.wait_kaasrelease(releases_count=2, interval=60)

    avail_kaas_releases = kaas_manager.get_kaasrelease_names()
    LOG.info('Available KaaS releases are {}'.format(avail_kaas_releases))

    assert len(avail_kaas_releases) > 1, (
        "There are no new kaas releases, current list: {}".format(
            avail_kaas_releases))


def test_change_date_check_update_started(kaas_manager: Manager):
    """
    Test to verify that the mgmt update started at the scheduled time.
    1. Check cluster is ready for testing
    2. Change time on seed node close to next update attempt
    3. Use seed as ntp server for mgmt cluster
    4. Check time synced
    5. Wait for update started at scheduled time
    """

    cluster = kaas_manager.get_mgmt_cluster()
    # Check cluster readiness before proceed
    cluster.check.check_cluster_readiness()
    mcc_upgrade_obj = cluster._manager.get_kaas_mcc_upgrade(name=mcc_upgrade_object_name)
    is_auto_delay_enabled = mcc_upgrade_obj.data.get('spec', {}).get('autoDelay', False)
    if not is_auto_delay_enabled:
        msg = "MCC upgrade auto-delay disabled. Nothing to check"
        LOG.info(msg)
        pytest.skip(msg)
    check_upgrade_auto_delay_schedule_correct(kaas_manager=kaas_manager)
    bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
    remote = bootstrap.remote_seed()
    ntp = ntp_manager.NTPManager(bootstrap_manager=bootstrap)
    seed_ip = ntp.seed_node_ip
    mccupgrade_obj = cluster._manager.get_kaas_mcc_upgrade(name=mcc_upgrade_object_name)
    next_attempt_date = mccupgrade_obj.data['status']['nextRelease']['scheduled']['startTime']
    seed_date = remote.check_call('date -u +"%Y-%m-%dT%H:%M:%SZ"').stdout_str
    LOG.info(f"Current seed date: {seed_date}")
    next_attempt_datetime_obj = datetime.strptime(next_attempt_date, "%Y-%m-%dT%H:%M:%SZ")
    seed_date_datetime_obj = datetime.strptime(seed_date, "%Y-%m-%dT%H:%M:%SZ")
    timediff = next_attempt_datetime_obj - seed_date_datetime_obj

    # We will set time as 30 min before upgrade should start
    # This gap is required to mgmt cluster sync time on nodes after configuring ntp
    time_reserve_minutes = 30
    timediff -= timedelta(minutes=time_reserve_minutes)
    days = timediff.days
    hours, remainder = divmod(timediff.seconds, 3600)
    minutes, seconds = divmod(remainder, 60)

    ntp.add_time(days=days, hours=hours, minutes=minutes, seconds=seconds)
    new_seed_date = remote.check_call('date -u +"%Y-%m-%dT%H:%M:%SZ"').stdout_str
    LOG.info(f"Date on seed nodes has been changed. Current date: {new_seed_date}")
    LOG.info(f"Next attempt date to upgrade mgmt is: {next_attempt_datetime_obj.isoformat()}")
    LOG.info(f"Configuring mgmt cluster to use seed node IP {seed_ip} as ntp server")

    cluster.set_or_update_ntp_servers(ntp_servers=[seed_ip])
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()

    LOG.info("Time is synchronised. Waiting for upgrade started")
    # Wait until KaaS downloads the second kaasrelease
    LOG.info("Waiting for downloading of new  kaasrelease")
    timeout_sec = 1800 + time_reserve_minutes*60
    kaas_manager.wait_kaasrelease(releases_count=2, timeout=timeout_sec, interval=60)
    avail_kaas_releases = kaas_manager.get_kaasrelease_names()
    LOG.info('Available KaaS releases are {}'.format(avail_kaas_releases))

    assert len(avail_kaas_releases) > 1, (
        "There are no new kaas releases, current list: {}".format(
            avail_kaas_releases))
