import base64
import dateutil.parser
import json
import os
import string
import urllib.parse
import math
import datetime
import re
import functools

import cachetools.func as cachetools_func
import exec_helpers
import yaml
from kubernetes.client.rest import ApiException
from si_tests.utils import packaging_version as version
from pyVim.task import WaitForTask
from retry import retry
from urllib3.exceptions import MaxRetryError, ProtocolError
from pyVmomi import vim
from functools import lru_cache

import si_tests.utils.templates as template_utils
from si_tests import logger
from si_tests import settings
from si_tests.clients.k8s import K8sCluster, kaas_vspherevmtemplates, diagnostic_diagnostic
from si_tests.clients.dashboard import mcc_dashboard_client
from si_tests.clients.dashboard import mke_dashboard_client
from si_tests.clients.docker.docker_cli import DockerCliClient
from si_tests.clients.k8s.nodes import K8sNode
from si_tests.clients.prometheus.prometheus_client import PrometheusClientOpenid
from si_tests.clients.grafana.grafana_client import GrafanaClient
from si_tests.deployments.utils.namespace import NAMESPACE
from si_tests.managers import clustercheck_manager
from si_tests.managers import clustercheck_mos_manager
from si_tests.managers import day2operations_manager
from si_tests.managers import provider_resource_manager as prm
from si_tests.managers import si_config_manager
from si_tests.managers import machine_deletion_policy_manager
from si_tests.managers import workaroundcheck_manager
from si_tests.managers import tungstenfafric_manager
from si_tests.managers import keycloak_manager
from si_tests.managers import runtime_manager
from si_tests.managers import ha_manager
from si_tests.utils import utils, waiters
from si_tests.utils.helpers import retry_for_rcs
from si_tests.clients.k8s.models import V1KaaSCluster
from si_tests.fixtures.cluster import collect_cluster_readiness

from typing import List, Optional, Union

LOG = logger.logger

system_namespaces = [
    'default', 'kube-system', 'kube-public', 'openstack-provider-system',
    'system', 'stacklight', 'ceph', 'kaas', 'lcm-system', 'istio-system',
    'kube-node-lease', 'local-path-storage', 'metallb-system',
    'node-feature-discovery']
if settings.CLUSTER_NAMESPACE not in system_namespaces:
    system_namespaces.append(settings.CLUSTER_NAMESPACE)


MachineProviders = Union["Machine", "OpenstackProviderMachine", "BaremetalProviderMachine",
                         "AwsProviderMachine", "ByoProviderMachine", "VsphereProviderMachine",
                         "EquinixMetalProviderMachine", "EquinixMetalV2ProviderMachine",
                         "AzureProviderMachine"]


class Manager(object):
    kubeconfig = None

    def __init__(self, kubeconfig=settings.KUBECONFIG_PATH):
        self._api = None
        self.kubeconfig = kubeconfig
        self.si_config = si_config_manager.SIConfigManager(self, si_config_path=settings.SI_CONFIG_PATH)
        self.__keycloak = None

    @property
    def api(self) -> K8sCluster:
        """
            :rtype: cluster.K8sCluster
        """
        if self._api is None:
            if not os.path.isfile(self.kubeconfig):
                raise FileNotFoundError(
                    'kubeconfig file={} not found!'.format(self.kubeconfig))
            self._api = K8sCluster(kubeconfig=self.kubeconfig)
        return self._api

    @property
    def keycloak(self):
        if not self.__keycloak:
            self.__keycloak = keycloak_manager.KeycloakManager(self)
        return self.__keycloak

    def get_regional_clusters(self):
        return [x for x in self.get_clusters(namespace=settings.REGION_NAMESPACE) if x.is_regional]

    def get_child_clusters_in_region(self, region_name=''):
        return [x for x in self.get_child_clusters() if x.region_name == region_name]

    def get_child_clusters(self):
        return [x for x in self.get_clusters() if x.is_child]

    @cachetools_func.ttl_cache(ttl=10)
    def get_kaasrelease(self, name, namespace=settings.CLUSTER_NAMESPACE):
        """Get KaaSRelease CRD object like kaas-0-1-0-rc"""
        return self.api.kaas_kaasreleases.get(name=name,
                                              namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_kaasreleases(self):
        return self.api.kaas_kaasreleases.list_all()

    @cachetools_func.ttl_cache(ttl=10)
    def get_active_kaasrelease(self):
        for kaasrelease in self.get_kaasreleases():
            kr_data = kaasrelease.data
            kr_meta = kr_data['metadata']['labels']
            if kr_meta:
                if 'true' in kr_meta.get('kaas.mirantis.com/active', ''):
                    return kaasrelease
        raise Exception("No kaasrelease found with 'kaas.mirantis.com/active' label")

    @cachetools_func.ttl_cache(ttl=10)
    def get_diagnosticreleases(self):
        return self.api.diagnostic_diagnosticreleases.list_all()

    @cachetools_func.ttl_cache(ttl=10)
    def get_diagnosticrelease(self, name):
        return self.api.diagnostic_diagnosticreleases.get(name=name)

    @cachetools_func.ttl_cache(ttl=10)
    def get_active_diagnosticrelease(self):
        for diagnosticrelease in self.get_diagnosticreleases():
            dr_data = diagnosticrelease.data
            dr_meta = dr_data['metadata']['labels']
            if dr_meta:
                if 'true' in dr_meta.get('diagnostic.mirantis.com/active', ''):
                    return diagnosticrelease
        raise Exception("No diagnosticrelease found with 'diagnostic.mirantis.com/active' label")

    @property
    def active_diagnostic_version(self):
        return self.get_active_diagnosticrelease().data.get('spec', {}).get('version')

    def get_kaasrelease_bootstrap_version(self):
        kaasrelease = self.get_active_kaasrelease()
        return kaasrelease.data.get('spec', {}).get('bootstrap', {}).get('version')

    def get_kaasrelease_names(self):
        releases = self.get_kaasreleases()
        return [release.name for release in releases]

    def get_kaas_mcc_upgrades(self):
        return self.api.kaas_mccupgrades.list_all()

    def get_kaas_mcc_upgrade(self, name):
        return self.api.kaas_mccupgrades.get(name)

    def get_kaas_mcc_upgrade_status(self, name):
        return self.get_kaas_mcc_upgrade(name).data.get(
            "status", {})

    def get_kaas_mcc_upgrade_status_condition(self, name, item):
        return self.get_kaas_mcc_upgrade(name).data.get(
            'status', {}).get('conditions', [{}])[0].get(item, None)

    def get_cluster_upgrades_status_crds(self):
        return [item.read() for item in self.api.kaas_clusterupgradestatus.list_all()]

    def get_cluster_upgrade_status_crd(self, name):
        return self.api.kaas_clusterupgradestatus.get(name)

    def get_cluster_deploy_status_crds(self):
        return [item.read() for item in self.api.kaas_clusterdeploystatus.list_all()]

    def get_cluster_deploy_status_crd(self, name):
        return self.api.kaas_clusterdeploystatus.get(name)

    def get_machine_upgrade_status_crds(self):
        return [item.read() for item in self.api.kaas_machineupgradestatus.list_all()]

    def get_machine_upgrade_status_crd(self, name, namespace=None):
        return self.api.kaas_machineupgradestatus.get(name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_clusterrelease(self, name, namespace=settings.CLUSTER_NAMESPACE):
        """Get ClusterRelease CRD object like kubernetes-0-3-1-1-15"""
        return self.api.kaas_clusterreleases.get(name=name,
                                                 namespace=namespace)

    @lru_cache(maxsize=12)
    def get_allowed_distributions(self, release_name):
        r_data = self.get_clusterrelease(release_name).data
        return utils.get_distribution_relevance(r_data)

    @cachetools_func.ttl_cache(ttl=10)
    def get_proxies(self):
        return self.api.kaas_proxies.list_all()

    def get_proxy_by_name(self, name, ns=None):
        if ns:
            return [proxy for proxy in self.get_proxies()
                    if name in proxy.name and proxy.namespace == ns]
        else:
            return [proxy for proxy in self.get_proxies()
                    if name in proxy.name]

    @cachetools_func.ttl_cache(ttl=10)
    def get_clusterreleases(self):
        return self.api.kaas_clusterreleases.list_all()

    @cachetools_func.ttl_cache(ttl=10)
    def get_ucp_tag(self, release_name):
        ucp_tag = set()
        ucp_tag_ctl = set(
            [x['params']['ucp_tag'] for x in self.get_clusterrelease(
                release_name).data['spec']['machineTypes']['control']
             if 'ucp_tag' in x['params']])
        ucp_tag.update(ucp_tag_ctl)
        ucp_tag_wkr = set(
            [x['params']['ucp_tag'] for x in self.get_clusterrelease(
                release_name).data['spec']['machineTypes']['worker']
             if 'ucp_tag' in x['params']])
        ucp_tag.update(ucp_tag_wkr)
        assert len(ucp_tag) == 1, f"Something went wrong. Different ucp tags are used for " \
                                  f"release {release_name}: {ucp_tag}"
        return ucp_tag.pop()

    @cachetools_func.ttl_cache(ttl=10)
    def get_all_ucp_tags(self):
        all_releases = self.get_clusterreleases()
        all_ucp_tags = []
        for r in all_releases:
            r_name = r.name
            r_ucp_tag = self.get_ucp_tag(r_name)
            all_ucp_tags.append(r_ucp_tag)
        return list(set(all_ucp_tags))

    def get_latest_available_clusterrelease_distro(self, clusterrelease_name):
        """Get the latest available distro name for the specified cluster release

        Sort releases by the string with version number before getting the latest one.
        """
        clusterrelease_data = self.get_clusterrelease(clusterrelease_name).data
        distros = clusterrelease_data.get('spec', {}).get('allowedDistributions', [])
        if not distros:
            raise Exception(f"Clusterrelease {clusterrelease_name} don't contain 'allowedDistributions' list")
        sorted_distros = sorted(distros, key=lambda x: x['version'])
        distro_names = [d['id'] for d in sorted_distros]
        latest_distro = distro_names[-1]
        LOG.debug(f"Latest distro for clusterrelease '{clusterrelease_name}' is '{latest_distro}' "
                  f"(from the following distro list: {distro_names})")
        return latest_distro

    @cachetools_func.ttl_cache(ttl=10)
    def get_helmbundles(self):
        return self.api.kaas_helmbundles.list_all()

    def get_helmbundle(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_helmbundles.get(name=name,
                                             namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_regional_cluster_by_region_name(self, region_name):
        # Get mgmt/region clusters from 'default' namespace
        clusters = self.get_clusters(namespace=settings.CLUSTER_NAMESPACE)
        if settings.CLUSTER_NAMESPACE != settings.REGION_NAMESPACE:
            clusters += self.get_clusters(namespace=settings.REGION_NAMESPACE)

        cluster = [
            cluster for cluster in clusters
            if (cluster.is_management or cluster.is_regional) and cluster.region_name == region_name
        ]
        if not cluster:
            raise Exception(f"Regional cluster for MCC region '{region_name}' not found")
        cluster = cluster[0]
        LOG.info(f"Found regional cluster '{cluster.namespace}/{cluster.name}' for region '{region_name}'")
        return cluster

    @cachetools_func.ttl_cache(ttl=10)
    def get_clusters(self, namespace=None):
        if namespace:
            clusters = self.api.kaas_clusters.list(namespace=namespace)
        else:
            clusters = self.api.kaas_clusters.list_all()
        return [Cluster(self, cluster) for cluster in clusters]

    @cachetools_func.ttl_cache(ttl=10)
    def _get_mgmt_cluster(self):
        """Get the KaaS management cluster

        Find the cluster where the following spec is set:
          spec:
            providerSpec:
              value:
                kaas:
                  management:
                    enabled: true  # or whatever
        """
        clusters = self.api.kaas_clusters.list(namespace=settings.CLUSTER_NAMESPACE)
        for cluster in clusters:
            c = Cluster(self, cluster)
            try:
                if c.is_management:
                    return c
            except ApiException as ex:
                # Skip processing just deleted clusters
                if ex.status == 404:
                    continue
                raise ex
        return None

    def get_mgmt_cluster(self):
        cluster = self._get_mgmt_cluster()
        if cluster:
            return cluster
        else:
            raise Exception("Management cluster was not found among these clusters"
                            " {}".format(','.join([c.name for c in self.get_clusters()])))

    def get_clusterrelease_names(self):
        releases = self.get_clusterreleases()
        return [release.name for release in releases]

    def get_supported_clusterreleases(self, provider, k_release=None):
        """Get all clusterreleases for the specified provider

        :rtype dict: {<version>: <data>} ,
            where <version> is an internal release version like '7.0.0',
            and <data> is the original dict with data for this release from
            the 'spec.supportedClusterReleases' list
        """
        assert provider is not None and provider != "", "Provider has to be set to get supported cluster releases"

        if not k_release:
            k_r = max(self.get_kaasreleases(),
                      key=lambda x: version.parse(x.name))
        else:
            k_r = self.get_kaasrelease(k_release)

        cl_releases_data = k_r.data['spec']['supportedClusterReleases']
        cl_releases_filtered = [
            x
            for x in cl_releases_data
            if not x.get('providers', {}).get('supported', [])  # [] means all providers are allowed
            or provider in x.get('providers', {}).get('supported', [])
        ]
        cl_releases = {
            x['version']: x
            for x in cl_releases_filtered
        }

        assert (len(cl_releases.keys()) == len(cl_releases_filtered)), (
            f"len of cl_keys {len(cl_releases.keys())} and data {cl_releases.keys()}"
            f"Duplicate clusterrelease 'version' number found in "
            f"spec.supportedClusterReleases for kaasrelease "
            f"{k_r.data['spec']['version']}, please check:\n"
            f"len {len(cl_releases_data)} {yaml.dump(cl_releases_data)}"
        )
        return cl_releases

    def get_supported_clusterrelease_names(self, provider, k_release=None):
        """Get the list of cluster release names"""
        supported_crs = self.get_supported_clusterreleases(
            provider=provider, k_release=k_release)
        return [data['name'] for data in sorted(supported_crs.values(),
                                                key=lambda x: version.parse(x["version"]))]

    def get_supported_clusterrelease_version_by_name(self, clusterrelease_name, provider, k_release=None):
        """Get numeric ClusterRelease version 'x.y.z' by it's name"""
        supported_crs = self.get_supported_clusterreleases(provider=provider, k_release=k_release)
        filtered_crs = [data['version'] for data in supported_crs.values() if data['name'] == clusterrelease_name]
        assert len(filtered_crs) == 1, (
            f"Unexpected number of supported cluster releases found for ClusterRelease {clusterrelease_name}. "
            f"Provider: {provider}. Expected one item, but found: {filtered_crs}")
        return filtered_crs[0]

    @cachetools_func.ttl_cache(ttl=10)
    def get_machines(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_machines.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_machine(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_machines.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_lcmmachines(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_lcmmachines.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_lcmmachine(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_lcmmachines.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ansibleextras(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_ansibleextras.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ansibleextra(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_ansibleextras.get(name=name, namespace=namespace)

    def get_baremetalhosts(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_baremetalhosts.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_baremetalhost(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_baremetalhosts.get(name=name, namespace=namespace)

    def get_baremetalhostinventories(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_baremetalhostinventories.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_baremetalhostinventory(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_baremetalhostinventories.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=5)
    def get_baremetalhostcredentials(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_baremetalhostscredentials.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=5)
    def get_baremetalhostcredential(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_baremetalhostscredentials.get(name=name, namespace=namespace)

    # START: metallb
    @cachetools_func.ttl_cache(ttl=5)
    def get_metallbconfig(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_metallbconfig.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=5)
    def get_metallbconfigs(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_metallbconfig.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=5)
    def get_metallbconfigtemplate(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_metallbconfigtemplate.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=5)
    def get_metallbconfigtemplates(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_metallbconfigtemplate.list(namespace=namespace)

    # ENV: metallb

    @cachetools_func.ttl_cache(ttl=10)
    def get_ipam_hosts(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.ipam_hosts.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ipam_host(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.ipam_hosts.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ipam_ipaddrs(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.ipam_ipaddrs.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ipam_ipaddr(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.ipam_ipaddrs.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ipam_subnets(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.ipam_subnets.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_ipam_subnet(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.ipam_subnets.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_bgpadvertisements(self, namespace=settings.BM_METALLB_NS):
        return self.api.bgpadvertisement.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_bgpadvertisement(self, name, namespace=settings.BM_METALLB_NS):
        return self.api.bgpadvertisement.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_bgppeers(self, namespace=settings.BM_METALLB_NS):
        return self.api.bgppeer.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_bgppeer(self, name, namespace=settings.BM_METALLB_NS):
        return self.api.bgppeer.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_addresspools(self, namespace=settings.BM_METALLB_NS):
        return self.api.addresspool.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_addresspool(self, name, namespace=settings.BM_METALLB_NS):
        return self.api.addresspool.get(name=name, namespace=namespace)

    def get_multirack_clusters(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_multirack_cluster.list(namespace=namespace)

    def get_multirack_cluster(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_multirack_cluster.get(name=name, namespace=namespace)

    def get_multirack_racks(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_racks.list(namespace=namespace)

    def get_multirack_rack(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.kaas_racks.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_l2templates(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.l2templates.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_l2template(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.l2templates.get(name=name, namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_baremetalhostprofiles(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.baremetalhostprofiles.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_baremetalhostprofile(self, name,
                                 namespace=settings.CLUSTER_NAMESPACE):
        return self.api.baremetalhostprofiles.get(name=name,
                                                  namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_byocredentials(self):
        return self.api.kaas_byocredentials.list_all()

    def create_namespace(self, namespace):
        ns = self.api.namespaces.create(
            namespace=namespace,
            body={
                "apiVersion": "v1",
                "kind": "Namespace",
                "metadata": {
                    "name": namespace
                }
            }
        )
        return Namespace(self, ns)

    def create_namespace_raw(self, data):
        ns = self.api.namespaces.create(
            namespace=data['metadata']['name'],
            body=data,
        )
        LOG.info("create_namespace_raw")
        return Namespace(self, ns)

    @cachetools_func.ttl_cache(ttl=10)
    def get_or_create_namespace(self, namespace):
        try:
            LOG.info("Getting namespace %s", namespace)
            ns = self.get_namespace(namespace)
            ns_uid = ns.uid
            LOG.info("Got a namespace %s with uid - %s", namespace, ns_uid)
            return ns
        except ApiException as ex:
            if ex.status != 404:
                raise ex
            LOG.info("Namespace %s doesn't exists. Going to create", namespace)
            try:
                return self.create_namespace(namespace)
            except ApiException as ex:
                if ex.status == 409:
                    if ex.reason == "Conflict":
                        LOG.info("Namespace %s was created in parallel thread", namespace)
                        ns = self.get_namespace(namespace)
                        ns_uid = ns.uid
                        LOG.info("Got a namespace %s with uid - %s", namespace, ns_uid)
                        return ns
                raise ex

    @cachetools_func.ttl_cache(ttl=10)
    def get_namespace(self, namespace: str) -> "Namespace":
        ns = self.api.namespaces.get(name=namespace, namespace=namespace)
        return Namespace(self, ns)

    @cachetools_func.ttl_cache(ttl=10)
    def get_namespaces(
            self,
            hide_system_namespaces: bool = True) -> List["Namespace"]:
        namespaces = self.api.namespaces.list_all()
        if hide_system_namespaces:
            namespaces = [
                ns for ns in namespaces
                if ns.name not in system_namespaces
            ]
        return [Namespace(self, ns) for ns in namespaces]

    def get_events(self, namespace, event_prefix=None, sort=True):
        """Return dict, where values contain lists of dicts with events data for the namespace"""
        events = self.api.events.list(namespace=namespace)
        # return self.parse_events(events, event_prefix, sort)
        return utils.parse_events(events, event_prefix, sort)

    def get_events_by_uid(self, namespace, uid, event_prefix=None, sort=True):
        """Return list of events for the specified object"""
        events = self.get_events(namespace, event_prefix, sort)
        filtered_events = []
        for group, values in events.items():
            if values[0]['object_uid'] == uid:
                filtered_events.extend(values)
        return filtered_events

    def get_events_by_uid_str(self, namespace, uid, event_prefix=None,
                              sort=True):
        events = self.get_events_by_uid(namespace, uid, event_prefix, sort)
        messages = []
        for event in events:
            data = event['data']
            messages.append(
                f"{data.last_timestamp or data.event_time}  "
                f"{data.type}  {data.reason}  {data.message}")
        return '\n'.join(messages)

    def get_system_events(self):
        result = {}
        for ns in system_namespaces:
            result.update(self.get_events(namespace=ns, sort=True))
        return result

    def wait_kaasrelease(self, releases_count, timeout=30 * 60,
                         interval=30):

        clusters = [self.get_mgmt_cluster()] + self.get_regional_clusters()

        def check_available_kaas_releases():
            status_msg = ""
            for cluster in clusters:
                cluster_data = cluster.data
                cluster_provider_status = (cluster_data.get("status") or {}).get("providerStatus", {})
                clusterrelease_version = cluster.clusterrelease_version
                cluster_ready = cluster_provider_status.get("ready")
                cluster_warnings = cluster_provider_status.get("warnings", "")
                status_msg += (f"[{cluster.namespace}/{cluster.name}]  {clusterrelease_version}  "
                               f"Ready: {cluster_ready}  Warnings: '{cluster_warnings}'\n")

            releases = self.get_kaasrelease_names()
            LOG.info(f'Found kaasreleases: {releases}\n{status_msg}\n')
            if len(releases) != releases_count:
                return False
            return True

        waiters.wait(check_available_kaas_releases,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg='Expected count: {0}, got {1}'.format(
                         releases_count, len(self.get_kaasrelease_names())))

    def wait_kaasrelease_label(self, release, timeout=3400, interval=60):
        """Wait until the specified kaasrelease has the 'active' label

        During waiting, show the current and next clusterrelease versions
        for all the management and region clusters, and LCMMachine status
        to track the KaaS update process.
        """
        clusters = [self.get_mgmt_cluster()] + self.get_regional_clusters()
        lcmmachines = {}
        for cluster in clusters:
            lcmmachines[cluster.name] = cluster.get_lcmmachines()

        def check_label():
            # Get releases from mgmt and region clusters with lcmmachine status
            try:
                status_msg = ""
                for cluster in clusters:
                    current = cluster.clusterrelease_version
                    available = cluster.available_clusterrelease_version
                    lcmstates = [
                        (lcmmachine.data.get('status') or {}).get('state', '-')
                        for lcmmachine in lcmmachines[cluster.name]
                    ]
                    status_msg += (f"[{cluster.namespace}/{cluster.name}] "
                                   f"Processing: {current}  Next: {available}"
                                   f"  {lcmstates}\n")
                # Get kaasrelease and check if it is active
                kr_data = self.get_kaasrelease(release).data
                LOG.debug("Release_details {}".format(kr_data))
                kr_meta = kr_data['metadata']['labels']
                LOG.info(
                    f"Release: {release}. Waiting for the ACTIVE label. "
                    f"Current {kr_meta}\n{status_msg}"
                )
                if kr_meta:
                    if 'true' in kr_meta.get('kaas.mirantis.com/active', ''):
                        return True
                return False
            except (ApiException, MaxRetryError, ProtocolError):
                return False

        waiters.wait(check_label,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg="Missed active label for release {}".format(
                         release))

    @staticmethod
    def get_bmApiversion(release_name):
        """
        For all kaas-releases  we use new format
        :param release_name:
        :return:
        """
        return 'baremetal.k8s.io/v1alpha1'

    def get_aws_credential(self, name_prefix, namespace, region=''):
        """Find AWS credentials for the specified cluster in the namespace

        :return: tuple (aws_key_id, aws_secret_access_key)
        """
        creds_list = self.api.kaas_awscredentials.list(
            namespace=namespace)

        cred_names = [cred.name for cred in creds_list]
        # Filter creds that match the specified cluster_name
        creds = [cred.data for cred in creds_list
                 if cred.name.startswith(name_prefix)]
        if region:
            # Filter creds that match the specified region
            creds = [cred for cred in creds
                     if cred.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '') == region]

        assert creds, (
            f"AWSCredential object not found for the cluster "
            f"{namespace}/{name_prefix} : {cred_names}")

        cred = creds[0]
        aws_key_id = cred["spec"]["accessKeyID"]
        secret_ak = cred["spec"]["secretAccessKey"]
        if "value" in secret_ak:
            aws_secret_access_key = secret_ak["value"]
        else:
            secret_name = secret_ak["secret"]["name"]
            secret = self.api.secrets.get(name=secret_name,
                                          namespace=namespace)
            value = secret.read().data["value"]
            aws_secret_access_key = base64.b64decode(value).decode('utf-8')
        return aws_key_id, aws_secret_access_key

    def get_secret_data(self, secret_name, namespace, data_key):
        secret = self.api.secrets.get(
            secret_name, namespace=namespace)
        secret_data = secret.read().to_dict().get('data', {})
        if secret_data and data_key in secret_data:
            data = base64.b64decode(secret_data[data_key]).decode("utf-8")
        else:
            data = None
        LOG.debug(f"Secret {namespace}/{secret_name} data from {data_key}:"
                  f" {data}")
        return data

    def get_secret_all_data(self, secret_name, namespace):
        secret = self.api.secrets.get(
            secret_name, namespace=namespace)
        secret_data = secret.read().to_dict().get('data', {})
        data = {
            key: base64.b64decode(value).decode("utf-8")
            for key, value in secret_data.items()
        }
        LOG.debug(f"Secret {namespace}/{secret_name} data:"
                  f" {data}")
        return data

    @cachetools_func.ttl_cache(ttl=10)
    def get_keycloak_ip(self):
        # Get Keycloak hostname from Management cluster status.providerStatus.tls.keycloak.hostname
        mgmt_cluster = self.get_mgmt_cluster()
        tls_statuses = mgmt_cluster.get_tls_statuses()
        tls_keycloak_hostname = tls_statuses.get('keycloak', {}).get('hostname')
        if tls_keycloak_hostname:
            LOG.debug(f"Keycloak hostname from TLS status: {tls_keycloak_hostname}")
            return tls_keycloak_hostname
        # Fallback to IP address from the 'iam-keycloak-http' service
        keycloak_svc_status = self.api.api_core.read_namespaced_service(
            name='iam-keycloak-http',
            namespace='kaas').to_dict()['status']
        keycloak_svc = keycloak_svc_status['load_balancer']['ingress'][0]
        keycloak_ip = \
            keycloak_svc['ip'] or keycloak_svc['hostname']
        LOG.debug(f"Keycloak IP: {keycloak_ip}")
        return keycloak_ip

    @cachetools_func.ttl_cache(ttl=10)
    def get_scopes(self):
        return self.api.kaas_scopes.list_all()

    @cachetools_func.ttl_cache(ttl=10)
    def get_scope(self, name):
        return self.api.kaas_scopes.get(name=name)

    @cachetools_func.ttl_cache(ttl=10)
    def get_iamusers(self, name_prefix=None, request_timeout=300):
        return self.api.iam_users.list_all(name_prefix=name_prefix,
                                           _request_timeout=request_timeout)

    @cachetools_func.ttl_cache(ttl=10)
    def get_iamuser(self, name):
        return self.api.iam_users.get(name=name)

    @cachetools_func.ttl_cache(ttl=10)
    def get_iamglobalrolebindings(self):
        return self.api.iam_globalrolebindings.list_all()

    @cachetools_func.ttl_cache(ttl=10)
    def create_iamglobalrolebinding(self, name, role, user):
        """
        Create IAMGlobalRoleBinding object

        :param name: IAMGlobalRoleBinding's name
        :param role: Role name that should be assigned
        :param user: Username that should be associated with role

        :return: IAMGlobalRoleBindings object
        """
        body = {
            "apiVersion": "iam.mirantis.com/v1alpha1",
            "kind": "IAMGlobalRoleBinding",
            "metadata": {
                "name": name,
            },
            "role": {
                "name": role,
            },
            "user": {
                "name": user,
            },
        }
        return self.api.iam_globalrolebindings.create(body=body)

    def create_tlsconfig(self, cluster, name, namespace, hostname, cert_pem, key_pem, ca_pem=None):
        """
        Create TLSConfig object

        :param cluster: Cluster object
        :param name: TLSConfig's name
        :param namespace: TLSConfig's namespace
        :param cert_pem: server certificate authenticates server's identity to a client.
        :param key_pem: private key for a server.
        :param hostname: Hostname of a server
        :param ca_pem: the certificate that issued the server certificate

        :return: TLSConfig object
        """
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "TLSConfig",
            "metadata": {
                "name": name,
                "namespace": namespace,
            },
            "spec": {
                "serverName": hostname,
                "serverCertificate": base64.b64encode(cert_pem.encode('utf-8')).decode(),
                "privateKey": {
                    "value": key_pem,
                }
            },
        }

        data = cluster.data
        owner = {
            "apiVersion": data["api_version"],
            "kind": data["kind"],
            "name": cluster.name,
            "uid": cluster.uid,
        }
        body["metadata"]["ownerReferences"] = [owner]

        if ca_pem:
            body["spec"]["caCertificate"] = base64.b64encode(ca_pem.encode('utf-8')).decode()
        return self.api.kaas_tlsconfigs.create(name=name, namespace=namespace, body=body)

    def get_tls_config(self, name, namespace):
        return self.api.kaas_tlsconfigs.get(name=name, namespace=namespace)

    def get_tls_configs(self, namespace):
        return self.api.kaas_tlsconfigs.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_iamglobalrolebinding(self, name):
        return self.api.iam_globalrolebindings.get(name=name)

    @cachetools_func.ttl_cache(ttl=10)
    def get_iamrolebindings(self, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.iam_rolebindings.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_iamrolebinding(self, name, namespace=settings.CLUSTER_NAMESPACE):
        return self.api.iam_rolebindings.get(name=name, namespace=namespace)

    def get_vsphere_credential(self, name_prefix, namespace, region=''):
        """Find VSpere credentials for the specified cluster in the namespace

        """

        creds_list = self.api.kaas_vspherecredentials.list(
            namespace=namespace)
        cred_names = [cred.name for cred in creds_list]
        creds = [cred.data for cred in creds_list
                 if cred.name.startswith(name_prefix)]

        if region:
            # Filter creds that match the specified region
            creds = [cred for cred in creds
                     if cred.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '') == region]
        assert creds, (
            f"VsphereCredential object not found for the cluster "
            f"{namespace}/{name_prefix} : {cred_names}")

        one_cred = creds[0]
        cluster_api_creds = one_cred['spec']['clusterApi']
        cloud_provider_creds = one_cred['spec']['cloudProvider']
        vsphere = one_cred['spec']['vsphere']
        vsphere_creds = {
            "cluster_api_user": cluster_api_creds['username'],
            "cluster_api_pwd": settings.KAAS_VSPHERE_CAPI_PROVIDER_PASSWORD,
            "cloud_provider_user": cloud_provider_creds['username'],
            "cloud_provider_pwd":
                settings.KAAS_VSPHERE_CLOUD_PROVIDER_PASSWORD,
            "datacenter_name": vsphere['datacenter'],
            "server_insecure": vsphere['insecure'],
            "vsphere_server_ip": vsphere['server'],
            "vsphere_server_port": vsphere.get('port', None)
        }

        return vsphere_creds

    def get_rhel_license(self, name_prefix, namespace, region=''):
        licenses_list = self.api.kaas_rhellicenses.list(
            namespace=namespace)
        licenses_names = [lic.name for lic in licenses_list]
        licenses = [lic.data for lic in licenses_list
                    if lic.name.startswith(name_prefix)]

        if region:
            # Filter licenses that match the specified region
            licenses = [lic for lic in licenses
                        if lic.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '') == region]
        assert licenses, (
            f"RHELLicense object not found for the cluster "
            f"{namespace}/{name_prefix} : {licenses_names}")

        one_license = licenses[0]

        return one_license

    def find_rhel_license(self, name_prefix, namespace, region=''):
        licenses_list = self.api.kaas_rhellicenses.list(
            namespace=namespace)
        licenses_names = [lic.name for lic in licenses_list]
        licenses = [lic.data for lic in licenses_list
                    if lic.name.startswith(name_prefix)]

        if region:
            # Filter licenses that match the specified region
            licenses = [lic for lic in licenses
                        if lic.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '') == region]
        assert licenses, (
            f"RHELLicense object not found for the cluster "
            f"{namespace}/{name_prefix} : {licenses_names}")

        one_license = licenses[0]

        return one_license

    def update_mccupgrade(self, name, blockuntil=None, schedule=None, timezone='UTC', autodelay=None):
        body = {
            "spec": {
                "timeZone": timezone,

            }
        }
        if blockuntil:
            body["spec"].update({"blockUntil": f"{blockuntil}Z"})

        if autodelay is not None:
            body["spec"].update({"autoDelay": autodelay})

        if schedule:
            body["spec"].update({"schedule": [schedule]})
        mcc_upgrade_upd = self.api.kaas_mccupgrades.update(name=name, body=body)
        return mcc_upgrade_upd

    def get_license(self, license_name='license'):
        """
        Get cluster license by name
        Args:
            license_name: license object name

        Returns: si_tests.clients.k8s.license.License object

        """
        return self.api.license.get(name=license_name)

    def get_licenses(self):
        """
        Get all cluster licenses
        Returns: List of si_tests.clients.k8s.license.License objects

        """
        return self.api.license.list_all()

    def update_license(self, license_value, license_name='license'):
        """
        Update license object with license value
        Args:
            license_value: license data
            license_name: license object name

        Returns: si_tests.clients.k8s.license.License object

        """
        body = {
            "spec": {
                "license": {
                    "value": license_value
                }
            }
        }

        return self.api.license.update(name=license_name, body=body)

    def get_certificate_requests(self, name_prefix, namespace=None):
        """
        Get list of CertificateRequest objects
        Args:
            name_prefix: specify name
            namespace: specify namespace or None to use all namespaces

        Returns: list of si_tests.clients.k8s.models.v1_certificate_request.V1CertificateRequest

        """
        return self.api.certificate_requests.list(name_prefix=name_prefix, namespace=namespace)

    def get_containerregistry(self, namespace=None):
        """
        Get all cluster container registries
        Args:
            namespace: specify namespace or None to use all namespaces
        Returns: List of si_tests.clients.k8s.containerregistry.ContainerRegistry objects

        """
        return self.api.containerregistry.list(namespace=namespace)

    def get_containerregistry_by_name(self, name, ns=None):
        if ns:
            return [containerregistry for containerregistry in self.get_containerregistry(ns)
                    if name in containerregistry.name and containerregistry.namespace == ns]
        else:
            return [containerregistry for containerregistry in self.get_containerregistry()
                    if name in containerregistry.name]

    def update_containerregistry_ca(self, registry_ca, registry_name, namespace=None):
        """
        Update ContainerRegistry object with registry domain ca

        Args:
            registry_ca: registry ca
            registry_name: ContainerRegistry object name
            namespace: specify namespace or None to use all namespaces
        Returns: si_tests.clients.k8s.containerregistry.ContainerRegistry object
        """
        body = {
            "spec": {
                "CACert": registry_ca
            }
        }
        return self.api.containerregistry.update(name=registry_name, namespace=namespace, body=body)

    def update_containerregistry_domain(self, registry_domain, registry_name, namespace=None):
        """
        Update ContainerRegistry object with registry domain

        Args:
            registry_domain: registry domain
            registry_name: ContainerRegistry object name
            namespace: specify namespace or None to use all namespaces
        Returns: si_tests.clients.k8s.containerregistry.ContainerRegistry object
        """
        body = {
            "spec": {
                "domain": registry_domain
            }
        }
        return self.api.containerregistry.update(name=registry_name, namespace=namespace, body=body)

    def list_all_bootstrapregions(self):
        """
        List bootstrapregions of bv2 cluster in all ns

        :return:
        """
        return self.api.kaas_bootstrapregions.list_all()

    def create_bootstrapregion(self, name, provider, namespace='default'):
        """
        Create a bootstrapregion with specific provider
        :param name: Bootstrapregion name
        :param namespace: Bootstrapregion namespace (should be 'default')
        :param provider: Provider name (openstack, azure, aws, etc)
        :return:
        """
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "BootstrapRegion",
            "metadata": {
                "name": name,
                "namespace": namespace
            },
            "spec": {
                "provider": provider
            }
        }
        return self.api.kaas_bootstrapregions.create(name, namespace, body)

    def list_all_vvmt(self) -> List[kaas_vspherevmtemplates.KaaSVsphereVMTemplate]:
        """
        List vspherevmtemplates objects.

        :return: list of VsphereVMTemplate object
        :rtype: List[si_tests.clients.k8s.kaas_vspherevmtemplates.KaaSVsphereVMTemplate]
        """
        return self.api.vvmt.list_all()

    def hostosconfigurationmodule_is_present(self, name) -> bool:
        if self.api.kaas_hostosconfigurationmodules.present_all(name=name):
            return True
        return False

    def get_hostosconfigurationmodule(self, name):
        return self.api.kaas_hostosconfigurationmodules.get(name=name)

    def get_hostosconfigurationmodules(self):
        return self.api.kaas_hostosconfigurationmodules.list_all()

    def create_hostosconfigurationmodule_raw(self, data):
        return self.api.kaas_hostosconfigurationmodules.create(name=data['metadata']['name'], body=data)


class Namespace(object):
    __manager = None
    __namespace = None

    def __init__(self, manager: Manager, k8s_namespace):
        """
        manager: <Manager> instance
        k8s_namespace: <K8sNamespace> instance
        """
        self.__manager = manager
        self.__namespace = k8s_namespace

    @property
    def name(self):
        """Namespace name"""
        return self.__namespace.name

    @property
    def uid(self):
        """Namespace uid"""
        return self.data['metadata']['uid']

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind,

        metadata, spec, status or items
        """
        return self.__namespace.read().to_dict()

    def delete(self):
        """Deletes the current namespace"""
        self.__namespace.delete()

    def is_existed(self):
        """Verifies the current namespace deletion"""
        all_ns = self.__manager.get_namespaces()
        for n in all_ns:
            if n.name == self.name:
                LOG.info("Cluster namespace found")
                return True
        return False

    def wait_for_deletion(self, timeout=2400, interval=15):
        """Wait for the current namespace deletion"""
        waiters.wait(lambda: not self.is_existed(),
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting for namespace deletion')
        LOG.info("Namespace has been deleted.")

    def get_publickeys(self):
        return self.__manager.api.kaas_publickeys.list(namespace=self.name)

    def get_publickey(self, name):
        return self.__manager.api.kaas_publickeys.get(name=name,
                                                      namespace=self.name)

    def create_publickey(self, name, key):
        """Create public key in the current namespace

        :param name: str, public key name
        :param key: str, public key content
        """
        return self.__manager.api.kaas_publickeys.create(
            namespace=self.name,
            body={
                "apiVersion": "kaas.mirantis.com/v1alpha1",
                "kind": "PublicKey",
                "metadata": {
                    "name": name,
                },
                "spec": {
                    "publicKey": key
                }
            }
        )

    def create_publickey_raw(self, data):
        return self.__manager.api.kaas_publickeys.create(namespace=self.name,
                                                         name=data['metadata']['name'], body=data)

    def get_secrets(self):
        return self.__manager.api.secrets.list(namespace=self.name)

    def get_secret(self, name):
        return self.__manager.api.secrets.get(name=name, namespace=self.name)

    def create_openstack_secret(self, name, clouds_yaml_str, region=""):
        encoded_clouds_yaml_str = base64.b64encode(
            clouds_yaml_str.encode('utf-8')).decode()
        data = {
            'clouds.yaml': encoded_clouds_yaml_str
        }
        return self.create_credentials_secret(name, data, region=region)

    def create_baremetalhostcredential(self, name, data, region="",
                                       provider=settings.BAREMETAL_PROVIDER_NAME):

        # Ugly hack, since old secret allow to work with base64- but monitoring only with raw.
        # All SI in base64, and we dont wont to re-write it all.
        data_decoded = {
            'username': base64.b64decode(data['username']).decode("utf-8"),
            'password': {
                'value': base64.b64decode(data['password']).decode("utf-8"),
            },
            'monitoringUsername': base64.b64decode(data.get('monitoringUsername', '')).decode("utf-8") or '',
            'monitoringPassword': base64.b64decode(data.get('monitoringPassword', '')).decode("utf-8") or '',
        }

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "BareMetalHostCredential",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {
                    "kaas.mirantis.com/provider": provider,
                },
            },
            "spec": data_decoded,
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        return self.__manager.api.kaas_baremetalhostscredentials.create(namespace=self.name,
                                                                        name=name, body=body)

    def create_baremetalhostcredential_raw(self, data):
        return self.__manager.api.kaas_baremetalhostscredentials.create(namespace=self.name,
                                                                        name=data['metadata']['name'], body=data)

    def create_metallbconfig_raw(self, data):
        return self.__manager.api.kaas_metallbconfig.create(namespace=self.name,
                                                            name=data['metadata']['name'], body=data)

    def create_metallbconfig(self, name, labels, spec):
        """
        Create [kaas.mirantis.com/v1alpha1/]MetalLBConfig object

        :param name: MetalLBConfig object name
        :param labels: object labels map
        :param spec: map of metallb.io/v1beta1 object templates
        :return: MetalLBConfig object
        """
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "MetalLBConfig",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": labels,
            },
            "spec": spec,
        }
        return self.__manager.api.kaas_metallbconfig.create(
            namespace=self.name,
            body=body
        )

    def create_metallbconfigtemplate_raw(self, data):
        self.__manager.api.kaas_metallbconfigtemplate.create(namespace=self.name,
                                                             name=data['metadata']['name'], body=data)

    def create_metallbconfigtemplate(self, name, labels, spec):

        body = {
            "apiVersion": "ipam.mirantis.com/v1alpha1",
            "kind": "MetalLBConfigTemplate",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": labels,
            },
            "spec": spec,
        }
        self.__manager.api.kaas_metallbconfigtemplate.create(namespace=self.name,
                                                             name=name, body=body)

    def create_secret(self, name, data, labels=None):
        """Create Secret object on Management cluster in the current Namespace"""
        body = {
            "apiVersion": "v1",
            "kind": "Secret",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": labels or {},
            },
            "data": data
        }
        return self.__manager.api.secrets.create(namespace=self.name, body=body)

    def create_credentials_secret(self, name, data, region='', provider=settings.OPENSTACK_PROVIDER_NAME):
        labels = {
            "kaas.mirantis.com/credentials": "true",
            "kaas.mirantis.com/provider": provider
        }
        if region:
            labels["kaas.mirantis.com/region"] = region
        return self.create_secret(name=name, data=data, labels=labels)

    def delete_secret(self, name, timeout=60, interval=5):
        """Delete Secret object from Management cluster in the current Namespace"""
        if self.__manager.api.secrets.present(name=name, namespace=self.name):
            secret = self.get_secret(name)
            secret.delete()
            timeout_msg = f"Secret {name} was not deleted in {timeout}s"
            waiters.wait(lambda: not bool(self.__manager.api.secrets.present(name=name, namespace=self.name)),
                         timeout=timeout,
                         interval=interval,
                         timeout_msg=timeout_msg)
            LOG.info(f"Secret {name} has been succesfully deleted")
        else:
            LOG.warning(f"Secret {name} not found, nothing to delete")

    def create_openstack_credential(self, name, clouds_yaml_str, region="", validate=True):
        clouds = yaml.load(clouds_yaml_str,
                           Loader=yaml.SafeLoader).get("clouds", {})
        if len(clouds) > 1:
            raise Exception("Multiple clouds are present in OpenStack config")
        for cloud in clouds.values():
            auth = cloud.get("auth", {})
            body = {
                "apiVersion": "kaas.mirantis.com/v1alpha1",
                "kind": "OpenStackCredential",
                "metadata": {
                    "name": name,
                    "namespace": self.name,
                    "labels": {},
                },
                "spec": {
                    "auth": {
                        "authURL": auth.get("auth_url", ""),
                        "userName": auth.get("username", ""),
                        "userID": auth.get("user_id", ""),
                        "password": {
                            "value": auth.get("password", "")
                        },
                        "projectDomainName": auth.get(
                            "project_domain_name", ""),
                        "projectDomainID": auth.get(
                            "project_domain_id", ""),
                        "projectName": auth.get("project_name", ""),
                        "projectID": auth.get("project_id", ""),
                        "userDomainName": auth.get(
                            "user_domain_name", ""),
                        "userDomainID": auth.get("user_domain_id", ""),
                        "domainName": auth.get("domain_name", ""),
                        "domainID": auth.get("domain_id", ""),
                        "defaultDomain": auth.get("default_domain", "")

                    },
                    "regionName": cloud.get("region_name", ""),
                }
            }
            if region:
                body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

            res = self.__manager.api.kaas_openstackcredentials.create(
                namespace=self.name,
                body=body
            )
            if validate:
                # keep in mind that object may be differ to 'res'
                # after validation
                self.wait_provider_cred_is_valid(
                    credential_name=name,
                    provider=settings.OPENSTACK_PROVIDER_NAME)
            return res
        raise Exception("There are no clouds provided in OpenStack config")

    def wait_provider_cred_is_valid(self,
                                    credential_name,
                                    provider,
                                    retries=60,
                                    interval=60):
        def wait_cred_status():
            try:
                LOG.info("Wait until credential {} become valid".format(
                    credential_name))
                if provider == utils.Provider.openstack.provider_name:
                    cred = self.get_openstackcredential(
                        name=credential_name).data
                elif provider == utils.Provider.vsphere.provider_name:
                    cred = self.__manager.api.kaas_vspherecredentials.get(
                        name=credential_name,
                        namespace=self.name).data
                elif provider == utils.Provider.aws.provider_name:
                    cred = self.__manager.api.kaas_awscredentials.get(
                        name=credential_name,
                        namespace=self.name).data
                elif provider == utils.Provider.equinixmetal.provider_name:
                    cred = self.__manager.api.kaas_equinixmetalcredentials.get(
                        name=credential_name,
                        namespace=self.name).data
                elif provider == utils.Provider.azure.provider_name:
                    cred = self.__manager.api.kaas_azurecredentials.get(
                        name=credential_name,
                        namespace=self.name).data
                else:
                    raise Exception("Unsupported provider for cred validation")

                LOG.info('cred data {}'.format(cred))
                if cred['status']:
                    if cred['status']['valid']:
                        return True
            except ApiException as e:
                LOG.debug(
                    'Failed to wait credential '
                    'become valid {}'.format(e))
            return False

        waiters.wait(lambda: wait_cred_status(),
                     timeout=retries * interval,
                     interval=interval,
                     timeout_msg="Credential {0} isn't valid "
                                 "after {1} retries".format(
            credential_name, retries
        ))

    def create_rhel_license(self, name, rhel_creds, region="", is_license_key=False):

        if is_license_key:
            (key, org, rpm_url) = rhel_creds
            spec = {
                "activationKey": {
                    "value": key
                },
                "orgID": org,
                "rpmUrl": rpm_url,
            }
        else:
            (username, password) = rhel_creds
            spec = {
                "username": username,
                "password": {
                    "value": password
                }
            }

        body = {
                "apiVersion": "kaas.mirantis.com/v1alpha1",
                "kind": "RHELLicense",
                "metadata": {
                    "name": name,
                    "namespace": self.name,
                    "labels": {}
                },
                "spec": spec
            }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        return self.__manager.api.kaas_rhellicenses.create(
            namespace=self.name,
            body=body
        )

    def create_vsphere_credential(self, name, vsphere_creds, region="", validate=True):

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "VsphereCredential",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {}
            },
            "spec": {
                "clusterApi": {
                    "username": vsphere_creds['cluster_api_user'],
                    "password": {
                        "value": vsphere_creds['cluster_api_pwd']
                    }
                },
                "cloudProvider": {
                    "username": vsphere_creds['cloud_provider_user'],
                    "password": {
                        "value": vsphere_creds['cloud_provider_pwd']
                    },
                },
                "vsphere": {
                    "server": vsphere_creds['vsphere_server_ip'],
                    "insecure": vsphere_creds['server_insecure'],
                    "datacenter": vsphere_creds['datacenter_name']
                }
            }
        }

        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if vsphere_creds.get('vsphere_server_port'):
            body["spec"]["vsphere"]["port"] = vsphere_creds[
                'vsphere_server_port']

        res = self.__manager.api.kaas_vspherecredentials.create(
            namespace=self.name,
            body=body)

        if validate:
            # keep in mind that object may be differ to 'res' after validation
            self.wait_provider_cred_is_valid(
                credential_name=name,
                provider=utils.Provider.vsphere.provider_name)
        return res

    def create_aws_credential(self, name, aws_key_id, aws_secret_access_key, region="", validate=True):
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "AWSCredential",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {}
            },
            "spec": {
                "accessKeyID": aws_key_id,
                "secretAccessKey": {
                    "value": aws_secret_access_key
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        res = self.__manager.api.kaas_awscredentials.create(namespace=self.name, body=body)

        if validate:
            # keep in mind that object may be differ to 'res' after validation
            self.wait_provider_cred_is_valid(
                credential_name=name,
                provider=utils.Provider.aws.provider_name)
        return res

    def create_equinix_credential(self, name, api_token, project_id, region, validate=True):
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "EquinixMetalCredential",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {}
            },
            "spec": {
                "apiToken": {
                    "value": api_token
                },
                "projectID": project_id,
            }
        }

        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        res = self.__manager.api.kaas_equinixmetalcredentials.create(namespace=self.name, body=body)
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if validate:
            # keep in mind that object may be differ to 'res' after validation
            self.wait_provider_cred_is_valid(
                credential_name=name,
                provider=utils.Provider.equinixmetal.provider_name)
        return res

    def create_azure_credential(self,
                                name,
                                subscription_id,
                                tenant_id,
                                client_id,
                                client_secret,
                                region="",
                                validate=True):
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "AzureCredential",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {}
            },
            "spec": {
                "subscriptionID": subscription_id,
                "tenantID": tenant_id,
                "clientID": client_id,
                "clientSecret": {
                    "value": client_secret
                },
            }
        }

        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        res = self.__manager.api.kaas_azurecredentials.create(namespace=self.name, body=body)
        if validate:
            # keep in mind that object may be differ to 'res' after validation
            self.wait_provider_cred_is_valid(
                credential_name=name,
                provider=utils.Provider.azure.provider_name)
        return res

    def get_events(self):
        return self.__manager.get_events(namespace=self.name)

    def print_events(self):
        events_msg = utils.create_events_msg(self.get_events())
        LOG.info(events_msg)

    def get_openstackresource(self, name):
        return self.__manager.api.kaas_openstackresources.get(
            name=name, namespace=self.name)

    def get_openstackcredential(self, name):
        return self.__manager.api.kaas_openstackcredentials.get(
            name=name, namespace=self.name)

    def get_openstackresources(self):
        return self.__manager.api.kaas_openstackresources.list(
            namespace=self.name)

    def get_clusters(self):
        clusters = self.__manager.api.kaas_clusters.list(namespace=self.name)
        return [Cluster(self.__manager, cluster) for cluster in clusters]

    def get_cluster(self, name: str) -> "Cluster":
        cluster = self.__manager.api.kaas_clusters.get(name=name,
                                                       namespace=self.name)
        return Cluster(self.__manager, cluster)

    def get_provider_credentials(self, provider, region=''):
        if provider == utils.Provider.aws:
            creds = self.__manager.api.kaas_awscredentials.list_all()
        elif provider == utils.Provider.openstack:
            creds = self.__manager.api.kaas_openstackcredentials.list_all()
        elif provider == utils.Provider.vsphere:
            creds = self.__manager.api.kaas_vspherecredentials.list_all()
        elif provider == utils.Provider.equinixmetal:
            creds = self.__manager.api.kaas_equinixmetalcredentials.list_all()
        else:
            creds = None
        if region and creds is not None:
            creds = [cred for cred in creds if
                     cred.data.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '') == region]
        return creds

    def create_byocredential(self, name, ucp_host, cacert,
                             clientcert, clientkey, ucp_kubeconfig,
                             region=''):
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "BYOCredential",
            "metadata": {
                "name": name,
                "namespace": self.name,
                'labels': {
                    'kaas.mirantis.com/provider': 'byo'
                }
            },
            "spec": {
                "docker": {
                    "host": ucp_host,
                    "caCert": cacert,
                    "clientCert": clientcert,
                    "clientKey": {"value": clientkey}
                },
                "kubeConfig": {'value': ucp_kubeconfig}
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        return self.__manager.api.kaas_byocredentials.create(namespace=self.name, body=body)

    def get_byocredential(self, name):
        return self.__manager.api.kaas_byocredentials.get(name=name,
                                                          namespace=self.name)

    def get_byocredentials(self):
        return self.__manager.api.kaas_byocredentials.list(namespace=self.name)

    def get_cephclusters(self):
        return self.__manager.api.kaas_cephclusters.list(
            namespace=self.name)

    def get_cephcluster(self, name):
        return self.__manager.api.kaas_cephclusters.get(
            name=name, namespace=self.name)

    def get_helmbundle(self, name):
        return self.__manager.api.kaas_helmbundles.get(name=name,
                                                       namespace=self.name)

    def get_helmbundles(self):
        return self.__manager.api.kaas_helmbundles.list(namespace=self.name)

    def attach_ucp_cluster(self,
                           name,
                           creds_name,
                           region='',
                           lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
                           provider_name="byo"):
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Cluster",
            "metadata": {
                "name": name,
                "namespace": self.name,
                'labels': {
                    'kaas.mirantis.com/provider': provider.provider_name
                },
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": f"byo.kaas.mirantis.com/{provider.api_version}",
                        "kind": provider.cluster_spec,
                        "credentials": creds_name,

                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if not lma_enabled:
            LOG.info("LMA is disabled")
            # Explicitly disable LMA for BYO cluster
            body["spec"]["providerSpec"]["value"]["helmReleases"] = [
                {
                    "name": "stacklight",
                    "enabled": False
                }
            ]
        else:
            stacklight_body = {
                "name": "stacklight",
                "enabled": True,
                "values": {
                    "alertmanagerSimpleConfig": {
                        "email": {
                            "enabled": False
                        },
                        "slack": {
                            "enabled": False
                        }
                    },
                    "highAvailabilityEnabled": True,
                    "logging": {
                        "enabled": settings.STACKLIGHT_ENABLE_LOGGING
                    },
                    "prometheusServer": {
                        "customAlerts": [],
                        "persistentVolumeClaimSize": "16Gi",
                        "retentionSize": "15GB",
                        "retentionTime": "15d",
                        "watchDogAlertEnabled": False
                    }
                }
            }

            lma_logging = {
                "elasticsearch": {
                    "logstashRetentionTime": "30",
                    "persistentVolumeClaimSize": "30Gi",
                    "retentionTime": {
                        "logstash": "3d",
                        "events": "2w",
                        "notifications": "1m",
                    }
                }
            }

            if settings.STACKLIGHT_ENABLE_LOGGING:
                utils.merge_dicts(stacklight_body['values'], lma_logging)

            body["spec"]["providerSpec"]["value"]["helmReleases"] = [
                stacklight_body
            ]

        LOG.info(f"Attach cluster with {body}")
        return self.__manager.api.kaas_clusters.create(namespace=self.name,
                                                       body=body)

    def create_cluster(self,
                       name,
                       release_name,
                       credentials_name="",
                       external_network_id="",
                       region="",
                       provider="openstack",
                       aws_region="us-east-2",
                       aws_az="us-east-2a",
                       aws_priv_subnet_cidr="10.0.0.0/24",
                       aws_pub_subnet_cidr="10.0.1.0/24",
                       loadbalancer_host=None,
                       metallb_ip_range=None,
                       services_cidr="10.96.0.0/16",
                       pods_cidr="192.168.0.0/16",
                       nodes_cidr="10.10.10.0/24",
                       dns=settings.KAAS_CHILD_CLUSTER_DNS,
                       lma_enabled=True,
                       lma_extra_options=None,
                       extra_helm_releases=None,
                       secure_overlay=False,
                       dedicated_controlplane=True,
                       public_key_name=None,
                       proxy_name=None,
                       ipam_enabled=False,
                       ipam_cidr=None,
                       ipam_gw=None,
                       ipam_ns=None,
                       ipam_incl_range=None,
                       container_registy_name=None,
                       boot_from_volume=False,
                       boot_volume_size=None,
                       max_worker_prepare_count=settings.KAAS_CHILD_WORKER_PREPARE_COUNT,
                       max_worker_upgrade_count=settings.KAAS_CHILD_WORKER_UPGRADE_COUNT,
                       bastion=None,
                       metallb_pool_protocol='layer2',
                       metallb_peers=None,
                       metallb_speaker=None,
                       use_bgp_announcement=None,
                       calico_mtu=None):

        if extra_helm_releases is None:
            extra_helm_releases = []

        # both options cant be in True
        if all([settings.KAAS_BM_AIO_ON_CHILD,
                settings.STACKLIGHT_ENABLE_HA,
                provider == 'baremetal']):
            raise Exception("You have wrong test configuration for bm cluster:\n"
                            f"STACKLIGHT_ENABLE_HA={settings.STACKLIGHT_ENABLE_HA}\n"
                            f"KAAS_BM_AIO_ON_CHILD={settings.KAAS_BM_AIO_ON_CHILD}")

        if public_key_name:
            public_key = {"name": public_key_name}
        else:
            public_key = {}
            LOG.warning("Cluster will be created without public key")

        boot_from_volume_template = {
            "enabled": boot_from_volume,
            "volumeSize": boot_volume_size
        }

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Cluster",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "annotations": {
                    "kaas.mirantis.com/lcm": "true"
                },
                "labels": {
                    "kaas.mirantis.com/region": region,
                    "kaas.mirantis.com/provider": provider
                }
            },
            "spec": {
                "clusterNetwork": {
                    "services": {
                        "cidrBlocks": [
                            services_cidr
                        ]
                    },
                    "pods": {
                        "cidrBlocks": [
                            pods_cidr
                        ]
                    },
                    "serviceDomain": ""
                },
                "providerSpec": {
                    "value": {
                        "apiVersion": "openstackproviderconfig.k8s.io/v1alpha1",  # noqa
                        "kind": "OpenstackClusterProviderSpec",
                        "dnsNameservers": dns.split(','),
                        "externalNetworkId": external_network_id,
                        "nodeCidr": nodes_cidr,
                        "release": release_name,
                        "publicKeys": [
                            public_key
                        ],
                        "credentials": credentials_name,
                        "secureOverlay": secure_overlay,
                        "dedicatedControlPlane": dedicated_controlplane,
                        "helmReleases": extra_helm_releases
                    }
                }
            }
        }

        # calico_mtu for Wireguard must be not greater than "underlay_mtu-60".
        # calico_mtu w/o Wireguard must be not greater than "underlay_mtu-50".
        # default underlay_mtu is 1500, default calico_mtu is 1450.
        # if you have bigger underlay_mtu value, please adjust calico_mtu if needed.
        if calico_mtu is not None:
            body['spec']['providerSpec']['value']['calico'] = {
                "mtu": calico_mtu
            }

        if use_bgp_announcement is not None:
            body['spec']['providerSpec']['value']['useBGPAnnouncement'] = \
                use_bgp_announcement

        lma_defaults = {
            "name": "stacklight",
            "enabled": True,
            "values": {
                "alertmanagerSimpleConfig": {
                    "email": {
                        "enabled": False
                    },
                    "slack": {
                        "enabled": False
                    }
                },
                "highAvailabilityEnabled": settings.STACKLIGHT_ENABLE_HA,
                "logging": {
                    "enabled": settings.STACKLIGHT_ENABLE_LOGGING
                },
                "prometheusServer": {
                    "customAlerts": [],
                    "persistentVolumeClaimSize": "16Gi",
                    "retentionSize": "15GB",
                    "retentionTime": "15d",
                    "watchDogAlertEnabled": False
                }
            }
        }

        lma_defaults_turned_off = {
            "name": "stacklight",
            "enabled": False,
            "values": {
                "highAvailabilityEnabled": settings.STACKLIGHT_ENABLE_HA,
                "logging": {
                    "enabled": settings.STACKLIGHT_ENABLE_LOGGING
                },
                "prometheusServer": {
                    "persistentVolumeClaimSize": "0Gi",
                }
            }
        }

        lma_logging = {
            "elasticsearch": {
                "logstashRetentionTime": "30",
                "persistentVolumeClaimSize": "30Gi",
                "retentionTime": {
                    "logstash": "3d",
                    "events": "2w",
                    "notifications": "1m",
                }
            }
        }

        enable_enforce_mod = {
            "name": "policy-controller",
            "enabled": True,
            "values": {
                "policy": {
                    "mode": "enforce"
                }
            }
        }

        if settings.STACKLIGHT_ENABLE_LOGGING:
            utils.merge_dicts(lma_defaults['values'], lma_logging)

        mgmt_cluster = self.__manager.get_mgmt_cluster()
        actual_kaasrelease = mgmt_cluster.get_kaasrelease_version()
        metallb_helm_release = {
                "name": "metallb",
                "values": {}
        }

        if metallb_ip_range:
            metallb_helm_release['values']['configInline'] = {
                "address-pools": [
                    {
                        "addresses": [
                            metallb_ip_range
                        ],
                        "name": "default",
                        "protocol": metallb_pool_protocol
                    }
                ],
            }
        if metallb_peers:
            metallb_helm_release['values']['configInline']["peers"] = metallb_peers
        if metallb_speaker:
            metallb_helm_release['values']['speaker'] = metallb_speaker

        aws_subnet_defaults = {
            "subnets": [
                {
                    "cidrBlock": aws_priv_subnet_cidr,
                    "isPublic": False,
                    "availabilityZone": aws_az
                },
                {
                    "cidrBlock": aws_pub_subnet_cidr,
                    "isPublic": True,
                    "availabilityZone": aws_az
                }
            ]
        }

        if proxy_name:
            body['spec']['providerSpec']['value']['proxy'] = proxy_name

        if container_registy_name:
            body['spec']['providerSpec']['value']['containerRegistries'] = container_registy_name
        if settings.AUDITD_ENABLE:
            body['spec']['providerSpec']['value']['audit'] = {
                "auditd": {
                    "enabled": True,
                    "enabledAtBoot": True,
                    "backlogLimit": 1,
                    "maxLogFile": 1,
                    "maxLogFileAction": "rotate",
                    "maxLogFileKeep": 4,
                    "mayHaltSystem": True,
                }
            }
        # TODO(tleontovich) Delete if condition after 2-28, when we do not support those releases in upd case
        if version.parse(actual_kaasrelease) >= version.parse("kaas-2-26-0-rc"):
            if (version.parse(release_name) >= version.parse("mosk-17-1-0-rc-24-1")) \
                    or (version.parse(release_name) >= version.parse("mke-16-1-0-rc-3-7-4")):
                if not body["spec"]["providerSpec"]["value"].get("audit"):
                    body["spec"]["providerSpec"]["value"]["audit"] = {}
                body["spec"]["providerSpec"]["value"]["audit"]["kubernetes"] = {
                    "apiServer": {
                        "enabled": True,
                    }
                }
        if settings.K8S_PROFILING_ENABLE and \
                version.parse(release_name) >= version.parse("mosk-17-1-0-rc-24-1"):
            body['spec']['providerSpec']['value']['profiling'] = {"enabled": True}
        if settings.ETCD_STORAGE_QOUTA:
            body['spec']['providerSpec']['value']['etcd'] = {'storageQuota': settings.ETCD_STORAGE_QOUTA}

        if lma_enabled:
            if lma_extra_options:
                assert type(lma_extra_options) == dict, (
                    "Please, use correct format for lma_extra_options. "
                    "Use next dict as a reference: \n{}").format(
                    yaml.dump(lma_defaults, default_flow_style=False))
                lma_options = utils.merge_dicts(lma_defaults,
                                                lma_extra_options)
                body.get("spec").get("providerSpec").get(
                    "value").get("helmReleases").append(lma_options)
            else:
                body.get("spec").get("providerSpec").get(
                    "value").get("helmReleases").append(lma_defaults)
        else:
            body.get("spec").get("providerSpec").get(
                "value").get("helmReleases").append(lma_defaults_turned_off)

        if provider == utils.Provider.baremetal.provider_name:
            body['spec']['providerSpec']['value']['helmReleases'].append(metallb_helm_release)
            body['spec']['providerSpec']['value'][
                'apiVersion'] = self.__manager.get_bmApiversion(release_name)
            body['spec']['providerSpec']['value']['kind'] = utils.Provider.baremetal.cluster_spec
            if loadbalancer_host:
                body['spec']['providerSpec']['value']['loadBalancerHost'] = loadbalancer_host
            body['spec']['providerSpec']['value']['region'] = region
            if settings.PARALLEL_UPGRADE_ENABLED:
                body['spec']['providerSpec']['value']['maxWorkerPrepareCount'] = max_worker_prepare_count
                body['spec']['providerSpec']['value']['maxWorkerUpgradeCount'] = max_worker_upgrade_count
            if settings.ENFORCE_ENABLED:
                body['spec']['providerSpec']['value']['helmReleases'].append(enable_enforce_mod)

        elif provider == utils.Provider.aws.provider_name:
            body['spec']['providerSpec']['value'][
                'apiVersion'] = f"aws.kaas.mirantis.com/{utils.Provider.aws.api_version}"
            body['spec']['providerSpec']['value'][
                'kind'] = utils.Provider.aws.cluster_spec
            body['spec']['providerSpec']['value'][
                'region'] = aws_region
            body['spec']['providerSpec']['value'][
                'networkSpec'] = aws_subnet_defaults
            if not settings.STACKLIGHT_ENABLE_HA and settings.PARALLEL_UPGRADE_ENABLED:
                body['spec']['providerSpec']['value']['maxWorkerPrepareCount'] = max_worker_prepare_count
                body['spec']['providerSpec']['value']['maxWorkerUpgradeCount'] = max_worker_upgrade_count

        elif provider == utils.Provider.vsphere.provider_name:
            body['spec']['providerSpec']['value'].update(
                {
                    "apiVersion": f"vsphere.cluster.k8s.io/{utils.Provider.vsphere.api_version}",
                    "kind": utils.Provider.vsphere.cluster_spec,
                    "loadBalancerHost": loadbalancer_host,
                    "vsphere": {
                        "cloudProviderDatastore":
                            settings.KAAS_VSPHERE_DATASTORE_NAME,
                        "clusterApiDatastore":
                            settings.KAAS_VSPHERE_DATASTORE_NAME,
                        "clusterApiDatastoreFolder":
                            settings.KAAS_VSPHERE_DATASTORECLUSTER_PATH,
                        "machineFolderPath":
                            settings.KAAS_VSPHERE_MACHINES_FOLDER,
                        "networkPath": settings.KAAS_VSPHERE_NETWORK_PATH,
                        "resourcePoolPath":
                            settings.KAAS_VSPHERE_RESOURCE_POOL_PATH,
                        "scsiControllerType": "pvscsi"
                    },

                }
            )
            if not settings.STACKLIGHT_ENABLE_HA and settings.PARALLEL_UPGRADE_ENABLED:
                body['spec']['providerSpec']['value'].update(
                    {
                        "maxWorkerPrepareCount":
                            max_worker_prepare_count,
                        "maxWorkerUpgradeCount":
                            max_worker_upgrade_count
                    }
                )
            body.get("spec").get("providerSpec").get(
                "value").get("helmReleases").append(metallb_helm_release)

            if ipam_enabled:
                LOG.info("Going to update provider spec")
                body.get("spec").get("providerSpec").get(
                    "value")["clusterNetwork"] = {
                    "clusterNetwork": True,
                    "cidr": ipam_cidr,
                    "ipamEnabled": ipam_enabled,
                    "gateway": ipam_gw,
                    "includeRanges": [ipam_incl_range],
                    # Exclude ranges aren't use now in tests. But defined
                    # in original teamples. Let's left them are disabled.
                    # "excludeRanges": [],
                    "nameservers": [ipam_ns]
                }

        elif provider == utils.Provider.equinixmetal.provider_name:
            machines_count = \
                int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT) + \
                int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
            equinix_region = ''
            regional_cluster = self.__manager.get_regional_cluster_by_region_name(region)
            if regional_cluster.provider in [utils.Provider.equinixmetal, utils.Provider.equinixmetalv2]:
                regional_cluster_facility = regional_cluster.spec['providerSpec']['value'].get('facility')
                assert regional_cluster_facility, (
                    f"Regional cluster {regional_cluster.namespace}/{regional_cluster.name} "
                    f"don't have 'facility' key in the providerSpec.value")
                metro_scoped_facilities = utils.discover_equinix_metros_and_facilities(
                    settings.KAAS_EQUINIX_USER_API_TOKEN)
                # Equinix region is one of: AMER, EMEA or APAC
                equinix_region = utils.get_equinix_region_by_facility(metro_scoped_facilities,
                                                                      regional_cluster_facility)
                LOG.info(f"Regional cluster located in the equinix facility '{regional_cluster_facility}', "
                         f"assuming Equinix region '{equinix_region}'")
            facility = utils.get_available_equinix_metro(
                settings.KAAS_EQUINIX_USER_API_TOKEN,
                settings.KAAS_EQUINIX_MACHINE_TYPE,
                machines_count,
                region_scoped=equinix_region)

            body['spec']['providerSpec']['value'][
                'apiVersion'] = f"equinix.kaas.mirantis.com/{utils.Provider.equinixmetal.api_version}"
            body['spec']['providerSpec']['value'][
                'kind'] = utils.Provider.equinixmetal.cluster_spec
            body['spec']['providerSpec']['value'][
                'facility'] = facility
            body['spec']['providerSpec']['value']['ceph'] = {
                'manualConfiguration': settings.KAAS_EQUINIX_CEPH_MANUAL_CONFIGURATION
            }
            if settings.KAAS_EQUINIX_PROJECT_SSH_KEYS:
                body['spec']['providerSpec']['value'][
                    'projectSSHKeys'] = settings.KAAS_EQUINIX_PROJECT_SSH_KEYS.split(",")

        elif provider == utils.Provider.azure.provider_name:
            body['spec']['providerSpec']['value'][
                'apiVersion'] = f"azure.kaas.mirantis.com/{utils.Provider.azure.api_version}"
            body['spec']['providerSpec']['value'][
                'kind'] = utils.Provider.azure.cluster_spec
            body['spec']['providerSpec']['value'][
                'location'] = settings.KAAS_AZURE_CLUSTER_LOCATION
            if not settings.STACKLIGHT_ENABLE_HA and settings.PARALLEL_UPGRADE_ENABLED:
                body['spec']['providerSpec']['value']['maxWorkerPrepareCount'] = max_worker_prepare_count
                body['spec']['providerSpec']['value']['maxWorkerUpgradeCount'] = max_worker_upgrade_count

        elif provider == utils.Provider.equinixmetalv2.provider_name:
            machines_count = \
                int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT) + \
                int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
            # getting equinixmetalv2 cluster network configuration
            si_network_config = self.__manager.si_config.data['network_config']
            cluster_network_config = utils.equinixmetalv2_get_network_config(cluster_namespace=self.name,
                                                                             cluster_name=name,
                                                                             mcc_network_config=si_network_config,
                                                                             machines_amount=machines_count)
            body['spec']['providerSpec']['value'][
                'apiVersion'] = f"equinix.kaas.mirantis.com/{utils.Provider.equinixmetalv2.api_version}"
            body['spec']['providerSpec']['value'][
                'kind'] = utils.Provider.equinixmetalv2.cluster_spec
            body['spec']['providerSpec']['value'][
                'facility'] = cluster_network_config['metro']
            body['spec']['providerSpec']['value'][
                'network'] = cluster_network_config['network_config']['networkSpec']
            body['spec']['providerSpec']['value']['ceph'] = {
                'manualConfiguration': settings.KAAS_EQUINIX_CEPH_MANUAL_CONFIGURATION
            }
            if settings.KAAS_EQUINIX_PROJECT_SSH_KEYS:
                body['spec']['providerSpec']['value'][
                    'projectSSHKeys'] = settings.KAAS_EQUINIX_PROJECT_SSH_KEYS.split(",")

        elif provider == utils.Provider.openstack.provider_name:
            if not settings.STACKLIGHT_ENABLE_HA and settings.PARALLEL_UPGRADE_ENABLED:
                body['spec']['providerSpec']['value']['maxWorkerPrepareCount'] = max_worker_prepare_count
                body['spec']['providerSpec']['value']['maxWorkerUpgradeCount'] = max_worker_upgrade_count
            if boot_from_volume:
                body['spec']['providerSpec']['value']['bootFromVolume'] = boot_from_volume_template

            if settings.ENFORCE_ENABLED:
                body['spec']['providerSpec']['value']['helmReleases'].append(enable_enforce_mod)

        # add custom bastion parameters
        if bastion and provider in [utils.Provider.aws.provider_name, utils.Provider.openstack.provider_name]:
            body['spec']['providerSpec']['value']['bastion'] = bastion

        # Creates the file 'artifacts/cluster_name'
        utils.save_cluster_name_artifact(self.name, name)

        cluster = self.__manager.api.kaas_clusters.create(
            namespace=self.name, body=body)
        return Cluster(self.__manager, cluster)

    def create_cluster_raw(self, data):
        cluster = self.__manager.api.kaas_clusters.create(namespace=self.name,
                                                          name=data['metadata']['name'], body=data)
        return Cluster(self.__manager, cluster)

    def _get_machines_with_label(self, label, machines=None, strict=False):
        """

        :param label: check for label exist
        :param strict: check label value also
        :param machines: optional list with machines data as dict
        :return: list with machines
        """
        if machines:
            machines_data = machines
        else:
            machines_data = [machine.read().to_dict() for machine in
                             self.get_machines()]
        label_name = tuple(label.keys())[0]
        label_value = tuple(label.values())[0]
        catched = []
        for machine in machines_data:
            labels = machine.get('metadata', {}).get('labels', {})
            if strict and labels.get(label_name) == label_value:
                catched.append(machine)
            if not strict and labels.get(label_name):
                catched.append(machine)
        return catched

    def _get_storage_machines(self):
        """
        Get machines,that marked with label for ceph `storage`
        :param self:
        :return:
        """
        # labels hardcoded in kaas/core/frontend/src/consts.js
        # Aligned with Machine.machine_type
        hostlabels = utils.get_labels_by_type('storage')
        return self._get_machines_with_label(hostlabels)

    def create_ceph_cluster(self, name,
                            cluster_name,
                            nodes_data=None,
                            cephversion='',
                            monitors_count=3,
                            manager_count=3,
                            hdd_type="hdd",
                            pools=None,
                            rgw=None,
                            cephfs=None,
                            rook_config=None,
                            network=None,
                            tolerations=None,
                            dashboard=False,
                            miraceph_name="rook-ceph",
                            prevent_cluster_destroy=True):
        """
        :param name:
        :param cluster_name:
        :param nodes_data:
        :param cephversion:
        :param monitors_count:
        :param manager_count:
        :param hdd_type:
        :param pools: The ceph pools data
        :param rgw: The ceph rgw data
        :param rook_config: The ceph config options override
        :param network: The ceph network configuration
        :param tolerations: Add tolerations to run ceph pods on
        :param dashboard: Deploy Ceph Dashboard, but IT IS NOT SUPPORTED yet
        :param miraceph_name: MiraCeph object name
        :param prevent_cluster_destroy: If True then Ceph cluster won't be deleted because of MiraCeph object deletion
        :return:
        """
        # TODO(alexz) Must be refactored, as soon bmhp will
        #  support disk_labeling.
        # also skip second disk, as default for local-volume-provisioner
        system_disk_pattern = ['/dev/vda', '/dev/hda', '/dev/sda',
                               '/dev/vdb', '/dev/hdb', '/dev/sdb']
        LVM_SUBSTR = '-lvm-'
        assert 0 < monitors_count <= 3, (
            "Num of monitors could not be more then 3 "
            "and should be at least 1")
        assert 0 < manager_count <= 3, (
            "Num of managers could not be more then 3 "
            "and should be at least 1")
        actual_cluster = self.get_cluster(cluster_name)
        if not nodes_data:
            if actual_cluster.workaround.skip_kaascephcluster_usage():
                nodes_data = []
            else:
                nodes_data = {}
            roles = ['mgr', 'mon', 'rgw', 'mds']
            storage_machines_data = self._get_storage_machines()
            for storage_machine in storage_machines_data:
                storage_machine_status = storage_machine.get('status') or {}
                storage_devices = storage_machine_status.get(
                    'providerStatus', {}).get('hardware', {}).get('storage', {})
                storage_devices_names: list[str] = list()
                for device in storage_devices:
                    device_name: str = device.get('name', '')
                    device_by_id: str = device.get('byID', '').strip('/dev/disk/by-id/').casefold()
                    device_by_path: str = device.get('byPath', '').strip('/dev/disk/by-path/').casefold()

                    if device_name in system_disk_pattern or \
                            device_name.startswith('/dev/dm-') or \
                            LVM_SUBSTR in device_by_id or \
                            LVM_SUBSTR in device_by_path:
                        continue

                    storage_devices_names.append(device_name.split('/')[-1])

                if actual_cluster.workaround.skip_kaascephcluster_usage():
                    nodes_item = {"name": storage_machine.get('status', {}).get(
                        'instanceName', {}), 'roles': roles, 'devices': []}
                    for storage_device in storage_devices_names:
                        nodes_item.get('devices', []).append(
                            {'name': storage_device, 'config': {
                                'deviceClass': hdd_type}})
                    nodes_data.append(nodes_item)
                else:
                    nodes_data[storage_machine.get('metadata', {}).get(
                        'name', {})] = {'roles': roles, 'storageDevices': []}
                    for storage_device in storage_devices_names:
                        nodes_data.get(storage_machine.get('metadata', {}).get(
                            'name', {})).get('storageDevices', {}).append(
                            {'name': storage_device, 'config': {
                                'deviceClass': hdd_type}})

        if pools is None:
            pools = [
                {
                    "deviceClass": hdd_type,
                    "name": "kubernetes",
                    "role": "kubernetes",
                    # NOTE(vsaienko): allow pool to survive when
                    # shutting down 1 osd.
                    # related https://mirantis.jira.com/browse/PRODX-9815
                    "replicated": {
                        "size": 2
                    },
                    "default": True
                }
            ]

        if actual_cluster.workaround.skip_kaascephcluster_usage():
            body_miraceph = {
                'apiVersion': "lcm.mirantis.com/v1alpha1",
                'kind': "MiraCeph",
                'metadata': {
                    "name": miraceph_name,
                    "namespace": settings.CEPH_LCM_MIRANTIS_NS,
                },
                "spec": {
                        "nodes": nodes_data,
                        "pools": pools,
                        "rookNamespace": settings.ROOK_CEPH_NS,
                        "dashboard": dashboard,
                        "extraOpts": {"preventClusterDestroy": prevent_cluster_destroy, },
                },
            }

            if network:
                body_miraceph["spec"]["network"] = network

            if rgw:
                body_miraceph["spec"]["objectStorage"] = {"rgw": rgw}

            if cephfs:
                body_miraceph["spec"]["sharedFilesystem"] = {"cephFS": [cephfs]}

            if tolerations:
                body_miraceph["spec"]["hyperconverge"] = {
                    "tolerations": tolerations
                }

            if rook_config:
                body_miraceph["spec"]["rookConfig"] = rook_config

            LOG.info("Creating Ceph cluster through MiraCeph with next data: {}".format(
                body_miraceph))

        else:
            body_kcc = {
                'apiVersion': "kaas.mirantis.com/v1alpha1",
                'kind': "KaaSCephCluster",
                'metadata': {
                    "name": name,
                    "namespace": self.name,
                },
                "spec": {
                    "cephClusterSpec": {
                        "failureDomain": "host",
                        "nodes": nodes_data,
                        "pools": pools,
                        "version": cephversion,
                    },
                    "k8sCluster": {
                        "name": actual_cluster.name,
                        "namespace": self.name
                    }
                },
            }

            if network:
                body_kcc["spec"]["cephClusterSpec"]["network"] = network

            if rgw:
                body_kcc["spec"]["cephClusterSpec"]["objectStorage"] = {"rgw": rgw}

            if cephfs:
                body_kcc["spec"]["cephClusterSpec"]["sharedFilesystem"] = {"cephFS": [cephfs]}

            if tolerations:
                body_kcc["spec"]["cephClusterSpec"]["hyperconverge"] = {
                    "tolerations": tolerations
                }

            if rook_config:
                body_kcc["spec"]["cephClusterSpec"]["rookConfig"] = rook_config

            LOG.info("Creating Ceph cluster through kcc with next data: {}".format(
                body_kcc))

        if rgw and rgw.get('SSLCertInRef', False):
            cl_version = actual_cluster.clusterrelease_version
            if version.parse(cl_version) >= version.parse("mosk-17-4-0-rc-25-1"):
                LOG.info("SSLCertInRef is enabled. Creating own ssl secret for rgw")
                with open(settings.CEPH_RGW_CUSTOM_SSL_SECRET_PATH, 'r') as ssl_secret:
                    data = yaml.safe_load(ssl_secret)
                    actual_cluster.k8sclient.secrets.create(namespace=settings.ROOK_CEPH_NS, body=data)
            else:
                LOG.info(f"SSLCertInRef is not supported in Cluster {cl_version}. Skipping")
                rgw['SSLCertInRef'] = False
        if actual_cluster.workaround.skip_kaascephcluster_usage():
            actual_cluster.k8sclient.miracephs.create(namespace=settings.CEPH_LCM_MIRANTIS_NS, body=body_miraceph)
        else:
            return self.__manager.api.kaas_cephclusters.create(namespace=self.name, body=body_kcc)

    def create_ceph_cluster_raw(self, data):
        return self.__manager.api.kaas_cephclusters.create(namespace=self.name,
                                                           name=data['metadata']['name'], body=data)

    def get_machines(self):
        return self.__manager.get_machines(namespace=self.name)

    def get_machine(self, name):
        return self.__manager.get_machine(name=name, namespace=self.name)

    def get_lcmmachines(self):
        return self.__manager.get_lcmmachines(namespace=self.name)

    def get_lcmmachine(self, name):
        return self.__manager.get_lcmmachine(name=name, namespace=self.name)

    def get_ansibleextras(self):
        return self.__manager.get_ansibleextra(namespace=self.name)

    def get_ansibleextra(self, name):
        return self.__manager.get_ansibleextra(name=name, namespace=self.name)

    def get_baremetalhosts(self):
        return self.__manager.get_baremetalhosts(namespace=self.name)

    def get_baremetalhost(self, name):
        return self.__manager.get_baremetalhost(name=name, namespace=self.name)

    def get_baremetalhostinventories(self):
        return self.__manager.get_baremetalhostinventories(namespace=self.name)

    def get_baremetalhostinventory(self, name):
        return self.__manager.get_baremetalhostinventory(name=name, namespace=self.name)

    def get_baremetalhostcredential(self, name):
        return self.__manager.get_baremetalhostcredential(name=name, namespace=self.name)

    def get_baremetalhostcredentials(self):
        return self.__manager.get_baremetalhostcredentials(namespace=self.name)

    # metallb
    def get_metallbconfig(self, name):
        return self.__manager.get_metallbconfig(name=name, namespace=self.name)

    def get_metallbconfigs(self):
        return self.__manager.get_metallbconfigs(namespace=self.name)

    def get_metallbconfigtemplate(self, name):
        return self.__manager.get_metallbconfigtemplate(name=name, namespace=self.name)

    def get_metallbconfigtemplates(self):
        return self.__manager.get_metallbconfigtemplates(namespace=self.name)

    #

    def get_ipam_hosts(self):
        return self.__manager.get_ipam_hosts(namespace=self.name)

    def get_ipam_host(self, name):
        return self.__manager.get_ipam_host(name=name, namespace=self.name)

    def get_ipam_ipaddrs(self):
        return self.__manager.get_ipam_ipaddrs(namespace=self.name)

    def get_ipam_ipaddr(self, name):
        return self.__manager.get_ipam_ipaddr(name=name, namespace=self.name)

    def get_ipam_subnets(self):
        return self.__manager.get_ipam_subnets(namespace=self.name)

    def get_ipam_subnet(self, name):
        return self.__manager.get_ipam_subnet(name=name, namespace=self.name)

    def get_multirack_clusters(self):
        return self.__manager.get_multirack_clusters(namespace=self.name)

    def get_multirack_cluster(self, name):
        return self.__manager.get_multirack_cluster(name=name, namespace=self.name)

    def get_multirack_racks(self):
        return self.__manager.get_multirack_racks(namespace=self.name)

    def get_multirack_rack(self, name):
        return self.__manager.get_multirack_rack(name=name, namespace=self.name)

    def create_bm_statics(self, child_data):
        """
        Create BM static resources, in NS:
        bmh
        subnets
        l2templates
        - only create new resources, never updated already existence.
        """
        subnets_data = child_data.get('ipam_subnets', [])
        subnets_data_raw = child_data.get('subnets_raw', [])
        bmhp_data = child_data.get('bmh_profiles', {})
        bmhp_data_raw = child_data.get('bmh_profiles_raw', [])
        l2templates = child_data.get('l2templates', [])
        l2templates_raw = child_data.get('l2templates_raw', [])
        ipaddrs = child_data.get('ipaddrs', [])
        metallb_config = child_data.get('metallb_config', {})
        metallb_config_raw = child_data.get('metallb_config_raw', {})
        metallb_configtemplate = child_data.get('metallb_configtemplate', {})
        metallb_configtemplate_raw = child_data.get('metallb_configtemplate_raw', {})
        racks = child_data.get('racks', list())
        racks_raw = child_data.get('racks_raw', list())
        multirack_cluster = child_data.get('multirack_cluster', dict())
        multirack_cluster_raw = child_data.get('multirack_cluster_raw', dict())
        hostosconfigurations_raw = child_data.get('hostosconfiguration_raw', list())

        if subnets_data:
            existing_subnets = [p.name for p in self.get_ipam_subnets()]
            for subnet in subnets_data:
                if subnet['name'] in existing_subnets:
                    LOG.info(f"Subnet {subnet['name']} already exists")
                else:
                    LOG.info(f"Creating Subnet '{subnet['name']}'")
                    self.create_ipam_subnet(subnet['name'], subnet['cidr'],
                                            labels=subnet.get('labels'),
                                            include_ranges=subnet.get(
                                                'includeRanges'),
                                            exclude_ranges=subnet.get(
                                                'excludeRanges'),
                                            gateway=subnet.get('gateway'),
                                            nameservers=subnet.get(
                                                'nameservers'))
        if subnets_data_raw:
            existing_subnets = [p.name for p in self.get_ipam_subnets()]
            for subnet in subnets_data_raw:
                if subnet['metadata']['name'] in existing_subnets:
                    LOG.info(f"Subnet {subnet['metadata']['name']} already exists")
                else:
                    LOG.info(f"Creating Subnet '{subnet['metadata']['name']}'")
                    self.create_ipam_subnet_raw(subnet)

        if bmhp_data:
            existing_bmhp = [p.name for p in self.get_baremetalhostprofiles()]
            for n, bmhp in bmhp_data.items():
                if n in existing_bmhp:
                    LOG.info(f"BareMetalHostProfile {n} already exists")
                else:
                    LOG.info(f"Creating bmhp '{n}'")
                    self.create_baremetalhostprofile(name=n, spec=bmhp['spec'],
                                                     labels=bmhp['labels'])
        if bmhp_data_raw:
            existing_bmhp = [p.name for p in self.get_baremetalhostprofiles()]
            for bmhp in bmhp_data_raw:
                if bmhp['metadata']['name'] in existing_bmhp:
                    LOG.info(f"BareMetalHostProfile {bmhp['metadata']['name']} already exists")
                else:
                    LOG.info(f"Creating bmhp '{bmhp['metadata']['name']}'")
                    self.create_baremetalhostprofile_raw(bmhp)

        if l2templates:
            existing_l2templates = [l2t.name for l2t in self.get_l2templates()]
            for l2template in l2templates:
                if l2template['name'] not in existing_l2templates:
                    LOG.debug(f"L2Template to create:\n{l2template}")
                    LOG.info(f"Creating L2Template: {l2template['name']}")
                    self.create_l2template(
                        l2template['name'],
                        l2template.get('npTemplate', None),
                        labels=l2template.get('labels'),
                        cluster_ref=l2template.get('clusterRef', None),
                        if_mapping=l2template.get('ifMapping'),
                        l3_layout=l2template.get('l3Layout', None),
                        auto_if_mapping_prio=l2template.get(
                            'autoIfMappingPrio'),
                        spec=l2template.get('spec', None),
                    )
                    LOG.info(f"L2Template '{l2template['name']}' "
                             f"has been created")
                else:
                    LOG.warning(
                        f'l2template {l2template["name"]} already exist,'
                        f'skipping')

        if l2templates_raw:
            existing_l2templates = [l2t.name for l2t in self.get_l2templates()]
            for l2template in l2templates_raw:
                if l2template['metadata']['name'] not in existing_l2templates:
                    self.create_l2template_raw(l2template)
                else:
                    LOG.warning(
                        f"l2template {l2template['metadata']['name']} already exist, skipping")

        if ipaddrs:
            existing_ipaddr = [ipaddr.name for ipaddr in
                               self.get_ipam_ipaddrs()]
            for ipaddr in ipaddrs:
                if ipaddr['name'] not in existing_ipaddr:
                    LOG.debug(f"ipaddrs to create:\n{ipaddr}")
                    LOG.info(f"Creating ipaddr: {ipaddr['name']}")
                    self.create_ipam_ipaddr(ipaddr['name'],
                                            spec=ipaddr['spec'],
                                            labels=ipaddr['labels'])
                    LOG.info(f"ipaddr '{ipaddr['name']}' has been created")
                else:
                    LOG.warning(
                        f'ipaddr {ipaddr["name"]} already exist, skipping')

        if metallb_config:
            config_name = metallb_config.get('name', 'child-metallb-config')
            existing_config_names = [config.name for config in
                                     self.get_metallbconfigs()]
            if config_name in existing_config_names:
                LOG.info(f"MetalLBConfig '{config_name}' already exists")
            else:
                self.create_metallbconfig(name=config_name,
                                          labels=metallb_config.get('labels', {}),
                                          spec=metallb_config.get('spec', {}))
                LOG.info(f"MetalLBConfig '{config_name}' has been created")

        if metallb_config_raw:
            existing_config_names = [config.name for config in
                                     self.get_metallbconfigs()]
            config_name = metallb_config_raw['metadata']['name']
            if config_name in existing_config_names:
                LOG.info(f"MetalLBConfig '{config_name}' already exists")
            else:
                self.create_metallbconfig_raw(metallb_config_raw)
                LOG.info(f"MetalLBConfig '{config_name}' has been created")

        if metallb_configtemplate:
            config_name = metallb_configtemplate.get('name', 'child-metallb-config')
            existing_config_names = [config.name for config in
                                     self.get_metallbconfigtemplates()]
            if config_name in existing_config_names:
                LOG.info(f"MetalLBConfigTemplate '{config_name}' already exists")
            else:
                self.create_metallbconfigtemplate(name=config_name,
                                                  labels=metallb_configtemplate.get('labels', {}),
                                                  spec=metallb_configtemplate.get('spec', {}))
                LOG.info(f"MetalLBConfigTemplate '{config_name}' has been created")

        if metallb_configtemplate_raw:
            existing_config_names = [config.name for config in
                                     self.get_metallbconfigtemplates()]
            config_name = metallb_configtemplate_raw['metadata']['name']
            if config_name in existing_config_names:
                LOG.info(f"MetalLBConfigTemplate '{config_name}' already exists")
            else:
                self.create_metallbconfigtemplate_raw(metallb_configtemplate_raw)
                LOG.info(f"MetalLBConfigTemplate '{config_name}' has been created")

        if multirack_cluster or multirack_cluster_raw:
            existing_multirack_cluster_names = [x.name for x in self.get_multirack_clusters()]
            if multirack_cluster_raw:
                multirack_cluster_name = multirack_cluster_raw['metadata']['name']
                if multirack_cluster_name in existing_multirack_cluster_names:
                    LOG.info(f"MultiRackCluster '{multirack_cluster_name}' already exists.")
                else:
                    self.create_multirack_cluster_raw(multirack_cluster_raw)
                    LOG.info(f"MultiRackCluster '{multirack_cluster_name}' created.")
            else:
                multirack_cluster_name = multirack_cluster.get('name')
                if multirack_cluster_name in existing_multirack_cluster_names:
                    LOG.info(f"MultiRackCluster '{multirack_cluster_name}' already exists.")
                else:
                    self.create_multirack_cluster(
                        name=multirack_cluster_name,
                        labels=multirack_cluster.get('labels', {}),
                        spec=multirack_cluster.get('spec', {}))

        for hoc in hostosconfigurations_raw:
            existing_hoc_names = [x.name for x in self.get_hostosconfigurations()]
            hoc_name = hoc['metadata']['name']
            if hoc_name in existing_hoc_names:
                LOG.info(f"hostosconfiguration obj '{hoc_name}' already exists.")
            else:
                self.create_hostosconfiguration_raw(hoc)
                LOG.info(f"hostosconfiguration obj '{hoc_name}' has been created")

        if racks_raw or racks:
            existing_rack_names = [x.name for x in self.get_multirack_racks()]
            if racks_raw:
                for raw_rack in racks_raw:
                    rack_name = raw_rack['metadata']['name']
                    if rack_name in existing_rack_names:
                        LOG.info(f"Rack obj '{rack_name}' already exists.")
                    else:
                        self.create_rack_raw(raw_rack)
                        LOG.info(f"Rack obj '{rack_name}' has been created")
            else:
                for rack in racks:
                    rack_name = rack.get('name')
                    if rack_name in existing_rack_names:
                        LOG.info(f"Rack obj '{rack_name}' already exists.")
                    else:
                        self.create_rack(name=rack_name,
                                         labels=rack.get('labels', {}),
                                         spec=rack.get('spec', {}))

    def create_rack(self, name, labels, spec):
        """
        Create [ipam.mirantis.com/v1alpha1/]Rack object

        :param name: Rack object name
        :param labels: object labels map
        :param spec: Rack object spec
        :return: Rack object
        """
        body = {
            "apiVersion": "ipam.mirantis.com/v1alpha1",
            "kind": "Rack",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": labels,
            },
            "spec": spec,
        }
        return self.__manager.api.kaas_racks.create(
            namespace=self.name,
            body=body
        )

    def create_rack_raw(self, data):
        return self.__manager.api.kaas_racks.create(namespace=self.name,
                                                    name=data['metadata']['name'], body=data)

    def create_multirack_cluster(self, name, labels, spec):
        """
        Create [ipam.mirantis.com/v1alpha1/]MultiRackCluster object

        :param name: MultiRackCluster object name
        :param labels: object labels map
        :param spec: MultiRackCluster object spec
        :return: MultiRackCluster object
        """
        body = {
            "apiVersion": "ipam.mirantis.com/v1alpha1",
            "kind": "MultiRackCluster",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": labels,
            },
            "spec": spec,
        }
        return self.__manager.api.kaas_multirack_cluster.create(
            namespace=self.name,
            body=body
        )

    def create_multirack_cluster_raw(self, data):
        return self.__manager.api.kaas_multirack_cluster.create(namespace=self.name,
                                                                name=data['metadata']['name'], body=data)

    def create_ipam_subnet_raw(self, data):
        return self.__manager.api.ipam_subnets.create(namespace=self.name,
                                                      name=data['metadata']['name'], body=data)

    def create_ipam_subnet(self, name, cidr,
                           labels=None,
                           include_ranges=None,
                           exclude_ranges=None,
                           gateway=None,
                           nameservers=None):
        """
        Create IPAM Subnet object

        :param name: Subnet's name
        :param cidr: network address with mask in form of 0.0.0.0/0
        :param labels: Dict with labels for Subnet object
        :param include_ranges: list of pairs (firstIP,lastIP) to choose IP from
        :param exclude_ranges: list of single IP and/or ranges of IPs
                               (pairs of (firstIP, lastIP)) to exclude from
                               distribution
        :param gateway: Gateway IP in the subnet
        :param nameservers: list of IPs represents nameservers in the subnet

        :return: SubnetIpamMirantis object
        """
        labels = labels or {}
        body = {
            "apiVersion": "ipam.mirantis.com/v1alpha1",
            "kind": "Subnet",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {
                    **labels
                }
            },
            "spec": {
                "cidr": cidr,
            },
        }
        if include_ranges:
            body['spec']['includeRanges'] = include_ranges
        if exclude_ranges:
            body['spec']['excludeRanges'] = exclude_ranges
        if gateway:
            body['spec']['gateway'] = gateway
        if nameservers:
            body['spec']['nameservers'] = nameservers
        return self.__manager.api.ipam_subnets.create(
            namespace=self.name,
            body=body
        )

    def create_ipam_ipaddr(self, name, spec, labels=None):
        """
        Create IPAM ipaddr object

        :param name: Ipaddr's name
        :param spec: spec
        :param labels: labels
        :return: IPaddrIpamMirantis object
        """
        labels = labels or {}
        body = {
            "apiVersion": "ipam.mirantis.com/v1alpha1",
            "kind": "IPaddr",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {
                    **labels
                }
            },
            "spec": {
                **spec
            },
        }
        return self.__manager.api.ipam_ipaddrs.create(
            namespace=self.name,
            body=body
        )

    def get_iamrolebindings(self):
        return self.__manager.get_iamrolebindings(namespace=self.name)

    def get_iamrolebinding(self, name):
        return self.__manager.get_iamrolebinding(name=name,
                                                 namespace=self.name)

    def create_iamrolebinding(self, name, role, user):
        """
        Create IAMRoleBinding object

        :param name: IAMRoleBinding's name
        :param role: Role name that should be assigned
        :param user: Username that should be associated with role

        :return: IAMRoleBindings object
        """
        body = {
            "apiVersion": "iam.mirantis.com/v1alpha1",
            "kind": "IAMRoleBinding",
            "metadata": {
                "name": name,
                "namespace": self.name,
            },
            "role": {
                "name": role,
            },
            "user": {
                "name": user,
            },
        }
        return self.__manager.api.iam_rolebindings.create(namespace=self.name,
                                                          body=body)

    def create_iamclusterrolebinding(self, name, role, user, cluster):
        """
        Create IAMRoleBinding object

        :param name: IAMRoleBinding's name
        :param role: Role name that should be assigned
        :param user: Username that should be associated with role
        :param cluster: Cluster name

        :return: IAMClusterRoleBindings object
        """
        body = {
            "apiVersion": "iam.mirantis.com/v1alpha1",
            "kind": "IAMClusterRoleBinding",
            "metadata": {
                "name": name,
                "namespace": self.name,
            },
            "role": {
                "name": role,
            },
            "user": {
                "name": user,
            },
            "cluster": {
                "name": cluster,
            },
        }
        return self.__manager.api.iam_clusterrolebindings.create(
            namespace=self.name,
            body=body)

    def get_l2templates(self):
        return self.__manager.get_l2templates(namespace=self.name)

    def get_l2template(self, name):
        return self.__manager.get_l2template(name=name, namespace=self.name)

    def get_baremetalhostprofile(self, name):
        return self.__manager.get_baremetalhostprofile(name=name,
                                                       namespace=self.name)

    def get_baremetalhostprofiles(self):
        return self.__manager.get_baremetalhostprofiles(namespace=self.name)

    def create_l2template(self, name, np_template,
                          labels=None,
                          cluster_ref=None,
                          if_mapping=None,
                          l3_layout=None,
                          auto_if_mapping_prio=None,
                          spec=None):
        """
        Create L2Template object

        :param name: L2Template's name
        :param np_template: Netplan template as a string
        :param labels: Dict with labels for L2Template object
        :param cluster_ref: Name of cluster to apply the template. 'default'
                           means all cluster in the namespace
        :param if_mapping: list of interfaces to use in npTemplate
        :param l3_layout: list of subnets and subnet-pools to use in npTemplate
        :param auto_if_mapping_prio: list of prefixes to form ifMapping
                                     automatically in this order
        :param spec: Full spec wo any checks
        :return: L2TemplateIpamMirantis object
        """
        labels = labels or {}
        if spec:
            body = {
                "apiVersion": "ipam.mirantis.com/v1alpha1",
                "kind": "L2Template",
                "metadata": {
                    "name": name,
                    "namespace": self.name,
                    "labels": {
                        **labels
                    }
                },
                "spec": {**spec},
            }
            return self.__manager.api.l2templates.create(
                namespace=self.name,
                body=body)
        body = {
            "apiVersion": "ipam.mirantis.com/v1alpha1",
            "kind": "L2Template",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {
                    **labels
                }
            },
            "spec": {
                "npTemplate": np_template,
            },
        }
        if cluster_ref:
            body['spec']['clusterRef'] = cluster_ref
        if if_mapping and isinstance(if_mapping, list):
            body['spec']['ifMapping'] = if_mapping
        elif auto_if_mapping_prio and isinstance(auto_if_mapping_prio, list):
            body['spec']['autoIfMappingPrio'] = auto_if_mapping_prio
        else:
            # Set default autoIfMappingPrio if both 'if_mapping' and
            # 'auto_if_mapping_prio' have not been provided
            body['spec']['autoIfMappingPrio'] = [
                'provision',  # is used in test envs so added to simplify life
                'eno',
                'ens',
                'enp',
            ]
        if l3_layout:
            body['spec']['l3Layout'] = l3_layout
        return self.__manager.api.l2templates.create(
            namespace=self.name,
            body=body
        )

    def create_l2template_raw(self, data):
        return self.__manager.api.l2templates.create(namespace=self.name,
                                                     name=data['metadata']['name'], body=data
                                                     )

    def create_baremetalhostprofile(self, name, spec, labels=None):
        body = {
            "apiVersion": "metal3.io/v1alpha1",
            "kind": "BareMetalHostProfile",
            "metadata": {
                "name": name,
                "namespace": self.name,
            },
            "spec": {**spec,
                     },
        }
        if labels:
            body['metadata']['labels'] = {**labels}
        mgmt_cluster = self.__manager.get_mgmt_cluster()
        # WR PRODX-47202 - check-wait each time before creation
        LOG.info('Wait for dnsmasq on mgmt before proceed with bmhp')
        mgmt_cluster.check.check_k8s_pods(target_namespaces='kaas', pods_prefix='dnsmasq')
        return self.__manager.api.baremetalhostprofiles.create(
            namespace=self.name,
            body=body
        )

    def create_baremetalhostprofile_raw(self, data):
        return self.__manager.api.baremetalhostprofiles.create(namespace=self.name,
                                                               name=data['metadata']['name'], body=data)

    def create_baremetalhost(self,
                             bmh_name,
                             bmh_secret,
                             bmh_mac,
                             bmh_ipmi,
                             hardwareProfile,
                             labels=None,
                             annotations=None,
                             bootUEFI=True,
                             bmhi_credentials_name=''):
        """
        Create node with kind: BareMetalHost
        :param bmh_name:
        :param bmh_secret:
        :param bmh_mac:
        :param bmh_ipmi:
        :return:
        """
        bootmode = 'UEFI' if bootUEFI else 'legacy'
        if "port" in bmh_ipmi:
            bmh_ipmi_address = "{0}:{1}".format(bmh_ipmi['ipmi_ip'],
                                                bmh_ipmi['port'])
        else:
            bmh_ipmi_address = bmh_ipmi['ipmi_ip']
        disablecertverif = bmh_ipmi.get('disableCertificateVerification', False)

        annotations = annotations or {}

        body = {
            "apiVersion": "metal3.io/v1alpha1",
            "kind": "BareMetalHost",
            "metadata": {
                "name": bmh_name,
                "labels": labels if labels else {},
                "annotations": annotations,
                "namespace": self.name,
            },
            "spec": {
                "online": True,
                "bootMode": bootmode,
                "bootMACAddress": bmh_mac,
                "bmc": {
                    "address": bmh_ipmi_address,
                    "credentialsName": bmh_secret,
                    "disableCertificateVerification": disablecertverif,
                }
            }
        }
        if hardwareProfile:
            body['spec']['hardwareProfile'] = hardwareProfile

        self._create_baremetalhost(body, bmhi_credentials_name=bmhi_credentials_name)

    def _create_baremetalhost(self, body, bmhi_credentials_name=''):
        """Creates BMH or BMHInventory object depending on USE_BMH_INVENTORY flag and MCC version"""

        mgmt_cluster = self.__manager.get_mgmt_cluster()
        actual_kaasrelease = mgmt_cluster.get_kaasrelease_version()
        # WR PRODX-47202 - check-wait each time before creation
        LOG.info('Wait for dnsmasq on mgmt before proceed with bmh')
        mgmt_cluster.check.check_k8s_pods(target_namespaces='kaas', pods_prefix='dnsmasq')
        # Create BMH using a gitops-compatible BaremetalHostInventory object
        if settings.USE_BMH_INVENTORY and version.parse(actual_kaasrelease) >= version.parse("kaas-2-29-0-rc"):
            LOG.info(f"Creating <BareMetalHostInventory> '{self.name}/{body['metadata']['name']}'")
            # Remove annotation 'kaas.mirantis.com/baremetalhost-credentials-name'
            annotations = body['metadata']['annotations'].copy()
            bmh_credentials_name_annotation = annotations.pop('kaas.mirantis.com/baremetalhost-credentials-name', '')

            # Create BareMetalHostInventory
            body['apiVersion'] = "kaas.mirantis.com/v1alpha1"
            body['kind'] = "BareMetalHostInventory"
            body['metadata']['annotations'] = annotations
            body['spec']['bmc']['credentialsName'] = ""
            if not body['spec']['bmc'].get('bmhCredentialsName'):
                # Check bmhi_credentials_name
                bmhi_credentials_name = bmhi_credentials_name or bmh_credentials_name_annotation
                assert bmhi_credentials_name, (
                    f"No bmhi_credentials_name specified to create BMH Inventory object "
                    f"{self.name}/{body['metadata']['name']}")
                body['spec']['bmc']['bmhCredentialsName'] = bmhi_credentials_name

            return self.__manager.api.kaas_baremetalhostinventories.create(
                # bmh objects must be set in default ns.
                namespace=self.name,
                body=body
            )
            # Wait for creating BMH object by provider
            timeout_msg = (f"BareMetalHost {self.name}/{body['metadata']['name']} was not created by controller"
                           f"for the just created BareMetalHostInventory")
            waiters.wait(lambda: bool(self.__manager.api.kaas_baremetalhosts.present(name=body['metadata']['name'],
                                                                                     namespace=self.name)),
                         timeout=600,
                         timeout_msg=timeout_msg)

        # Deprecating way, create BMH object directly
        else:
            LOG.info(f"Creating <BareMetalHost> '{self.name}/{body['metadata']['name']}'")
            return self.__manager.api.kaas_baremetalhosts.create(
                # bmh objects must be set in default ns.
                namespace=self.name,
                body=body
            )

    def create_baremetalhost_raw(self, data):
        return self.__manager.api.kaas_baremetalhosts.create(namespace=self.name,
                                                             name=data['metadata']['name'], body=data)

    def delete_baremetalhost(self, name):
        # Try to delete BareMetalHostInventory first
        if (self.__manager.api.kaas_baremetalhostinventories.available and
                self.__manager.api.kaas_baremetalhostinventories.present(name=name, namespace=self.name)):
            LOG.info(f"Delete BareMetalHostInventory {self.name}/{name} and wait until it is deleted")
            bmhi = self.get_baremetalhostinventory(name=name)
            bmhi.delete()
            timeout_msg = (f"BareMetalHostInventory {self.name}/{name} is still not deleted")
            waiters.wait(lambda: not bool(self.__manager.api.kaas_baremetalhostinventories.present(
                                          name=name, namespace=self.name)),
                         timeout=1200,
                         timeout_msg=timeout_msg)
        else:
            LOG.info(f"Delete BareMetalHost {self.name}/{name}")
            bmh = self.get_baremetalhost(name=name)
            bmh.delete()

    def create_baremetalhostinventories_raw(self, data):
        return self.__manager.api.kaas_baremetalhostinventories.create(namespace=self.name,
                                                                       name=data['metadata']['name'], body=data)

    def get_baremetalhost_status(self, name):
        return self.get_baremetalhost(name).data.get("status", {}).get("provisioning", {}).get("state")

    def check_baremetalhost(self, bmh_name, wait_bmh_cred=False, bm_hosts_data=None):
        bm_hosts_data = bm_hosts_data or []
        node_name = bmh_name.split('-')[0]
        bmh_credential = next((node.get('bmh_annotations', {}).get(
            'kaas.mirantis.com/baremetalhost-credentials-name', '')
            for node in bm_hosts_data if node['name'] == node_name), None)
        if not bmh_credential:
            LOG.info(f"BareMetalHostCredential annotation not found for BMH {bmh_name}")
        bmh = [x for x in self.get_baremetalhosts() if x.name == bmh_name]
        if not bmh:
            LOG.info("BMH %s hasn't been found in cluster, "
                     "looks like it was deleted", bmh_name)
            if wait_bmh_cred and bmh_credential:
                LOG.info(f"Check for BareMetalHostCredential {bmh_credential} deleted")
                if not self.__manager.api.kaas_baremetalhostscredentials.present(name=bmh_credential,
                                                                                 namespace=self.name):
                    LOG.info(f"Credential {bmh_credential} was deleted succesfully")
                else:
                    LOG.info(f"BareMetalHostCredential {bmh_credential} was not deleted successfully yet")
                    return False
            return True

        LOG.info("BMH %s has not been deleted yet and has status - %s",
                 bmh_name, self.get_baremetalhost_status(bmh_name))
        return False

    def wait_all_bmhs_deletion_by_names(self, bmh_names: list, timeout=60 * 60, wait_bmh_cred=False,
                                        bm_hosts_data=None):
        """
        Wait till all bmh by bmh_names will be deleted
        """
        statuses = dict()
        bm_hosts_data = bm_hosts_data or []

        def status_msg():
            """To show the remaining machines status after wait timeout"""
            bmhs = [bmh.read() for bmh in self.get_baremetalhosts()]
            bmhs_status = {m.metadata.name: m.status.get('provisioning', {}).get('state') for m in bmhs}
            return f"Baremetal hosts status: {bmhs_status}"

        def check_deleted():

            nonlocal statuses
            statuses = {}
            for bmh_name in bmh_names:
                statuses[bmh_name] = self.check_baremetalhost(bmh_name,
                                                              wait_bmh_cred=wait_bmh_cred,
                                                              bm_hosts_data=bm_hosts_data)
            if not all(statuses.values()):
                raise Exception("Not all BMHs have been deleted")

        timeout_msg = ("Machines have not been deleted during "
                       "timeout {} seconds\n{}").format(timeout,
                                                        (bmh_name for bmh_name in statuses
                                                         if not statuses[bmh_name]))
        waiters.wait_pass(lambda: check_deleted(), timeout=timeout,
                          interval=15,
                          timeout_msg=timeout_msg,
                          status_msg_function=status_msg)

    def wait_all_bmhs_deletion(self, bmhs_for_delete, timeout=900, wait_bmh_cred=False, bm_hosts_data=None):
        bm_hosts_data = bm_hosts_data or []
        bmh_names = [bmh.name for bmh in bmhs_for_delete]
        self.wait_all_bmhs_deletion_by_names(bmh_names, timeout, wait_bmh_cred, bm_hosts_data)

    def wait_baremetalhost_deletion(self, bmh_name, timeout=600, wait_bmh_cred=False, bm_hosts_data=None):
        bm_hosts_data = bm_hosts_data or []

        def status_msg():
            """To show the remaining machines status after wait timeout"""
            bmhs = [bmh.read()
                    for bmh in self.get_baremetalhosts()]
            bmhs_status = {
                m.metadata.name: m.status.get('provisioning', {}).get('state') for m in bmhs}
            return "Baremetal hosts status: {0}".format(bmhs_status)

        def check_deleted():

            if not self.check_baremetalhost(bmh_name, wait_bmh_cred=wait_bmh_cred,
                                            bm_hosts_data=bm_hosts_data):
                raise Exception(f"BMH {bmh_name} has not been deleted yet")

        timeout_msg = ("Machine {} has not been deleted during "
                       "timeout {} seconds").format(bmh_name, timeout)
        waiters.wait_pass(lambda: check_deleted(), timeout=timeout,
                          interval=15,
                          timeout_msg=timeout_msg,
                          status_msg_function=status_msg)

    def wait_openstack_resources_available(self, credential_name,
                                           retries=20,
                                           interval=15):

        def wait_resource():
            LOG.info("Wait until openstack resources are available")
            try:
                resource = [x for x in self.get_openstackresources()
                            if x.name == credential_name]
                if resource:
                    resource[0].read()
                else:
                    raise ApiException
                LOG.debug("Available openstackresource {0}"
                          .format(credential_name))
                return True
            except Exception as e:
                LOG.debug("Waiting for openstackresource {0}: {1}"
                          .format(credential_name, e))
            return False

        waiters.wait(lambda: wait_resource(),
                     timeout=retries * interval,
                     interval=interval,
                     timeout_msg="Openstack resource {0} is still "
                                 "not available after {1} retries".format(
            credential_name, retries
        ))

    def wait_baremetalhosts_statuses(self, retries=10, interval=60,
                                     wait_status='provisioned',
                                     wait_nodes_msg=None,
                                     nodes=None,
                                     check_error=True, autotime=False, self_check=False):
        """nodes: list of node names
         ['cz7987-child-controller-1', 'cz7842-child-worker-1']
         check_error: raise, if any of bmh in error state.
         wait_status: str with status, or set of strings.
         autotime: guess retries count dynamically
        """
        # convert, to make it bw-compatible
        if isinstance(wait_status, str):
            wait_status = {wait_status}
        _self_check_msg = "Not found any alive bmh to watch! Something wrong!"
        bmhs = self.get_baremetalhosts()
        if nodes:
            _self_check_msg += f"Was looking for \n{nodes}"
            bmhs = [bmh for bmh in bmhs if bmh.name in nodes]
        else:
            nodes = [bmh.name for bmh in bmhs]
        if self_check:
            assert len(bmhs) >= 1, _self_check_msg
        if autotime:
            interval = 60
            nodescount = len(nodes)
            # we assume, 30min+20 for transition period, based on ironic-inspect timeout 20min default
            # but with case for cleanup|provision in case big disk amount = 30min.
            # Otherwise, we have envs were we run 2+ child deployments in parallel - and test cant guess timeouts
            # properly for such cases. so, moving those parameter to options.
            transition = settings.KAAS_BM_WAIT_STATUS_TRANSITION
            # NIT: yup, we mix here transition as "count" in seconds, just not to math in formula
            retries = (math.ceil(nodescount / settings.KAAS_BM_PROVISIONING_LIMIT) * transition)
            calctime = datetime.timedelta(seconds=retries * interval)
            LOG.debug(f"wait_baremetalhosts_statuses: calculated timeout:{calctime}, based on:\n"
                      f"nodescount:{nodescount} KAAS_BM_PROVISIONING_LIMIT:{settings.KAAS_BM_PROVISIONING_LIMIT} "
                      f"KAAS_BM_WAIT_STATUS_TRANSITION: {transition} retries:{retries} interval:{interval}")

        if 'ready' in wait_status and len(wait_status) == 1:
            wait_status = {'preparing', 'ready', 'available'}
            LOG.warning(f"Switching  to wait multiply statuses "
                        f"as allowed: {wait_status}")

        def get_statuses(bmhs, wait_nodes_msg):
            error_count_threshold = 3
            bm_statuses = {}
            if not wait_nodes_msg:
                wait_nodes_msg = "BMH {name} has <{status}> provisioning status"
            for bm in bmhs:
                bm_data = bm.read()
                name = bm_data.metadata.name
                if not bm_data.status:
                    LOG.info(f"BMH {name} status isn't ready yet")
                    bm_statuses[name] = f"BMH {name} status isn't ready yet"
                    continue
                pstatus = bm_data.status.get('provisioning').get('state', 'Nil')
                LOG.info(wait_nodes_msg.format(name=name, status=pstatus))
                bm_statuses[name] = pstatus
                if check_error:
                    error_count = int(bm_data.status.get('errorCount', 0))
                    if error_count >= error_count_threshold:
                        error_msg = bm_data.status.get('errorMessage')
                        raise Exception(
                            f"Aborting due to excessive error count "
                            f"({error_count} >= {error_count_threshold}) "
                            f"for BMH {name}, errorMessage: {error_msg}")
            return bm_statuses

        total_time = retries * interval
        LOG.info(f"Wait {total_time} seconds until BMH will be in the "
                 f"status(es) <{wait_status}>")
        try:
            waiters.wait(
                lambda: set(
                    get_statuses(bmhs, wait_nodes_msg).values()).issubset(wait_status),
                timeout=total_time, interval=interval)
        except TimeoutError as e:
            self.print_events()
            LOG.error(f"Not all baremetalhosts are in {wait_status} "
                      f"status(es)")
            raise e

    def create_proxyobject(self, name=settings.KAAS_PROXYOBJECT_NAME,
                           region='',
                           proxy_str=settings.KAAS_EXTERNAL_PROXY_ACCESS_STR,
                           no_proxy=settings.KAAS_PROXYOBJECT_NO_PROXY,
                           ca_cert=None):
        """Create a proxy inside child cluster namespaces

        :param no_proxy: No proxy variable
        :param name: Name of proxyobject
        :param region: Region of proxyobject
        :param proxy_str: Generated access string
        :param ca_cert: Custom proxy certificate
        :return:
        """

        LOG.info('Creating proxy object with name - {0}'.format(name))
        options = {
            'KAAS_PROXYOBJECT_NAME': name,
            'KAAS_PROXYOBJECT_REGION': region,
            'KAAS_PROXYOBECT_NAMESPACE': self.name,
            'KAAS_PROXYOBJECT_ACCESS_STR': proxy_str,
            'KAAS_PROXYOBJECT_NO_PROXY': no_proxy,
            'KAAS_PROXYOBJECT_CA_CERT': ca_cert

        }

        templates_prx = template_utils.render_template(
            settings.KAAS_PROXYOBJECT_YAML, options)
        LOG.debug(templates_prx)
        proxy_obj = self.__manager.api.kaas_proxies.create(
            name=name,
            namespace=self.name,
            body=yaml.load(templates_prx, Loader=yaml.SafeLoader))
        return proxy_obj

    def create_registryobject(self, name=settings.KAAS_REGISTRYOBJECT_NAME,
                              container_registry_doman=settings.KAAS_REGISTRYOBJECT_DOMAIN,
                              container_registry_cert=settings.KAAS_REGISTRYOBJECT_CERT):

        LOG.info('Creating container registry object with name - {0}'.format(name))

        with open(settings.KAAS_REGISTRYOBJECT_CERT, 'r') as f:
            container_registry_cert_base64 = base64.b64encode(
                f.read().encode("utf-8")).decode()
        options = {
            'KAAS_REGISTRYOBJECT_NAME': name,
            'KAAS_REGISTRYOBECT_NAMESPACE': self.name,
            'KAAS_REGISTRYOBJECT_DOMAIN':  container_registry_doman,
            'KAAS_REGISTRYOBJECT_CERT': container_registry_cert_base64

        }

        templates_prx = template_utils.render_template(
            settings.KAAS_REGISTRYOBJECT_YAML, options)
        LOG.debug(templates_prx)
        registryobject = self.__manager.api.containerregistry.create(
            name=name,
            namespace=self.name,
            body=yaml.load(templates_prx, Loader=yaml.SafeLoader))
        return registryobject

    def create_gracefulrebootrequest_object(self, name, ns, machine_list=None):

        machines = '[]'
        if machine_list is not None:
            machines = json.dumps(machine_list)
        LOG.info('Creating graceful reboot request object with name - {0}'.format(name))
        options = {
            'KAAS_GRACEFULREBOOTREQUEST_NAME': name,
            'KAAS_GRACEFULREBOOTREQUEST_NAMESPACE': ns,
            'KAAS_GRACEFULREBOOTREQUEST_MACHINES': machines
        }

        templates_prx = template_utils.render_template(
            settings.KAAS_GRACEFULREBOOTREQUEST_YAML, options)
        LOG.debug(templates_prx)
        request_object = self.__manager.api.gracefulrebootrequest.create(
            name=name,
            namespace=ns,
            body=yaml.load(templates_prx, Loader=yaml.SafeLoader))
        return request_object

    def node_bmc_data(self, node):
        """
        return BMC data(host,user,password) for specific node in namespace
        """
        bmh_name = node.data['metadata'][
            'annotations']['metal3.io/BareMetalHost'].split("/")[1]
        bmh = self.get_baremetalhost(name=bmh_name)
        bmc_host = bmh.data['spec']['bmc']['address'].split(":")[0]
        bmc_cred_name = bmh.data['spec']['bmc']['credentialsName']
        bmc_cred_data = self.get_secret(bmc_cred_name)
        bmc_username = base64.b64decode(bmc_cred_data.read().to_dict()
                                        ['data']['username']).decode("utf-8")
        bmc_password = base64.b64decode(bmc_cred_data.read().to_dict()
                                        ['data']['password']).decode("utf-8")
        return bmc_host, bmc_username, bmc_password

    def create_metallbconfig_substitutive_inlineconfig(self, cluster_name: str, region: str,
                                                       ip_range: str, provider_name: str) -> None:
        """
        Create an object of kaas.mirantis.com/v1alpha1/metallbconfigs resource explicitly
        instead of implicit cluster spec's inlineConfig.

        Skips creation if an objects with the same name
        already exists within the namespace.

        :param cluster_name: The name of a cluster that should be referenced by the object,
        the object's name will be the same
        :type cluster_name: str

        :param region: The name of a region that should be referenced by the object
        :type region: str

        :param ip_range: MetalLB services IP Address range in format '10.0.0.1-10.0.0.10'
        :type ip_range: str

        :param provider_name: Name of a provider for which the object is being created,
        only 'vsphere','equinixmetalv2' and 'baremetal' will be considered as eligible,
        for 'baremetal' value an additional object of
        ipam.mirantis.com/v1alpha1metallbconfigtemplates resource
        will be created
        :type provider_name: str

        :return: Nothing
        :rtype: None
        """
        if provider_name not in [settings.BAREMETAL_PROVIDER_NAME,
                                 settings.EQUINIXMETALV2_PROVIDER_NAME,
                                 settings.VSPHERE_PROVIDER_NAME]:
            LOG.info(f"Not eligible provider '{provider_name}' for MetalLBConfig creation given")
            return

        object_name = cluster_name

        if not ip_range:
            LOG.info(f"Empty IP Address range given to create {object_name} MetalLBConfig from, skipping")
            return

        existing_configs = [mc.name for mc in self.get_metallbconfigs()]
        if object_name in existing_configs:
            LOG.info(f"MetalLBConfig '{object_name}' already exists")
            return

        labels = {
            "kaas.mirantis.com/provider": provider_name,
            "kaas.mirantis.com/region": region,
            "cluster.sigs.k8s.io/cluster-name": cluster_name,
        }

        TEMPLATE_NAME = f"{object_name}-template"
        if provider_name == settings.BAREMETAL_PROVIDER_NAME:
            tpl_spec = {
                "templates": {
                    "ipAddressPools": {
                        "layer2": yaml.dump([{
                            "name": "default",
                            "spec": {
                                "addresses": [ip_range]
                            }
                        }]),
                    },
                    "l2Advertisements": yaml.dump([{
                        "name": "services",
                        "spec": {
                            "ipAddressPools": """{{ipAddressPoolNames "layer2"}}"""
                        },
                    }]).replace("'{", "{").replace("}'", "}"),
                }
            }
            self.create_metallbconfigtemplate(TEMPLATE_NAME, labels, tpl_spec)

        if provider_name == settings.BAREMETAL_PROVIDER_NAME:
            spec = {
                "templateName": TEMPLATE_NAME
            }
        else:
            spec = {
                "l2Advertisements": [{
                    "name": "default",
                    "spec": {
                        "ipAddressPools": ["default"]
                    }
                }],
                "ipAddressPools": [{
                    "name": "default",
                    "spec": {
                        "addresses": [ip_range],
                        "autoAssign": True,
                        "avoidBuggyIPs": False,
                    },
                }]
            }
        self.create_metallbconfig(object_name, labels, spec)

    def hostosconfiguration_is_present(self, name=None):
        return self.__manager.api.kaas_hostosconfigurations.present(name=name, namespace=self.name)

    def get_hostosconfiguration(self, name=None):
        return self.__manager.api.kaas_hostosconfigurations.get(name=name, namespace=self.name)

    def get_hostosconfigurations(self):
        return self.__manager.api.kaas_hostosconfigurations.list(namespace=self.name)

    def create_hostosconfiguration_raw(self, data):
        return self.__manager.api.kaas_hostosconfigurations.create(namespace=data['metadata']['namespace'],
                                                                   name=data['metadata']['name'], body=data)

    def _check_config_state_items_statuses_in_hoc(self, hoc_name, expected_status):
        hostcfg_data = self.get_hostosconfiguration(name=hoc_name).data
        machines_dict = hostcfg_data.get('status', {}).get('machinesStates', {}) if hostcfg_data else {}
        if not machines_dict:
            LOG.info(f"No machines in {hoc_name}, continue waiting...")
            return False
        items_states = {}
        states_list = []
        for machine, data in machines_dict.items():
            package_list = data.get('configStateItemsStatuses', {})
            if not package_list:
                LOG.info("No packages in configStateItemsStatuses dictionary")
                return False
            for package, data in package_list.items():
                for phase, state_data in data.items():
                    state = state_data.get('state', '')
                    items_states.setdefault(machine, {})[phase] = state
                    states_list.append(state == expected_status)
        LOG.info(f"Current states:\n{yaml.dump(items_states)}")

        if expected_status == "Failed":
            if any(states_list):
                LOG.info(f"One HOC object is in {expected_status} state. This is enough to mark all HOC as Failed."
                         f"Finish checking procedure.")
                return True
        elif expected_status == "Success":
            if all(states_list):
                LOG.info(f"All HOC objects are in {expected_status} state, Finish checking procedure.")
                return True
        else:
            LOG.info(f"Unsupported state: {expected_status}. Check parameters")
            return False
        LOG.info(f"Not all HOC objects are in {expected_status} state, continue waiting...")
        return False

    def wait_hostosconfiguration_config_state_items_status(self, hoc_name, expected_status, timeout=1200, interval=30):
        """Wait for state Failed inside only one field configStateItemsStatuses for all machines in HOC object
           you can use this checker after changing HOC object
        Args:
            hostoscfg: KaaSHostOSConfiguration object
            expected_status: KaaSHostOSConfiguration expected status: Success or Failed
            timeout: timeout to wait
            interval: time between checks
        Returns: None
        """
        timeout_msg = (f'Not all procedures for hoc object {hoc_name} inside configStateItemsStatuses '
                       f'have status: Success')
        waiters.wait(lambda: self._check_config_state_items_statuses_in_hoc(hoc_name, expected_status),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    # Netchecker methods
    def infraconnectivitymonitor_is_present(self, name=None):
        return self.__manager.api.kaas_infraconnectivitymonitors.present(name=name, namespace=self.name)

    def get_infraconnectivitymonitor(self, name=None):
        return self.__manager.api.kaas_infraconnectivitymonitors.get(name=name, namespace=self.name)

    def get_infraconnectivitymonitors(self):
        return self.__manager.api.kaas_infraconnectivitymonitors.list(namespace=self.name)

    def create_infraconnectivitymonitor_raw(self, data):
        return self.__manager.api.kaas_infraconnectivitymonitors.create(namespace=data['metadata']['namespace'],
                                                                        name=data['metadata']['name'], body=data)

    def update_infraconnectivitymonitor_raw(self, name, data):
        return self.__manager.api.kaas_infraconnectivitymonitors.update(namespace=self.name,
                                                                        name=name, body=data)


class Cluster(object):
    def __init__(self, manager: Manager, kaas_cluster: V1KaaSCluster):
        """
        manager: <Manager> instance
        kaas_cluster: <KaaSCluster> instance
        """
        self.__manager = manager
        self.__cluster = kaas_cluster
        self.__metadata = None
        self.__bastion_ip = None
        self.__k8sclient = None
        self.__dockerclient = None
        self.__mkedashboardclient = None
        self.__prometheusclient = None
        self.__grafanaclient = None
        self.__provider_resources = None
        self.__expected_pods = None
        self.__expected_docker_objects = None
        self.__ssh_user = None
        self.__private_key = None
        self.__key_file_var = None
        self.__mcp_docker_registry = None
        self.__cluster_type = None
        self.__provider = None
        self.__region_name = None
        self.check = clustercheck_manager.ClusterCheckManager(self)
        self.mos_check = clustercheck_mos_manager.ClusterCheckMosManager(self)
        self.workaround = workaroundcheck_manager.WorkaroundCheckManager(self)
        self.day2operations = day2operations_manager.Day2OperationsManager(self)
        self.runtime = runtime_manager.RuntimeManager(self)
        self.ha = ha_manager.HAManager(self)

    @property
    def _manager(self):
        return self.__manager

    @property
    @retry(KeyError, delay=10, tries=30, logger=LOG)
    def _cluster_type(self):
        if not self.__cluster_type:
            if self.provider == utils.Provider.byo:
                self.__cluster_type = 'byo_child'
                return self.__cluster_type
            kaas_spec = self.spec['providerSpec']['value'].get('kaas')
            if not kaas_spec:
                self.__cluster_type = 'child'
                return self.__cluster_type
            mgmt = kaas_spec.get('management', {}).get('enabled', False)
            if mgmt:
                self.__cluster_type = 'management'
            elif 'regional' in kaas_spec:
                self.__cluster_type = 'regional'
            else:
                self.__cluster_type = 'child'
        return self.__cluster_type

    @property
    def k8sclient(self):
        """Cached cluster k8sclient"""
        if self.__k8sclient is None:
            if self.is_management:
                self.__k8sclient = self.__manager.api
            else:
                if settings.SI_OPENID_AUTH:
                    self.__k8sclient = self.__get_openid_k8sclient()
                else:
                    _, kubeconfig = self.get_kubeconfig_from_secret()
                    config_data = yaml.load(kubeconfig, Loader=yaml.SafeLoader)
                    self.__k8sclient = K8sCluster(config_data=config_data)
        return self.__k8sclient

    @property
    @retry(KeyError, delay=10, tries=30, logger=LOG)
    def mkedashboardclient(self):
        """Cached docker ssh client"""
        if self.__mkedashboardclient is None:
            self.__mkedashboardclient = self.get_mke_dashboardclient()
        return self.__mkedashboardclient

    @property
    def prometheusclient(self):
        if self.__prometheusclient is None:
            self.__prometheusclient = self.get_prometheus_client()
        return self.__prometheusclient

    @property
    def grafanaclient(self):
        if self.__grafanaclient is None:
            self.__grafanaclient = self.get_grafana_client()
        return self.__grafanaclient

    @property
    def ssh_user(self):
        """Cached ssh_user property"""
        if self.__ssh_user is None:
            if self.is_management:
                self.__ssh_user = settings.KAAS_MGMT_CLUSTER_SSH_LOGIN
            elif self.is_regional:
                self.__ssh_user = settings.KAAS_REGIONAL_CLUSTER_SSH_LOGIN
            else:
                self.__ssh_user = settings.KAAS_CHILD_CLUSTER_SSH_LOGIN
        return self.__ssh_user

    @ssh_user.setter
    def ssh_user(self, ssh_user):
        self.__ssh_user = ssh_user

    @property
    def private_key(self):
        """Cached ssh_user property"""
        if self.__private_key is None:
            if self.is_management:
                self.__private_key = \
                    settings.KAAS_MGMT_CLUSTER_PRIVATE_KEY_FILE
                LOG.info("Use KAAS_MGMT_CLUSTER_PRIVATE_KEY_FILE for Management cluster")
            elif self.is_regional:
                self.__private_key = \
                    settings.KAAS_REGIONAL_CLUSTER_PRIVATE_KEY_FILE
                LOG.info("Use KAAS_REGIONAL_CLUSTER_PRIVATE_KEY_FILE for Regional cluster")
            else:
                self.__private_key = \
                    settings.KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE
                LOG.info("Use KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE for Child cluster")
        return self.__private_key

    @private_key.setter
    def private_key(self, private_key):
        self.__private_key = private_key

    @property
    def key_file_var(self):
        """Cached key_file_var property"""
        if self.__key_file_var is None:
            if self.is_management:
                self.__key_file_var = 'KAAS_MGMT_CLUSTER_PRIVATE_KEY_FILE'
            elif self.is_regional:
                self.__key_file_var = 'KAAS_REGIONAL_CLUSTER_PRIVATE_KEY_FILE'
            else:
                self.__key_file_var = 'KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE'
        return self.__key_file_var

    @property
    def load_balancer_ip(self):
        """Cluster load balancer ip address"""
        address = self.data.get("status", {}).get("providerStatus", {}).get("loadBalancerIP", "")
        LOG.info(f"Get LoadBalancer from cluster.providerStatus.loadBalancerIP = {address}")
        return address

    @property
    def load_balancer_host(self):
        """Cluster load balancer fqdn address"""
        address = self.data.get("status", {}).get("providerStatus", {}).get("loadBalancerHost", "")
        LOG.info(f"Get LoadBalancer from cluster.providerStatus.loadBalancerHost = {address}")
        return address

    @property
    def ucp_dashboard(self):
        """Cluster UCP dashboard address"""
        return self.data['status']['providerStatus']['ucpDashboard']

    @property
    def alerta_url(self):
        """Cluster alerta address"""
        return self.data['status']['providerStatus']['helm']['releases']['stacklight']['alerta']['url']

    @property
    def grafana_url(self):
        """Cluster grafana address"""
        return self.data['status']['providerStatus']['helm']['releases']['stacklight']['grafana']['url']

    @property
    def prometheus_url(self):
        """Cluster prometheus address"""
        return self.data['status']['providerStatus']['helm']['releases']['stacklight']['prometheus']['url']

    @property
    def kibana_url(self):
        """Cluster kibana address"""
        return self.data['status']['providerStatus']['helm']['releases']['stacklight']['kibana']['url']

    @property
    def alertmanager_url(self):
        """Cluster alertmanager address"""
        return self.data['status']['providerStatus']['helm']['releases']['stacklight']['alertmanager']['url']

    @property
    def mcc_cache(self):
        """Cluster mcc-cache address"""
        return self.data['status']['providerStatus']['helm']['releases']['decc']['cache']['url']

    @property
    def max_worker_upgrade_count(self) -> int:
        return self.data['spec'].get('providerSpec', {}).get('value', {}).get('maxWorkerUpgradeCount', 1)

    @key_file_var.setter
    def key_file_var(self, key_file_var):
        self.__key_file_var = key_file_var

    def get_parent_cluster(self):
        """Returns parent Cluster object for current child"""
        if self.is_management:
            raise Exception("You are trying to get parent cluster "
                            "of management cluster.")
        else:
            if self.is_regional:
                LOG.debug("{} is regional cluster".format(self.name))
                return self
            provider_region_labels = ['kaas.mirantis.com/provider',
                                      'kaas.mirantis.com/region']
            # let's use cached metadata
            labels = {k: v for k, v in self.metadata['labels'].items()
                      if k in provider_region_labels}
            # return mgmt cluster as parent for byo
            if labels['kaas.mirantis.com/provider'] == 'byo':
                return self.__manager.get_mgmt_cluster()

            # Get mgmt/region clusters from 'default' namespace
            clusters = self.__manager.get_clusters(namespace=settings.CLUSTER_NAMESPACE)
            if settings.CLUSTER_NAMESPACE != settings.REGION_NAMESPACE:
                clusters += self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)

            cluster_list = [x for x in clusters
                            if x.name != self.name and (x.is_management or x.is_regional)]
            if len(cluster_list) == 1:
                # we have only 1 cluster and it is mgmt
                return cluster_list[0]

            cluster_provider = labels['kaas.mirantis.com/provider']
            parents = []
            for cl in cluster_list:
                cl_data = cl.data
                candidate_provider = cl_data[
                    'metadata']['labels']['kaas.mirantis.com/provider']
                candidate_region = cl_data.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '')

                # if region doesn't match - skip
                if candidate_region != labels.get('kaas.mirantis.com/region', ''):
                    continue
                # check if regional cluster is a parent itself
                elif candidate_provider == cluster_provider:
                    parents.append(cl)
                else:
                    # check if regional cluster contains
                    # desired provider as nested
                    nested_providers = cl_data[
                        'spec']['providerSpec']['value']['kaas']['regional']
                    for p in nested_providers:
                        if p['provider'] == cluster_provider:
                            parents.append(cl)
                            break

            if len(parents) != 1:
                parents = [x.metadata['name'] for x in parents]
                raise Exception(f"Found {len(parents)} parent clusters: "
                                f"{parents} but should have found 1")

            parent_name = parents[0].metadata['name']
            LOG.info("Cluster {0} belongs to parent cluster {1}".format(
                self.name, parent_name))
            return parents[0]

    @retry(Exception, delay=30, tries=10, logger=LOG)
    def __get_openid_k8sclient(self):
        LOG.info("Getting openid kubeconfig for {} cluster".format(self.name))
        providerStatus = self.data['status']['providerStatus']
        oidc = providerStatus['oidc']
        assert oidc['ready'], (f"OIDC is not ready for {self.name}")
        ca = providerStatus['apiServerCertificate']
        external_ip = providerStatus['loadBalancerHost']
        if settings.SI_OPENID_IDP_CA:
            idp_ca = settings.SI_OPENID_IDP_CA
        else:
            idp_ca = oidc['certificate']
        username = settings.SI_OPENID_USERNAME
        password = settings.SI_OPENID_PASSWORD
        # initiate openid kubeconfig
        if not password:
            password = self.__manager.si_config.get_keycloak_user_password(username)
        return K8sCluster(user=username,
                          ca=ca,
                          password=password,
                          idp_issuer_url=oidc['issuerUrl'],
                          host=external_ip,
                          idp_ca=idp_ca,
                          client_id=oidc['clientId'],
                          cluster_name=self.name)

    def kubeconfig_secret_exists(self, secret_name=''):
        """Check if kubeconfig secret exists

        :param secret_name: str, (optional) use the specified secret
                            name instead of generated from cluster_name
        """
        client = self.get_parent_client()
        secret_name = secret_name or "{}-kubeconfig".format(self.name)
        LOG.debug("Fetching kubeconfig of {} cluster".format(self.name))
        kubeconfig_secret = [
            x for x in client.secrets.list(namespace=self.namespace)
            if x.name == secret_name
        ]
        return True if kubeconfig_secret else False

    def get_parent_client(self):
        """Returns parent client.
           If parent is a regional cluster, openid client will be returned.
           Note: openid client may not have access to some objects.
        """
        client = self.__manager.api
        try:
            if not self.is_management:
                parent = self.get_parent_cluster()
                if not parent.is_management:
                    # get openid config for regional cluster only
                    client = parent.__get_openid_k8sclient()
        except Exception as e:
            LOG.error("An error occurred {}".format(e))
            LOG.error("Fallback to old behavior (use mgmt client for "
                      "get_kubeconfig_from_secret)")
        return client

    def get_kubeconfig(self, secret_name=''):
        """Get kubeconfig for the cluster from openid provider status or
           from secret

        :param secret_name: str, (optional) use the specified secret
                            name instead of generated from cluster_name
        """
        if settings.SI_OPENID_AUTH:
            return self.get_kubeconfig_from_openid()
        else:
            return self.get_kubeconfig_from_secret(secret_name)

    def get_kubeconfig_from_openid(self):
        """Get kubeconfig for the cluster from openid provider status
           Returns tuple with kubeconfig name and content in yaml format
        """
        k8scluster = self.__get_openid_k8sclient()
        config_yaml = yaml.dump(k8scluster.generate_config_data(offline=True))
        return "{}-kubeconfig".format(self.name), config_yaml

    def get_kubeconfig_from_secret(self, secret_name=''):
        """Get kubeconfig for the cluster from secret

        :param secret_name: str, (optional) use the specified secret
                            name instead of generated from cluster_name
            Returns tuple with kubeconfig name and content in yaml format
        """
        client = self.get_parent_client()
        secret_name = secret_name or "{}-kubeconfig".format(self.name)
        external_ip = self.data[
            'status']['providerStatus']['loadBalancerHost']
        LOG.debug("Fetching kubeconfig of {} cluster".format(self.name))
        assert client.secrets.present(secret_name, namespace=self.namespace), (
            f"Secret {self.namespace}/{secret_name} not found")
        kubeconfig_secret = client.secrets.get(
            secret_name, namespace=self.namespace)
        kubeconfig = kubeconfig_secret.read()
        assert kubeconfig.data, f"Secret {self.namespace}/{secret_name} is empty"
        kubeconfig = kubeconfig.data.get('admin.conf')
        assert kubeconfig, f"Secret {self.namespace}/{secret_name} doesn't contain 'admin.conf'"
        kubeconfig = base64.b64decode(kubeconfig)
        kubeconfig = yaml.load(kubeconfig, Loader=yaml.SafeLoader)
        endpoint = "https://{}".format(external_ip)
        if endpoint not in kubeconfig['clusters'][0]['cluster']['server']:
            api_url = 'https://{external_ip}:443'.format(
                external_ip=external_ip
            )
            LOG.info("Replace public endpoint to %s", api_url)
            kubeconfig['clusters'][0]['cluster']['server'] = api_url
        else:
            curr_address = kubeconfig['clusters'][0]['cluster']['server']
            LOG.info("Current server address in kubeconfig "
                     "will not be replaced: {}".format(curr_address))
        kubeconfig = yaml.dump(kubeconfig)

        name = secret_name
        content = kubeconfig
        return (name, content)

    def get_dockerclient(self, private_key=None):
        """
        Get docker ssh client working directly or through bastion node

        Args:
            private_key: private key for ssh connection

        Returns: DockerCliClient

        """

        return DockerCliClient(self, private_key=private_key)

    def get_mke_dashboardclient(self):
        cluster_status = self.data.get('status') or {}
        dashboard_url = cluster_status.get('providerStatus', {}).get('ucpDashboard')
        if not dashboard_url:
            raise Exception("Cannot find ucpDashboard url in "
                            "cluster status: {}".format(self.data))

        LOG.info(">>> Cluster {0}/{1} dashboard URL: {2}"
                 .format(self.namespace,
                         self.name,
                         dashboard_url))
        if self.is_management or settings.FEATURE_FLAGS.enabled("force-plain-dashboard-auth"):
            # TODO(ddmitriev): enable openid when allowed, PRODX-16245
            # client_id = 'kaas'
            secret_name = "ucp-admin-password-{0}".format(self.name)
            secret = self.__manager.api.secrets.get(
                secret_name, namespace=self.namespace)
            pwd_data = secret.read().to_dict()['data']['ucpAdminPassword']
            pwd = base64.b64decode(pwd_data).decode("utf-8")
            LOG.info(f">>> Use the UCP admin password '{pwd}' "
                     f"from the secret {secret_name}")

            d_client = mke_dashboard_client.MKEDashboardClientPlain(
                dashboard_url, settings.UCP_UI_USER, pwd)

        else:
            client_id = 'k8s'
            keycloak_ip = self.__manager.get_keycloak_ip()
            writer_password = self.__manager.si_config.get_keycloak_user_password('writer')
            d_client = mke_dashboard_client.MKEDashboardClientOpenid(
                dashboard_url, keycloak_ip,
                'writer', writer_password,
                client_id=client_id)

        return d_client

    def get_mcc_dashboardclient(self, user='writer', password=None):
        child_client = self.k8sclient

        # Get kubernetes ip and port
        service = child_client.services.get(name='kaas-kaas-ui',
                                            namespace='kaas')
        ip = service.get_external_ip()
        https_service_port = [s for s in service.get_ports() if s.name == 'https']
        # TODO(tleontovich) Delete http after 2.12 release
        http_service_port = [s for s in service.get_ports() if s.name == 'http']
        assert len(https_service_port) or len(http_service_port) > 0, (
            "No http/https ports found for the service 'kaas-kaas-ui'")
        if https_service_port:
            port = https_service_port[0].port
            # Get objects from kubernetes dashboard
            dashboard_url = "https://{0}:{1}".format(ip, port)
            LOG.info(f"MCC UI found at {dashboard_url}")
        # TODO(tleontovich) Delete http after 2.12 release
        if http_service_port:
            port = http_service_port[0].port
            # Get objects from kubernetes dashboard
            dashboard_url = "http://{0}:{1}".format(ip, port)
            LOG.info(f"MCC UI found at {dashboard_url}")

        if password is None:
            password = self.__manager.si_config.get_keycloak_user_password('writer')
        d_client = mcc_dashboard_client.MCCDashboardClient(dashboard_url, user, password)
        return d_client

    def get_prometheus_client(self, user='operator', password=None, proto="https"):
        keycloak_ip = self.__manager.get_keycloak_ip()
        if password is None:
            password = self.__manager.si_config.get_keycloak_user_password(user)
        svc = self.k8sclient.services.get(name='iam-proxy-prometheus', namespace='stacklight')
        svc_ip = svc.get_external_addr()
        svc_port = next((s.port for s in svc.get_ports() if s.name == proto), None)
        client = PrometheusClientOpenid(host=svc_ip, port=svc_port, proto=proto, keycloak_ip=keycloak_ip,
                                        username=user, password=password)
        return client

    def get_grafana_client(self, user='operator', password=None, proto="https"):
        keycloak_ip = self.__manager.get_keycloak_ip()
        if password is None:
            password = self.__manager.si_config.get_keycloak_user_password(user)
        svc = self.k8sclient.services.get(name='iam-proxy-grafana', namespace='stacklight')
        svc_ip = svc.get_external_addr()
        svc_port = next((s.port for s in svc.get_ports() if s.name == proto), None)
        client = GrafanaClient(
            host=svc_ip, port=svc_port, proto=proto, keycloak_ip=keycloak_ip,
            username=user, password=password
        )
        return client

    def get_load_balancer_address(self):
        """
        Get LoadBalancer FQDN or IP

        Returns:
          - return FQDN if set
          - return IP address if set
          - raise exception if none of them is set
        """
        if lb_host := self.load_balancer_host:
            return lb_host
        elif lb_ip := self.load_balancer_ip:
            return lb_ip
        else:
            raise Exception("loadBalancerHost or loadBalancerIP should be set on cluster.providerStatus")

    @property
    def provider_resources(self):
        """Cached cluster provider class"""
        try:
            if self.__provider_resources is None:
                if self.provider == utils.Provider.openstack:
                    self.__provider_resources = \
                        prm.OSProviderResources(self.__manager, self)
                elif self.provider == utils.Provider.aws:
                    self.__provider_resources = \
                        prm.AWSProviderResources(self.__manager, self)
                elif self.provider == utils.Provider.vsphere:
                    self.__provider_resources = \
                        prm.VsphereProviderResources(self.__manager, self)
                else:
                    self.__provider_resources = \
                        prm.ProviderResources(self.__manager, self)
            return self.__provider_resources
        except Exception as e:
            LOG.error(e)

    def is_patchrelease_upgrade(self, clusterrelease_version_before: str,
                                clusterrelease_version_after: str,
                                kaas_release_version=None):
        """
        Determine if the cluster upgrade path versions are patchreleases.
        Return True if the first 2 numbers in the numeric release version are the same.
          11.7.0 -> 14.0.0 == upgrade 11.7 to 14.0 = major
          11.7.0 -> 14.0.1 == upgrade 11.7 to 14.0 = major
          14.0.0 -> 14.0.1 == upgrade с 14.0 to 14.0 patch-release

        For BYO clusters always return False.

        Args:
            clusterrelease_version_before: current ClusterRelease version (eg. mke-11-7-0-3-5-7)
            clusterrelease_version_after: ClusterRelease update version (eg. mke-14-0-1-3-6-5)
            kaas_release_version: KaaS release verion to check for update path (e.g. kaas-2-28-4)

        Returns: boolean

        """

        skipMaintenance = self.get_skipMaintenance_flag(
            cr_before=clusterrelease_version_before,
            target_clusterrelease=clusterrelease_version_after,
            kaasrelease_version=kaas_release_version)

        if self.is_byo_child:
            LOG.info("The cluster is defined as BYO, is_patchrelease=False")
            return False

        numeric_version_before = self.__manager.get_supported_clusterrelease_version_by_name(
            clusterrelease_version_before, self.provider.provider_name)
        LOG.debug(f"Found version {numeric_version_before} for the ClusterRelease name "
                  f"{clusterrelease_version_before} for current cluster")
        numeric_version_before = numeric_version_before.replace("-rc", "")

        numeric_version_after = self.__manager.get_supported_clusterrelease_version_by_name(
            clusterrelease_version_after, self.provider.provider_name)
        LOG.debug(f"Found version {numeric_version_after} for the ClusterRelease name "
                  f"{clusterrelease_version_after} for upgrade")
        numeric_version_after = numeric_version_after.replace("-rc", "")

        comparison_version_before = ".".join(numeric_version_before.split(".")[:2])
        comparison_version_after = ".".join(numeric_version_after.split(".")[:2])
        LOG.debug(f"Versions for comparison: current_version={comparison_version_before}; "
                  f"version_for_upgrade={comparison_version_after}")

        if comparison_version_before == comparison_version_after and skipMaintenance:
            LOG.info(f"Upgrade path from '{clusterrelease_version_before}' ({numeric_version_before}) to "
                     f"'{clusterrelease_version_after}' ({numeric_version_after}), is_patchrelease=True")
            return True
        else:
            LOG.info(f"Upgrade path from '{clusterrelease_version_before}' ({numeric_version_before}) "
                     f"to '{clusterrelease_version_after}' ({numeric_version_after}), "
                     "is_patchrelease=False")
            return False

    @property
    def is_management(self):
        return True if self._cluster_type == 'management' else False

    @property
    def is_regional(self):
        """Returns True only if this is a standalone regional cluster"""
        return True if self._cluster_type == 'regional' else False

    @property
    def is_child(self):
        """Returns True only if this is a child cluster"""
        return True if self._cluster_type == 'child' else False

    @property
    def is_mosk(self):
        """Returns True only if this is a MOSK cluster"""
        return True if 'mos' in self.clusterrelease_version else False

    @property
    def is_byo_child(self):
        """Returns True only if this is a BYO child cluster"""
        return True if self._cluster_type == 'byo_child' else False

    def is_existed(self):
        ns = self.__manager.get_namespace(self.namespace)
        try:
            all_clusters = ns.get_clusters()
        except ApiException as e:
            if e.status == 404:
                return False
            raise e
        for c in all_clusters:
            if c.name == self.name:
                LOG.debug(f"Cluster object '{self.namespace}/{self.name}' found in Management cluster")
                return True
        LOG.debug(f"Cluster object '{self.namespace}/{self.name}' not found in Management cluster")
        return False

    def is_ready(self, exp_provider_status, expected_fails):
        """
        Check if all cluster components and machines are ready
        :param exp_provider_status: expected overall cluster status
        :type exp_provider_status: bool
        :param expected_fails: list of strings with
        expectedly failed deployment names
        :type expected_fails: List[str]
        :rtype bool: bool
        """
        data = self.data
        if data['status']:
            unexpected_fails = []
            provider_status = data['status'] \
                .get('providerStatus', {}) \
                .get('ready')
            not_ready_obj = data['status'] \
                .get('providerStatus', {}). \
                get('notReadyObjects', {})
            LOG.debug(f"Provider ready status: {provider_status}")
            LOG.debug(f"Not ready objects: {not_ready_obj}")
            LOG.debug(f"Deployments expected to fail: {expected_fails}")
            not_ready_depl = not_ready_obj.get('deployments', [])

            if not_ready_depl:
                unexpected_fails = [d for d in not_ready_depl
                                    if d.get('name', '') not in expected_fails]

            if (provider_status and not unexpected_fails) or \
                    (provider_status == exp_provider_status
                     and not unexpected_fails):
                LOG.info(f"Provider ready status is {provider_status} "
                         f"as expected and no unexpected fails. "
                         f"Expected fails are: {expected_fails}")
                return True
            else:
                error_msg = "Current status: "
                if provider_status != exp_provider_status:
                    error_msg = error_msg + f"provider ready status " \
                                            f"{provider_status} expected " \
                                            f"to be {exp_provider_status} "
                if unexpected_fails:
                    error_msg = error_msg + f"unexpected fails " \
                                            f"{unexpected_fails}"
                LOG.error(error_msg)
                return False
        else:
            raise ValueError(f"No status in provider data, "
                             f"provider data: {data}")

    def get_custom_hostnames_enabled(self):
        """Check Regional (or Management) cluster configmap for customHostnamesEnabled flag"""
        if self.is_management:
            parent_cluster = self
        else:
            parent_cluster = self.get_parent_cluster()

        provider_name = self.provider.provider_name
        region_name = self.region_name
        parent_provider_cm_name = f"provider-config-{provider_name}"
        if region_name:
            parent_provider_cm_name = f"provider-config-{provider_name}-{region_name}"
        if not parent_cluster.k8sclient.configmaps.present(
                name=parent_provider_cm_name, namespace='kaas'):
            raise Exception(f"Configmap 'kaas/{parent_provider_cm_name}' not found in Cluster "
                            f"'{parent_cluster.namespace}/{parent_cluster.name}'")
        parent_provider_cm = parent_cluster.k8sclient.configmaps.get(
            name=parent_provider_cm_name, namespace='kaas')
        conf_data = parent_provider_cm.data["data"].get("conf")
        if not conf_data:
            raise Exception(f"Configmap 'kaas/{parent_provider_cm_name}' key 'conf' is empty or missing "
                            f"for Cluster '{parent_cluster.namespace}/{parent_cluster.name}'")
        conf = yaml.safe_load(conf_data)
        custom_hostnames_enabled = conf.get("customHostnamesEnabled")
        LOG.debug(f"customHostnamesEnabled = '{custom_hostnames_enabled}'")
        return custom_hostnames_enabled

    def set_custom_hostnames_enabled(self, flag=False, provider_name=None):
        """Set customHostnamesEnabled flag for Cluster regional provider and wait for applying"""
        if self.is_management:
            parent_cluster = self
        else:
            parent_cluster = self.get_parent_cluster()

        old_flag = self.get_custom_hostnames_enabled()
        if old_flag == flag:
            LOG.info(f"'customHostnamesEnabled' is already set to {flag} "
                     f"for Cluster '{parent_cluster.namespace}/{parent_cluster.name}' "
                     f"region '{parent_cluster.region_name}'")
            return

        LOG.info(f"Setting 'customHostnamesEnabled' to '{flag}' for Cluster "
                 f"'{parent_cluster.namespace}/{parent_cluster.name}' "
                 f"region '{parent_cluster.region_name}'")

        # 1. Update Cluster spec on Management cluster
        # 2. Wait for configmap changes on Region cluster
        spec = parent_cluster.data['spec']
        regional_spec = spec['providerSpec']['value']['kaas']['regional']

        for provider_spec in regional_spec:
            if provider_name and provider_spec['provider'] != provider_name:
                LOG.debug(f"Skipping provider '{provider_spec['provider']}' for setting customHostnamesEnabled")
                continue
            LOG.debug(f"Checking provider '{provider_spec['provider']}' to set customHostnamesEnabled")
            for helm_release in provider_spec["helmReleases"]:
                if helm_release["name"] == f"{provider_spec['provider']}-provider":
                    LOG.info(f"Setting customHostnamesEnabled={flag} for provider '{provider_spec['provider']}'")
                    config = helm_release["values"]["config"]
                    config["customHostnamesEnabled"] = flag
                    break
            else:
                raise Exception(f"Regional helm release with name '{provider_spec['provider']}' not found "
                                f"in the cluster '{parent_cluster.namespace}/{parent_cluster.name}'")
        body = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'kaas': {
                            'regional': regional_spec,
                        }
                    }
                }
            }
        }
        parent_cluster.patch(body=body)

        LOG.info(f"Wait until 'customHostnamesEnabled: {flag}' is applied in configmap")
        waiters.wait(lambda: self.get_custom_hostnames_enabled() == flag, timeout=200, interval=10)
        LOG.info(f"Wait until providers pods are restarted on Cluster "
                 f"'{parent_cluster.namespace}/{parent_cluster.name}'")
        parent_cluster.check.check_k8s_pods(target_namespaces=['kaas'])

    @collect_cluster_readiness
    def get_conditions_status(self, data, expected_fails=None, verbose=False, skip_reboot_conditions=True):
        expected_fails = expected_fails or {}
        result = {
            'ready': [],
            'not_ready': [],
            'skipped': [],
        }
        data_status = data.get('status') or {}
        conditions = data_status.get('providerStatus', {}).get('conditions', [])
        if not conditions:
            result['not_ready'].append("NO CONDITIONS IN PROVIDER STATUS")

        for condition in conditions:
            ready = condition['ready']
            condition_type = condition.get('type')
            condition_message = condition.get('message')
            if skip_reboot_conditions and utils.is_reboot_condition(condition_type):
                LOG.info(f"Skipping status.providerStatus.conditions.{condition_type} "
                         f"ready={ready}, message={condition_message}")

                result['skipped'].append(condition_type)
            elif (not ready and
                    condition_type in expected_fails.keys() and
                    expected_fails[condition_type] in condition_message):
                result['skipped'].append(condition_type)
            elif ready is True:
                result['ready'].append(condition_type)
            else:
                result['not_ready'].append(condition_type)
        if verbose:
            message = (f"{data['kind']} {self.namespace}/"
                       f"{data['metadata']['name']}: {result}")
            LOG.info(message)
        return result

    def get_cluster_condition_by_type(self, condition_type):
        """
        Searches for condition by type
        Returns condition
        """
        condition = {
            'type': condition_type,
            'ready': False,
            'message': 'NOT FOUND',
        }

        data_status = self.data.get('status') or {}
        conditions = data_status.get('providerStatus', {}).get('conditions', [])
        if not conditions:
            LOG.info(f"Cluster {self.name} does not have conditions in status")
            return condition

        for condition in conditions:
            if condition.get('type') != condition_type:
                continue

            condition['ready'] = condition.get('ready')
            condition['message'] = condition.get('message')
            LOG.info(f"Condition: {condition_type}, Ready: {condition['ready']}, Message: {condition['message']}")
            return condition

        LOG.info(f"Condition {condition_type} is not present in {self.name} cluster status")
        return condition

    def wait_for_cluster_condition(self, condition_type, timeout=300, interval=10):
        """Wait that Cluster condition to be ready"""
        waiters.wait(lambda: self.get_cluster_condition_by_type(condition_type).get('ready'),
                     timeout=timeout, interval=interval,
                     timeout_msg=(f"Cluster {self.name} condition {condition_type} is not ready yet"))

    @property
    def etcd_storage_quota(self):
        # Default 2GB quota is set by etcd. If no etcd storage quota configured in
        # cluster spec, then it will always be 2GB
        return self.data['spec'].get('providerSpec', {}).get('value', {}).get('etcd', {}).get('storageQuota', '2GB')

    def set_etcd_storage_quota(self, quota_size='2GB', wait_for_applied=False):
        # quota_size examples: 8GB(suggested maximum), 2GB, 500MB
        # https://etcd.io/docs/v3.3/dev-guide/limit/
        # Can't be empty if already set
        body = {
            'spec': {
              'providerSpec': {
                'value': {
                  'etcd': {
                    'storageQuota': quota_size}}}}}
        self.patch(body)
        if wait_for_applied:
            waiters.wait(self.check.check_etcd_quota_applied, interval=180, timeout=1200)

    def create_diagnostic_object(self, object_name: str) -> diagnostic_diagnostic.DiagnosticDiagnostic:
        """
        Create a diagnostic.mirantis.com/v1alpha1, Kind=Diagnostic object.
        Namespace is inherited from the Cluster instance.
        Cluster name argument is inherited from the Cluster instance.
        :param object_name: name of the object to create.
        :type object_name: str

        :return: a Diagnostic object
        :rtype: si_tests.clients.k8s.diagnostic_diagnostic.DiagnosticDiagnostic
        """

        body = {
            "apiVersion": "diagnostic.mirantis.com/v1alpha1",
            "kind": "Diagnostic",
            "metadata": {
                "name": object_name,
                "namespace": self.namespace
            },
            "spec": {
                "cluster": self.name,
            }
        }

        return self._manager.api.diagnostic_diagnostics.create(name=object_name, namespace=self.namespace, body=body)

    def create_cache_warmup_request(self, name, cluster_releases, openstack_releases,
                                    openstack_only, namespace='default', clients_per_endpoint=2):
        """
        Create a cachewarmuprequest
        :param name: management or regional cluster name
        :param cluster_releases: list of cluster releases for warmup, e.g. ['mosk-12-7-4-23-1-4']
        :param openstack_releases: list of openstack release for warmup, e.g. ['yoga']
        :param openstack_only: warmup only openstack images
        :param clients_per_endpoint: is a number of clients to use for fetching per each MCC Cache endpoint.
               default=2 - is default as per 2.28 version.
        :return:
        """
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "CacheWarmupRequest",
            "metadata": {
                "name": name,
                "namespace": namespace
            },
            "spec": {
                "clusterReleases": cluster_releases,
                "clientsPerEndpoint": clients_per_endpoint,
            }
        }

        if openstack_releases:
            body['spec']['openstackReleases'] = openstack_releases

        if openstack_only:
            body['spec']['openstackOnly'] = True

        return self.__manager.api.kaas_cachewarmuprequests.create(name, namespace, body)

    def get_warmup_job_name(self, namespace) -> str:
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        jobs = client.jobs.list_raw(namespace=namespace).to_dict()['items']
        for job in jobs:
            job_metadata = job['metadata']
            job_release = job_metadata.get('annotations', {}).get('meta.helm.sh/release-name', '')
            if job_release == 'mcc-cache-warmup':
                return job_metadata['name']

        return ''

    def wait_warmup_job_creation(self, namespace, timeout=300, interval=10):
        """Wait for warmup job to be created"""
        waiters.wait(lambda: self.get_warmup_job_name(namespace) != '',
                     timeout=timeout, interval=interval,
                     timeout_msg=('Warmup job is not found'))

    def get_warmup_result(self, warmup_request_creation_timestamp) -> bool:
        warmup_status = \
            self.data.get('status', {}).get('providerStatus', {}).get('regionalStatus', {}).get('cacheWarmup', {})
        warmup_started_at = warmup_status.get('startedAt', '')

        if warmup_started_at:
            warmup_started_at_date = dateutil.parser.parse(warmup_started_at)
            if warmup_request_creation_timestamp > warmup_started_at_date:
                # Request is created after job is started,
                # so regionalStatus contains information about old warmup run
                LOG.info('warmup status contains information about old warmup run')
                return False

            LOG.info(f"Warmup job is in progress. Started at {warmup_started_at}")
        else:
            LOG.info('No status for warmup job yet')

        return warmup_status.get('success', False)

    def wait_for_cluster_warmup_success(self, warmup_request_creation_timestamp, timeout=1200, interval=30):
        """Wait that Cluster condition to be ready"""
        waiters.wait(lambda: self.get_warmup_result(warmup_request_creation_timestamp),
                     timeout=timeout, interval=interval,
                     timeout_msg=(f"Warmup for cluster {self.name} is not success in {timeout} seconds"))

    def run_job_from_cronjob(self, namespace, cronjob_name, wait_for_completed=False):
        cronjob = [j for j in self.k8sclient.cronjobs.list_all() if j.name == cronjob_name]
        assert cronjob, f"Cronjob with name {cronjob_name} not found"
        job_spec = cronjob[0].read().spec.job_template.spec
        result = self.k8sclient.jobs.create(namespace=namespace,
                                            body={'metadata': {'generateName': cronjob_name + '-manual-'},
                                                  'spec': job_spec})
        job_name = result.name
        LOG.info(f"Running job {job_name} in namespace {namespace}")
        if wait_for_completed:
            LOG.info(f"Waiting for job {job_name} completed")
            result.wait_succeded()
            LOG.info(f"Job {job_name} completed successfully")

    def are_conditions_ready(self, expected_fails=None, verbose=False):
        """
        Check if all Cluster conditions in ready status
        :param expected_fails: None or dict like the following
            {<condition.type>: <expected message pattern>, ...}
            If expected message pattern is empty, then the whole
            condition is ignored. If it is not empty - then
            this pattern is matched to the message from the cluster
            to check if the message is expected or not.
        :rtype bool: bool
        """
        data = self.data
        result = self.get_conditions_status(data, expected_fails, verbose)
        if result['not_ready']:
            return False
        else:
            return True

    def is_k8s_available(self):
        """Check if the k8s API is available on the Cluster, even if not all the nodes are ready"""
        result = self.get_conditions_status(self.data)
        if "LoadBalancer" in result["ready"]:
            try:
                k8s_version = self.get_k8s_version()
                LOG.debug(f"Cluster '{self.namespace}/{self.name}' k8s API version: {k8s_version}")
                return True
            except Exception:
                pass
        return False

    @property
    def name(self):
        """Cluster name"""
        return self.__cluster.name

    @property
    def namespace(self):
        """Cluster namespace"""
        return self.__cluster.namespace

    @property
    def metadata(self):
        """Cached cluster metadata"""
        if self.__metadata is None:
            self.__metadata = self.data['metadata']
        return self.__metadata

    @property
    def spec(self):
        """Cluster spec"""
        return self.data['spec']

    @property
    def uid(self):
        """Cluster uid"""
        return self.metadata['uid']

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind,

        metadata, spec, status or items
        """
        return self.__cluster.read().to_dict()

    def patch(self, *args, **kwargs):
        self.__cluster.patch(*args, **kwargs)

    def replace(self, *args, **kwargs):
        self.__cluster.replace(*args, **kwargs)

    def delete(self):
        """Deletes the current cluster"""
        # TODO(ddmitriev) delete cluster nodes first
        self.__cluster.delete()
        # TODO(ddmitriev) wait for deletion complete

    @property
    def cluster_status(self):
        """Determine the cluster status

        :rtype string: status of the cluster,
                       one of ('Pending', 'Updating', 'Ready')
        """
        status = self.data['status']
        if not status:
            return 'Pending'
        nodes = status.get('providerStatus', {}).get('nodes', {})
        if not nodes:
            return 'Pending'
        if nodes['ready'] != nodes['requested']:
            return 'Updating'
        return 'Ready'

    @property
    def clusterrelease_version(self):
        """Determine the cluster clusterrelease version

        :rtype string: clusterrelease of the cluster
        """
        status = self.data.get('status')
        if not status or type(status) is not dict:
            return None
        current_clusterrelease = status.get('providerStatus', {}).get(
            'releaseRefs', {}).get('current', {}).get('name', '')
        if not current_clusterrelease:
            return None
        return current_clusterrelease

    @property
    def cluster_spec_release_version(self):
        """Determine the cluster clusterrelease version

        :rtype string: clusterrelease of the cluster
        """
        if not self.spec or type(self.spec) is not dict:
            return None
        spec_clusterrelease = self.spec.get("providerSpec", {}).get(
            "value", {}).get("release", "")
        if not spec_clusterrelease:
            return None
        return spec_clusterrelease

    @property
    def clusterrelease_actual_version(self):
        """Determine the cluster clusterrelease actual version

        :rtype string: clusterrelease of the cluster
        """
        status = self.data['status']
        if not status:
            return None
        current_clusterrelease = status.get('providerStatus', {}).get(
            'releaseRefs', {}).get('current', {}).get('version', '')
        if not current_clusterrelease:
            return None
        return current_clusterrelease

    @property
    def available_clusterrelease_version(self):
        """Determine first available clusterrelease version

        :rtype string clusterrelease of the cluster or None
        """
        status = self.data['status']
        if not status:
            return None
        available_clusterreleases_list = status.get('providerStatus', {}).get(
            'releaseRefs', {}).get('available', [{}])
        if len(available_clusterreleases_list) == 0:
            return None
        available_clusterrelease = available_clusterreleases_list[0].get(
            'name', '')
        if not available_clusterrelease:
            return None
        return available_clusterrelease

    @property
    def lcm_type(self):
        """Determine the cluster LCM engine type

        :rtype string: LCM type of the cluster, or "kubespray" as default,
                       or None if no status yet
        """
        status = self.data['status']
        if not status:
            return None
        current_release = status.get('providerStatus', {}).get(
            'releaseRefs', {}).get('current', {})
        if not current_release:
            return None
        return current_release.get('lcmType', 'kubespray')

    @property
    def lcm_type_is_ucp(self):
        return 'ucp' in self.lcm_type

    @property
    def lcm_type_is_kubespray(self):
        return 'kubespray' in self.lcm_type

    @property
    def cluster_proxy_name(self):
        current_cluster_proxy_name = self.spec.get('providerSpec', {}).get('value', {}).get('proxy')
        return current_cluster_proxy_name

    @property
    def cluster_container_registy_name(self):
        current_cluster_container_registy_name = self.spec['providerSpec']['value'].get(
            'сontainerRegistries', {}).get('name')
        return current_cluster_container_registy_name

    @property
    def provider(self):
        """Determine the cluster provider

        Returns: utils.Provider
        """
        if self.__provider is None:
            kind = self.spec['providerSpec']['value']['kind']
            api_version = self.spec['providerSpec']['value']['apiVersion'].split("/")[-1]
            provider = utils.Provider.get_provider_by_cluster(kind, api_version)
            self.__provider = provider
            assert self.__provider is not None, \
                f"Cannot determinate provider with kind '{kind}' and version '{api_version}' " \
                f"for cluster '{self.namespace}/{self.name}'"

            assert self.__provider in utils.Provider, \
                (f"Provider '{self.__provider}' for the Cluster '{self.namespace}/{self.name}' "
                 f"is not supported, please update tests")

        return self.__provider

    @property
    def region_name(self):
        """Determine the cluster region"""

        mgmt_cluster = self
        if not self.is_management:
            mgmt_cluster = self.__manager.get_mgmt_cluster()

        if utils.version_greater_than_2_26_0(mgmt_cluster.get_kaasrelease_version()):
            self.__region_name = ''
        elif self.__region_name is None:
            if self.is_management:
                self.__region_name = self.data.get(
                    'metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '')
            else:
                self.__region_name = mgmt_cluster.region_name

        return self.__region_name

    def get_kaasrelease_version(self):
        if self.is_management or self.is_regional:
            cluster_data = self.data
            return cluster_data['spec'][
                'providerSpec']['value']['kaas']['release']
        else:
            LOG.warning(f"Cannot get kaasrelease version "
                        f"for cluster: {self.name} - "
                        f"its non management|regional cluster => using information from mgmt cluster.")
            return self.__manager.get_mgmt_cluster().get_kaasrelease_version()

    def get_cluster_annotations(self):
        cluster_data = self.data
        return cluster_data['metadata'][
            'annotations']

    def update_proxy(self, name, update_proxy, ca_cert=None):

        body = {
            "spec": {
                "httpProxy": update_proxy,
                "httpsProxy": update_proxy,
                "caCertificate": ca_cert
            }
        }
        proxy_object = self.__manager.api.kaas_proxies.update(
            name=name, namespace=self.namespace, body=body)
        return proxy_object

    def get_cluster_annotation(self, annotation_key):
        """
        ---------
        metadata:
          annotations:
            kaas.mirantis.com/lcm: "true"
        ---------
        :param annotation_key: "kaas.mirantis.com/lcm"  # example value
        :return: {kaas.mirantis.com/lcm: "true"} # example return
        """
        annotation = None
        all_annotations = self.get_cluster_annotations()
        for ak, av in all_annotations.items():
            if ak == annotation_key:
                annotation = {ak: av}
        return annotation

    def add_cluster_annotation(self, annotation_key, annotation_value):
        """
        Adds the annotation "annotation_key" with value
        "annotation_value" and will overwrite the "annotation_key"
        label if it already exists.
        :return: api result
        """
        body = {
            "metadata": {
                "annotations": {
                    annotation_key: annotation_value}
            }
        }
        api_responce = self.__manager.api.kaas_clusters.update(
            name=self.name,
            namespace=self.namespace,
            body=body
        )
        return api_responce

    def remove_cluster_annotation(self, annotation_key):
        """
        For remove annotation, we should add annotation
        "annotation_key" with "annotation_value" = None
        """
        return self.add_cluster_annotation(annotation_key, None)

    def get_k8s_version(self):
        try:
            ver = self.k8sclient.api_version.get_code()
        except OSError as e:
            LOG.error(e)
            ver = self.k8sclient.api_version.get_code()
        except ApiException as e:
            if e.status == 401 and e.reason == 'Unauthorized':
                LOG.error(e)
                self.k8sclient.login()
                self.k8sclient.init_apis()
                ver = self.k8sclient.api_version.get_code()
            else:
                LOG.warning("There is an error happened during k8s api access")
                raise e
        k8s_version = "{0}.{1}.x".format(ver.major, ver.minor).replace("+", "")
        return k8s_version

    def _save_service_external_ip(self, service_name, service_namespace, filename=None):
        filename = filename or f"service_{service_name}_external_ip.txt"
        client = self.k8sclient
        if client.services.present(name=service_name, namespace=service_namespace):
            service = self.k8sclient.services.get(name=service_name, namespace=service_namespace)
            external_ip = service.get_external_ip()
            artifact_path = os.path.join(settings.ARTIFACTS_DIR, filename)
            LOG.info(f"Save the service '{service_namespace}/{service_name}' "
                     f"external IP '{external_ip}' to {artifact_path}")
            with open(artifact_path, 'w') as f:
                f.write(str(external_ip))
        else:
            LOG.warning(f"Service '{service_namespace}/{service_name}' not found "
                        f"in the Cluster '{self.namespace}/{self.name}', skip saving artifact")

    def store_k8s_artifacts(self, secret_name=''):

        kubeconfig_name, kubeconfig = self.get_kubeconfig(secret_name)

        kubeconfig_path = os.path.join(
            settings.ARTIFACTS_DIR, kubeconfig_name)

        LOG.info("Save cluster kubeconfig to %s", kubeconfig_path)
        with open(kubeconfig_path, 'w') as f:
            f.write(kubeconfig)

        k8s_version = self.get_k8s_version()
        kubeconfig_version_path = os.path.join(
            settings.ARTIFACTS_DIR, kubeconfig_name + "_k8s_version")
        LOG.info("Save child cluster k8s version {0} to {1}"
                 .format(k8s_version, kubeconfig_version_path))
        with open(kubeconfig_version_path, 'w') as f:
            f.write(k8s_version)

        LOG.info("Save child cluster uid")
        cluster_uid_path = "{0}/child_cluster_uid".format(settings.ARTIFACTS_DIR)
        kaas_uid_meta = self.get_cluster_annotation(
            "kaas.mirantis.com/uid")
        assert kaas_uid_meta["kaas.mirantis.com/uid"], "No kaas.mirantis.com/uid found"
        cluster_uid_test = kaas_uid_meta["kaas.mirantis.com/uid"]
        with open(cluster_uid_path, 'w') as fp:
            fp.write(str(cluster_uid_test))

        # Creates the file 'artifacts/cluster_name'
        utils.save_cluster_name_artifact(self.namespace, self.name)

        # For EM2/BM cases, external IP address for dhcp-relay service
        self._save_service_external_ip(service_name='dhcp-lb',
                                       service_namespace='kaas',
                                       filename="service_dhcp-lb_external_ip.txt")

    def store_machines_artifacts(self,
                                 machine_types=None,
                                 public_ip=True):
        artifacts_dir = settings.ARTIFACTS_DIR
        m_types = machine_types or ["control", "worker"]
        LOG.info(f"Store IPs for the following machine types in the cluster "
                 f"{self.name}: {m_types}")

        for m_type in m_types:
            typed_machines = self.get_machines(machine_type=m_type)
            addrs = []
            for machine in typed_machines:
                if public_ip:
                    addrs.append(machine.public_ip)
                else:
                    addrs.append(machine.internal_ip)
            filename = f"{artifacts_dir}/machines_{m_type}"
            with open(filename, "w") as f:
                f.write(",".join(addrs))
            LOG.info(f"'{m_type}' machine IPs have been saved to {filename}")

    def update_cluster(self, update_release_name):

        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "release": update_release_name
                    }
                }
            }
        }

        cluster = self.__manager.api.kaas_clusters.update(
            name=self.name, namespace=self.namespace, body=body)
        return Cluster(self.__manager, cluster)

    def update_cluster_proxy(self, update_proxy_name=None):
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "proxy": update_proxy_name
                    }
                }
            }
        }

        self.__manager.api.kaas_clusters.update(
            name=self.name, namespace=self.namespace, body=body)

    def update_tlsconfigref_for_app(self, app, tlsconfigref):
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "tls": {
                            app: {
                                "tlsConfigRef": tlsconfigref,
                            }
                        }
                    }
                }
            }
        }

        self.__manager.api.kaas_clusters.update(
            name=self.name, namespace=self.namespace, body=body)

    def describe_public_services(self):
        """Find the services with external IP and http/https ports

        return: dict, key/values where key is service.name,
                value is list of URLs:
                {'iam-proxy-alerta': ['http://172.19.116.138'], ...}
        """
        services = self.k8sclient.services.list_all()
        result = {}
        for service in services:
            ip = service.get_external_ip()
            ports = service.get_ports()
            if ip:
                urls = []
                for port in ports:
                    if port.target_port in ['http', 'https']:
                        urls.append("{0}://{1}".format(port.target_port, ip))
                if urls:
                    result[service.name] = urls
        return result

    def get_helmbundles(self):
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        return client.kaas_helmbundles.list(
            namespace=self.namespace)

    def get_helmbundle(self, name=None, namespace=None):
        """Return specified helmbundle.
        By default helmbundle for current cluster is returned
        """
        if not namespace:
            namespace = self.namespace
        if not name:
            name = self.name
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        hb = client.kaas_helmbundles.get(name=name, namespace=namespace)
        if hb is None:
            raise LookupError(f"Helmbundle {namespace}/{name} not found")
        return hb

    def get_clusterworkloadlocks(self):
        return self.k8sclient.clusterworkloadlocks.list_all()

    def get_clusterworkloadlock(self, name):
        cwl = [lock for lock in self.get_clusterworkloadlocks() if lock.name == name]
        return cwl[0] if cwl else None

    def get_clustermaintenancerequests(self):
        return self.k8sclient.clustermaintenancerequests.list_all()

    def get_clustermaintenancerequest(self, name):
        cmr = [cr for cr in self.get_clustermaintenancerequests() if cr.name == name]
        return cmr[0] if cmr else None

    def get_nodeworkloadlocks(self):
        return self.k8sclient.nodeworkloadlocks.list_all()

    def get_nodeworkloadlock(self, name):
        nwl = [lock for lock in self.get_nodeworkloadlocks() if lock.name == name]
        return nwl[0] if nwl else None

    def get_nodemaintenancerequests(self, **kwargs):
        return self.k8sclient.nodemaintenancerequests.list_all(**kwargs)

    def get_nodemaintenancerequest(self, name):
        nmr = [nr for nr in self.get_nodemaintenancerequests() if nr.name == name]
        return nmr[0] if nmr else None

    def get_gracefulrebootrequests(self):
        return self.__manager.api.gracefulrebootrequest.list(
            namespace=self.namespace)

    def get_gracefulrebootrequest(self):
        grr = [r for r in self.get_gracefulrebootrequests() if r.name == self.name]
        return grr[0] if grr else None

    def get_default_update_group(self):
        return self.__manager.api.updategroups.get(
            f'{self.name}-default',
            self.namespace
        )

    def get_control_update_group(self):
        return self.__manager.api.updategroups.get(
            f'{self.name}-controlplane',
            self.namespace
        )

    def get_update_group_by_name(self, name):
        return self.__manager.api.updategroups.get(name, self.namespace)

    def get_update_groups(self):
        return self.__manager.api.updategroups.list_all()

    def create_update_group(self, name, cluster_name, index=1, concurrent_updates=1, namespace='default'):
        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "UpdateGroup",
            "metadata": {
                "name": name,
                "namespace": namespace,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": cluster_name}
            },
            "spec": {
                "index": index,
                "concurrentUpdates": concurrent_updates
            }
        }
        return self.__manager.api.updategroups.create(name=name, namespace=namespace, body=body)

    def create_update_group_raw(self, data):
        return self.__manager.api.updategroups.create(namespace=self.namespace,
                                                      name=data['metadata']['name'], body=data)

    def update_group_present(self, name):
        """Check that specified group name exists in the namespace"""
        return any([group.name for group in self.get_update_groups()
                    if group.name == name])

    def delete_update_group(self, name, timeout=60, interval=5):
        """Delete Update Group object in the current Namespace"""
        if self.__manager.api.updategroups.get(name, self.namespace):
            group = self.get_update_group_by_name(name)
            group.delete()
            timeout_msg = f"Update Group {name} was not deleted in {timeout}s"
            waiters.wait(lambda: not bool(self.update_group_present(name=name)),
                         timeout=timeout,
                         interval=interval,
                         timeout_msg=timeout_msg)
            LOG.info(f"Group {name} has been succesfully deleted")
        else:
            LOG.warning(f"Group {name} not found, nothing to delete")

    def get_nodedeletionrequests(self):
        return self.k8sclient.nodedeletionrequests.list_all()

    def get_nodedeletionrequest(self, name):
        nmr = [nr for nr in self.get_nodedeletionrequests() if nr.name == name]
        return nmr[0] if nmr else None

    def get_nodedisablenotifications(self):
        return self.k8sclient.nodedisablenotifications.list_all()

    def get_nodedisablenotification(self, name):
        ndn = [dn for dn in self.get_nodedisablenotifications() if dn.name == name]
        return ndn[0] if ndn else None

    def get_rookcephclusters(self, namespace='rook-ceph'):
        return self.k8sclient.rookcephclusters.list(namespace=namespace)

    def get_kaascephclusters(self):
        return self.__manager.api.kaas_cephclusters.list(
            namespace=self.namespace)

    def get_cephoperationrequests(self):
        return self.k8sclient.kaas_cephoperationrequests.list_all()

    def get_cephoperationrequest(self, name):
        return self.k8sclient.kaas_cephoperationrequests.get(name)

    def get_mcc_certificate_requests(self, name_prefix=None, namespace=None):
        """
        Get list of MCCCertificateRequest objects
        Args:
            name_prefix: prefix for name
            namespace: specify namespace or None to use all namespaces

        Returns: list of si_tests.clients.k8s.models.v1_mcc_certificate_request.V1MCCCertificateRequest

        """
        return self.k8sclient.mcc_certificate_requests.list(name_prefix=name_prefix, namespace=namespace)

    def get_mcc_certificate_request(self, name, namespace=None):
        """
        Get MCCCertificateRequest object
        Args:
            name: specify name
            namespace: specify namespace or None to use all namespaces

        Returns: si_tests.clients.k8s.models.v1_mcc_certificate_request.V1MCCCertificateRequest

        """
        return self.k8sclient.mcc_certificate_requests.get(name, namespace)

    def get_certificate_configuration(self, name, namespace=None):
        """
        Get CertificateConfiguration object
        Args:
            name: specify name
            namespace: specify namespace

        Returns: si_tests.clients.k8s.models.v1_certificate_configuration.V1CertificateConfiguration

        """
        return self.k8sclient.certificat_configurations.get(name=name, namespace=namespace)

    def get_allowed_parallel_nmr_cnt(self):
        # TODO: Here we should check, how many parallel NodeMaintenanceRequests may be created in current cluster.
        # This feature not yet implemented(PRODX-19088), and this function temporarily return 1 by default
        allowed_parallel_nmr_cnt = 1
        return allowed_parallel_nmr_cnt

    def get_cluster_deploy_stages_filtered_by_cluster_name(self, cluster_name):
        all_clusterdeploystatuses = self.__manager.get_cluster_deploy_status_crds()
        LOG.debug(f'All crds with deploy stages:\n{all_clusterdeploystatuses}')
        stages = [depl.stages for depl in all_clusterdeploystatuses
                  if cluster_name in depl.to_dict()['metadata']['name']]
        if stages:
            LOG.info(f'Have next stages {stages} in cluster {cluster_name}')
            return stages[0]
        else:
            return []

    def get_cluster_upgrade_stages_filtered_by_cluster_name(self, cluster_name):
        all_clusterupgstatuses = self.__manager.get_cluster_upgrades_status_crds()
        LOG.debug(f'All crds with upgrades stages:\n{all_clusterupgstatuses}')
        stages = [upg.stages for upg in all_clusterupgstatuses
                  if cluster_name in upg.to_dict()['metadata']['name']]
        if stages:
            LOG.info(f'Have next stages {stages} in cluster {cluster_name}')
            return stages[0]
        else:
            return []

    def get_machine_upgrade_stages_filtered_by_machine_name(self, machine_name):
        all_machineupgstatus = self.__manager.get_machine_upgrade_status_crds()
        LOG.debug(f'All crds with upgrades stages:\n{all_machineupgstatus}')
        stages = [upg.stages for upg in all_machineupgstatus
                  if machine_name in upg.to_dict()['metadata']['name']]
        if stages:
            LOG.info(f'Have next stages {stages} in machine {machine_name}')
            return stages[0]
        else:
            return []

    def get_machine_upgrade_stages(self, machine_name, namespace=None):
        machine_upgrade_status = self.__manager.get_machine_upgrade_status_crd(machine_name, namespace=namespace)
        stages = machine_upgrade_status.data.get('stages', [])
        LOG.info(f'Have next stages {stages} in machine {machine_name}')
        return stages

    def get_rookcephcluster(self, cluster_name='rook-ceph', namespace='rook-ceph'):
        rookcephcluster = [cl for cl in self.get_rookcephclusters(namespace) if cl.name == cluster_name]
        assert rookcephcluster, (f"No rookcephcluster with name {cluster_name} found")
        return rookcephcluster[0]

    def get_kaascephcluster(self, name=None):
        # for future if we have 2 ceph clusters for one child
        # name will be required
        kaascephclusters = self.get_kaascephclusters()
        num_clusters = len(kaascephclusters)
        if name:
            # Use the specified KaasCephCluster name
            clusters = [cluster for cluster in kaascephclusters if cluster.name == name]
            assert clusters, (f"No kaascephclusters found in {num_clusters} clusters, "
                              f"expected '{self.namespace}/{name}'")
            assert len(clusters) == 1, (f"More then one kaascephcluster found "
                                        f"with name '{self.namespace}/{name}' : {clusters}")
        else:
            # Try to find the KaasCephCluster for the current Cluster object
            clusters = [cluster for cluster in kaascephclusters
                        if cluster.data['spec']['k8sCluster']['name'] == self.name]
            assert clusters, (f"No kaascephclusters found in {num_clusters} clusters "
                              f"for the Cluster '{self.namespace}/{self.name}'")
            assert len(clusters) == 1, (f"More then one kaascephcluster found "
                                        f"with spec.k8sCluster '{self.namespace}/{self.name}' : {clusters}")
        return clusters[0]

    def get_miracephcluster_osd_number(self, name='rook-ceph'):
        ceph_cluster_data = self.get_miracephcluster(name).data
        ceph_nodes = ceph_cluster_data.get('spec', {}).get('nodes', {})
        osd_number = 0
        for node in ceph_nodes:
            ceph_devices = node.get('devices', [])
            for device in ceph_devices:
                osd_per_device = device.get('config', {}).get('osdsPerDevice')
                if osd_per_device:
                    osd_number += int(osd_per_device)
                else:
                    osd_number += 1
        return osd_number

    def get_miracephcluster_min_replica_size(self, name='rook-ceph'):
        ceph_cluster_data = self.get_miracephcluster(name).data
        ceph_pools = ceph_cluster_data.get('spec', {}).get('pools', [])
        min_replica_size = 0
        for pool in ceph_pools:
            pool_size = pool.get('replicated', {}).get('size')
            if pool_size:
                if min_replica_size == 0 or int(pool_size) < min_replica_size:
                    min_replica_size = int(pool_size)
        return min_replica_size

    def get_kaascephcluster_osd_number(self, name=None):
        ceph_cluster_data = self.get_kaascephcluster(name).data
        ceph_nodes = ceph_cluster_data.get('spec', {}).get(
            'cephClusterSpec', {}).get('nodes', {})
        osd_number = 0
        for _, node in ceph_nodes.items():
            ceph_devices = node.get('storageDevices', [])
            for device in ceph_devices:
                osd_per_device = device.get('config', {}).get('osdsPerDevice')
                if osd_per_device:
                    osd_number += int(osd_per_device)
                else:
                    osd_number += 1
        return osd_number

    def get_kaascephcluster_min_replica_size(self, name=None):
        ceph_cluster_data = self.get_kaascephcluster(name).data
        ceph_pools = ceph_cluster_data.get('spec', {}).get(
            'cephClusterSpec', {}).get('pools', [])
        min_replica_size = 0
        for pool in ceph_pools:
            pool_size = pool.get('replicated', {}).get('size')
            if pool_size:
                if min_replica_size == 0 or int(pool_size) < min_replica_size:
                    min_replica_size = int(pool_size)
        return min_replica_size

    def get_ceph_version(self, name=None):
        kaascephcluster_status = self.get_kaascephcluster(name).data.get('status') or {}
        ceph_version = kaascephcluster_status.get('fullClusterInfo', {}).get(
            'clusterStatus', {}).get('version', {}).get('version')
        return ceph_version

    def get_miracephhealth_version(self, name='rook-ceph'):
        miracephhealth_status = self.get_miracephhealth(name).data.get('status') or {}
        ceph_version = miracephhealth_status.get('fullClusterStatus', {}).get(
            'clusterStatus', {}).get('version', {}).get('version')
        return ceph_version

    def get_miraceph_version(self, cluster_name='rook-ceph', namespace='ceph-lcm-mirantis'):
        """
        Get version of the specified Ceph cluster from the MiraCeph resource.
        Args:
            cluster_name: Name of the Ceph cluster to get.
            namespace: The Kubernetes namespace where the MiraCeph resource is located.

        Returns: Ceph cluster version as string.
        """
        miracephcluster = self.get_miracephcluster(cluster_name, namespace)
        assert miracephcluster, (f"No miracephcluster with name {cluster_name} found")
        miraceph_version = miracephcluster.data.get('status', {}).get('clusterVersion')
        return miraceph_version

    def get_cluster_events(self):
        """Return all Cluster events"""
        return self.k8sclient.events.list_all()

    def get_parent_events(self):
        """Return all events from the Cluster namespace in the Region or Management cluster"""
        return self.get_parent_client().events.list(namespace=self.namespace)

    def wait_machine_deletion(self, machine_name,
                              retries=30, interval=30):
        def status_msg():
            """To show the remaining machines status after wait timeout"""
            machines_status = {m.name: m.data['status'] for m in self.get_machines()}
            return "Machines status: {0}".format(machines_status)

        def check_deleted():
            machine = self.get_machine(machine_name)
            if not machine:
                LOG.info("Machine %s was not found in cluster, "
                         "looks like it was deleted", machine_name)
                return True
            LOG.info("Machine %s has not been deleted yet and has status - %s",
                     machine_name, machine.machine_status)
            raise Exception("Machine has been deleted")

        total_time = retries * interval
        LOG.info("Wait {0} seconds until machine will be deleted"
                 .format(total_time))

        timeout_msg = ("Machine has not been deleted during "
                       "timeout {} seconds").format(total_time)
        waiters.wait_pass(lambda: check_deleted(), timeout=total_time,
                          timeout_msg=timeout_msg,
                          status_msg_function=status_msg)

    def get_lcmmachines(self, states=None):
        """Return list of lcmmachines filtered by states list"""
        all_lcmmachines = self.get_all_lcmmachines(states=states, namespace=self.namespace)
        cluster_lcmmachines = [
            x for x in all_lcmmachines if self.name == x.data.get('spec', {}).get('clusterName', '')]
        LOG.debug(f"LCMMachines of cluster {self.name} are: {[x.name for x in cluster_lcmmachines]} "
                  f"from the total {len(all_lcmmachines)} lcmmachines in the namespace {self.namespace}")
        return cluster_lcmmachines

    def get_all_lcmmachines(self, states=None, namespace=None):
        """Return list of all lcmmachines filtered by states
        and namespace name
        """
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        if namespace:
            lcmm = client.kaas_lcmmachines.list_all(states=states)
            return [x for x in lcmm if x.namespace == namespace]
        else:
            return client.kaas_lcmmachines.list_all(states=states)

    def get_cluster_lcmmachine(self, name, namespace):
        """Return lcmmachine"""
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        lcmm = client.kaas_lcmmachines.get(name=name, namespace=namespace)
        return lcmm

    def get_cluster_lcmmachines_by_type(self, name, namespace, machine_type):
        """Return lcmmachines of certain cluster with specified type"""
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        target_machines = []
        all_machines = client.kaas_lcmmachines.list(namespace=namespace)
        for m in all_machines:
            if m.data.get("spec", {}).get("clusterName") != name:
                continue
            if m.data.get("spec", {}).get("type") != machine_type:
                continue
            target_machines.append(m)
        return target_machines

    def get_lcm_cluster(self, name, namespace):
        """
        Get KaaS LCM Cluster
        Args:
            name: lcm cluster name
            namespace: required namespace

        Returns: Lcmcluster object

        """
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        lcm_cluster = client.kaas_lcmclusters.get(name=name, namespace=namespace)
        return lcm_cluster

    def get_lcm_cluster_mke_config(self, name, namespace):
        """
        Get KaaS LCM Cluster Mke Config
        Args:
            name: lcm cluster name
            namespace: required namespace

        Returns: Lcmclustermkeconfig object

        """
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        lcm_cluster_mke_config = client.kaas_lcmclustermkeconfigs.get(name=name, namespace=namespace)
        return lcm_cluster_mke_config

    def get_lcmcluster_states(self, namespace=None, name_prefix=None):
        """
        Get KaaS LCM Cluster States
        Args:
            namespace: required namespace
            name_prefix: item prefix name

        Returns: list of Lcmclusterstates objects

        """
        client = self.__manager.api
        if not self.is_management:
            client = self.get_parent_cluster().k8sclient
        states = client.kaas_lcmcluster_states.list(namespace=namespace, name_prefix=name_prefix)
        return states

    def get_machine(self, name):
        all_machines = self.get_machines()
        ret = next((m for m in all_machines if m.name == name), None)
        return ret

    def get_machine_uncached(self, name):
        all_machines = self.get_machines_uncached()
        ret = next((m for m in all_machines if m.name == name), None)
        return ret

    def get_machine_by_k8s_name(self, name):
        all_machines = self.get_machines()
        ret = next((m for m in all_machines if m.data["status"]["nodeRef"]["name"] == name), None)
        return ret

    def get_machines_by_label(self, label):
        all_machines = self.get_machines()
        return [m for m in all_machines if m.data['metadata'].get('labels', {}).get(label, '')]

    def get_machines(self, machine_type=None, machine_status=None, k8s_labels=None, raise_if_empty=True):
        """Return list of Machine objects and set bastion_ip"""
        cluster_machines = self._get_machines(raise_if_empty=raise_if_empty)
        if machine_type is not None:
            cluster_machines = [m for m in cluster_machines
                                if m.is_machine_type(machine_type)]
        if machine_status is not None:
            cluster_machines = [m for m in cluster_machines
                                if m.machine_status == machine_status]
        if k8s_labels is not None:
            cluster_machines = [m for m in cluster_machines
                                if m.has_k8s_labels(k8s_labels)]
        return cluster_machines

    def get_machines_uncached(self) -> List[MachineProviders]:
        """Return list of Machine objects of the current cluster without using cache"""
        label_selector = f"cluster.sigs.k8s.io/cluster-name={self.name}"
        machines = self.__manager.api.kaas_machines.list(namespace=self.namespace, label_selector=label_selector)
        provider_machines = {
            'openstack': OpenstackProviderMachine,
            'baremetal': BaremetalProviderMachine,
            'aws': AwsProviderMachine,
            'byo': ByoProviderMachine,
            'vsphere': VsphereProviderMachine,
            'equinixmetal': EquinixMetalProviderMachine,
            'equinixmetalv2': EquinixMetalV2ProviderMachine,
            'azure': AzureProviderMachine
        }
        if self.provider.provider_name not in provider_machines:
            raise NotImplementedError(f"Provider {self.provider.provider_name} is not supported")
        machine_class = provider_machines[self.provider.provider_name]
        return [machine_class(self.__manager, self, m) for m in machines]

    def get_machines_reboot_required_status(self, machines=None):
        if not machines:
            machines = self.get_machines()
        machines_map = {}
        for machine in machines:
            machines_map.update({machine.name: {'is_reboot_required': machine.is_reboot_required()[0]}})
        return machines_map

    def get_machine_openstack_name(self, machine=None):
        """ Return name of instance in Openstack cloud
        :param machine: Machine Name or Object
        """
        if isinstance(machine, str):
            os_machine = self.get_machine(machine)
        else:
            os_machine = machine

        if self.get_custom_hostnames_enabled():
            LOG.info(f"Use custom machine name: {os_machine.name}")
            return os_machine.name
        else:
            LOG.info("Use kaas.mirantis.com/uid for machine name")
            openstack_instance_uid = os_machine.data['metadata'][
                'annotations']['kaas.mirantis.com/uid']
            openstack_instance_name = "kaas-node-" + openstack_instance_uid
            return openstack_instance_name

    def get_cluster_lcmmachines_timestamps(self):
        """
        Returns map of all cluster lcmmachines with timestamps
        """
        lcmmachines = [m for m in self.get_lcmmachines()
                       if m.data.get('spec', {}).get('type', '') != 'bastion']
        return {m.name: self.get_lcmmachine_timestamps(m) for m in lcmmachines}

    def get_lcmmachine_timestamps(self, lcmmachine):
        """
        Returns lcmmachine timestamps with upgrade index and machine type
        """
        m_name = lcmmachine.name
        timestamps = {'phases': {}, 'upgradeIndex': None, 'updateGroup': {}, 'machine_type': ''}
        lcmmachine_status = lcmmachine.data.get('status') or {}
        statuses = lcmmachine_status.get('stateItemStatuses', {})
        machine = self.get_machine(m_name)
        machine_status = machine.data.get('status') or {}
        upgrade_index = machine_status.get('providerStatus', {}).get('upgradeIndex', None)
        update_group_name = machine.metadata.get('labels', {}).get('kaas.mirantis.com/update-group', '')
        update_group = self.get_update_group_by_name(name=update_group_name)
        update_group_index = update_group.data['spec'].get('index', None)
        update_group_concurrent_updates = update_group.data['spec'].get('concurrentUpdates', 1)
        timestamps['upgradeIndex'] = upgrade_index
        timestamps['updateGroup']['Name'] = update_group_name
        timestamps['updateGroup']['Index'] = update_group_index
        timestamps['updateGroup']['Concurrency'] = update_group_concurrent_updates
        timestamps['machine_type'] = machine.machine_type
        timestamps['latest_startedAt'] = ''
        timestamps['latest_finishedAt'] = ''
        for k, v in statuses.items():
            startedAt = v.get('startedAt', '')
            finishedAt = v.get('finishedAt', '')
            timestamps['phases'].update(
                {k: {'startedAt': startedAt, 'finishedAt': finishedAt}})
            if startedAt > timestamps['latest_startedAt']:
                timestamps['latest_startedAt'] = startedAt
            if finishedAt > timestamps['latest_finishedAt']:
                timestamps['latest_finishedAt'] = finishedAt
        return timestamps

    def get_cluster_machines_ansible_versions(self):
        """
        Returns map of ansible_versions for each machine in cluster
        """
        lcmmachines = [m for m in self.get_lcmmachines()
                       if m.data.get('spec', {}).get('type', '') != 'bastion'
                       and not m.data.get('spec', {}).get('disable', False)]
        return {m.name: self.get_ansible_version(m) for m in lcmmachines}

    def get_ansible_version(self, lcmmachine):
        """
        Returns lcmmachine timestamps with upgrade index and machine type
        """
        lcmmachine_spec = [spec for spec in lcmmachine.data.get('spec', {}).get('stateItems', [])
                           if spec.get('name', '') == "download lcm ansible"]
        assert lcmmachine_spec, 'stateItems with name: \'download lcm ansible\' wasn\'t found in lcmmachine spec'
        ansible_version = lcmmachine_spec[0].get('params', {}).get('path', '').split("/")[-1]
        return ansible_version

    def get_related_cephclusters(self):
        return self.__manager.api.kaas_cephclusters.list(
            namespace=self.namespace)

    def get_controller_ip(self):
        machines = self._get_machines()
        for machine in machines:
            if machine.is_machine_type('control'):
                return machine.public_ip
        raise Exception("No controllers found in the cluster")

    def _get_machines(self, raise_if_empty=True) -> List[MachineProviders]:
        """Return list of Machine objects which are belong to the cluster"""
        # collect raw data of all machines in namespace
        machines = self.__manager.api.kaas_machines.list_raw(
            namespace=self.namespace).to_dict()['items']
        cl_name = 'cluster.sigs.k8s.io/cluster-name'
        # filter out machines not from current cluster
        cl_machines = [x for x in machines if x['metadata'].get('labels', {}).get(cl_name, '') == self.name]

        # assuming all machines in cluster has the same provider
        if cl_machines:
            # get MachineClass based on the 1st machine in list
            MachineClass = self._get_machine_class(cl_machines[0])
        elif raise_if_empty:
            raise Exception(f"No Machines have been found for the cluster {self.namespace}/{self.name}")
        else:
            return []
        cl_machine_names = [x['metadata']['name'] for x in cl_machines]

        # collect kaas_machines objects
        machines_obj = [
            x for x in self.__manager.get_machines(namespace=self.namespace)
            if x.name in cl_machine_names
        ]

        cluster_machines = []
        for machine in machines_obj:
            cluster_machines.append(
                MachineClass(self.__manager, self, machine))
        return cluster_machines

    def _get_machine_class(self, kaasmachine) -> MachineProviders:
        provider_machines = {
            'openstack': OpenstackProviderMachine,
            'baremetal': BaremetalProviderMachine,
            'aws': AwsProviderMachine,
            'byo': ByoProviderMachine,
            'vsphere': VsphereProviderMachine,
            'equinixmetal': EquinixMetalProviderMachine,
            'equinixmetalv2': EquinixMetalV2ProviderMachine,
            'azure': AzureProviderMachine
        }
        machine_spec = kaasmachine['spec']
        machine_provider = machine_spec['providerSpec']['value']['kind']
        machine_api_version = machine_spec['providerSpec']['value']['apiVersion'].split("/")[-1]
        provider = utils.Provider.get_provider_by_machine(machine_provider, machine_api_version)
        if provider is None:
            raise Exception(
                "Provider 'spec/providerSpec/value/kind: {0}' and version {1} is not"
                " supported in si-tests, please"
                " update the code".format(machine_provider, machine_api_version))
        adapter_class = provider_machines[provider.provider_name]
        return adapter_class

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def bastion_ip(self):
        if self.__bastion_ip:
            return self.__bastion_ip

        provider_status = self.data["status"].get("providerStatus", {})
        if 'bastion' in provider_status and provider_status.get('bastion'):
            bastion = provider_status["bastion"]
            # We have different field names for different providers right now
            if "publicIp" in bastion:
                self.__bastion_ip = bastion["publicIp"]
            elif "publicIP" in bastion:
                self.__bastion_ip = bastion["publicIP"]
            else:
                self.__bastion_ip = ""
        elif self.provider == utils.Provider.aws:
            # AWS machines never have public IPs
            raise Exception("No <bastion> data in the status "
                            "for cluster {0}".format(self.name))
        elif self.provider == utils.Provider.azure:
            # Azure machine access using host API server IP address as bastion
            assert 'loadBalancerHost' in provider_status, \
                (f"loadBalancerHost must present in cluster {self.name} "
                 "provider status as bastion host for azure machines")
            self.__bastion_ip = provider_status['loadBalancerHost']
            return self.__bastion_ip
        elif self.provider == utils.Provider.baremetal:
            return None
        else:
            # When bastion is not available
            cluster_machines = self._get_machines()
            # Use a k8s control node with public_ip for a bastion_ip
            # if any of control nodes have public_ip and is in Ready status
            for m in cluster_machines:
                LOG.debug("Machine {0} has type {1} and status {2}"
                          .format(m.name, m.machine_type, m.machine_status))
                if all([m.is_machine_type('control'),
                        m.machine_status == "Ready",
                        m.public_ip is not None]):
                    self.__bastion_ip = m.public_ip
                    break
        return self.__bastion_ip

    @bastion_ip.setter
    def bastion_ip(self, ip):
        self.__bastion_ip = ip

    def _get_bastion_remote(self, ssh_login="mcc-user",
                            ssh_key=None, key_file_var="Private key"):
        if not self.bastion_ip:
            raise Exception(
                "Bastion IP is not set for cluster <{0}>,"
                " SSHClient cannot be initialized".format(self.name))

        if not os.path.isfile(ssh_key):
            raise Exception(
                "{2} is not set for bastion "
                "IP {0} in the cluster <{1}>, "
                "SSHClient cannot be "
                "initialized".format(self.bastion_ip, self.name, key_file_var))

        keys = utils.load_keyfile(ssh_key)
        pkey = utils.get_rsa_key(keys['private'])

        auth = exec_helpers.SSHAuth(username=ssh_login,
                                    password='', key=pkey)
        ssh = exec_helpers.SSHClient(host=self.bastion_ip, port=22, auth=auth)
        ssh.logger.addHandler(logger.console)

        # Upload private key to the bastion node, if missing
        priv_key_dir = os.path.join("/home",
                                    ssh_login,
                                    ".ssh")
        priv_key_path = os.path.join(priv_key_dir, self.name)
        res = ssh.execute("test -f {0}".format(priv_key_path), verbose=False)
        if res.exit_code != 0:
            ssh.check_call("mkdir -p {0}; chmod 0700 {0}"
                           .format(priv_key_dir), verbose=False)
            ssh.upload(ssh_key,
                       priv_key_path)
            ssh.check_call("mkdir -p {0}; chmod 0600 {0}"
                           .format(priv_key_path), verbose=False)
        return ssh

    def get_bastion_remote(self):
        return self._get_bastion_remote(ssh_login=self.ssh_user,
                                        ssh_key=self.private_key,
                                        key_file_var=self.key_file_var)

    def download_kubernetes_logs(self, name=None, raise_on_error=False):
        """Try to download kubernetes logs, print exception to the log

        :param name: archive name (without .tar.gz)
        """
        try:
            self._download_kubernetes_logs(name=name)
        except Exception as e:
            LOG.error(e)
            LOG.error("Failed to download kubernetes logs for cluster <{0}>"
                      .format(self.name))
            if raise_on_error:
                raise

    def get_boot_time_dict(self, exclude_bastion=False, lcm_machines=None):
        """Return a dict of lcm machines: name -> boot time"""
        if not lcm_machines:
            lcm_machines = self.get_lcmmachines()
        boot_time = dict()
        for lcmm in lcm_machines:
            if exclude_bastion and lcmm.data.get('spec', {}).get('type') == 'bastion':
                continue
            lcmm_status = lcmm.data.get('status') or {}
            boot_time[lcmm.name] = lcmm_status.get('hostInfo', {}).get('bootTime')
        return boot_time

    def get_runtime_dict(self, exclude_bastion=False, machines=None):
        """Return a dict of machines: name -> runtime"""
        if not machines:
            machines = self.get_machines()
        runtime = dict()
        for machine in machines:
            if exclude_bastion and machine.lcmmachine.data.get('spec', {}).get('type') == 'bastion':
                continue
            assert machine.runtime, f'No machine runtime found for machine {machine}, check cluster version'
            runtime[machine.name] = machine.runtime.split(':')[0]
        return runtime

    def _download_kubernetes_logs(self, name=None):
        """Download pod logs and describe kubernetes resources

        Based on:
        https://github.com/openstack/openstack-helm-infra/tree/master/\
            roles/describe-kubernetes-objects/tasks
        https://github.com/openstack/openstack-helm-infra/tree/master/\
            roles/gather-pod-logs/tasks
        """
        if not name:
            if self.is_management:
                name = "mgmt_k8s_{0}".format(self.name)
            else:
                name = "child_k8s_{0}".format(self.name)

        control_nodes = self.get_machines(machine_type='control',
                                          machine_status='Ready')
        if not control_nodes:
            raise Exception("No control nodes in 'Ready' status found "
                            "in the cluster {0}".format(self.name))

        ctl = control_nodes[0]
        kube_dir = "{0}_{1}".format(name, self.cluster_status)
        ctl.run_cmd("sudo rm -rf /tmp/{0}* || true;"
                    "mkdir /tmp/{1}/;"
                    "sudo cp /root/ucp-bundle/kube.yml /tmp/{1}/kubeconfig &&"
                    " sudo chmod o+r /tmp/{1}/kubeconfig;"
                    .format(name, kube_dir),
                    verbose=True, check_exit_code=True)

        # TODO: In case vSphere supports Ubuntu need to find another way to
        # detect host OS
        if ctl.provider == "vsphere":
            ctl.run_cmd("sudo yum install -y rsync", verbose=False)
        else:
            ctl.run_cmd("sudo apt-get -qq install -y rsync", verbose=False)

        logs_dir_command = ("""
          set -ex
          export LOGS_DIR=/tmp/{0}
          export KUBECONFIG=/tmp/{1}/kubeconfig
        """.format(kube_dir, kube_dir))

        pod_logs_commands = ("""
PARALLELISM_FACTOR=4
function get_namespaces () {
  kubectl get namespaces -o name | awk -F '/' '{ print $NF }'
}
function get_pods () {
  NAMESPACE=$1
  kubectl get pods -n ${NAMESPACE} -o name | \
  awk -F '/' '{ print $NF }' | \
  xargs -L1 -P 1 -I {} echo ${NAMESPACE} {}
}
export -f get_pods
function get_pod_logs () {
  NAMESPACE=${1% *}
  POD=${1#* }
  INIT_CONTAINERS=$(kubectl get pod/$POD -n ${NAMESPACE} \
      -o jsonpath={.spec.initContainers[*].name})
  CONTAINERS=$(kubectl get pod/$POD -n ${NAMESPACE} \
      -o jsonpath={.spec.containers[*].name})
  for CONTAINER in ${INIT_CONTAINERS} ${CONTAINERS}; do
    echo "${NAMESPACE}/${POD}/${CONTAINER}"
    mkdir -p "${LOGS_DIR}/pod-logs/${NAMESPACE}/${POD}"
    mkdir -p "${LOGS_DIR}/pod-logs/failed-pods/${NAMESPACE}/${POD}"
    kubectl logs ${POD} -n ${NAMESPACE} -c ${CONTAINER} \
     > "${LOGS_DIR}/pod-logs/${NAMESPACE}/${POD}/${CONTAINER}.txt"
    kubectl logs --previous ${POD} -n ${NAMESPACE} -c ${CONTAINER} \
     > "${LOGS_DIR}/pod-logs/failed-pods/${NAMESPACE}/${POD}/${CONTAINER}.txt"\
     || true
  done
}
export -f get_pod_logs
get_namespaces | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
    bash -c 'get_pods "$@"' _ {} | \
  xargs -r -n 2 -P ${PARALLELISM_FACTOR} -I {} \
    bash -c 'get_pod_logs "$@"' _ {} \
  > ${LOGS_DIR}/dump_pod_logs_commands.log 2>&1
  """)

        non_namespaced_objects_commands = ("""
export OBJECT_TYPES=$(kubectl api-resources -o name --namespaced=false)
export PARALLELISM_FACTOR=4

function list_objects () {
  echo ${OBJECT_TYPES} | xargs -d ' ' -I {} -P1 -n1 bash -c 'echo "$@"' _ {}
}
export -f list_objects

function name_objects () {
  export OBJECT=$1
  kubectl get ${OBJECT} -o name | xargs -L1 -I {} -P1 -n1 \
      bash -c 'echo "${OBJECT} ${1#*/}"' _ {}
}
export -f name_objects

function get_objects () {
  input=($1)
  export OBJECT=${input[0]}
  export NAME=${input[1]#*/}
  echo "${OBJECT}/${NAME}"
  DIR="${LOGS_DIR}/objects/cluster/${OBJECT}"
  mkdir -p ${DIR}
  kubectl get ${OBJECT} ${NAME} -o yaml > "${DIR}/${NAME}.yaml"
  kubectl describe ${OBJECT} ${NAME} > "${DIR}/${NAME}.txt"
}
export -f get_objects

list_objects | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
      bash -c 'name_objects "$@"' _ {} | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
      bash -c 'get_objects "$@"' _ {} \
  > ${LOGS_DIR}/dump_non_namespaced_objects_commands.log 2>&1""")

        namespaced_objects_commands = ("""
export OBJECT_TYPES=$(kubectl api-resources -o name --namespaced=true)
export PARALLELISM_FACTOR=4
function get_namespaces () {
  kubectl get namespaces -o name | awk -F '/' '{ print $NF }'
}

function list_namespaced_objects () {
  export NAMESPACE=$1
  echo ${OBJECT_TYPES} | xargs -d ' ' -I {} -P1 -n1 \
      bash -c 'echo "${NAMESPACE} $@"' _ {}
}
export -f list_namespaced_objects

function name_objects () {
  input=($1)
  export NAMESPACE=${input[0]}
  export OBJECT=${input[1]}
  kubectl get -n ${NAMESPACE} ${OBJECT} -o name | xargs -L1 -I {} -P1 -n1 \
      bash -c 'echo "${NAMESPACE} ${OBJECT} $@"' _ {}
}
export -f name_objects

function get_objects () {
  input=($1)
  export NAMESPACE=${input[0]}
  export OBJECT=${input[1]}
  export NAME=${input[2]#*/}
  echo "${NAMESPACE}/${OBJECT}/${NAME}"
  DIR="${LOGS_DIR}/objects/namespaced/${NAMESPACE}/${OBJECT}"
  mkdir -p ${DIR}
  kubectl get -n ${NAMESPACE} ${OBJECT} ${NAME} -o yaml > "${DIR}/${NAME}.yaml"
  kubectl describe -n ${NAMESPACE} ${OBJECT} ${NAME} > "${DIR}/${NAME}.txt"
}
export -f get_objects

get_namespaces | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
      bash -c 'list_namespaced_objects "$@"' _ {} | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
      bash -c 'name_objects "$@"' _ {} | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
      bash -c 'get_objects "$@"' _ {} \
  > ${LOGS_DIR}/dump_namespaced_objects_commands.log 2>&1""")

        get_objects_commands = ("""
export OBJECT_TYPES=$(kubectl api-resources -o name)
export PARALLELISM_FACTOR=4

function list_resources () {
  echo ${OBJECT_TYPES} | xargs -d ' ' -I {} -P1 -n1 bash -c 'echo "$@"' _ {}
}
export -f list_resources

function get_objects () {
  export OBJECT=$1
  kubectl get ${OBJECT} -o wide --all-namespaces > \
      ${LOGS_DIR}/get_${OBJECT}.txt || true
  if [ ! -s ${LOGS_DIR}/get_${OBJECT}.txt ]; then
      rm ${LOGS_DIR}/get_${OBJECT}.txt
  fi
}
export -f get_objects

list_resources | \
  xargs -r -n 1 -P ${PARALLELISM_FACTOR} -I {} \
      bash -c 'get_objects "$@"' _ {} \
  > ${LOGS_DIR}/dump_get_objects_commands.log 2>&1""")

        res = ctl.run_cmd('hostname', verbose=False, verbose_info=True)
        LOG.info("Get kubernetes logs from the cluster <{0}> on Machine <{1}> "
                 "with local hostname <{2}>"
                 .format(self.name, ctl.name, res.stdout_str))

        # Download pod logs
        logs_target = "{0}/dump_pod_logs_commands.log".format(kube_dir)
        ret = ctl.run_cmd("{0}{1}"
                          .format(logs_dir_command, pod_logs_commands),
                          verbose=False, check_exit_code=False)
        if ret.exit_code != 0:
            LOG.error("Failed collecting pod logs from cluster <{0}> with "
                      "exit code <{1}>, see {2} for details"
                      .format(self.name, ret.exit_code, logs_target))

        # Download non-namespaced objects
        logs_target = "{0}/dump_non_namespaced_objects_commands.log".format(
            kube_dir)
        ret = ctl.run_cmd("{0}{1}"
                          .format(logs_dir_command,
                                  non_namespaced_objects_commands),
                          verbose=False, check_exit_code=False)
        if ret.exit_code != 0:
            LOG.error("Failed collecting non-namespaced k8s objects from "
                      "cluster <{0}> with exit code <{1}>, see {2} for details"
                      .format(self.name, ret.exit_code, logs_target))

        # Download namespaced objects
        logs_target = "{0}/dump_namespaced_objects_commands.log".format(
            kube_dir)
        ret = ctl.run_cmd("{0}{1}"
                          .format(logs_dir_command,
                                  namespaced_objects_commands),
                          verbose=False, check_exit_code=False)
        if ret.exit_code != 0:
            LOG.error("Failed collecting namespaced k8s objects from cluster "
                      "<{0}> with exit code <{1}>, see {2} for details"
                      .format(self.name, ret.exit_code, logs_target))

        # List all objects
        logs_target = "{0}/dump_get_objects_commands.log".format(
            kube_dir)
        ret = ctl.run_cmd("{0}{1}"
                          .format(logs_dir_command,
                                  get_objects_commands),
                          verbose=False, check_exit_code=False)
        if ret.exit_code != 0:
            LOG.error("Failed to get k8s object lists from cluster "
                      "<{0}> with exit code <{1}>, see {2} for details"
                      .format(self.name, ret.exit_code, logs_target))

        # Archive the logs
        ctl.run_cmd(
            "tar --absolute-names --warning=no-file-changed"
            " -C /tmp/{0} -czf /tmp/{0}.tar.gz ./".format(kube_dir),
            verbose=False)

        # Copy kubernetes logs from k8s master node to the bastion node
        # if the bastion node is not the same node as k8s master
        bastion_remote = self.get_bastion_remote()
        if self.bastion_ip != ctl.public_ip:
            priv_key_path = os.path.join("/home",
                                         settings.KAAS_CHILD_CLUSTER_SSH_LOGIN,
                                         ".ssh", self.name)
            LOG.info("Copy kubernetes logs from {0} to the bastion node {1}"
                     .format(ctl.name, self.bastion_ip))
            bastion_remote.check_call(
                "rsync -aruvz -e 'ssh -o StrictHostKeyChecking=no -i {0}' "
                "{1}:/tmp/{2}.tar.gz "
                "/tmp/{2}.tar.gz".format(priv_key_path,
                                         ctl.internal_ip or ctl.public_ip,
                                         kube_dir),
                verbose=False)

        # Download the logs
        artifact_name = os.path.join(settings.ARTIFACTS_DIR,
                                     "{0}.tar.gz".format(kube_dir))
        bastion_remote.download(destination="/tmp/{0}.tar.gz"
                                .format(kube_dir), target=artifact_name)
        LOG.info("Kubernetes logs were downloaded to {0}"
                 .format(os.path.join(artifact_name)))

    def download_system_logs(self, name=None, raise_on_error=False):
        """Try to download system logs, print exception to the log

        :param name: archive name (without .tar.gz)
        """
        try:
            self._download_system_logs(name=name)
        except Exception as e:
            LOG.error(e)
            LOG.error("Failed to download system logs for cluster <{0}>"
                      .format(self.name))
            if raise_on_error:
                raise

    def _download_system_logs(self, name=None):
        """Collect system logs from all cluster nodes

        1. rsync logs from cluster nodes to the node with bastion_ip
        2. tar.gz logs on bastion_ip node and download the archive
        """
        if not name:
            if self.is_management:
                name = "mgmt_system_logs_{0}".format(self.name)
            else:
                name = "child_system_logs_{0}".format(self.name)

        dump_commands = (
            r"mkdir /tmp/{0}/;"
            r"rsync -aruv --exclude journal /var/log /tmp/{0}/;"
            r"rsync -aruv --exclude /etc/*shadow* /etc /tmp/{0}/;"
            r"cp -r /root/lcm-ansible* /tmp/{0}/;"
            r"hostname -f > /tmp/{0}/hostname.txt;"
            r"dpkg -l > /tmp/{0}/dump_dpkg_l.txt ||"
            r" rpm -qa > /tmp/{0}/dump_rpm_l.txt;"
            r"df -h > /tmp/{0}/dump_df.txt;"
            r"free > /tmp/{0}/dump_free.txt;"
            r"cat /proc/cpuinfo > /tmp/{0}/dump_cpuinfo.txt;"
            r"mount > /tmp/{0}/dump_mount.txt;"
            r"blkid -o list > /tmp/{0}/dump_blkid_o_list.txt;"
            r"iptables -t nat -S > "
            r"  /tmp/{0}/dump_iptables_nat.txt;"
            r"iptables -S > /tmp/{0}/dump_iptables.txt;"
            r"ps auxwwf > /tmp/{0}/dump_ps.txt;"
            r"vgdisplay > /tmp/{0}/dump_vgdisplay.txt;"
            r"lvdisplay > /tmp/{0}/dump_lvdisplay.txt;"
            r"ip a > /tmp/{0}/dump_ip_a.txt;"
            r"ip r > /tmp/{0}/dump_ip_r.txt;"
            r"netstat -anp > /tmp/{0}/dump_netstat.txt;"
            r"brctl show > /tmp/{0}/dump_brctl_show.txt ||"
            r" bridge link show > /tmp/{0}/dump_bridge_show.txt;"
            r"arp -an > /tmp/{0}/dump_arp.txt;"
            r"uname -a > /tmp/{0}/dump_uname_a.txt;"
            r"lsmod > /tmp/{0}/dump_lsmod.txt;"
            r"cat /proc/interrupts > "
            r"  /tmp/{0}/dump_interrupts.txt;"
            r"cat /etc/*-release > /tmp/{0}/dump_release.txt;"
            r"timedatectl status > /tmp/{0}/timedata.txt;"
            r"systemctl --state=failed --no-pager --all > "
            r"  /tmp/{0}/systemctl_failed.txt;"
            r"journalctl -p err --no-pager > /tmp/{0}/journalctl_errors.txt;")

        dump_controller_commands = (
            r"etcdctl --key-file /etc/kubernetes/ssl/apiserver-etcd-client.key"
            r"      --cert-file /etc/kubernetes/ssl/apiserver-etcd-client.crt"
            r"      --ca-file /etc/kubernetes/ssl/etcd/ca.crt"
            r"      --endpoint=https://127.0.0.1:2379"
            r"      cluster-health > /tmp/{0}/etcd_cluster_health.txt;"
            r"calicoctl node status > /tmp/{0}/calico_node_status.txt;")

        bastion_remote = self.get_bastion_remote()
        # Prepare target dir on the bastion node
        cluster_dir = "{0}_{1}".format(name, self.cluster_status)
        bastion_remote.check_call("sudo rm -rf /tmp/{0}* || true;"
                                  "mkdir /tmp/{1}"
                                  .format(name, cluster_dir),
                                  verbose=False)
        # Set private key path on the bastion node
        priv_key_path = os.path.join("/home",
                                     settings.KAAS_CHILD_CLUSTER_SSH_LOGIN,
                                     ".ssh", self.name)

        cluster_machines = self.get_machines()
        for m in cluster_machines:
            res = m.run_cmd('hostname', verbose=False, verbose_info=True)
            node_dir = "{0}_{1}_{2}".format(m.machine_type, m.name,
                                            m.machine_status)
            LOG.info("Get system logs from the Machine <{0}> with local "
                     "hostname <{1}>"
                     .format(m.name, res.stdout_str))
            # Re-create the directory to store the artifacts
            m.run_cmd("sudo rm -rf /tmp/{0}_{1}_* || true;"
                      "mkdir /tmp/{2}".format(m.machine_type,
                                              m.name, node_dir),
                      verbose=False)
            m.run_cmd("sudo apt-get -qq install -y rsync", verbose=False)

            # Collect node logs. Do not fail on errors
            m.run_cmd('sudo bash -cx "{0}" > /tmp/{1}/dump_commands.log 2>&1'
                      .format(dump_commands.format(node_dir), node_dir),
                      verbose=False, check_exit_code=False)
            if m.is_machine_type('control'):
                # run k8s controller-specific commands
                m.run_cmd('sudo bash -cx "{0}" > '
                          '/tmp/{1}/dump_controller_commands.log 2>&1'
                          .format(dump_controller_commands.format(node_dir),
                                  node_dir),
                          verbose=False, check_exit_code=False)
            m.run_cmd('sudo chmod o+rx -R /tmp/{0}/*'.format(node_dir),
                      verbose=False)

            # Copy system logs from nodes to the bastion node
            # Try to use internal IP first
            bastion_remote.check_call(
                "rsync -aruvz -e 'ssh -o StrictHostKeyChecking=no -i {3}' "
                "{0}:/tmp/{1} /tmp/{2}/".format(m.internal_ip or m.public_ip,
                                                node_dir, cluster_dir,
                                                priv_key_path),
                verbose=False)
        # Archive the logs
        bastion_remote.check_call(
            "tar --absolute-names --warning=no-file-changed"
            " -C /tmp/{0} -czf /tmp/{0}.tar.gz ./".format(cluster_dir),
            verbose=False)
        # Download the logs
        artifact_name = os.path.join(settings.ARTIFACTS_DIR,
                                     "{0}.tar.gz".format(cluster_dir))
        bastion_remote.download(destination="/tmp/{0}.tar.gz"
                                .format(cluster_dir), target=artifact_name)
        LOG.info("System logs were downloaded to {0}"
                 .format(os.path.join(artifact_name)))

    def get_ceph_storage_nodes(self):
        """
        Total ceph-osd pods must be equal to Total osd disks in cluster
        """
        kubectl_client = self.k8sclient
        # Name and namespace hardcoded in rook
        rookcephcls = \
            [x for x in
             kubectl_client.rookcephclusters.list(namespace='rook-ceph')
             if x.name == 'rook-ceph']
        if not rookcephcls:
            LOG.info("Rookcephcluster 'rook-ceph' was not found")
            return []
        else:
            rookceph = rookcephcls[0].read().to_dict()
            return rookceph['spec']['storage']['nodes']

    def _render_expected_pods_template(self):
        """Render the template with expected pods for the cluster release"""

        all_machines = self.get_machines()

        ctl_nodes = [x.data['metadata']['name'] for x in all_machines
                     if x.is_machine_type('control')]
        num_ctl_nodes = len(ctl_nodes)

        num_worker_nodes = len(
            [x for x in all_machines
             if x.is_machine_type('worker')])
        num_storage_nodes = len(
            [x for x in all_machines
             if x.is_machine_type('storage')]
        )
        worker_or_storage_nodes = [
            x.data['metadata']['name'] for x in all_machines
            if x.is_machine_type('worker') or x.is_machine_type('storage')]
        # in dedicatedControlPlane=False the storage might be collocated
        # with ctl
        num_worker_or_storage_nodes = len(
            set(worker_or_storage_nodes) - set(ctl_nodes))
        if num_storage_nodes == 0:
            # for mgmt cluster we won't have machines w/ 'storage'
            # labels so let's set default value to num_ctl_nodes (3)
            LOG.debug(
                f"Number of storage nodes is set to {num_ctl_nodes} "
                "(equals to number of control nodes)"
            )
            num_storage_nodes = num_ctl_nodes
        num_nodes = len(all_machines)
        machines_info = \
            self.__manager.api.kaas_machines.list_raw(
                namespace=self.namespace).items
        os_compute_label = {'key': 'openstack-compute-node',
                            'value': 'enabled'}
        os_computes_num = len([
            x for x in machines_info
            if os_compute_label in x.spec['providerSpec']
            ['value'].get('nodeLabels', [])
        ])

        clusterrelease_version = (
            self.data["spec"]["providerSpec"]["value"].get("release", "") or
            self.data["status"]["providerStatus"].get(
                "releaseRefs", {}).get("current", {}).get("name", "")
        ).replace("-rc", "")

        if self.sl_ha_enabled():
            LOG.debug("HA in SL is enabled")
            alertmanager_num = 2
            prometheus_server_num = 2
            elasticsearch_master_num = 3
            patroni_num = 3
            fluentd_notifications_num = 2
            iam_proxy_num = 2
            kube_state_metrics_num = 2
            blackbox_exporter_num = 2
        else:
            LOG.debug("HA in SL is disabled")
            alertmanager_num = 1
            prometheus_server_num = 1
            elasticsearch_master_num = 1
            patroni_num = 1
            fluentd_notifications_num = 1
            iam_proxy_num = 1
            kube_state_metrics_num = 1
            blackbox_exporter_num = 1

        if self.logging_enabled():
            LOG.debug("Logging in SL is enabled")
            logging_enabled = True
        else:
            LOG.debug("Logging in SL is disabled")
            logging_enabled = False

        if self.openstack_lma_enabled():
            LOG.debug("Openstack monitoring in SL is enabled")
            openstack_lma_enabled = True
        else:
            LOG.debug("Openstack monitoring in SL is disabled")
            openstack_lma_enabled = False

        if self.tf_monitoring_enabled():
            LOG.debug("TF monitoring in SL is enabled")
            tf_monitoring_enabled = True
        else:
            LOG.debug("TF monitoring in SL is disabled")
            tf_monitoring_enabled = False

        if self.pushgateway_enabled():
            LOG.debug("Pushgateway in SL is enabled")
            pushgateway_enabled = True
        else:
            LOG.debug("Pushgateway in SL is disabled")
            pushgateway_enabled = False

        tf_v2_enabled = False
        if self.tf_enabled():
            LOG.debug("TF is enabled")
            tf_enabled = True
            if self.tf_v2_enabled():
                LOG.debug("TF is enabled")
                tf_v2_enabled = True
        else:
            LOG.debug("This is OVS deployment")
            tf_enabled = False

        if self.tf_analytics_enabled():
            LOG.debug("TF analytics is enabled")
            tf_analytics_enabled = True
        else:
            LOG.debug("TF analytics is disabled")
            tf_analytics_enabled = False

        if self.is_os_deployed():
            LOG.debug("OS is deployed")
            os_deployed = True
        else:
            LOG.debug("OS is not deployed")
            os_deployed = False

        if self.netchecker_enabled():
            LOG.debug("NetChecker is enabled")
            netchecker_enabled = True
        else:
            LOG.debug("NetChecker is disabled")
            netchecker_enabled = False

        if (self.workaround.skip_kaascephcluster_usage()
                and self.is_child
                and self.is_mosk):
            miracephcluster = self.get_miracephcluster()
            cephcluster = None
        elif self.provider == utils.Provider.aws and self.is_mosk:
            miracephcluster = self.get_miracephcluster()
            cephcluster = None
        else:
            cephcluster = self.get_cephcluster()
            miracephcluster = None

        mon = []
        osd_nodes_num = 0
        crashcollector_num = 0
        rgw_pods = []
        rgw_pods_num = 0
        mds_pods_num = 0
        ceph_osd = 0
        if cephcluster or miracephcluster:
            if cephcluster:
                cephspecs = cephcluster.data['spec']['cephClusterSpec']
                if cephspecs.get('nodeGroups', {}):
                    for group_name in cephspecs['nodeGroups']:
                        if 'mon' in cephspecs['nodeGroups'][group_name]['spec'].get('roles', []):
                            mon.extend(cephspecs['nodeGroups'][group_name]['nodes'])
                else:
                    mon = [node for node in cephspecs['nodes'].values() if 'mon' in node.get('roles', [])]
            else:
                cephspecs = miracephcluster.data['spec']
                mon = [node for node in cephspecs['nodes'] if 'mon' in node.get('roles', [])]

            ceph_nodes = self.get_ceph_storage_nodes()
            for node in ceph_nodes:
                # calculate nodes with storage osd config
                node_devices = node.get('devices', [])
                if len(node_devices) > 0:
                    osd_nodes_num += 1
                for device in node_devices:
                    osdsPerDevice = device.get('config', {}).get('osdsPerDevice', '1')
                    try:
                        ceph_osd += int(osdsPerDevice)
                    except ValueError:
                        raise Exception("Incorrect osdsPerDevice value in KaaSCephCluster spec, expected integer")

            crashcollector_num = len(cephspecs['nodes'])
            rgwspecs = cephspecs.get('rgw', {})
            if cephspecs.get('objectStorage') is not None:
                rgwspecs = cephspecs.get('objectStorage', {}).get('rgw', {})

            rgw_pods_num = rgwspecs.get('gateway', {}).get('instances', 0)
            for i in range(0, rgw_pods_num):
                store_name = rgwspecs['name']
                rgw_pods.append(
                    f"rook-ceph-rgw-{store_name}-{string.ascii_letters[i]}")

            cephfs_specs = cephspecs.get('sharedFilesystem', {}).get('cephFS', [])
            for cephfs in cephfs_specs:
                # rook doubles active count and create pods
                mds_pods_num += cephfs.get('metadataServer', {}).get('activeCount', 0) * 2
        # ntpd constant
        ntpd_num = 2

        mgr_nodes = 2

        options = {
            'logging_enabled': logging_enabled,
            'openstack_lma_enabled': openstack_lma_enabled,
            'tf_monitoring_enabled': tf_monitoring_enabled,
            'pushgateway_enabled': pushgateway_enabled,
            'num_ctl_nodes': num_ctl_nodes,
            'lma_child_enabled': self.is_sl_enabled(),
            'num_worker_nodes': num_worker_nodes,
            'num_storage_nodes': num_storage_nodes,
            'num_worker_or_storage_nodes': num_worker_or_storage_nodes,
            'num_nodes': num_nodes,
            'ntpd_num': ntpd_num,
            'num_ceph_osd_pods': ceph_osd,
            'crashcollector_num': crashcollector_num,
            'num_ceph_mgr_nodes': mgr_nodes,
            'num_ceph_mon_nodes': len(mon),
            'num_ceph_osd_nodes': osd_nodes_num,
            'num_ceph_rgw_pods': rgw_pods_num,
            'num_ceph_mds_pods': mds_pods_num,
            'rgw_enabled': rgw_pods_num > 0,
            'cephfs_enabled': mds_pods_num > 0,
            'prometheus_alertmanager_num': 2,
            'alertmanager_num': alertmanager_num,
            'prometheus_server_num': prometheus_server_num,
            'elasticsearch_master_num': elasticsearch_master_num,
            'patroni_num': patroni_num,
            'fluentd_notifications_num': fluentd_notifications_num,
            'iam_proxy_num': iam_proxy_num,
            'kube_state_metrics_num': kube_state_metrics_num,
            'blackbox_exporter_num': blackbox_exporter_num,
            'rgw_pods': rgw_pods,
            'os_computes_num': os_computes_num,
            'tf_enabled': tf_enabled,
            'tf_v2_enabled': tf_v2_enabled,
            'tf_analytics_enabled': tf_analytics_enabled,
            'os_deployed': os_deployed,
            'aio_mgmt': settings.KAAS_BM_AIO_ON_MGMT or settings.KAAS_AIO_CLUSTER,
            'aio_child': settings.KAAS_BM_AIO_ON_CHILD or settings.KAAS_AIO_CLUSTER,
            'ceph_enabled': settings.MOSK_CHILD_DEPLOY_CEPH,
            'ucp_byo_cloud_provider': settings.UCP_BYO_CLOUD_PROVIDER,
            'netchecker_enabled': netchecker_enabled,
        }

        templates = template_utils.render_template(
            settings.EXPECTED_PODS_TEMPLATES_DIR +
            clusterrelease_version +
            '.yaml', options
        )

        json_body = yaml.load(templates, Loader=yaml.SafeLoader)
        return json_body

    def _render_expected_rolebindings_template(self):
        """Render the template with expected rolebindins for the cluster release"""

        clusterrelease_version = \
            self.data['spec']['providerSpec'][
                'value']['release'].replace("-rc", "")

        if self.logging_enabled():
            LOG.debug("Logging in SL is enabled")
            logging_enabled = True
        else:
            LOG.debug("Logging in SL is disabled")
            logging_enabled = False

        if self.tf_enabled():
            LOG.debug("TF is enabled")
            tf_enabled = True
        else:
            LOG.debug("This is OVS deployment")
            tf_enabled = False

        machines_names = self.get_machines_names()
        regions = []
        regions.extend(
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.azure.provider_name} for r in
             self.azure_regions()] +
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.vsphere.provider_name} for r in
             self.vsphere_regions()] +
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.equinixmetal.provider_name} for r in
             self.equinix_regions()] +
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.equinixmetalv2.provider_name} for r in
             self.equinixmetalv2_regions()] +
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.aws.provider_name} for r in
             self.aws_regions()] +
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.openstack.provider_name} for r in
             self.openstack_regions()] +
            [{"name": r[0], "cluster-name": r[1], "provider": utils.Provider.baremetal.provider_name} for r in
             self.baremetal_regions()]
        )

        options = {
            'cluster_name': self.name,
            'logging_enabled': logging_enabled,
            'lma_child_enabled': self.is_sl_enabled(),
            'tf_enabled': tf_enabled,
            'machines_names': machines_names,
            'regions': regions,
        }

        templates = template_utils.render_template(
            settings.EXPECTED_ROLEBINDINGS_TEMPLATES_DIR +
            clusterrelease_version +
            '.yaml', options
        )

        json_body = yaml.load(templates, Loader=yaml.SafeLoader)
        return json_body

    @property
    def expected_pods(self):
        if not self.__expected_pods:
            self._refresh_expected_objects()
        return self.__expected_pods

    @property
    def expected_docker_objects(self):
        if not self.__expected_docker_objects:
            self._refresh_expected_objects()
        return self.__expected_docker_objects

    def _refresh_expected_objects(self):
        """Force update expected objects after cluster scale"""
        self.__expected_pods, self.__expected_docker_objects = \
            self.get_expected_objects()

    def get_expected_objects(self, exclude_pods=None):
        """Returns list of expected pods (and their quantity for the cluster)
           plus all docker objects
        """
        if self.is_regional:
            cluster_type = 'regional'
        elif self.is_management:
            cluster_type = 'management'
        else:
            cluster_type = 'child'

        json_body = self._render_expected_pods_template()
        all_pods = {}
        all_objects = {}

        provider_name = self.provider.provider_name

        if exclude_pods:
            LOG.warning("These pods will be excluded from "
                        "expected pod list {}".format(exclude_pods))
        else:
            exclude_pods = []
        for ns in json_body[cluster_type][provider_name]:
            pods = json_body[cluster_type][provider_name][ns]
            pods = {k: v for k, v in pods.items() if k not in exclude_pods}
            if ns not in all_pods:
                all_pods[ns] = pods
            else:
                all_pods[ns].update(pods)

        for component in self.get_components():
            if component in json_body[cluster_type]['components']:
                for ns in json_body[cluster_type]['components'][component]:
                    pods = json_body[cluster_type]['components'][component][ns]
                    pods = {k: v for k, v in pods.items()
                            if k not in exclude_pods}
                    if ns not in all_pods:
                        all_pods[ns] = pods
                    else:
                        all_pods[ns].update(pods)
            else:
                LOG.error("Component {} was not found in "
                          "expected pod templates".format(component))

        # fetching docker objects
        cluster_type = f"docker_{cluster_type}"
        for opt in json_body[cluster_type][provider_name]:
            objects = json_body[cluster_type][provider_name][opt]
            if opt not in all_objects:
                all_objects[opt] = objects
            else:
                all_objects[opt].update(objects)
        LOG.info("Expected docker objects:\n{0}"
                 .format(json.dumps(all_objects, indent=4)))

        LOG.info(json.dumps(all_pods, indent=4))
        return (all_pods, all_objects)

    def get_expected_rolebindings(self, exclude_rolebindings: list = None) -> dict:
        """Return expect rolebindings objects"""

        cluster_type = self._cluster_type
        expected_rolebindings = self._render_expected_rolebindings_template()
        ret_rolebinds = {}

        provider = self.provider

        if exclude_rolebindings:
            LOG.warning("These rolebindigs will be excluded from "
                        "expected list %s", exclude_rolebindings)
        else:
            exclude_rolebindings = []

        for ns in expected_rolebindings[cluster_type][provider.provider_name]:
            rolebindings = expected_rolebindings[cluster_type][provider.provider_name][ns]
            rolebindings = {k: v for k, v in rolebindings.items() if k not in exclude_rolebindings}
            if ns not in ret_rolebinds:
                ret_rolebinds[ns] = rolebindings
            else:
                ret_rolebinds[ns].update(rolebindings)

        for component in self.get_components():
            if component in expected_rolebindings[cluster_type]['components']:
                for ns in expected_rolebindings[cluster_type]['components'][component]:
                    rolebindings = expected_rolebindings[cluster_type]['components'][component][ns]
                    rolebindings = {k: v for k, v in rolebindings.items() if k not in exclude_rolebindings}
                    if ns not in ret_rolebinds:
                        ret_rolebinds[ns] = rolebindings
                    else:
                        ret_rolebinds[ns].update(rolebindings)
            else:
                LOG.error("Component %s was not found in expected rolebindings templates", component)

        LOG.info("Expected rolebindings: %s", json.dumps(ret_rolebinds, indent=4))

        return ret_rolebinds

    def get_expected_kernel_version(self, target_cr=None):
        """Return expect kernel version"""

        if not target_cr:
            clusterrelease = self.clusterrelease_version
        else:
            clusterrelease = target_cr
        cluster_type = self._cluster_type
        provider = self.provider
        template_file = os.path.join(settings.EXPECTED_KERNEL_VERSIONS_DIR,
                                     clusterrelease.replace("-rc", "") + '.yaml')
        exp_kernels_data = yaml.safe_load(
            template_utils.render_template(template_file))

        expected_versions_map = exp_kernels_data.get(cluster_type, {}).get(
            provider.provider_name, {})
        assert expected_versions_map, \
            "Invalid or empty expected info in yaml file for " + \
            f"clusterrelese: {clusterrelease}, provider: {provider}" + \
            f"({cluster_type})"

        versions_json = json.dumps(expected_versions_map)

        LOG.info(f"Expected kernel versions:\n{versions_json}"
                 f" for provider: {provider}, clusterrelease: {clusterrelease}\n")
        return expected_versions_map

    def get_components(self):
        """Returns all components defined for current cluster
          (istio, harbor, etc)
        """
        release_names = []
        available_releases = self.spec['providerSpec']['value'].get(
            'helmReleases', [])
        if available_releases:
            release_names = [x['name'] for x in available_releases
                             if x.get('enabled', True)]
        if self.provider == utils.Provider.equinixmetal and not \
                self.is_regional and not self.is_management:
            release_names.append("ceph-controller")
            release_names.append("storage-discovery")
        elif self.provider == utils.Provider.equinixmetalv2:
            release_names.append("ceph-controller")
        if self.provider == utils.Provider.aws and self.is_child and self.is_mosk:
            release_names.append("ceph-controller")
            release_names.append("openstack-redis")

        return release_names

    def is_sl_enabled(self):
        """Check if StackLight enabled for the cluster"""
        return 'stacklight' in self.get_components()

    def sl_ha_enabled(self):
        if 'stacklight' not in self.get_components():
            LOG.warning("Stacklight component is not enabled")
            return None

        hb = self.get_helmbundle().data
        for chart in hb['spec']['releases']:
            if chart['name'] == 'stacklight':
                if 'highAvailabilityEnabled' in chart['values']:
                    value = bool(chart['values']['highAvailabilityEnabled'])
                    LOG.debug(f"Found StackLight chart with 'highAvailabilityEnabled' value: {value}")
                    return value
                else:
                    LOG.warning("Stacklight chart don't have 'highAvailabilityEnabled' value")
                    break
        else:
            LOG.warning("Stacklight chart not found in the helmbundle spec")

        if self.is_management or self.is_regional:
            LOG.info("Stacklight chart not found, assuming HA is enabled for Management/Regional cluster")
            return True

        LOG.warning("No Stacklight chart data found for a child cluster in the helmbundle, set HA to False")
        return None

    @lru_cache()
    def tf_enabled(self):
        if self.is_management:
            return False
        os_deployed = self.is_os_deployed()
        if os_deployed:
            osdpl = self.k8sclient.openstackdeployment.get(
                name=settings.OSH_DEPLOYMENT_NAME,
                namespace=settings.OSH_NAMESPACE)
            backend = osdpl.data.get('spec').get('features', {}).get('neutron', {}).get('backend', '')
            return 'tungstenfabric' in backend
        return False

    @lru_cache()
    def tf_v2_enabled(self):
        if self.is_management or self.is_regional:
            return False
        if self.tf_enabled():
            child_kubeconfig_name, child_kubeconfig = self.get_kubeconfig_from_secret()
            with open('child_conf_tf', 'w') as f:
                f.write(child_kubeconfig)
            tf_manager = tungstenfafric_manager.TFManager(kubeconfig='child_conf_tf')
            return tf_manager.apiv2
        else:
            return False

    def aws_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.aws])

        return []

    def azure_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.azure])

        return []

    def vsphere_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.vsphere])

        return []

    def equinix_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.equinixmetal])

        return []

    def equinixmetalv2_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.equinixmetalv2])

        return []

    def openstack_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.openstack])

        return []

    def baremetal_regions(self):
        if self.is_management:
            clusters = self.__manager.get_clusters(namespace=settings.REGION_NAMESPACE)
            regions = [c for c in clusters if c.is_regional]

            return set([(r.region_name, r.name) for r in regions if r.provider == utils.Provider.baremetal])

        return []

    def get_machines_names(self):
        machines = self.get_machines()
        names = [n.name for n in machines]
        return names

    def logging_enabled(self):
        if 'stacklight' not in self.get_components():
            LOG.warning("Stacklight component is not enabled")
            return None

        hb = self.get_helmbundle().data
        for chart in hb['spec']['releases']:
            if chart['name'] == 'stacklight':
                if 'logging' in chart['values']:
                    return bool(chart['values']['logging']['enabled'])
                # in old versions parameter is absent, but logging is enabled
                else:
                    return True

        LOG.warning("Stacklight release is not found in helmbundle")
        return None

    def openstack_lma_enabled(self):
        if 'stacklight' not in self.get_components():
            LOG.warning("Stacklight component is not enabled")
            return None
        if self.is_management or self.is_regional:
            return False

        hb = self.get_helmbundle().data
        for chart in hb['spec']['releases']:
            if chart['name'] == 'stacklight':
                if 'openstack' in chart['values']:
                    return bool(chart['values']['openstack']['enabled'])
                else:
                    return False

        LOG.warning("Stacklight release is not found in helmbundle")
        return None

    def tf_monitoring_enabled(self):
        if 'stacklight' not in self.get_components():
            LOG.warning("Stacklight component is not enabled")
            return None
        if self.is_management or self.is_regional:
            return False

        hb = self.get_helmbundle().data
        for release in hb['spec']['releases']:
            if release['name'] == 'stacklight':
                if 'tungstenFabricMonitoring' in release['values']:
                    return bool(release['values']['tungstenFabricMonitoring']['enabled'])
                else:
                    return False

        LOG.warning("Stacklight release is not found in helmbundle")
        return None

    def netchecker_enabled(self):
        if self.provider != utils.Provider.baremetal:
            return False

        disabled = self.data['spec']['providerSpec']['value'].get('disableInfraConnectivityMonitor', False)
        return not disabled

    @lru_cache()
    def tf_analytics_enabled(self):
        if self.is_management or self.is_regional:
            return False
        if self.tf_enabled():
            child_kubeconfig_name, child_kubeconfig = self.get_kubeconfig_from_secret()
            with open('child_conf_tf_analytics', 'w') as f:
                f.write(child_kubeconfig)
            tf_manager = tungstenfafric_manager.TFManager(kubeconfig='child_conf_tf_analytics')
            if tf_manager.is_analytics_enabled():
                return True
            else:
                return False
        else:
            return False

    def pushgateway_enabled(self):
        if 'stacklight' not in self.get_components():
            LOG.warning("Stacklight component is not enabled")
            return None

        hb = self.get_helmbundle().data
        for chart in hb['spec']['releases']:
            if chart['name'] == 'stacklight':
                if 'prometheusPushgateway' in chart['values']:
                    return bool(chart['values']['prometheusPushgateway']['enabled'])
                else:
                    return False

        LOG.warning("Stacklight release is not found in helmbundle")
        return None

    def get_allowed_node_labels(self):
        cluster_cr = self.clusterrelease_version or \
            self.data['spec']['providerSpec']['value']['release']
        cr_version = \
            self.__manager.get_clusterrelease(name=cluster_cr)
        if 'allowedNodeLabels' in cr_version.data['spec'].keys():
            allowed_node_labels = {x['key']: x.get('value', '') for x in cr_version.data['spec']['allowedNodeLabels']}
            LOG.debug(f"Cluster {self.name} allowedNodeLabels: {allowed_node_labels}")
            return allowed_node_labels
        else:
            LOG.info("This cluster doesn't support Machine NodeLabels")
            return {}

    def create_os_machine(self,
                          node_type,
                          node_flavor,
                          node_image,
                          node_az,
                          boot_from_volume=False,
                          boot_volume_size=None,
                          node_sg=None,
                          region="",
                          provider_name="openstack",
                          node_labels=None,
                          upgrade_index=None,
                          day1_deployment='auto'):
        name_prefix = "{0}-{1}-".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        boot_from_volume_template = {
            "enabled": boot_from_volume,
            "volumeSize": boot_volume_size
        }

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Machine",
            "metadata": {
                "generateName": name_prefix,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": f"openstackproviderconfig.k8s.io/{provider.api_version}",  # noqa
                        "kind": provider.machine_spec,
                        "flavor": node_flavor,
                        "image": node_image,
                        "availabilityZone": node_az,
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if node_labels:
            body['spec']['providerSpec']['value']['nodeLabels'] = node_labels

        if node_sg:
            body['spec']['providerSpec']['value']['securityGroups'] = node_sg

        if boot_from_volume:
            body['spec']['providerSpec']['value']['bootFromVolume'] = boot_from_volume_template

        if upgrade_index:
            body['spec']['providerSpec']['value']['upgradeIndex'] = upgrade_index

        if settings.FEATURE_FLAGS.enabled('machine-pauses'):
            cr_version = self.get_desired_clusterrelease_version()
            if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cr_version):
                if not settings.MACHINE_PAUSE_DURING_CREATION_ENABLED:
                    # Override to turn off the pause if feature is disabled
                    day1_deployment = 'auto'
                body['spec']['providerSpec']['value']['day1Deployment'] = day1_deployment

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machines.create(
            namespace=self.namespace,
            body=body
        )

    def create_aws_machine(self,
                           node_type,
                           instance_type,
                           ami,
                           root_device_size=40,
                           root_device_type="gp3",
                           region="",
                           provider_name="aws",
                           node_labels=None,
                           upgrade_index=None,
                           day1_deployment='auto'):
        name_prefix = "{0}-{1}".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Machine",
            "metadata": {
                "generateName": name_prefix,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/region": region,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": f"aws.kaas.mirantis.com/{provider.api_version}",  # noqa
                        "kind": provider.machine_spec,
                        "instanceType": instance_type,
                        "ami": {
                            "id": ami
                        },
                        "rootDeviceSize": root_device_size,
                        "rootDeviceType": root_device_type
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if node_labels:
            body['spec']['providerSpec']['value']['nodeLabels'] = node_labels

        if upgrade_index:
            body['spec']['providerSpec']['value']['upgradeIndex'] = upgrade_index

        if settings.FEATURE_FLAGS.enabled('machine-pauses'):
            cr_version = self.get_desired_clusterrelease_version()
            if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cr_version):
                if not settings.MACHINE_PAUSE_DURING_CREATION_ENABLED:
                    # Override to turn off the pause if feature is disabled
                    day1_deployment = 'auto'
                body['spec']['providerSpec']['value']['day1Deployment'] = day1_deployment

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machines.create(
            namespace=self.namespace,
            body=body
        )

    def create_vsphere_machine(self,
                               node_type,
                               template,
                               rhel_license,
                               root_device_size=0,
                               region="",
                               provider_name="vsphere",
                               num_cpu=settings.KAAS_VSPHERE_MACHINE_CPU,
                               ram_mib=settings.KAAS_VSPHERE_MACHINE_RAM,
                               node_labels=None,
                               upgrade_index=None):

        name_prefix = "{0}-{1}".format(self.name, node_type)
        if settings.KAAS_CHILD_CLUSTER_MACHINES_PREFIX:
            name_prefix = "{}-{}-".format(
                settings.KAAS_CHILD_CLUSTER_MACHINES_PREFIX,
                name_prefix)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Machine",
            "metadata": {
                "generateName": name_prefix,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": f"vsphere.cluster.k8s.io/{provider.api_version}",
                        "kind": provider.machine_spec,
                        "rhelLicense": rhel_license,
                        "template": template,
                        "numCPUs": num_cpu,
                        "memoryMiB": ram_mib,
                    },
                    "labels": {
                        "cluster.sigs.k8s.io/cluster-name": self.name,
                        "kaas.mirantis.com/region": region,
                        "kaas.mirantis.com/provider": provider.provider_name
                    }
                }
            }
        }

        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        # Starting from 2.23, if diskGiB is unset, the disk size from VM template is used
        if root_device_size:
            body['spec']['providerSpec']['value']['diskGiB'] = root_device_size

        if node_labels:
            body['spec']['providerSpec']['value']['nodeLabels'] = node_labels

        if upgrade_index:
            body['spec']['providerSpec']['value']['upgradeIndex'] = upgrade_index

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machines.create(
            namespace=self.namespace,
            body=body
        )

    def create_equinix_machine(self,
                               node_type,
                               billing_cycle,
                               machine_type,
                               provider_name="equinixmetal",
                               region="",
                               os=None,
                               facility=None,
                               node_labels=None,
                               upgrade_index=None):
        name_prefix = "{0}-{1}".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)
        facility = facility or settings.KAAS_EQUINIX_FACILITY_REGION

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Machine",
            "metadata": {
                "generateName": name_prefix,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": f"equinix.kaas.mirantis.com/{provider.api_version}",  # noqa
                        "kind": provider.machine_spec,
                        "OS": os,
                        "billingCycle": billing_cycle,
                        "machineType": machine_type,
                        "facility": facility
                    }
                }
            }
        }

        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if os:
            body['spec']['providerSpec']['value']['OS'] = os

        if node_labels:
            body['spec']['providerSpec']['value']['nodeLabels'] = node_labels

        if upgrade_index:
            body['spec']['providerSpec']['value']['upgradeIndex'] = upgrade_index

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machines.create(
            namespace=self.namespace,
            body=body
        )

    def create_azure_machine(self,
                             node_type,
                             os,
                             vm_size,
                             vm_disk_size,
                             region="",
                             provider_name=settings.AZURE_PROVIDER_NAME,
                             node_labels=None,
                             upgrade_index=None):
        name_prefix = "{0}-{1}".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Machine",
            "metadata": {
                "generateName": name_prefix,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": f"azure.kaas.mirantis.com/{provider.api_version}",  # noqa
                        "kind": provider.machine_spec,
                        "vmSize": vm_size,
                        "osDisk": {
                            "osType": os,
                            "diskSizeGB": vm_disk_size,
                            }
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if node_labels:
            body['spec']['providerSpec']['value']['nodeLabels'] = node_labels

        if upgrade_index:
            body['spec']['providerSpec']['value']['upgradeIndex'] = upgrade_index

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machines.create(
            namespace=self.namespace,
            body=body
        )

    def create_baremetal_machine(self,
                                 node_pubkey_name,
                                 labels,
                                 genname,
                                 matchlabels=False,
                                 baremetalhostprofile=False,
                                 node_labels=False,
                                 l2TemplateSelector=None,
                                 distribution=None,
                                 upgrade_index=None,
                                 day1_provisioning='auto',
                                 day1_deployment='auto',
                                 provider_name="baremetal"):
        """
        :param genname: Machine generateName
        :param node_pubkey_name:
        :param matchlabels:  Dict with extra labels,  to be passed into
        hostSelector
        :param labels: Dict with labels to be passed into Machine
        :param node_labels: Dict with labels to be passed into
         providerSpec.nodeLabels
        :param labels: Dict with labels to be passed into Machine obj
        :param l2TemplateSelector: Dict with l2TemplateSelector data
        :param provider_name: Provider name
        :param day1_provisioning:
        'auto' allows automatic progression through the provisioning workflow.
        'manual' or empty string requires explicit approval before proceeding
        with provisioning.
        empty string is the same as 'manual'
        The value might be overridden depending on the feature flag settings
        and MACHINE_PAUSE_DURING_CREATION_ENABLED value
        :param day1_deployment:
        'auto' allows automatic progression through the deployment workflow.
        'manual' or empty string requires explicit approval before proceeding
        with deployment.
        Empty string is the same as 'manual'
        The value might be overridden depending on the feature flag settings and
        MACHINE_PAUSE_DURING_CREATION_ENABLED value
        :return:
        """
        mlabels = dict()
        if matchlabels:
            mlabels = matchlabels

        # define actual version of kaasrelease, which needed only to guess
        # bmApiversion
        cluster = self.__manager.get_mgmt_cluster()
        actual_kaasrelease = cluster.get_kaasrelease_version()
        LOG.info(f"Actual (deployed) kaasrelease version is\n"
                 f"{actual_kaasrelease}")
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "cluster.k8s.io/v1alpha1",
            "kind": "Machine",
            "metadata": {
                "generateName": f"{genname}-",
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    **labels}
            },
            "spec": {
                "providerSpec": {
                    "value": {
                        "apiVersion": self.__manager.get_bmApiversion(
                            actual_kaasrelease),
                        "kind": provider.machine_spec,
                        "hostSelector": {
                            "matchLabels": {**mlabels}
                        },
                        "publicKeys": [
                            {"name": node_pubkey_name},
                        ]
                    }
                }
            }
        }
        if baremetalhostprofile:
            body['spec']['providerSpec']['value'][
                'bareMetalHostProfile'] = baremetalhostprofile
        if l2TemplateSelector:
            body['spec']['providerSpec']['value'][
                'l2TemplateSelector'] = l2TemplateSelector
        if distribution:
            body['spec']['providerSpec']['value']['distribution'] = distribution
        if upgrade_index:
            body['spec']['providerSpec']['value']['upgradeIndex'] = upgrade_index
        if settings.FEATURE_FLAGS.enabled('machine-pauses'):
            cr_version = self.get_desired_clusterrelease_version()
            if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cr_version):
                if not settings.MACHINE_PAUSE_DURING_CREATION_ENABLED:
                    # Override to turn off pauses if feature is disabled
                    day1_provisioning = 'auto'
                    day1_deployment = 'auto'
                body['spec']['providerSpec']['value']['day1Provisioning'] = day1_provisioning
                body['spec']['providerSpec']['value']['day1Deployment'] = day1_deployment
        # Process nodeLabels
        nlabels = dict()
        if node_labels:
            nlabels = node_labels
        set_node_labels = []
        allowed_node_labels = self.get_allowed_node_labels()
        for k, v in nlabels.items():
            if k.startswith('hotfix/'):
                continue
            if k in allowed_node_labels.keys() and v == allowed_node_labels[k]:
                set_node_labels.append({'key': k, 'value': v})
            else:
                set_node_labels.append({'key': k, 'value': v})
                LOG.error("Key/value {0}:{1} is not in "
                          "allowed node label list".format(k, v))
        if set_node_labels:
            body['spec']['providerSpec']['value'][
                'nodeLabels'] = set_node_labels

        return self.__manager.api.kaas_machines.create(
            namespace=self.namespace,
            body=body
        )

    def create_baremetal_machine_raw(self, data):
        return self.__manager.api.kaas_machines.create(namespace=self.namespace,
                                                       name=data['metadata']['name'], body=data)

    def create_machine(self,
                       node_type,
                       region,
                       **data):
        node_labels = data.get('node_labels', None)
        provider_name = self.provider
        if provider_name == utils.Provider.openstack:
            flavor = data['flavor']
            image = data['image']
            az = data['az']
            sg = data.get('sg', None)
            machine = self.create_os_machine(node_type=node_type,
                                             node_flavor=flavor,
                                             node_image=image,
                                             node_az=az,
                                             node_sg=sg,
                                             region=region,
                                             node_labels=node_labels)
        elif provider_name == utils.Provider.aws:
            instance_type = data['instance_type']
            ami = data['ami']
            root_device_size = data.get('root_device_size', 40)
            machine = self.create_aws_machine(node_type=node_type,
                                              region=region,
                                              instance_type=instance_type,
                                              ami=ami,
                                              root_device_size=root_device_size,
                                              node_labels=node_labels)
        elif provider_name == utils.Provider.vsphere:
            template = data['template']
            rhel_license = data['rhel_license']
            root_device_size = data.get('root_device_size', 0)
            num_cpu = data.get('num_cpu', settings.KAAS_VSPHERE_MACHINE_CPU)
            ram_mib = data.get('memory_mib', settings.KAAS_VSPHERE_MACHINE_RAM)
            machine = self.create_vsphere_machine(node_type=node_type,
                                                  region=region,
                                                  rhel_license=rhel_license,
                                                  template=template,
                                                  root_device_size=root_device_size,
                                                  node_labels=node_labels,
                                                  num_cpu=num_cpu,
                                                  ram_mib=ram_mib)
        elif provider_name in (utils.Provider.equinixmetal, utils.Provider.equinixmetalv2):
            billing_cycle = data['billing_cycle']
            machine_type = data['machine_type']
            os = data.get('os', None)
            facility = data.get('facility', None)
            machine = self.create_equinix_machine(node_type=node_type,
                                                  region=region,
                                                  provider_name=provider_name.provider_name,
                                                  billing_cycle=billing_cycle,
                                                  machine_type=machine_type,
                                                  os=os,
                                                  facility=facility,
                                                  node_labels=node_labels)
        elif provider_name == utils.Provider.azure:
            os = data['os']
            vm_size = data['vm_size']
            vm_disk_size = data['vm_disk_size']
            machine = self.create_azure_machine(node_type=node_type,
                                                region=region,
                                                os=os,
                                                vm_size=vm_size,
                                                vm_disk_size=vm_disk_size,
                                                node_labels=node_labels)
        elif provider_name == utils.Provider.baremetal:
            raise NotImplementedError("Baremetal machine is not supported by SI tests yet")
        else:
            raise RuntimeError("Unknown provider")
        LOG.info("Created machine {0} in the namespace {1} for provider {2}".format(machine.name,
                                                                                    self.namespace,
                                                                                    provider_name))
        return machine

    def create_openstack_machinepool(self,
                                     replicas,
                                     node_type,
                                     region="",
                                     deletePolicy='never',
                                     provider_name="openstack",
                                     node_labels=None,
                                     node_flavor=None,
                                     node_image=None,
                                     node_az=None,
                                     node_sg=None):
        name = "{0}-{1}-machinepool".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "MachinePool",
            "metadata": {
                "name": name,
                "namespace": self.namespace,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "replicas": replicas,
                "deletePolicy": deletePolicy,
                "machineSpec": {
                    "providerSpec": {
                        "value": {
                            "apiVersion": f"openstackproviderconfig.k8s.io/{provider.api_version}",  # noqa
                            "kind": provider.machine_spec,
                            "flavor": node_flavor,
                            "image": node_image,
                            "availabilityZone": node_az,
                        }
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if node_labels:
            body['spec']['machineSpec']['providerSpec']['value']['nodeLabels'] = node_labels

        if node_sg:
            body['spec']['machineSpec']['providerSpec']['value']['securityGroups'] = node_sg

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machinepools.create(
            namespace=self.namespace,
            body=body
        )

    def create_aws_machinepool(self,
                               replicas,
                               node_type,
                               region="",
                               deletePolicy='never',
                               provider_name="aws",
                               instance_type=None,
                               ami=None,
                               root_device_size=40,
                               node_labels=None):
        name = "{0}-{1}-machinepool".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "MachinePool",
            "metadata": {
                "name": name,
                "namespace": self.namespace,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "replicas": replicas,
                "deletePolicy": deletePolicy,
                "machineSpec": {
                    "providerSpec": {
                        "value": {
                            "apiVersion": f"aws.kaas.mirantis.com/{provider.api_version}",  # noqa
                            "kind": provider.machine_spec,
                            "instanceType": instance_type,
                            "ami": {
                                "id": ami
                            },
                            "rootDeviceSize": root_device_size
                        }
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if node_labels:
            body['spec']['machineSpec']['providerSpec']['value']['nodeLabels'] = node_labels

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machinepools.create(
            namespace=self.namespace,
            body=body
        )

    def create_vsphere_machinepool(self,
                                   replicas,
                                   node_type,
                                   region="",
                                   deletePolicy='never',
                                   provider_name="vsphere",
                                   rhel_license=None,
                                   template=None,
                                   root_device_size=0,
                                   node_labels=None,):
        name = "{0}-{1}-machinepool".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "MachinePool",
            "metadata": {
                "name": name,
                "namespace": self.namespace,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "replicas": replicas,
                "deletePolicy": deletePolicy,
                "machineSpec": {
                    "providerSpec": {
                        "value": {
                            "apiVersion": f"vsphere.cluster.k8s.io/{provider.api_version}",
                            "kind": provider.machine_spec,
                            "rhelLicense": rhel_license,
                            "template": template,
                            "numCPUs": 8,
                            "memoryMiB": 16384
                        }
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        # Starting from 2.23, if diskGiB is unset, the disk size from VM template is used
        if root_device_size:
            body['spec']['providerSpec']['value']['diskGiB'] = root_device_size

        if node_labels:
            body['spec']['machineSpec']['providerSpec']['value']['nodeLabels'] = node_labels

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machinepools.create(
            namespace=self.namespace,
            body=body
        )

    def create_azure_machinepool(self,
                                 replicas,
                                 node_type,
                                 region="",
                                 deletePolicy='never',
                                 provider_name=settings.AZURE_PROVIDER_NAME,
                                 os=None,
                                 vm_size=None,
                                 vm_disk_size=None,
                                 node_labels=None):
        name = "{0}-{1}-machinepool".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "MachinePool",
            "metadata": {
                "name": name,
                "namespace": self.namespace,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "replicas": replicas,
                "deletePolicy": deletePolicy,
                "machineSpec": {
                    "providerSpec": {
                        "value": {
                            "apiVersion": f"azure.kaas.mirantis.com/{provider.api_version}",  # noqa
                            "kind": provider.machine_spec,
                            "vmSize": vm_size,
                            "osDisk": {
                                "osType": os,
                                "diskSizeGB": vm_disk_size,
                            }
                        }
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if node_labels:
            body['spec']['machineSpec']['providerSpec']['value']['nodeLabels'] = node_labels

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machinepools.create(
            namespace=self.namespace,
            body=body
        )

    def create_equinix_machinepool(self,
                                   replicas,
                                   node_type,
                                   region="",
                                   deletePolicy='never',
                                   provider_name="equinixmetal",
                                   billing_cycle=None,
                                   machine_type=None,
                                   os=None,
                                   facility=None,
                                   node_labels=None,):
        name = "{0}-{1}-machinepool".format(self.name, node_type)
        provider = utils.Provider.get_provider_by_name(provider_name)
        facility = facility or settings.KAAS_EQUINIX_FACILITY_REGION

        body = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "MachinePool",
            "metadata": {
                "name": name,
                "namespace": self.namespace,
                "labels": {
                    "cluster.sigs.k8s.io/cluster-name": self.name,
                    "kaas.mirantis.com/provider": provider.provider_name
                }
            },
            "spec": {
                "replicas": replicas,
                "deletePolicy": deletePolicy,
                "machineSpec": {
                    "providerSpec": {
                        "value": {
                            "apiVersion": f"equinix.kaas.mirantis.com/{provider.api_version}",  # noqa
                            "kind": provider.machine_spec,
                            "OS": os,
                            "billingCycle": billing_cycle,
                            "machineType": machine_type,
                            "facility": facility
                        }
                    }
                }
            }
        }
        if region:
            body["metadata"]["labels"]["kaas.mirantis.com/region"] = region

        if os:
            body['spec']['machineSpec']['providerSpec']['value']['OS'] = os

        if node_labels:
            body['spec']['machineSpec']['providerSpec']['value']['nodeLabels'] = node_labels

        if node_type == "master":
            labels = body['metadata']['labels']
            labels['cluster.sigs.k8s.io/control-plane'] = 'true'
        elif node_type == "node":
            pass  # for possible future changes
        else:
            raise NotImplementedError("Machine type {0} is not "
                                      "supported".format(node_type))

        return self.__manager.api.kaas_machinepools.create(
            namespace=self.namespace,
            body=body
        )

    def create_machinepool(self,
                           replicas,
                           node_type,
                           region,
                           deletePolicy='never',
                           **data):
        node_labels = data.get('node_labels', None)
        provider_name = self.provider
        if provider_name == utils.Provider.openstack:
            flavor = data['flavor']
            image = data['image']
            az = data['az']
            sg = data.get('sg', None)
            machinepool = self.create_openstack_machinepool(replicas, node_type, region, deletePolicy,
                                                            node_flavor=flavor,
                                                            node_image=image,
                                                            node_az=az,
                                                            node_sg=sg,
                                                            node_labels=node_labels)
        elif provider_name == utils.Provider.aws:
            instance_type = data['instance_type']
            ami = data['ami']
            root_device_size = data.get('root_device_size', 40)
            machinepool = self.create_aws_machinepool(replicas, node_type, region, deletePolicy,
                                                      instance_type=instance_type,
                                                      ami=ami,
                                                      root_device_size=root_device_size,
                                                      node_labels=node_labels)
        elif provider_name == utils.Provider.vsphere:
            template = data['template']
            rhel_license = data['rhel_license']
            root_device_size = data.get('root_device_size', 0)
            machinepool = self.create_vsphere_machinepool(replicas, node_type, region, deletePolicy,
                                                          rhel_license=rhel_license,
                                                          template=template,
                                                          root_device_size=root_device_size,
                                                          node_labels=node_labels)
        elif provider_name in (utils.Provider.equinixmetal, utils.Provider.equinixmetalv2):
            billing_cycle = data['billing_cycle']
            machine_type = data['machine_type']
            os = data['os', None]
            facility = data['facility', None]
            machinepool = self.create_equinix_machinepool(replicas, node_type, region, deletePolicy, provider_name,
                                                          billing_cycle=billing_cycle,
                                                          machine_type=machine_type,
                                                          os=os,
                                                          facility=facility,
                                                          node_labels=node_labels)
        elif provider_name == utils.Provider.azure:
            os = data['os']
            vm_size = data['vm_size']
            vm_disk_size = data['vm_disk_size']
            machinepool = self.create_azure_machinepool(replicas, node_type, region, deletePolicy,
                                                        os=os,
                                                        vm_size=vm_size,
                                                        vm_disk_size=vm_disk_size,
                                                        node_labels=node_labels)
        elif provider_name == utils.Provider.baremetal:
            raise NotImplementedError("Baremetal machinepools is not supported by SI tests yet")
        else:
            raise RuntimeError("Unknown provider")
        # TODO(vryzhenkin): Support BM machinepools
        LOG.info("Created machinepool {0} in the namespace {1} for provider {2}".format(machinepool.name,
                                                                                        self.namespace,
                                                                                        provider_name))
        return machinepool

    def get_machinepool_controlled_machines(self, machinepool_name):
        """Get machines controlled by defined machinepool

        :param machinepool_name: MachinePool name to filter
        :return: Filtered Machine List
        """
        return self.__manager.api.kaas_machines.filter(namespace=self.namespace,
                                                       cluster_name=self.name,
                                                       mp_name=machinepool_name)

    def get_machinepools(self, node_type=None):
        """Get machinepools in cluster by node type or without filtering

        :param node_type:
        :return: Filtered Machinepool list
        """
        return self.__manager.api.kaas_machinepools.filter(namespace=self.namespace,
                                                           cluster_name=self.name,
                                                           node_type=node_type)

    def get_existing_machine_spec(self, machine):
        """Get machine spec from existing machine for requested provider

        :param machine: Existing machine object
        :return data: Dict object with key/value machine params for requested provider
        """
        provider = self.provider
        data = {'node_labels': machine.data['spec']['providerSpec']['value'].get('nodeLabels', None)}
        if provider == utils.Provider.openstack:
            data['flavor'] = machine.data['spec']['providerSpec']['value']['flavor']
            data['image'] = machine.data['spec']['providerSpec']['value']['image']
            data['az'] = machine.data['spec']['providerSpec']['value']['availabilityZone']
            data['sg'] = machine.data['spec']['providerSpec']['value'].get('securityGroups', None)
        elif provider == utils.Provider.aws:
            data['instance_type'] = machine.data['spec']['providerSpec']['value']['instanceType']
            data['ami'] = machine.data['spec']['providerSpec']['value']['ami']['id']
            data['root_device_size'] = machine.data['spec']['providerSpec']['value']['rootDeviceSize']
        elif provider == utils.Provider.vsphere:
            data['template'] = machine.data['spec']['providerSpec']['value']['template']
            data['rhel_license'] = machine.data['spec']['providerSpec']['value'].get('rhelLicense')
            data['root_device_size'] = machine.data['spec']['providerSpec']['value'].get('diskGiB', 0)
            data['num_cpu'] = machine.data['spec']['providerSpec']['value'].get('numCPUs', 0)
            data['memory_mib'] = machine.data['spec']['providerSpec']['value'].get('memoryMiB', 0)
        elif provider == utils.Provider.azure:
            data['os'] = machine.data['spec']['providerSpec']['value']['osDisk']['osType']
            data['vm_size'] = machine.data['spec']['providerSpec']['value']['vmSize']
            data['vm_disk_size'] = machine.data['spec']['providerSpec']['value']['osDisk']['diskSizeGB']
        elif provider in (utils.Provider.equinixmetal, utils.Provider.equinixmetalv2):
            data['billing_cycle'] = machine.data['spec']['providerSpec']['value'].get('billingCycle')
            data['machine_type'] = machine.data['spec']['providerSpec']['value'].get('machineType')
            data['os'] = machine.data['spec']['providerSpec']['value'].get('OS')
            data['facility'] = machine.data['spec']['providerSpec']['value'].get('facility')
        elif provider == utils.Provider.baremetal:
            raise NotImplementedError
        # TODO(vryzhenkin): Support BM machinepools
        else:
            raise RuntimeError("Unknown provider")
        return data

    def get_bmc_data(self, bmh):
        """
        return BMC data(host,port,user,password) for specific bmh
        """
        bmh_data = bmh.data
        bmc_port = 623
        # we need to guess, when we have raw IP on input, or when bmc-uri
        url_parse = urllib.parse.urlparse(bmh_data['spec']['bmc']['address'])
        if url_parse.scheme:
            bmc_host = url_parse.netloc.split(":")[0]
            # guess port, or use default
            try:
                bmc_port = int(url_parse.netloc.split(":")[1])
            except Exception as e:
                LOG.error(e)
        else:
            bmc_host = bmh_data['spec']['bmc']['address'].split(":")[0]
            # guess port, or use default
            try:
                bmc_port = int(bmh_data['spec']['bmc']['address'].split(":")[1])
            except Exception as e:
                LOG.error(e)
        bmc_cred_name = bmh_data['spec']['bmc']['credentialsName']
        bmc_cred_data = self.__manager.api.secrets.get(
            name=bmc_cred_name, namespace=self.namespace).read().to_dict()
        bmc_username = base64.b64decode(
            bmc_cred_data['data']['username']).decode("utf-8")
        bmc_password = base64.b64decode(
            bmc_cred_data['data']['password']).decode("utf-8")
        return {'host': bmc_host,
                'port': bmc_port,
                'username': bmc_username,
                'password': bmc_password}

    def describe_fixed_resources(self):
        """Describe the cluster resource properties

        Describe the resource properties that should not be changed
        during the Cluster update process
        """
        resources = [
            self.k8sclient.nodes,
            self.k8sclient.services,
        ]
        data = {}
        for resource in resources:
            data.update(resource.describe_fixed_resources())
        return data

    def docker_ps(self):
        """Show info from 'docker ps', 'top' and 'df'"""
        machines = self.get_machines()
        for machine in machines:
            # Note: use double escape (`\`) as command has to be passed as
            # pod cmd first, then as shell cmd
            cmd = ("docker ps --all; top -bn 1| head -n6; df -h|"
                   "grep -e 'Filesystem\\\\| /$'")
            LOG.info(f"Collecting docker container statuses from "
                     f"{machine.name}, see debug log for details")
            result = machine.exec_pod_cmd(cmd, get_events=False, verbose=False)
            LOG.debug(f"Machine {machine.name}:\n" + result["logs"].strip())

    def get_cluster_description(self, system_info=False):
        cluster_release = self.clusterrelease_version
        provider_types = []
        if self.is_management:
            cluster_name = f"Management cluster: '{self.name}'"
        elif self.is_regional:
            cluster_name = f"Region cluster: '{self.name}'"
        else:
            cluster_name = f"Child cluster: '{self.name}'"
        lcm_type = self.lcm_type
        machines = self.get_machines()
        mstatus = {}
        docker_versions = set()
        kernel_versions = set()
        system_versions = set()
        for machine in machines:
            if machine.provider not in provider_types:
                provider_types.append(machine.provider)
            if machine.machine_type not in mstatus:
                mstatus[machine.machine_type] = {}
            if machine.machine_status not in mstatus[machine.machine_type]:
                mstatus[machine.machine_type][machine.machine_status] = 0
            mstatus[machine.machine_type][machine.machine_status] += 1
            if machine.is_disabled():
                # There is no node in the cluster if machine is disabled,
                # host is supposed to be offline
                continue
            if system_info and self.provider != utils.Provider.byo:
                system_versions.add(utils.get_system_version(machine))
                kernel_versions.add(utils.get_kernel_version(machine))
                docker_versions.add(utils.get_docker_version(machine))
        msg = (f"{cluster_name} ({self.uid})"
               f" release: '{cluster_release}' provider: {provider_types}"
               f" lcm_type: '{lcm_type}'\n"
               f"    machines: {mstatus}\n")
        if system_info:
            msg += (f"    system versions: {system_versions}\n"
                    f"    kernel versions: {kernel_versions}\n"
                    f"    docker versions: {docker_versions}\n")
        return msg

    def apply_hotfix_labels(self, bm_nodes):
        LOG.info("Trying to apply hotfix labels")
        machines = self.get_machines()
        for machine in machines:
            hwid = machine.get_bmh_id()
            for bm_node in bm_nodes:
                k8s_new_labels = {}
                if hwid == bm_node[
                        'bmh_labels']['kaas.mirantis.com/baremetalhost-id']:
                    bm_node_labels = bm_node.get('node_labels')
                    if bm_node_labels:
                        for k, v in bm_node_labels.items():
                            if k.startswith('hotfix/'):
                                k8s_new_labels.update(
                                    {k.replace('hotfix/', ''): v}
                                )

                if k8s_new_labels:
                    k8s_node = machine.get_k8s_node()
                    LOG.info("Set new labels {0} for child K8S"
                             " node {1}, hwid {2}".format(k8s_new_labels,
                                                          k8s_node.name,
                                                          hwid))
                    k8s_node.patch({'metadata': {'labels': k8s_new_labels}})
        LOG.info("Applying hotfix labels is completed")

    def apply_nodes_annotations(self, bm_nodes):
        LOG.info("Trying to apply nodes annotations")
        machines = self.get_machines()
        for machine in machines:
            hwid = machine.get_bmh_id()
            for bm_node in bm_nodes:
                k8s_new_annotations = {}
                if hwid == bm_node[
                        'bmh_labels']['kaas.mirantis.com/baremetalhost-id']:
                    bm_node_annotations = bm_node.get('node_annotations', {})
                    if bm_node_annotations:
                        for k, v in bm_node_annotations.items():
                            k8s_new_annotations.update(
                                {k: v}
                            )
                if k8s_new_annotations:
                    k8s_node = machine.get_k8s_node()
                    LOG.info("Set new annotations {0} for child K8S"
                             " node {1}, hwid {2}".format(k8s_new_annotations,
                                                          k8s_node.name,
                                                          hwid))
                    machine.add_k8s_node_annotations(k8s_new_annotations)
        LOG.info("Applying annotations is completed")

    def remove_k8s_node_all_labels(self, bm_machine_name):
        machine = self.get_machine(name=bm_machine_name)
        k8s_node = machine.get_k8s_node()
        k8s_node.patch({'metadata': {'labels': {'nolabels': 'nolabels'}}})

    def add_osdpl_overrides(self, data):
        """
        Get data in format:
          child_data.yaml node['name']: <node overrides dict>
        Creates a secret in child cluster openstack namespace:
          data:
            kubernetes-node-name: base64(<node overrides dict>)
        """
        if not data:
            return
        LOG.info("Add osdpl overrides")
        machines = self.get_machines()
        kubectl_client_child = self.k8sclient
        secret_name = "osdpl-nodes-override"
        secret_data = {}
        for machine in machines:
            hwid = machine.get_bmh_id()
            for node_name, node_data in data.items():
                if hwid == node_name:
                    k8s_node = machine.get_k8s_node()
                    secret_data[k8s_node.name] = base64.b64encode(
                        json.dumps(node_data).encode('utf-8')).decode()
        kubectl_client_child.secrets.create(
            namespace=NAMESPACE.openstack,
            body={
                "apiVersion": "v1",
                "kind": "Secret",
                "metadata": {
                    "name": secret_name,
                    "namespace": NAMESPACE.openstack,
                },
                "data": secret_data
            }
        )
        LOG.info("Add osdpl overrides completed")

    def get_miracephcluster(self, name='rook-ceph', namespace='ceph-lcm-mirantis'):
        """
        Get MiraCeph cluster object.

        Args:
            cluster_name: Name of the MiraCeph cluster to get.
            namespace: The Kubernetes namespace where the MiraCeph resource is located.

        Returns: si_tests.clients.k8s.miracephs.Miracephs or None.
        """
        miraceph = [miracephcl for miracephcl in self.k8sclient.miracephs.list(namespace=namespace)
                    if miracephcl.name == name]
        return miraceph[0] if miraceph else None

    def get_miracephhealth(self, name='rook-ceph', namespace='ceph-lcm-mirantis'):
        """
        Get MiraCephHealth cluster object.

        Args:
            cluster_name: Name of the MiraCephHealth cluster to get.
            namespace: The Kubernetes namespace where the MiraCephHealth resource is located.

        Returns: si_tests.clients.k8s.miracephs.MiracephHealths or None.
        """
        miracephhealth = [miracephh for miracephh in self.k8sclient.miracephhealths.list(namespace=namespace)
                          if miracephh.name == name]
        return miracephhealth[0] if miracephhealth else None

    def get_miracephsecret(self, name='rook-ceph', namespace='ceph-lcm-mirantis'):
        """
        Get MiraCephSecret cluster object.

        Args:
            cluster_name: Name of the MiraCephSecret cluster to get.
            namespace: The Kubernetes namespace where the MiraCephSecret resource is located.

        Returns: si_tests.clients.k8s.miracephs.MiracephHealths or None.
        """
        miracephsecret = [miracephs for miracephs in self.k8sclient.miracephsecrets.list(namespace=namespace)
                          if miracephs.name == name]
        return miracephsecret[0] if miracephsecret else None

    def get_cephcluster(self):
        cephcluster = [
            cephcl for cephcl in self.__manager.api.kaas_cephclusters.list(namespace=self.namespace)
            if cephcl.data['spec']['k8sCluster']['name'] == self.name and
            cephcl.data['spec']['k8sCluster']['namespace'] == self.namespace
        ]
        return cephcluster[0] if cephcluster else None

    def patch_ceph_data(self, data, crd):
        LOG.info("crd: {0}: ".format(crd))
        crd.patch(body=data)

    def replace_ceph_data(self, data, crd):
        LOG.info("crd: {0}: ".format(crd))
        crd.replace(body=data)

    def get_ceph_tool_pod(self):
        rook_ns = settings.ROOK_CEPH_NS
        ceph_tools_pod = self.k8sclient.pods.list(
            namespace=rook_ns,
            name_prefix='rook-ceph-tools')
        assert len(ceph_tools_pod) > 0, (f"No pods found with prefix "
                                         f"rook-ceph-tools in namespace {rook_ns}")
        return ceph_tools_pod[0]

    def get_ceph_mgr_pod(self):
        rook_ns = settings.ROOK_CEPH_NS
        ceph_mgr_pod = self.k8sclient.pods.list(
            namespace=rook_ns,
            name_prefix='rook-ceph-mgr')
        assert len(ceph_mgr_pod) > 0, (f"No pods found with prefix "
                                       f"rook-ceph-mgr in namespace {rook_ns}")
        return ceph_mgr_pod[0]

    def get_ceph_operator_pod(self):
        ceph_operator_pod = self.k8sclient.pods.list(
            namespace=settings.ROOK_CEPH_NS,
            name_prefix='rook-ceph-operator')
        assert len(ceph_operator_pod) > 0, ("No pods found with prefix "
                                            "rook-ceph-operator in "
                                            "namespace {}".
                                            format(settings.ROOK_CEPH_NS))
        return ceph_operator_pod[0]

    def get_ceph_rgw_pod(self):
        ceph_rgw_pod = self.k8sclient.pods.list(
            namespace=settings.ROOK_CEPH_NS,
            name_prefix='rook-ceph-rgw')
        assert len(ceph_rgw_pod) > 0, ("No pods found with prefix "
                                       "rook-ceph-rgw in "
                                       "namespace {}".
                                       format(settings.ROOK_CEPH_NS))
        return ceph_rgw_pod[0]

    def get_ceph_rgw_user_info(self, rgw_username):
        """

        :param rgw_username: rgw user name to search
        """
        if self.workaround.skip_kaascephcluster_usage():
            secret_crd = self.get_miracephsecret().data or {}
            if len(secret_crd) == 0:
                return None
            ceph_secrets_info = secret_crd.get('status', {}).get(
                'secretInfo', {}).get(
                'rgwUserSecrets', [])
        else:
            ceph_crd = self.get_cephcluster().data or {}
            if len(ceph_crd) == 0:
                return None
            ceph_secrets_info = ceph_crd.get(
                'status', {}).get(
                'miraCephSecretsInfo', {}).get(
                'secretInfo', {}).get(
                'rgwUserSecrets', [])
        if len(ceph_secrets_info) == 0:
            return None
        for rgw_secret in ceph_secrets_info:
            if (rgw_secret.get('name') == rgw_username
                    and rgw_secret.get('secretName', '') != ''):
                return rgw_secret
        return None

    def get_ceph_rbd_client_info(self, client_name):
        """

        :param client_name: rbd client name to search
        """
        if self.workaround.skip_kaascephcluster_usage():
            secret_crd = self.get_miracephsecret().data or {}
            if len(secret_crd) == 0:
                return None
            ceph_secrets_info = secret_crd.get('status', {}).get(
                'secretInfo', {}).get(
                'clientSecrets', [])
        else:
            ceph_crd = self.get_cephcluster().data or {}
            if len(ceph_crd) == 0:
                return None
            ceph_secrets_info = ceph_crd.get(
                'status', {}).get(
                'miraCephSecretsInfo', {}).get(
                'secretInfo', {}).get(
                'clientSecrets', [])
        if len(ceph_secrets_info) == 0:
            return None
        for client_secret in ceph_secrets_info:
            if (client_secret.get('name') == client_name
                    and client_secret.get('secretName', '') != ''):
                return client_secret
        return None

    def determine_mcp_docker_registry(self,
                                      pod_name="helm-controller-",
                                      pod_ns="kube-system",
                                      default="mirantis.azurecr.io"):
        """Determine MCP docker registry from the specified pod"""
        if self.__mcp_docker_registry is None:
            self.__mcp_docker_registry = self.k8sclient.pods.extract_registry_url(
                pod_name=pod_name, pod_ns=pod_ns, default=default)
        LOG.info(f"Found the MCP docker registry for "
                 f"{self.namespace}/{self.name} at: "
                 f"{self.__mcp_docker_registry}")
        return self.__mcp_docker_registry

    def get_machines_uptime(self, verbose=False, dump_reboot_list=False):
        """
        Collect uptime for each machine in the cluster

        Args:
            verbose: print command output
            dump_reboot_list: print in console list of existing reboot

        Returns: dict of machine name (str) as a key and machine uptime (datetime) as a value

        """
        uptimes = {}
        LOG.info("Collect machines uptime")
        for machine in self.get_machines():
            if machine.is_disabled():
                continue
            machine_uptime = machine.get_uptime(verbose=verbose)
            uptimes[machine.name] = machine_uptime
            if dump_reboot_list:
                machine.get_reboot_list()
        return uptimes

    def get_nodes_kernel_and_tarfs_versions(self):
        data = {}
        provider = self.provider
        for m in self.get_machines():
            if m.is_disabled():
                continue
            versions_out = m.exec_pod_cmd(
                ". /etc/os-release;"
                "DIB=$([ -f /etc/dib_build.yaml ] && cat /etc/dib_build.yaml | awk ' /datetime:/ {print $2}');"
                "echo $ID'_'$VERSION_ID,$(uname -r),$DIB",
                verbose=False)['logs'].strip()
            data[m.name] = {
                'os_version': versions_out.split(",")[0],  # rhel_7.9, ubuntu_18.04, centos
                'kernel': versions_out.split(",")[1],  # 4.15.0-112-generic
                'dib_datetime': versions_out.split(",")[2] or None,  # 20211214122154
            }
            LOG.info(f"Current machine os versions {m.name}:\n"
                     f"{data[m.name]}")
            # -bm must always contain dib information
            if provider == utils.Provider.baremetal.provider_name:
                assert data[m.name]['dib_datetime'] is not None, \
                    f"Machine {m.name} doesnt have proper /etc/dib_build.yaml information"
        return data

    def check_bmhp_mapping(self):
        """
        Check for expected(in bmhp object) and actual fs-names on disks
        Idea here: exact bmhp, must contain exact si-test label, to match exact
        disk-labels on exact node, with exact part_label
        :return: boolean
        """
        match_label = 'si-test/test-bmhp-mapping'
        value_separator = '-si-separator-'
        _cmd = 'lsblk --json -M -p -O'
        machines_to_test = []
        bmhp_to_test = []
        if self.provider.name != utils.Provider.baremetal.provider_name:
            LOG.warning('Not -bm provider, skip check_bmhp_mapping')
            return
        for bmhp in self.__manager.get_baremetalhostprofiles(namespace=self.namespace):
            if [k for k in bmhp.data['metadata']['labels'].keys() if k.startswith(match_label)]:
                LOG.info(f"Bmhp {bmhp.name} supposed to be tested for {match_label}")
                bmhp_to_test.append(bmhp)
        if not bmhp_to_test:
            LOG.warning('Not found any suitable bmhp for check_bmhp_mapping.Skipping...')
            return
        for m in self.get_machines():
            _machine_bmhp = m.spec['providerSpec']['value'].get('bareMetalHostProfile', {})
            if _machine_bmhp and _machine_bmhp['name'] in [bmh.name for bmh in bmhp_to_test]:
                LOG.info(f"Machine {m.name} uses bmhp {_machine_bmhp} and will be tested for {match_label}")
                machines_to_test.append(m)
        for m in machines_to_test:
            # no default check here, since for that moment - data must be in place
            _machine_bmhp_name = m.spec['providerSpec']['value']['bareMetalHostProfile']['name']
            _bmhp = self.__manager.get_baremetalhostprofile(name=_machine_bmhp_name,
                                                            namespace=self.namespace)
            m_test = dict()
            for k, v in _bmhp.data['metadata']['labels'].items():
                if k.startswith(match_label):
                    m_test[k] = {'case': v.split(value_separator)[0],
                                 'value': v.split(value_separator)[1],
                                 'target': v.split(value_separator)[2]
                                 }
            LOG.info(f"Machine {m.name} test scenario:\n${m_test}")
            lsblk = yaml.safe_load(m.exec_pod_cmd(_cmd, verbose=False)['logs'])['blockdevices']
            LOG.debug(f"Machine {m.name} lsblk data:\n{lsblk}")

            def bmhp_mapping_case_data_checker(m_test_case, case_data):
                _pass = False
                for block in lsblk:
                    case_data_target = case_data['target']
                    case_data_value = case_data['value']
                    case = case_data['case']
                    partlabels = [i['partlabel'] for i in block.get('children', [])]
                    block_children = block.get('children', '')
                    block_value = block.get(case, '')
                    block_name = block['name']
                    LOG.debug(f'looking in block:{block}')
                    if block_value == case_data_value:
                        LOG.info(f"Block:{block_name} has {case} value:{block_value}")
                        if case_data_target in partlabels:
                            LOG.info(f"Block:{block_name} has value:{block_value}, "
                                     f"and has partition on it with partlabel {case_data_target}")
                            _pass = True
                        elif case_data_target == 'empty' and not block_children:
                            LOG.info(f"Block:{block_name} has value:{block_value}, "
                                     f"and has empty partlabel {case_data_target}")
                            _pass = True
                        else:
                            raise Exception(f"Machine {m.name} doesn't pass test:{m_test_case}:{case_data}, "
                                            f"because it has block_value: {block_value} and partlabels: {partlabels}")
                if not _pass:
                    raise Exception(f"Machine {m.name} doesn't pass test:{m_test_case}:{case_data}, because "
                                    f"this condition: (block_value == case_data_value) was failed")
                return True

            for m_test_case, case_data in m_test.items():
                LOG.info(f'Attempt to test:{m_test_case}:{case_data}')
                bmhp_mapping_case_data_checker(m_test_case, case_data)

        return True

    def update_requires_reboot_mgmt(self, cr_before,
                                    target_clusterrelease=None,
                                    kaasrelease_version=None):
        """
        Check that any step in possible upgrade path has rebootRequired: True

        Upgrade path can be multistep e.g. v1 -> v2 -> v3
        """
        # Reject rebooting for any provider except "baremetal", PRODX-20442
        if self.provider != utils.Provider.baremetal:
            return False

        # cr_before - current (before update) clusterrelease version
        # we need to define it explicitly because by the time
        # we are in this method, current cluster clusterrelease may be updated
        # ----
        # for mgmt and region clusters we must provide
        # target kaasrelease_version to determine
        # target clusterrelease version from it
        if self.is_management or self.is_regional:
            if not kaasrelease_version:
                raise Exception("kaasrelease version is not specified "
                                "for update_requires_reboot. Exiting")
            kr_data = \
                self.__manager.get_kaasrelease(kaasrelease_version).data
            target_clusterrelease = kr_data['spec']['clusterRelease']
        else:
            # for child cluster we just need to know
            # target clusterrelease version
            if not target_clusterrelease:
                raise Exception("target clusterrelease version is not "
                                "specified for update_requires_reboot. "
                                "Exiting")
            # get mgmt cluster object, because it has
            # needed kaasrelease_version
            mgmt_cluster = self.__manager.get_mgmt_cluster()
            kaasrelease_version = mgmt_cluster.get_kaasrelease_version()
            kr_data = \
                self.__manager.get_kaasrelease(kaasrelease_version).data
        supported_releases = kr_data['spec']['supportedClusterReleases']

        if target_clusterrelease == cr_before:
            # If clusterrelease version is not changed - do not expect any reboot
            return False

        # we have 'name' and 'version' fields for clusterrelease
        # let's find 'version' field, using 'name'
        target_cl_version = [x['version'] for x in supported_releases
                             if x['name'] == target_clusterrelease]
        if not target_cl_version:
            raise Exception(f"Cannot find 'version' for clusterrelease "
                            f"{target_clusterrelease}. supportedClusterReleases: "
                            f"{supported_releases}")
        else:
            LOG.debug(f"'version' for clusterrelease {target_cl_version}")
            target_cl_version = target_cl_version[0]

        # Recursively search for multistep upgrade path
        # Returns list of objects from availableUpgrades needed to perform
        # consequitive upgrade from 'start' version to 'end' version
        def get_upgrade_path(start, end):
            if start == end:
                # This is not upgrade already so exit
                return []

            result = []
            available_upgrades = []
            for ver in supported_releases:
                if ver['version'] == start:
                    available_upgrades = ver.get('availableUpgrades', [])
                    break
            for available_upgrade in available_upgrades:
                version = available_upgrade['version']
                if version == end:
                    result.append(available_upgrade)
                    break
                else:
                    next_releases = get_upgrade_path(version, end)
                    if next_releases:
                        result.append(available_upgrade)
                        result += next_releases
            return result

        cr_before_version = [x['version'] for x in supported_releases
                             if x['name'] == cr_before][0]
        upgrade_path = get_upgrade_path(cr_before_version, target_cl_version)
        LOG.debug(f"Upgrade path: {upgrade_path}")
        if upgrade_path:
            return any([x.get('rebootRequired', False) for x in upgrade_path])
        else:
            raise LookupError(
                f"There is no any path to upgrade from {cr_before} to {target_cl_version}.\n"
                f"Supported releases:\n{yaml.dump(supported_releases)}"
            )

    def update_requires_reboot(self, cr_before,
                               target_clusterrelease=None,
                               kaasrelease_version=None):
        # Reject rebooting for any provider except "baremetal", PRODX-20442
        if self.provider != utils.Provider.baremetal:
            return False

        # cr_before - current (before update) clusterrelease version
        # we need to define it explicitly because by the time
        # we are in this method, current cluster clusterrelease may be updated
        # ----
        # for mgmt and region clusters we must provide
        # target kaasrelease_version to determine
        # target clusterrelease version from it
        if self.is_management or self.is_regional:
            if not kaasrelease_version:
                raise Exception("kaasrelease version is not specified "
                                "for update_requires_reboot. Exiting")
            kr_data = \
                self.__manager.get_kaasrelease(kaasrelease_version).data
            target_clusterrelease = kr_data['spec']['clusterRelease']
        else:
            # for child cluster we just need to know
            # target clusterrelease version
            if not target_clusterrelease:
                raise Exception("target clusterrelease version is not "
                                "specified for update_requires_reboot. "
                                "Exiting")
            # get mgmt cluster object, because it has
            # needed kaasrelease_version
            mgmt_cluster = self.__manager.get_mgmt_cluster()
            kaasrelease_version = mgmt_cluster.get_kaasrelease_version()
            kr_data = \
                self.__manager.get_kaasrelease(kaasrelease_version).data
        supported_releases = kr_data['spec']['supportedClusterReleases']

        if target_clusterrelease == cr_before:
            # If clusterrelease version is not changed - do not expect any reboot
            return False

        # we have 'name' and 'version' fields for clusterrelease
        # let's find 'version' field, using 'name'
        target_cl_version = [x['version'] for x in supported_releases
                             if x['name'] == target_clusterrelease]
        if not target_cl_version:
            raise Exception(f"Cannot find 'version' for clusterrelease "
                            f"{target_clusterrelease}. supportedClusterReleases: "
                            f"{supported_releases}")
        else:
            LOG.debug(f"'version' for clusterrelease {target_cl_version}")
            target_cl_version = target_cl_version[0]

        aval_versions = \
            [x.get('availableUpgrades', [])
             for x in supported_releases if x['name'] == cr_before][0]
        for ver in aval_versions:
            if ver['version'] == target_cl_version:
                return ver.get('rebootRequired', False)
        else:
            raise LookupError(
                f"Target cluster version {target_cl_version} "
                f"wasn't found in the availableUpgrades from previous clusterrelease {cr_before}: {aval_versions}"
            )

    def get_skipMaintenance_flag(self, cr_before, target_clusterrelease=None,
                                 kaasrelease_version=None):
        if self.is_management or self.is_regional:
            if not kaasrelease_version:
                raise Exception("kaasrelease version is not specified. Exiting")
            kr_data = \
                self.__manager.get_kaasrelease(kaasrelease_version).data
            target_clusterrelease = kr_data['spec']['clusterRelease']
        else:
            # for child cluster we just need to know
            # target clusterrelease version
            if not target_clusterrelease:
                raise Exception("target clusterrelease version is not "
                                "specified. "
                                "Exiting")
            # get mgmt cluster object, because it has
            # needed kaasrelease_version
            mgmt_cluster = self.__manager.get_mgmt_cluster()
            kaasrelease_version = mgmt_cluster.get_kaasrelease_version()
            kr_data = \
                self.__manager.get_kaasrelease(kaasrelease_version).data
        supported_releases = kr_data['spec']['supportedClusterReleases']

        # we have 'name' and 'version' fields for clusterrelease
        # let's find 'version' field, using 'name'
        target_cl_version = [x['version'] for x in supported_releases
                             if x['name'] == target_clusterrelease]
        if not target_cl_version:
            raise Exception(f"Cannot find 'version' for clusterrelease "
                            f"{target_clusterrelease}. supportedClusterReleases: "
                            f"{supported_releases}")
        else:
            LOG.debug(f"'version' for clusterrelease {target_cl_version}")
            target_cl_version = target_cl_version[0]
        aval_versions = \
            [x.get('availableUpgrades', [])
             for x in supported_releases if x['name'] == cr_before][0]
        for ver in aval_versions:
            if ver['version'] == target_cl_version:
                return ver.get('skipMaintenance', False)

    def is_skip_maintenance_set(self, cr_before, target_clusterrelease, kaasrelease_version=None):
        if self.get_skipMaintenance_flag(cr_before,
                                         target_clusterrelease, kaasrelease_version):
            LOG.info('skipMaintenance flag is set in true. '
                     'Checking that there were not cordondrains')
            return True
        else:
            LOG.info('skipMaintenance flag is set in False')
            return False

    def get_cluster_repository_url(self):
        """
        Get mapped machine type to repository based on cluster release version

        Returns: [machine_type: repository_url]

        """
        clusterrelease_version = self.clusterrelease_version
        machine_types = self.__manager.get_clusterrelease(clusterrelease_version).data \
            .get("spec", {}).get("machineTypes", {})
        repo_url = {}
        for machine_type in machine_types:
            repo_url[machine_type] = next(
                (x['params']['kaas_ubuntu_repo'] for x in machine_types[machine_type] if
                 x['name'] == 'setup'), None)
        LOG.info(f"Release version: {clusterrelease_version}")
        LOG.info(f"Repository url: {repo_url}")
        return repo_url

    def get_cluster_dedicated_control_plane_status(self):
        if 'dedicatedControlPlane' in self.spec['providerSpec']['value']:
            return self.spec['providerSpec']['value']['dedicatedControlPlane']
        else:
            return True

    def cluster_maintenance_start(self):
        """
        Start Cluster maintenance
        Returns: Cluster object

        """
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "maintenance": True
                    }
                }
            }
        }

        cluster = self.__manager.api.kaas_clusters.update(name=self.name, namespace=self.namespace, body=body)
        LOG.info("Wait until Cluster Maintenance request created")
        self.check.wait_clustermaintenancerequest_created()

        LOG.info("Wait expected Cluster Maintenance status")
        self.check.wait_cluster_maintenance_status(expected_status=True)

        return Cluster(self.__manager, cluster)

    def cluster_maintenance_stop(self):
        """
        Stop Cluster maintenance
        Returns: Cluster object

        """
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "maintenance": False
                    }
                }
            }
        }

        cluster = self.__manager.api.kaas_clusters.update(name=self.name, namespace=self.namespace, body=body)

        LOG.info("Wait until Cluster Maintenance request deleted")
        self.check.wait_clustermaintenancerequest_not_existed()

        LOG.info("Wait expected Cluster Maintenance status")
        self.check.wait_cluster_maintenance_status(expected_status=False)

        return Cluster(self.__manager, cluster)

    def get_tls_statuses(self):
        """
        Get cluster TLS statuses
        Returns: dict of {'keycloak': {'expirationTime': '', 'hostname': '', 'renewalTime': ''}...}

        """
        cluster_status = self.data.get('status') or {}
        return cluster_status.get('providerStatus', {}).get('tls', {})

    def get_tls_status(self, component):
        """
        Get cluster TLS status based on component
        Args:
            component: component name

        Returns: dict {'expirationTime': '', 'hostname': '', 'renewalTime': ''}

        """
        component_lower = component.replace("-", "").lower()
        statuses = self.get_tls_statuses()
        return next(val for key, val in statuses.items() if key.lower() == component_lower)

    def is_os_deployed(self):
        """
        Check if OS deployed on current cluster
        Returns: True or False

        """
        os_ns = self.k8sclient.namespaces.present(settings.OSH_NAMESPACE)
        if os_ns:
            osdpl = self.k8sclient.openstackdeployment.get(
                name=settings.OSH_DEPLOYMENT_NAME,
                namespace=settings.OSH_NAMESPACE)
            try:
                osdpl.read()
            except ApiException:
                return False
            return True
        else:
            return False

    def get_k8s_nodes(self) -> List[K8sNode]:
        return self.k8sclient.nodes.list_raw().to_dict()['items']

    def get_k8s_node_names(self) -> List[str]:
        nodes = self.get_k8s_nodes()
        return [n['metadata']['name'] for n in nodes]

    def get_keepalive_master_machine(self) -> MachineProviders:
        """
        Find KaaS keepalive master machine.
        Search an interface with the Virtual IP address equal to the load balancer on each machine

        Returns: Machine object

        """
        lb_ip = self.load_balancer_host
        # Check if LB IP from cluster.providerStatus.loadBalancerHost is a valid IP
        if not utils.is_valid_ip(lb_ip):
            # If no - try to use LB IP from cluster.providerStatus.loadBalancerIP
            lb_ip = self.load_balancer_ip
            if not utils.is_valid_ip(lb_ip):
                raise Exception(f"Cluster LoadBalancer IP {lb_ip} is not valid IP address")

        LOG.info(f"LoadBalancer IP: {lb_ip}")
        machines = self.get_machines(machine_type='control')
        machine_vip = []
        for machine in machines:
            result = machine.exec_pod_cmd("ip a", verbose=False)
            ips = re.findall("(?<=inet )[0-9.]+", result['logs'])
            LOG.info("Machine {} with IPs: {}".format(machine.name, ", ".join(ips)))
            if lb_ip in ips:
                machine_vip.append(machine)
        assert machine_vip, "Cannot find machine with keepalive VIP"
        assert (len(machine_vip) == 1), "For some reason VIP is located on more than one node"

        LOG.info(f"Machine with VIP ({lb_ip}) is {machine_vip[0].name}")
        return machine_vip[0]

    def update_cluster_container_registry(self, registry_names: Optional[List[str]] = None):
        """
        Sets the cluster containerRegistries with value
        "registry_names" and will overwrite the "containerRegistries"
        if it already exists.
        :return: api result
        """
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "containerRegistries": registry_names
                    }
                }
            }
        }
        api_responce = self.__manager.api.kaas_clusters.update(
            name=self.name,
            namespace=self.namespace,
            body=body
        )
        return api_responce

    def remove_cluster_container_registry(self):
        """
        To remove containerRegistries, we should add containerRegistries
        with "registry_names" = None
        """
        return self.update_cluster_container_registry(None)

    @staticmethod
    def machine_deletion_policy_enabled():
        return settings.MACHINE_DELETION_POLICY_ENABLED

    def get_pods_for_node(self, node_name):
        """
        Return list of K8sPod object hosted on target node
        Args:
            node_name: target node name

        Returns: list of K8sPod

        """
        pods = []
        for p in self.k8sclient.pods.list_all():
            try:
                if p.data.get('spec', {}).get('node_name') == node_name:
                    pods.append(p)
            except ApiException as ex:
                LOG.warning(f"Raised exception while reading pod {p.namespace}/{p.name}:\n{ex}")
                continue
        return pods

    def get_leader_pods(self):
        """
        Get list of current leader pods based on ConfigMap data

        Returns: dict of <service_name>: <pod_name>

        """
        leader_postfix = "-leader-election"
        configmaps = self.k8sclient.configmaps.list_all()
        configmap_leader = [c for c in configmaps if c.name.endswith(leader_postfix)]
        result = dict()
        for cm in configmap_leader:
            leader_data = cm.data.get('metadata', {}).get('annotations', {})\
                .get('control-plane.alpha.kubernetes.io/leader', "")
            leader_pod_name = json.loads(leader_data).get('holderIdentity')
            result.update({cm.name.replace(leader_postfix, ""): leader_pod_name.split('_')[0]})
        return result

    @property
    def get_volumes_cleanup_enabled(self):
        return self.spec['providerSpec']['value'].get('volumesCleanupEnabled', False)

    def set_volumes_cleanup_enabled(self, enabled=True):
        body = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'volumesCleanupEnabled': enabled}}}}
        return self.patch(body)

    def set_postpone_distribution_upgrade(self, enabled=True):
        LOG.info(f"Set 'postponeDistributionUpdate' flag to '{enabled}' for cluster {self.namespace}/{self.name}")
        body = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'postponeDistributionUpdate': enabled
                    }
                }
            }
        }
        self.patch(body)

    @property
    def is_postpone_distribution_upgrade_enabled(self):
        return self.spec['providerSpec']['value'].get('postponeDistributionUpdate', False)

    def get_machines_by_distribution(self, target_distro):
        """Get Machines by the specified distribution

        1. Assert that distribution from Machine spec is the same as on the node
        2. Return touple of two lists: Machines that match the 'target_distro', and those which don't match.
        """
        machines = self.get_machines()
        distributions = self.get_machines_distributions_from_nodes()
        # Check if the cluster machines are ready for distribution upgrade
        wrong_distro_machines = [f"Machine {m.name} distribution name in spec: '{m.get_distribution()}' , "
                                 f"but on the node: '{distributions[m.name]}'" for m in machines
                                 if m.get_distribution() != distributions[m.name]]
        is_pending = self.is_postpone_distribution_upgrade_enabled
        if wrong_distro_machines:
            LOG.warning(f"Cluster postponeDistributionUpdate={is_pending}, while distribution upgrade is pending "
                        f"for the following Machines:\n" + '\n'.join(wrong_distro_machines))

        matched_machines = [m for m in machines if target_distro == distributions[m.name]]
        not_matched_machines = [m for m in machines if target_distro != distributions[m.name]]

        return matched_machines, not_matched_machines

    def get_machines_distributions_from_nodes(self):
        # Get distribution versions from machines hosts and convert it to distribution id
        perhostdata = self.get_nodes_kernel_and_tarfs_versions()
        expected_versions_map_current = self.get_expected_kernel_version()
        os_version_to_id = {}
        for item in expected_versions_map_current['allowedDistributions']:
            if not item.get('notgreenfield'):
                os_version_to_id[item['os_version']] = item['id']

        distributions = {
            # Default distribution version to "---", because RHEL distribution codes are undefined yet
            machine.name: os_version_to_id.get(perhostdata[machine.name]['os_version'], "---")
            for machine in self.get_machines() if not machine.is_disabled()
        }
        LOG.debug(f"Distributions found for cluster '{self.namespace}/{self.name}':\n{distributions}")
        return distributions

    @property
    def is_ceph_deployed(self):
        """
        Check if Ceph deployed on current cluster
        Returns: True or False

        """
        kubectl_client = self.k8sclient
        if (
            kubectl_client.rookcephclusters.available
            and len(kubectl_client.rookcephclusters.list(namespace='rook-ceph')) > 0
        ):
            return True
        return False

    def delete_pending_openstack_pods(self):
        def delete_pending_pvc():
            try:
                pvcs_for_delete = []
                openstack_pods = (self.k8sclient.pods.list(namespace='openstack'))
                openstack_pods.extend(self.k8sclient.pods.list(namespace='openstack-redis'))
                openstack_pods.extend(self.k8sclient.pods.list(namespace='tf'))
                for pod in openstack_pods:
                    if pod.data['status']['phase'] == 'Pending':
                        for pvc in self.k8sclient.pvolumeclaims.list_all():
                            if pod.data['metadata']['name'] in pvc.data['metadata']['name']:
                                LOG.info(f"Found PVC {pvc.data['metadata']['name']}, "
                                         f"for pod {pod.data['metadata']['name']}")
                                pvcs_for_delete.append(pvc)
                if pvcs_for_delete:
                    for pvc_for_delete in pvcs_for_delete:
                        LOG.info(f"Delete PVC {pvc_for_delete.data['metadata']['name']}")
                        pvc_for_delete.delete()
                    return False
                else:
                    LOG.info("Pending pods not found")
                    return True
            except Exception as e:
                LOG.info(f"Some objects not found during the collecting: {e}")
                return False

        waiters.wait(lambda: delete_pending_pvc(),
                     timeout=1800, interval=60,
                     timeout_msg="Timeout deletion Pending pvc objects")

    def set_or_update_ntp_servers(self, ntp_servers):
        """
        Set or update value of NTP servers for provider config.
        Can be used only already deployed clusters due some values in the cluster object
        tree appearing during deployment
        Potentially supported providers: ALL

        :param ntp_servers: list: List of NTP servers. If only one server exists - it should be list with one element
        :return:
        """
        LOG.info(f"Will update NTP servers value to {ntp_servers}")
        ntp_chunk = {
            "ntp": {
                "servers": ntp_servers
            }
        }

        regional_spec = (self.data['spec'].get('providerSpec', {}).get('value', {}).get('kaas', {}).
                         get('regional', {}))
        LOG.info(f">>> Regional spec before changes : {regional_spec}")

        for provider_spec in regional_spec:
            if provider_spec['provider'] != self.provider.provider_name:
                LOG.info(f"Skipping provider '{provider_spec['provider']}' for setting NTP config")
                continue
            LOG.info(f"Checking provider '{provider_spec['provider']}' to set NTP servers")
            for helm_release in provider_spec["helmReleases"]:
                if helm_release["name"] == f"{provider_spec['provider']}-provider":
                    LOG.info(f"Setting NTP servers for provider '{provider_spec['provider']}'")
                    config = helm_release["values"]["config"]
                    if not config.get('lcm', {}):
                        config['lcm'] = ntp_chunk
                    else:
                        if not config.get('ntp', {}):
                            config['lcm']['ntp'] = ntp_chunk['ntp']
                        else:
                            # If servers set - update value
                            config['lcm']['ntp']['servers'] = ntp_servers
                    break
            else:
                raise Exception(f"Regional helm release with name '{provider_spec['provider']}' not found "
                                f"in the cluster '{self.namespace}/{self.name}'")

        LOG.info('>>> Will apply this this regional spec for cluster')
        LOG.info(regional_spec)
        spec = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'kaas': {
                            'regional': regional_spec,
                        }
                    }
                }
            }
        }
        self.patch(body=spec)

    def wait_for_job_to_appear(self, name_prefix, namespace=None, timeout=300, interval=20):
        """Wait a job to be created. Main goal - to use it for scheduled cronjobs.

        :param name_prefix: Mandatory. Need some data to wait appearance.
        :param namespace: If set - filtering by ns. If not - cluster-wide
        :param timeout: Timeout of waiting
        :param interval: Interval of checks
        :return:
        """
        waiters.wait(lambda: self.k8sclient.jobs.list(namespace, name_prefix=name_prefix),
                     timeout=timeout, interval=interval, timeout_msg="Timed out waiting job for appear")

    def get_pods_by_job_name(self, job_name, namespace=None):
        if namespace:
            return [p for p in self.k8sclient.pods.list(namespace=namespace) if p.job_name == job_name]
        else:
            return [p for p in self.k8sclient.pods.list_all() if p.job_name == job_name]

    def get_cluster_runtime(self):
        """Return cluster runtime. If few runtime found on nodes - return 'mixed'"""
        runtimes = []
        for machine in self.get_machines():
            runtimes.append(machine.runtime)

        docker = any('docker' in runtime for runtime in runtimes)
        containerd = any('containerd' in runtime for runtime in runtimes)
        if docker and containerd:
            # that can be happen if migration in progress or partial
            runtime = 'mixed'
        elif docker:
            # default for 2.27 or upgraded to 2.28 and not migrated yet
            runtime = 'docker'
        elif containerd:
            # default for 2.28 clean or upgraded and migrated
            runtime = 'containerd'
        else:
            # weird case and not possible now but should be covered just in case
            runtime = 'unknown'
        return runtime

    def update_plan_get(self, version):
        status = self.data.get("status") or {}
        if not status or type(status) is not dict:
            return None
        if "+" in version:
            version = version[:version.index("+")]
        return self.__manager.api.kaas_clusterupdateplans.get(
            namespace=self.namespace, name=f"{self.name}-{version}")

    def update_plan_list(self):
        update_plans = self.__manager.api.kaas_clusterupdateplans.list(namespace=self.namespace)
        cluster_update_plans = [update_plan for update_plan in update_plans
                                if update_plan.data.get('spec', {}).get('cluster') == self.name]
        return cluster_update_plans

    def update_plan_set_steps_commence(self, version, steps_id, commence, step_key="id"):
        plan = self.update_plan_get(version)
        current_steps = plan.data.get("spec", {}).get("steps", [])
        for s in current_steps:
            step_id = s.get(step_key)
            if step_id in steps_id and s.get("commence") != commence:
                s["commence"] = commence
                LOG.info(f"Step '{step_id}' details:\n{yaml.dump(s)}\n")
        body = {
            "spec": {
                "steps": current_steps
            }
        }
        plan.patch(body=body, verbose=False)

    def update_plan_steps_wait_for_completion(
            self, version, steps_id, step_key="id", timeout=None, interval=30):
        plan = self.update_plan_get(version)

        if timeout:
            LOG.info(f"Using the specified timeout '{timeout}s' for steps '{steps_id}'")
        else:
            timeout = 0
            LOG.info(f"Extracting timeout from update plan 'eta' for steps '{steps_id}'")
            current_steps = plan.data.get("spec", {}).get("steps", [])
            for s in current_steps:
                step_id = s.get(step_key)
                if step_id in steps_id:
                    step_timeout = 0
                    eta = s.get('duration', {}).get('eta')
                    if eta:
                        parsed_time = utils.parse_time_value(eta)
                        if parsed_time:
                            step_timeout = (parsed_time["hours"] * 3600 +
                                            parsed_time["minutes"] * 60 +
                                            parsed_time["seconds"])

                            # W/A for PRODX-46983
                            LOG.info(f"PRODX-46983: Increase timeout for the step '{step_id}' by 1800sec")
                            step_timeout += 1800

                            LOG.info(f"Step '{step_id}' timeout from update plan 'eta': '{step_timeout}s'")

                    if not step_timeout:
                        # Default timeout for step
                        step_timeout = 3600
                        LOG.warning(f"No valid 'eta' found for the step '{step_id}', using default '{step_timeout}s'")
                    if step_timeout > settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT:
                        LOG.warning(f"Timeout '{step_timeout}s' for step '{step_id}' is too high, replace it with "
                                    f"timeout '{settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT}' "
                                    f"from KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT")
                        step_timeout = settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT
                    # Calculate total timeout for all specified steps in "steps_id"
                    timeout += step_timeout

        def wait_func():
            status = plan.data.get("status") or {}
            steps = status.get("steps", [])
            self.check.show_updateplan_steps_status(steps, step_key, steps_id)
            self.check.show_machines_conditions()

            steps_status = {s.get(step_key): s.get("status")
                            for s in steps
                            if s.get(step_key) in steps_id}
            LOG.info(f"Current steps status: '{steps_status}'")
            # Wait until all steps statuses are appeared in the clusterupdateplan status
            if len(steps_status.keys()) != len(steps_id):
                return False
            # Check that all steps statuses are completed
            return all(step_status == "Completed" for step_status in steps_status.values())

        LOG.info(f"Wait for completion of update plan steps {steps_id} with total timeout '{timeout}s'")
        waiters.wait(
            wait_func, timeout=timeout, interval=interval,
            timeout_msg=f"Timed out waiting for steps {steps_id} to complete")

    def get_desired_clusterrelease_version(self):
        """Get the version of the clusterrelease whose name is set in the cluster spec"""

        cluster_release_name = self.cluster_spec_release_version
        if not cluster_release_name:
            return None
        cluster_release = self._manager.get_clusterrelease(cluster_release_name)
        return cluster_release.data.get('spec', {}).get('version', None)

    def set_day1_provisioning(self, mode, machines):
        """Set spec.providerSpec.value.day1Provisioning for machines
        Args:
            mode (str): Mode to set to day1Provisioning
            'auto' allows automatic progression through the provisioning workflow.
            'manual' or empty string requires explicit approval before proceeding with provisioning.
            empty string is the same as 'manual'
            machines (list): List of machine objects to set pause for
            set for all machines
        """
        for machine in machines:
            current_value = (machine.data.get('spec', {}).get('providerSpec', {}).get('value', {})
                             .get('day1Provisioning', {}))
            if mode == current_value:
                continue
            patch = {
                "spec": {
                    "providerSpec": {
                        "value": {
                            "day1Provisioning": mode
                        }
                    }
                }
            }
            LOG.info(f"Setting day1Provisioning for machine {machine.namespace}/{machine.name} to {mode}")
            self._manager.api.kaas_machines.update(
                name=machine.name,
                namespace=machine.namespace,
                body=patch
            )
        LOG.info("Machines were successfully updated with day1Provisioning")

    def set_day1_deployment(self, mode, machines):
        """Set spec.providerSpec.value.day1Deployment for all machines
        Args:
            mode (str): Mode to set to day1Deployment
            'auto' allows automatic progression through the deployment workflow.
            'manual' or empty string requires explicit approval before proceeding with deployment.
            empty string is the same as 'manual'
            machines (list): List of machine objects to set pause for
            set for all machines
        """
        for machine in machines:
            current_value = (machine.data.get('spec', {}).get('providerSpec', {}).get('value', {})
                             .get('day1Deployment', {}))
            if mode == current_value:
                continue
            patch = {
                "spec": {
                    "providerSpec": {
                        "value": {
                            "day1Deployment": mode
                        }
                    }
                }
            }
            LOG.info(f"Setting day1Deployment for machine {machine.namespace}/{machine.name} to {mode}")
            self._manager.api.kaas_machines.update(
                name=machine.name,
                namespace=machine.namespace,
                body=patch
            )
        LOG.info("Machines were successfully updated with day1Deployment")

    def remove_kaascephcluster(self):
        """Remove KaaSCephCluster
        Args:
        """
        kcc = self.get_cephcluster()
        if kcc:
            if version.parse(self.clusterrelease_version) >= version.parse('mosk-17-0-4-25-1'):
                LOG.info(f"Adding annotation for removing KaasCephCluster '{kcc.namespace}/{kcc.name}'")
                metadata = {
                    'metadata': {
                        'annotations': {
                            'manual-get-rid-of-kaascephcluster': 'si-test',
                        }
                    }
                }
                kcc.patch(metadata)
            else:
                LOG.info(f"Cluster version: {version.parse(self.clusterrelease_version)} "
                         f"does not support KaaSCephCluster deletion procedure. "
                         f"Cluster version should be >= mosk-17-0-4-25-1")
        else:
            LOG.info(f"No KaaSCephCluster found to delete in cluster: '{self.name}/{self.namespace}'")


class Machine(object):
    """Provider independent methods for the kaas machine"""

    def __init__(self, manager, cluster, kaasmachine):
        """
        manager: <Manager> instance
        cluster: <Cluster> instance with attribute "bastion_ip"
                 "bastion_ip" is a public IP address of a jump host that
                 is connected to the internal network with other nodes
                 in the cluster
        kaasmachine: <KaaSMachine> or <KaaSBaremetalHosts> instance
        """
        self.__manager = manager
        self.__cluster = cluster
        self.__kaasmachine = kaasmachine
        self.__uid = None
        self.__metadata = None
        self.__spec = None
        self.__machine_type = None
        self.__machine_types = None

    @property
    def provider(self):
        raise NotImplementedError("Abstract method")

    @property
    def internal_ip(self):
        raise NotImplementedError("Abstract method")

    @property
    def public_ip(self):
        raise NotImplementedError("Abstract method ")

    def power_off(self):
        raise NotImplementedError("Method is not implemented for current provider yet")

    def power_on(self):
        raise NotImplementedError("Method is not implemented for current provider yet")

    def get_power_status(self):
        raise NotImplementedError("Method is not implemented for current provider yet")

    @property
    def _manager(self):
        return self.__manager

    @property
    def _cluster(self):
        return self.__cluster

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def name(self):
        """Machine name"""
        return self.__kaasmachine.name

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def namespace(self):
        """Machine namespace"""
        return self.__kaasmachine.namespace

    @property
    def metadata(self):
        """Cached machine metadata"""
        if self.__metadata is None:
            self.__metadata = self.data['metadata']
        return self.__metadata

    @property
    def spec(self):
        """Cached machine spec"""
        if self.__spec is None:
            self.__spec = self.data['spec']
        return self.__spec

    @property
    def uid(self):
        """Machine uid"""
        if self.__uid is None:
            self.__uid = self.metadata['uid']
        return self.__uid

    @property
    def status(self):
        return self.data.get('status') or {}

    @property
    def machine_type(self):
        """Cached machine type
        :rtype string: One of ('control', 'worker', 'storage')
        """
        if self.__machine_type is None:
            self.__machine_type = \
                utils.get_type_by_labels(self.metadata['labels'])
        return self.__machine_type

    @property
    def machine_types(self):
        """Cached machine types
        :rtype list: with any of valid machine types:
            'control', 'worker', 'storage'
        """
        if self.__machine_types is None:
            self.__machine_types = \
                utils.get_types_by_labels(self.metadata['labels'])
        return self.__machine_types

    @property
    def l2template_name(self):
        l2t_selector_name = self.data['spec']['providerSpec']['value'].get(
            'l2TemplateSelector', {}).get('name', '')
        l2t_selector_label = self.data['spec']['providerSpec']['value'].get(
            'l2TemplateSelector', {}).get('label', '')
        if not l2t_selector_name and not l2t_selector_label:
            return 'default'
        if l2t_selector_name:
            return l2t_selector_name
        if l2t_selector_label:
            all_l2t = self.__manager.get_l2templates(namespace=self.__cluster.namespace)
            l2t = [l2t.name for l2t in all_l2t if l2t.data['metadata'].get(
                'labels', {}).get(l2t_selector_label)]
            return l2t[0] if l2t else ''

    def is_machine_type(self, mtype):
        """
        :mtype string: One of ('control', 'worker', 'storage')
        """
        if mtype in self.machine_types:
            return True
        return False

    @property
    def machine_status(self):
        """Machine status from providerStatus

        Status object may disappear for a short periods, return None
        :rtype string: One of (None, 'Ready', 'Pending', 'Updating', ...)
        """
        machine_data = self.data or {}
        machine_status = machine_data.get('status') or {}
        provider_status = machine_status.get('providerStatus', {}).get('status')
        return provider_status or None

    @property
    def host_platform(self):
        if self._cluster.is_management:
            lcm_machine = self._manager.get_lcmmachine(name=self.name, namespace=self.namespace)
        else:
            parent_cluster = self._cluster.get_parent_cluster()
            lcm_machine = parent_cluster.get_cluster_lcmmachine(name=self.name, namespace=self.namespace)
        lcm_machine_status = lcm_machine.data.get('status') or {}
        return lcm_machine_status.get('hostInfo', {}).get('os', {}).get('platform', '')

    @property
    def lcmmachine(self):
        return self._cluster.get_cluster_lcmmachine(name=self.name, namespace=self.namespace)

    @property
    def lcmmachine_status(self):
        return self.lcmmachine.data.get('status') or {}

    @property
    def runtime(self):
        return self.get_k8s_node().runtime

    def is_lcmmachine_stuck(self, attempts_threshold=20):
        lcmmachine_status = self.lcmmachine_status
        lcm_stuck = lcmmachine_status.get('lcmOperationStuck', False)
        if lcm_stuck:
            return lcm_stuck

        # (ddmitriev): Workaround for PRODX-40790
        state_item_statuses = lcmmachine_status.get('stateItemStatuses', {})
        for state_name, state_data in state_item_statuses.items():
            attempts = state_data.get('attempt', 0)
            exit_code = state_data.get('exitCode', 0)
            if exit_code != 0 and attempts > attempts_threshold:
                message = (f"Flag 'lcmOperationStuck' is missing for LCMMachine "
                           f"'{self.namespace}/{self.name}', while state '{state_name}' "
                           f"has been re-tried {attempts} times and should be considered as 'stuck'")
                LOG.error(message)
                if self._cluster.workaround.prodx_40790():
                    LOG.warning(f"Consider LCMMachine '{self.namespace}/{self.name}' as 'stuck' "
                                f"because of a lot failed attempts")
                    return True
                else:
                    raise Exception(message)
        return False

    def get_k8s_node_name(self):
        if self.provider == 'byo':
            return self.data['status']['nodeRef']['name']
        if self.data and self.data['status']:
            instanceName = self.data['status'].get('instanceName', None)
            if instanceName is not None:
                return instanceName
        if self.metadata['annotations']:
            node_name = 'kaas-node-' + \
                        self.metadata['annotations']['kaas.mirantis.com/uid']
            return node_name
        else:
            raise Exception(f"Annotation 'kaas.mirantis.com/uid' not found in machine '{self.name}'")

    def get_k8s_node(self):
        node_name = self.get_k8s_node_name()
        if node_name:
            return self._cluster.k8sclient.nodes.get(
                name=node_name
            )

    def check_k8s_nodes_has_labels(self):
        node_meta = self.get_k8s_node().read().metadata
        nodeLabels = {x['key']: x['value'] for x in self.nodeLabels}
        not_assigned_labels = self._cluster.get_allowed_node_labels()
        result = {'Missing labels': [], 'Unexpected labels': []}
        if nodeLabels:
            for k, v in nodeLabels.items():
                if k not in node_meta.labels.keys() or \
                        v != node_meta.labels[k]:
                    result['Missing labels'].append({k: v})
                del not_assigned_labels[k]
        for k, v in not_assigned_labels.items():
            if k in node_meta.labels.keys():
                result['Unexpected labels'].append({k: v})

        if result['Unexpected labels'] or result['Missing labels']:
            LOG.warning(f"Labels mismatch on Machine {self.namespace}/{self.name}: {result}. "
                        f"K8S node labels: {node_meta.labels}")
            return False
        LOG.info(f"Labels on Machine {self.namespace}/{self.name} match labels on the Node {node_meta.name}")
        return True

    def get_bmh_id(self):
        _id = self.spec['providerSpec']['value'].get(
            'hostSelector', {}).get('matchLabels', {}).get(
            'kaas.mirantis.com/baremetalhost-id')
        if not _id:
            LOG.warning(f"HostSelector label for link <machine> and <bmh>"
                        f"not found for machine:{self.name}")
        return _id

    def get_bmh_name(self):
        _name = self.metadata['annotations'].get(
            'metal3.io/BareMetalHost')
        if not _name:
            LOG.warning(f"HostSelector label for link <machine> and <bmh>"
                        f"not found for machine:{self.name}")
        else:
            _name = _name.split('/')[1]
        return _name

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind,

        metadata, spec, status or items
        """

        return self.__kaasmachine.read().to_dict()

    def patch(self, *args, **kwargs):
        self.__kaasmachine.patch(*args, **kwargs)

    def replace(self, *args, **kwargs):
        self.__kaasmachine.replace(*args, **kwargs)

    @property
    def nodeLabels(self):
        spec = self.data['spec']
        if 'nodeLabels' in spec['providerSpec']['value']:
            node_labels = [{'key': label['key'], 'value': label.get('value', '')}
                           for label in spec['providerSpec']['value']['nodeLabels']]
            LOG.debug(f"Machine {self.name} nodeLabels: {node_labels}")
            return node_labels
        return []

    def add_labels(self, labels):
        """Add labels to spec.providerSpec.value.nodeLabels in Machine"""
        # WARNING(alexz): thats actuall nodeLabels,not machine labels!
        if self.nodeLabels:
            existing_labels = self.nodeLabels
            LOG.info("Existing labels: {}".format(existing_labels))
            # TODO maybe check for duplicates
            existing_labels += labels
            body = \
                {"spec": {
                    'providerSpec': {
                        'value': {'nodeLabels': existing_labels}}}}
        else:
            body = \
                {"spec": {
                    'providerSpec': {
                        'value': {'nodeLabels': labels}}}}
        self.__manager.api.kaas_machines.update(
            name=self.name, namespace=self.namespace, body=body)

    def add_k8s_node_labels(self, labels=None):
        """Add labels to k8s Node object (not Machine nodeLabels)"""
        if not labels:
            LOG.warning("Labels to add are not set. Skipping")
            return
        assert type(labels) is dict, "Please, use dict format for annotations"
        k8s_node = self.get_k8s_node()
        body = {'metadata': {'labels': labels}}
        k8s_node.patch(body=body)

    def add_machine_labels(self, labels=None):
        """Add labels to metadata.labels in Machine object"""
        if not labels:
            LOG.warning("Labels to add are not set. Skipping")
            return
        assert type(labels) is dict, "Please, use dict format for annotations"
        body = {'metadata': {'labels': labels}}
        self.__kaasmachine.patch(body=body)

    def has_k8s_labels(self, labels) -> bool:
        """Check labels on K8sNode"""
        k8s_node = self.get_k8s_node()
        node_labels = k8s_node.data.get('metadata', {}).get('labels', {})
        for label, value in labels.items():
            if label not in node_labels or value != node_labels[label]:
                return False
        return True

    def has_nodelabels(self, labels) -> bool:
        """Check labels on Machine spec.providerSpec.value.nodeLabels

        :param labels: list of dicts, contain labels to check. Same as nodeLabels structure
        """
        nodelabels = self.nodeLabels
        return all([
                    any([nodelabel == label for nodelabel in nodelabels])
                    for label in labels])

    def has_machine_labels(self, labels) -> bool:
        """Check labels on K8sNode"""
        machine_labels = self.data.get('metadata', {}).get('labels', {})
        for label, value in labels.items():
            if label not in machine_labels or value != machine_labels[label]:
                return False
        return True

    @property
    def machinelabels(self) -> dict:
        return self.data.get('metadata', {}).get('labels', {})

    @property
    def k8snodelabels(self) -> dict:
        k8s_node = self.get_k8s_node()
        return k8s_node.data.get('metadata', {}).get('labels', {})

    def get_k8s_node_annotations(self):
        k8s_node = self.get_k8s_node()
        return k8s_node.data.get('metadata', {}).get('annotations', {})

    def add_k8s_node_annotations(self, annotations=None):
        """
        Add annotation to k8s node in format
        {'annotation_name1': 'annotation_value1',
         'annotation_name2': 'annotation_value2',
         etc}
        """
        if not annotations:
            LOG.warning("Annotations to add are not set. Skipping")
            return
        assert type(annotations) is dict, "Please, use dict format for annotations"
        k8s_node = self.get_k8s_node()
        body = {'metadata': {'annotations': annotations}}
        k8s_node.patch(body=body)

    def remove_k8s_node_annotations(self, annotations=None):
        """
        TODO(alexz:) Substitute with utils.remove_k8s_obj_annotations ?
        Remove annotation from k8s node.
        use list of keys for removing. For example:
        annotations == {'annotation1': 'value1',
                        'annotation2': 'value2',
                        'annotation3': 'value3'}
        remove_k8s_node_annotations(annotations=['annotation1', 'annotation2'])

        """
        if not annotations:
            LOG.warning("Annotations to remove are not set. Skipping")
            return
        assert type(annotations) is list, "Please, use list of annotations keys for removing"
        k8s_node = self.get_k8s_node()
        k8s_node_name = k8s_node.name
        existing = k8s_node.data.get('metadata', {}).get('annotations', {})
        if existing:
            for annotation in annotations:
                if annotation in existing:
                    LOG.info(f"Annotation \"{annotation}:{existing[annotation]}\" "
                             f"will be removed from node {k8s_node_name}")
                    existing.pop(annotation)
                else:
                    LOG.info(f"Annotation {annotation} not found in existing annotations "
                             f"for node {k8s_node_name}. Skipping")
        else:
            LOG.warning(f"No annotations are existed in node {k8s_node_name}. Skipping")
        data = k8s_node.read()
        data.metadata.annotations = existing
        k8s_node.replace(body=data)

    def remove_labels(self, labels=None):
        """Remove labels from spec.providerSpec.value.nodeLabels in Machine object

        :param labels: list of dicts, contain labels to remove. Same as nodeLabels structure
        """
        if labels:
            updated_labels = [x for x in self.nodeLabels
                              if x not in labels]
            body = \
                {"spec": {
                    'providerSpec': {
                        'value': {'nodeLabels': updated_labels}}}}
        else:
            body = \
                {"spec": {
                    'providerSpec': {
                        'value': {'nodeLabels': None}}}}
        self.__manager.api.kaas_machines.update(
            name=self.name, namespace=self.namespace, body=body)

    def remove_k8s_node_labels(self, labels=None):
        """Remove labels from metadata.labels in K8sNode object

        :param labels: list of labels keys to remove
        """
        if not labels:
            LOG.warning("Labels to remove are not set. Skipping")
            return
        assert type(labels) is list, "Please, use list of labels keys for removing"
        k8s_node = self.get_k8s_node()
        k8s_node_name = k8s_node.name
        k8s_node_labels = k8s_node.data.get('metadata', {}).get('labels', {})
        if k8s_node_labels:
            for label in labels:
                if label in k8s_node_labels:
                    LOG.info(f"Label \"{label}:{k8s_node_labels[label]}\" "
                             f"will be removed from node {k8s_node_name}")
                    k8s_node_labels[label] = None
                else:
                    LOG.info(f"Label {label} not found in existing labels "
                             f"for node {k8s_node_name}. Skipping")
            body = {'metadata': {'labels': k8s_node_labels}}
            k8s_node.patch(body=body)
        else:
            LOG.warning(f"Node {k8s_node_name} doens't contain any labels. Skipping")

    def remove_machine_labels(self, labels=None):
        """Remove labels from metadata.labels in Machine object

        :param labels: list of labels keys to remove
        """
        if not labels:
            LOG.warning("Labels to remove are not set. Skipping")
            return
        assert type(labels) is list, "Please, use list of labels keys for removing"
        machine_labels = self.data.get('metadata', {}).get('labels', {})
        if machine_labels:
            for label in labels:
                if label in machine_labels:
                    LOG.info(f"Label \"{label}:{machine_labels[label]}\" "
                             f"will be removed from Machine {self.name}")
                    machine_labels[label] = None
                else:
                    LOG.info(f"Label {label} not found in existing labels "
                             f"for Machine {self.name}. Skipping")
            body = {'metadata': {'labels': machine_labels}}
            self.__kaasmachine.patch(body=body)
        else:
            LOG.warning(f"Machine {self.name} doens't contain any labels. Skipping")

    def set_baremetalhost_power(self, online=True, wait_expected_power=True, timeout=1200, interval=10):
        """
        online=True - Powered on
        online=False - Safe(soft) power off
        """
        bmh_name = self.get_bmh_name()
        if (self.__manager.api.kaas_baremetalhostinventories.available and
                self.__manager.api.kaas_baremetalhostinventories.present(name=bmh_name, namespace=self.namespace)):
            bmhi = self.__manager.get_baremetalhostinventory(name=bmh_name, namespace=self.namespace)
            LOG.info(f"Set BMH Inventory '{bmhi.namespace}/{bmhi.name}' power state to '{online}'")
            bmhi.patch({'spec': {'online': online}})
        else:
            bmh = self.__manager.get_baremetalhost(bmh_name, namespace=self.namespace)
            LOG.info(f"Set BMH '{bmh.namespace}/{bmh.name}' power state to '{online}'")
            bmh.patch({'spec': {'online': online}})

        if wait_expected_power:
            LOG.info(f"Wait until BMH '{self.namespace}/{bmh_name}' status poweredOn='{online}'")
            waiters.wait(
                lambda: self.get_power_status() == online,
                timeout=timeout,
                interval=interval,
                timeout_msg=f"Machine '{self.namespace}/{self.name}' didn't get power status '{online}'")
            LOG.info(f"Got BMH '{self.namespace}/{bmh_name}' expected power status '{online}'")

    def get_upgrade_index(self, default=None):
        """
        If ``upgradeIndex`` in the Machine's spec is set, status value is equal
        to the value in the spec. Otherwise its value shows the autogenerated order
        of upgrade.
        """
        return self.status.get('providerStatus', {}).get('upgradeIndex', default)

    def set_upgrade_index(self, u_index=1, wait_index_applyed=False):
        body = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'upgradeIndex': u_index}}}}
        self.__kaasmachine.patch(body)

        def _wait_index_applyed(timeout=60, interval=10):
            timeout_msg = f"Current index is not equeal index set after {timeout} sec"
            waiters.wait(lambda: self.data['status']['providerStatus'].get('upgradeIndex', None) == u_index,
                         timeout=timeout, interval=interval, timeout_msg=timeout_msg)
        if wait_index_applyed:
            LOG.info("Waiting upgrade index applyed")
            _wait_index_applyed()

    def set_distribution(self, distribution):
        body = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'distribution': distribution
                    }
                }
            }
        }
        self.__kaasmachine.patch(body)

    def set_distribution_and_migrate(self, distribution, runtime):
        """Migrate runtime of machine and set distribution

        This func needed only for covering case with simultaneous migration and distrib upgrade
        :param distribution:
        :param runtime:
        :return:
        """
        body = {
            'metadata': {
                'annotations': {
                    'kaas.mirantis.com/preferred-container-runtime': runtime
                }
            },
            'spec': {
                'providerSpec': {
                    'value': {
                        'distribution': distribution
                    }
                }
            }
        }
        self.__kaasmachine.patch(body)

    def get_distribution(self):
        return self.data['spec']['providerSpec']['value'].get('distribution')

    def set_deletion_policy(self, policy):
        body = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'deletionPolicy': policy
                    }
                }
            }
        }
        self.__kaasmachine.patch(body)

    def get_deletion_policy(self):
        return self.data['spec']['providerSpec']['value'].get('deletionPolicy')

    def get_delete_status(self):
        return self.data['status']['providerStatus']['delete']

    def get_prepare_deletion_phase(self):
        return self.data['status']['providerStatus'].get('prepareDeletionPhase', '')

    def delete(self, check_deletion_policy=False, new_delete_api=True):
        """Deletes the current machine"""
        if new_delete_api:
            body = {
                'spec': {
                    'providerSpec': {
                        'value': {
                            'delete': True
                        }
                    }
                }
            }
            delete_machine = functools.partial(self.__kaasmachine.patch, body)
        else:
            delete_machine = self.__kaasmachine.delete

        if check_deletion_policy:
            with machine_deletion_policy_manager.create(self._cluster, self) as policy_manager:
                delete_machine()
                policy_manager.check()
        else:
            delete_machine()

    def abort_graceful_deletion(self):
        policy = self.get_deletion_policy()
        if policy != 'graceful':
            raise Exception(f"cannot abort deletion with policy {policy}")

        self.__kaasmachine.patch(body={
            'spec': {
                'providerSpec': {
                    'value': {
                        'delete': False
                    }
                }
            }
        })

    def _run_cmd(self, cmd, verbose=True, timeout=None,
                 expected_codes=None, check_exit_code=True,
                 verbose_info=False, ssh_key=None, ssh_login="mcc-user",
                 sudo=False, reconnect=False):
        """Direct or through-bastion shell command execution"""

        expected_codes = expected_codes or [0]
        machine_ip = self.public_ip or self.internal_ip

        keys = utils.load_keyfile(ssh_key)
        pkey = utils.get_rsa_key(keys['private'])
        auth = exec_helpers.SSHAuth(username=ssh_login,
                                    password='', key=pkey)

        if self.public_ip or (self.internal_ip and
                              not self.__cluster.bastion_ip):
            # For public IP or directly accessible internal ip
            if verbose_info:
                LOG.info("SSH to the kaas node <{0}> using machine IP {1}"
                         .format(self.name, machine_ip))
            ssh = exec_helpers.SSHClient(
                host=machine_ip,
                port=22,
                auth=auth,
            )
            ssh.logger.addHandler(logger.console)
            if reconnect:
                ssh.reconnect()
            if sudo:
                with ssh.sudo(True):
                    result = ssh.execute(cmd, verbose=verbose, timeout=timeout)
            else:
                result = ssh.execute(cmd, verbose=verbose, timeout=timeout)
        elif self.internal_ip:
            if not self.__cluster.bastion_ip:
                raise Exception("Node {0} don't have bastion_ip to access the "
                                "internal address {1}"
                                .format(self.name, self.internal_ip))

            if verbose_info:
                LOG.info("SSH to the kaas node <{0}> using bastion IP '{1}' "
                         "and internal IP '{2}'"
                         .format(self.name,
                                 self.__cluster.bastion_ip,
                                 self.internal_ip))

            ssh = self.__cluster._get_bastion_remote(ssh_login=ssh_login,
                                                     ssh_key=ssh_key)
            result = ssh.execute_through_host(self.internal_ip, cmd,
                                              auth=auth, verbose=verbose,
                                              timeout=timeout)
        else:
            raise Exception("Node {0} don't have neither "
                            "public nor internal IP. "
                            "Current ones are Bastion {1} "
                            "internal {2}".format(self.name,
                                                  self.__cluster.bastion_ip,
                                                  self.internal_ip))

        if result.exit_code not in expected_codes and check_exit_code:
            message = (
                "Command {result.cmd!r} returned exit code "
                "{result.exit_code!s} while expected {expected!s}\n"
                "STDERR: {stderr!s}".format(
                    result=result, expected=expected_codes,
                    stderr=result.stderr_str
                )
            )
            LOG.error(msg=message)
            raise Exception(message)

        return result

    def run_cmd(self, cmd, verbose=True, timeout=None,
                expected_codes=None, check_exit_code=True,
                verbose_info=False, reconnect=False, ssh_key=None):
        """Direct or through-bastion shell command execution"""
        LOG.debug(f"Use user `{self.__cluster.ssh_user}` "
                  f"for remote command run")

        private_key = ssh_key if ssh_key else self.__cluster.private_key

        use_sudo = False
        if self.provider == 'vsphere':
            use_sudo = True

        return self._run_cmd(cmd, verbose, timeout, expected_codes,
                             check_exit_code, verbose_info,
                             ssh_key=private_key,
                             ssh_login=self.__cluster.ssh_user,
                             sudo=use_sudo, reconnect=reconnect)

    @retry_for_rcs(exceptions=ApiException, rcs=[500])
    def exec_pod_cmd(self, cmd, verbose=True, get_events=True, timeout=600, delete_pod=True, pod_type='base'):
        """Execute cmd in a privileged pod on the Machine

        Doesn't require SSH access to the Machine.

        * The {cmd} is executed in the privileged container from
          the chroot to the node's root filesystem /

        * A separated pod is started and removed for each {cmd}, so
          it will spend some time to pull the image and start the pod.

        * There is no way to interact with {cmd} from shell while it
          is executed. All required parameters must be set in {cmd}

        * There is no way to separate stdout from stderr, all the
          info is collected from the pod "logs".

        * String-based selector to choose between usual usage and calico-specific

        :rtype dict: {'logs': <str>, 'events': <str>, 'exit_code': <int>}
        """
        if self.is_disabled():
            raise Exception(f"Machine {self.name} is in 'Disabled' state, cannot execute command '{cmd}'")
        node_name = self.get_k8s_node_name()
        mcp_docker_registry = self._cluster.determine_mcp_docker_registry()
        render_options = {}
        if pod_type == 'calico':
            render_options['UCP_CERT_PATH'] = '/var/lib/docker/volumes/ucp-node-certs/_data'
            if self._cluster.workaround.prodx_37306() or self._cluster.workaround.prodx_51513():
                render_options['UCP_CERT_PATH'] = '/var/lib/docker/volumes/ucp-kv-certs/_data'
            pod_yaml = settings.CALICOCTL_PROVELEGED_POD_YAML
        else:
            pod_yaml = settings.MACHINE_PRIVELEGED_POD_YAML

        results = self._cluster.k8sclient.pods.exec_pod_cmd(
            cmd=cmd,
            registry=mcp_docker_registry,
            node_name=node_name,
            pod_template_path=pod_yaml,
            render_options=render_options,
            verbose=verbose,
            get_events=get_events,
            timeout=timeout,
            delete_pod=delete_pod)

        return results

    def are_conditions_ready(self, expected_fails=None, verbose=False):
        """
        Check if all Machine conditions in ready status
        :param expected_fails: None or dict like the following
            {<condition.type>: <expected message pattern>, ...}
            If expected message pattern is empty, then the whole
            condition is ignored. If it is not empty - then
            this pattern is matched to the message from the cluster
            to check if the message is expected or not.
        :rtype bool: bool
        """
        data = self.data
        result = self._cluster.get_conditions_status(
            data, expected_fails, verbose)
        if result['not_ready']:
            return False
        else:
            return True

    def machine_maintenance_start(self):
        """
        Start Node maintenance
        Returns: None

        """
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "maintenance": True
                    }
                }
            }
        }

        self.__manager.api.kaas_machines.update(name=self.name, namespace=self.namespace, body=body)

        dedicated_control_plane_status = self._cluster.get_cluster_dedicated_control_plane_status()
        if dedicated_control_plane_status and self.machine_type == "control":
            LOG.info("Node Maintenance request should not be created, dedicatedControlPlane=true on control node")
        else:
            node_name = self.get_k8s_node_name()
            LOG.info(f"Wait until 'nodemaintenancerequest' created for Node {node_name}  (Machine {self.name})")
            self._cluster.check.wait_nodemaintenancerequest_created(node_name=node_name)

        LOG.info("Wait expected Node Maintenance status")
        self._cluster.check.wait_machine_maintenance_status(self.name, expected_status=True)

    def machine_maintenance_stop(self):
        """
        Stop Node maintenance
        Returns: None

        """
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "maintenance": False
                    }
                }
            }
        }

        self.__manager.api.kaas_machines.update(name=self.name, namespace=self.namespace, body=body)

        node_name = self.get_k8s_node_name()
        LOG.info(f"Wait until 'nodemaintenancerequest' deleted for Node {node_name}  (Machine {self.name})")
        self._cluster.check.wait_nodemaintenancerequest_not_existed(node_name=node_name)

        LOG.info("Wait expected Machine Maintenance status")
        self._cluster.check.wait_machine_maintenance_status(self.name, expected_status=False)

    @property
    def is_in_maintenance(self):
        # TODO Find in tests and replace calls with usage of this property
        return self.data.get('spec', {}).get('providerSpec', {}).get('value', {}).get('maintenance', False)

    def disable(self):
        """Day 2 Operations: set flag to disable LCM operations for the Machine"""
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "disable": True
                    }
                }
            }
        }
        self.__kaasmachine.patch(body)
        LOG.info(f"Machine {self.namespace}/{self.name} was disabled")

    def enable(self):
        """Day 2 Operations: set flag to enable LCM operations for the Machine"""
        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "disable": False
                    }
                }
            }
        }
        self.__kaasmachine.patch(body)
        LOG.info(f"Machine {self.namespace}/{self.name} was enabled")

    def wait_for_status_presence(self, status: str, timeout=900, interval=30):
        """
        Wait for expected Machine status
        Args:
            status: machine status to wait
            timeout: timeout for wait
            interval: interval for wait
        """
        def check_status():
            machine_status = self.machine_status
            LOG.info(f"Machine {self.name} expected status presence: '{status}', actual status: '{machine_status}'")
            return machine_status == status

        waiters.wait(check_status, timeout=timeout, interval=interval,
                     timeout_msg=f"Timeout waiting for Machine {self.name} status '{status}' presence")

    def wait_for_status_absence(self, status: str, timeout=900, interval=30):
        """
        Wait for expected Machine status
        Args:
            status: machine status to wait
            timeout: timeout for wait
            interval: interval for wait
        """
        def check_status():
            machine_status = self.machine_status
            LOG.info(f"Machine {self.name} expected status absence: '{status}', actual status: '{machine_status}'")
            return machine_status != status

        waiters.wait(check_status, timeout=timeout, interval=interval,
                     timeout_msg=f"Timeout waiting for Machine {self.name} status '{status}' absence")

    def is_disabled(self):
        """Day 2 Operations: get flag which disable/enable LCM operations for the Machine"""
        return self.data.get('spec', {}).get('providerSpec', {}).get('value', {}).get('disable', False)

    def check_condition_cluster(self):
        """Check that Cluster conditions were changed according to the Machine maintenance status"""
        dedicated_control_plane = self._cluster.get_cluster_dedicated_control_plane_status()
        conditions_status = self._cluster.get_conditions_status(self.data)
        assert_message = f"{self.machine_type} machine_type machine {self.name} " \
                         f"with dedicatedControlPlane = {dedicated_control_plane} " \
                         f"mod has an invalid status\n Cluster state:{conditions_status}"
        if dedicated_control_plane and self.machine_type == 'control':
            assert conditions_status['not_ready'] == [], assert_message
        else:
            assert (
                conditions_status['not_ready'] in (['Kubelet'], self._cluster.workaround.prodx_47907())
            ), assert_message

    def wait_condition_cluster(self, timeout=60, interval=5):
        """Wait that Cluster conditions were changed according to the Machine maintenance status"""
        waiters.wait_pass(lambda: self.check_condition_cluster(),
                          timeout=timeout, interval=interval,
                          timeout_msg=(f"Cluster {self._cluster.name} conditions don't reflect "
                                       f"the Maintenance status for {self.name}"))

    def cordon_k8s_node(self):
        """
        Cordon appropriate k8s node

        Returns: None

        """
        k8s_node = self.get_k8s_node()
        LOG.info(f"Cordon node {k8s_node.name}")
        k8s_node.cordon()

    def uncordon_k8s_node(self):
        """
        Uncordon appropriate k8s node

        Returns: None

        """
        k8s_node = self.get_k8s_node()
        LOG.info(f"Uncordon node {k8s_node.name}")
        k8s_node.uncordon()

    def drain_k8s_node(self):
        """
        Drain k8s node.

        Returns: None

        """
        k8s_node = self.get_k8s_node()
        LOG.info(f"Drain node {k8s_node.name}")
        field_selector = f"spec.nodeName={k8s_node.name}"
        pods = self._cluster.k8sclient.pods.list_all(field_selector=field_selector)
        pods_name = [pod.name for pod in pods]
        LOG.info("{} pod(s) expected to be evicted".format(len(pods_name)))
        for pod in pods:
            pod.eviction()

        def _status_msg():
            not_evicted = [f"{pod.namespace}/{pod.name}" for pod in pods if pod.exists()]
            return "\n\nNot evicted pod(s):\n{}".format(yaml.dump(not_evicted))

        def _wait_eviction():
            not_evicted = [pod.name for pod in pods if pod.exists()]
            LOG.info("Remains to evict {} pod(s)".format(len(not_evicted)))
            return True if not_evicted == [] else False

        waiters.wait(_wait_eviction, timeout=1800,
                     timeout_msg='Pod eviction timeout', status_msg_function=_status_msg)

    def reboot_required_status_from_host(self):
        is_required = False
        reason_packages = []
        platform = self.host_platform
        if platform == 'ubuntu':
            res = self.exec_pod_cmd('test -f /var/run/reboot-required', verbose=False)
            if res.get('exit_code') == 0:
                is_required = True
                reason_packages = self.exec_pod_cmd(
                    'cat /var/run/reboot-required.pkgs', verbose=False).get('logs').splitlines()
        else:
            LOG.warning(
                f"Checking for reboot required is not implemented for {platform} "
                f"platform on machine {self.namespace}/{self.name}")
        return is_required, sorted(reason_packages)

    def is_reboot_required(self):
        machine_status = self.data.get('status') or {}
        is_required = machine_status.get('providerStatus', {}).get('reboot', {}).get('required', False)
        reason = machine_status.get('providerStatus', {}).get('reboot', {}).get('reason', "").splitlines()
        return is_required, sorted(reason)

    def get_reboot_count(self, verbose=False) -> str:
        """
        Get number of reboots
        Args:
            verbose: verbose command output

        Returns: command output as a string

        """
        output = self.exec_pod_cmd("journalctl --list-boots  2>/dev/null | wc -l", verbose=verbose)['logs']
        LOG.info(f"Current machine reboots count on {self.name}: {output}")
        return output

    def get_reboot_list(self, verbose=False) -> str:
        """
        Get list of existing OS reboots
        Args:
            verbose: verbose command output

        Returns: command output as a string

        """
        output = self.exec_pod_cmd("journalctl --list-boots  2>/dev/null", verbose=verbose)['logs']
        LOG.info(f"Machine '{self.name}' reboots:\n{output}")
        return output

    def get_uptime(self, verbose=False) -> datetime:
        """
        Get machine uptime since last boot
        Args:
            verbose: verbose command output

        Returns: datetime object

        """
        output = self.exec_pod_cmd("uptime -s", verbose=verbose)['logs']
        date = datetime.datetime.fromisoformat(output.rstrip())
        LOG.info(f"Machine '{self.name}' uptime since: {date}")
        return date

    def get_chrony_status(self, verbose=False):
        """
        Get chrony status of machine
        Args
            verbose: verbose command output

        Returns: dict contains chronyc tracking info

        Example of output:
        {
          'Reference ID': '52713529 (82.113.53.41)',
          'Stratum': '4',
          'Ref time (UTC)': 'Thu Apr 25 11:14:24 2024',
          'System time': '0.000212169 seconds slow of NTP time',
          'Last offset': '-0.000190820 seconds',
          'RMS offset': '0.000179011 seconds',
          'Frequency': '20.262 ppm fast',
          'Residual freq': '-0.002 ppm',
          'Skew': '0.024 ppm',
          'Root delay': '0.009191952 seconds',
          'Root dispersion': '0.001468957 seconds',
          'Update interval': '1040.4 seconds',
          'Leap status': 'Normal'
        }
        """
        LOG.debug(f"Getting chronyc tracking (NTP status) for machine {self.namespace}/{self.name}")
        output = self.exec_pod_cmd("chronyc tracking", verbose=verbose)['logs']
        chronyc_out = {}
        for line in output.strip().split("\n"):
            kv_line = line.split(':', 1)
            chronyc_out[kv_line[0].strip()] = kv_line[1].strip()
        return chronyc_out

    def get_netplan_from_host(self, verbose=False):
        return yaml.safe_load(self.exec_pod_cmd("netplan get 2>/dev/null", verbose=verbose)['logs'])

    def get_machine_ipaddresses_from_netplan(self):
        addresses_map = {}
        np_data = self.get_netplan_from_host().get('network', {})
        iftypes = ['bonds', 'bridges', 'ethernets', 'vlans']

        for iftype in iftypes:
            for k, v in np_data.get(iftype, {}).items():
                address = v.get('addresses', [])
                if address:
                    addresses_map[k] = ','.join(address)
        return addresses_map

    @property
    def annotations(self):
        return self.__kaasmachine.annotations

    def set_annotations(self, annotations):
        return self.__kaasmachine.set_annotations(annotations)


class OpenstackProviderMachine(Machine):
    """Openstack provider specific methods"""
    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.OPENSTACK_PROVIDER_NAME

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        if self.__internal_ip is not None:
            return self.__internal_ip
        annotations = self.metadata['annotations']
        if 'openstack-ip-address' in annotations:
            self.__internal_ip = annotations['openstack-ip-address']
            return self.__internal_ip
        if self.data and self.data['status'] and \
                self.data['status']['providerStatus'] and \
                self.data['status']['providerStatus']['privateIp']:
            provider_status = self.data['status']['providerStatus']
            self.__internal_ip = provider_status.get("privateIp")
            return self.__internal_ip
        else:
            return None

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def public_ip(self):
        # if self.machine_type != 'control':
        #     return None
        if self.__public_ip is not None:
            return self.__public_ip
        annotations = self.metadata['annotations']
        if 'openstack-floating-ip-address' in annotations:
            self.__public_ip = annotations['openstack-floating-ip-address']
            return self.__public_ip
        else:
            return None

    def power_off(self):
        openstack_instance_name = self.get_k8s_node_name()
        client = self._cluster.provider_resources.client
        server = client.get_server_by_name(server_name=openstack_instance_name)
        if server.status == 'ACTIVE':
            LOG.info(f"Power off {openstack_instance_name}")
            server.stop()
            client.wait_server_vm_state(server, state='shut down',
                                        timeout=180, interval=20)
            LOG.info(f"Machine {openstack_instance_name} powered off")
        else:
            raise Exception(f"Machine with name {openstack_instance_name} "
                            f"in status {server.status}, expected ACTIVE")

    def power_on(self):
        openstack_instance_name = self.get_k8s_node_name()
        client = self._cluster.provider_resources.client
        server = client.get_server_by_name(server_name=openstack_instance_name)
        if server.status == 'SHUTOFF':
            LOG.info(f"Power on {openstack_instance_name}")
            server.start()
            client.wait_server_vm_state(server, state='running',
                                        timeout=180, interval=20)
            LOG.info(f"Machine with name {openstack_instance_name} "
                     f"powered on")
        else:
            raise Exception(f"Machine with name {openstack_instance_name} "
                            f" in status {server.status}, expected SHUTOFF")


class BaremetalProviderMachine(Machine):
    """Baremetal provider specific methods"""
    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.BAREMETAL_PROVIDER_NAME

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        if self.__internal_ip is not None:
            return self.__internal_ip
        # PRODX-14564 providerStatus.privateIp will have the address
        machine_status = self.data.get('status') or {}
        self.__internal_ip = machine_status.get('providerStatus', {}).get('privateIp', None)
        LOG.debug(f"Machine internal ip is {self.__internal_ip}")

        if not self.__internal_ip:
            # fallback to lcmmachine info
            lcmm_status = \
                [x for x in self._cluster.get_lcmmachines()
                 if x.name == self.name][0].data.get('status') or {}
            LOG.debug(f"lcmmachine {self.name} status: {lcmm_status}")
            ip = [x.get('address', None)
                  for x in lcmm_status.get('addresses', [])
                  if x.get('type', '') == 'InternalIP']
            if ip:
                LOG.info(f"Machine internal ip is "
                         f"taken from lcmmachine: {ip[0]}")
                self.__internal_ip = ip[0]
        return self.__internal_ip

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def public_ip(self):
        if self.__public_ip is not None:
            return self.__public_ip
        # PRODX-15440 we don't have PublicIP in addresses array
        self.__public_ip = self.internal_ip
        return self.__public_ip

    def get_k8s_node_name(self):
        # After PRODX-11497 we use custom names for k8s nodes,
        # and the "best" (at the moment) option to get it is only from lcmmachine
        # status.addresses object, there should be address with `type: Hostname`

        lcmmachine = self._cluster.get_cluster_lcmmachine(
            name=self.name, namespace=self.namespace)
        if lcmmachine.data:
            lcmmachine_status = lcmmachine.data.get('status') or {}
            lcmmachine_addresses = lcmmachine_status.get('addresses', [])
            hostnames = [addr.get('address', '') for addr in lcmmachine_addresses
                         if addr.get('type', '') == 'Hostname']
            if hostnames and hostnames[0]:
                # There should be only one address of type == 'Hostname' so
                # returning first one anyway
                return hostnames[0]
            LOG.warning("There is no address of type 'Hostname' in status"
                        f"of '{self.name}' lcmmachine")
        else:
            LOG.warning("There is no status yet, seems like machine still in provisioning state"
                        f"of '{self.name}' lcmmachine")

        # Fallback to the old aproach
        return super().get_k8s_node_name()

    def get_power_status(self):
        """
        Return BMH power status
        Returns: bool, False if poweredOff, or True if poweredOn
        """
        bmh = self._manager.get_baremetalhost(self.get_bmh_name(), namespace=self.namespace)
        bmh_status = bmh.read().status or {}
        power_state = bmh_status.get('poweredOn')
        LOG.info(f"BMH '{bmh.namespace}/{bmh.name}' power state: {power_state}")
        return power_state


class AwsProviderMachine(Machine):
    """AWS provider specific methods"""

    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.AWS_PROVIDER_NAME

    def get_k8s_node_name(self):
        machine_status = self.data.get('status') or {}
        if machine_status.get('providerStatus', {}).get('providerInstanceState', {}).get('id', None):
            provider_status = machine_status['providerStatus']
            _id = provider_status['providerInstanceState']['id']
            nodes = [x for x in
                     self._cluster.k8sclient.nodes.list_raw().items
                     if _id in x.spec.provider_id]
            if nodes:
                return nodes[0].metadata.name

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        if self.__internal_ip is not None:
            return self.__internal_ip
        if self.data and self.data['status'] and \
                self.data['status']['providerStatus'] and \
                self.data['status']['providerStatus']['privateIp']:
            provider_status = self.data['status']['providerStatus']
            self.__internal_ip = provider_status.get("privateIp")
            return self.__internal_ip
        else:
            return None

    @property
    def public_ip(self):
        LOG.debug("public_ip() is not implemented for AWS provider")
        return None

    def power_off(self):
        aws_instance_name = \
            self.data['metadata']['annotations']['kaas.mirantis.com/uid']
        client = self._cluster.provider_resources.client
        power_state = client.get_instance_state_by_name(
            instance_name=aws_instance_name)
        if power_state == 'running':
            client.instance_power_action(instance_name=aws_instance_name,
                                         action='stop')
            waiters.wait(lambda: client.get_instance_state_by_name(
                instance_name=aws_instance_name) == 'stopped', timeout=300,
                interval=30)
            power_state = client.get_instance_state_by_name(
                aws_instance_name)
            LOG.info(f"state after power_off instance: {power_state}\n")
        else:
            raise Exception(f"Machine with name {aws_instance_name} "
                            f"in status {power_state}, expected <running>")

    def power_on(self):
        aws_instance_name = \
            self.data['metadata']['annotations']['kaas.mirantis.com/uid']
        client = self._cluster.provider_resources.client
        power_state = client.get_instance_state_by_name(aws_instance_name)
        if power_state == 'stopped':
            client.instance_power_action(aws_instance_name, action='start')
            waiters.wait(lambda: client.get_instance_state_by_name(
                instance_name=aws_instance_name) == 'running', timeout=300,
                interval=30)
            power_state = client.get_instance_state_by_name(
                aws_instance_name)
            LOG.info(f"state after power_on instance: {power_state}\n")
        else:
            raise Exception(f"Machine with name {aws_instance_name} "
                            f"in status {power_state}, expected <stopped>")


class ByoProviderMachine(Machine):
    """BYO provider specific methods"""

    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.BYO_PROVIDER_NAME

    @property
    def internal_ip(self):
        LOG.debug("internal_ip() is not implemented for BYO provider")
        return None

    @property
    def public_ip(self):
        LOG.debug("public_ip() is not implemented for BYO provider")
        return None


class VsphereProviderMachine(Machine):
    """VSphere provider specific methods"""
    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.VSPHERE_PROVIDER_NAME

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        if self.__internal_ip is not None:
            return self.__internal_ip
        if self.data and self.data['status'] and \
                self.data['status']['providerStatus'] and \
                self.data['status']['providerStatus']['privateIp']:
            provider_status = self.data['status']['providerStatus']
            self.__internal_ip = provider_status.get("privateIp")
            return self.__internal_ip
        else:
            return None

    @property
    def public_ip(self):
        LOG.debug("public_ip() is not implemented for VSphere provider")
        return None

    def _task_status(self, task, status):
        LOG.info(f'Task {task} status: {status}')

    def get_power_status(self):
        """
        Return VM power status
        Returns: bool, False if poweredOff, or True if poweredOn

        """
        instance_name = self.get_k8s_node_name()
        client = self._cluster.provider_resources.client

        vm: vim.VirtualMachine = client.get_object_by_property(
            property_name='name',
            property_value=instance_name,
            obj_type=vim.VirtualMachine
        )

        power_state = vm.runtime.powerState
        LOG.info(f"Current power state: {power_state}")

        return (power_state == "poweredOn")

    @retry((ConnectionResetError,), delay=10, tries=3, logger=LOG)
    def power_off(self):
        instance_name = self.get_k8s_node_name()
        client = self._cluster.provider_resources.client

        vm: vim.VirtualMachine = client.get_object_by_property(
            property_name='name',
            property_value=instance_name,
            obj_type=vim.VirtualMachine
        )

        LOG.info("Found: {0}".format(vm.name))
        LOG.info("The current powerState is: {0}".format(vm.runtime.powerState))
        LOG.info("Attempting to power off {0}".format(vm.name))

        task = vm.PowerOffVM_Task()
        LOG.info("Waiting for a task to complete")
        WaitForTask(task, onProgressUpdate=self._task_status)

        self.get_power_status()

    def power_on(self):
        instance_name = self.get_k8s_node_name()
        client = self._cluster.provider_resources.client
        from pyVmomi import vim
        vm: vim.VirtualMachine = client.get_object_by_property(
            property_name='name',
            property_value=instance_name,
            obj_type=vim.VirtualMachine
        )
        LOG.info("Found: {0}".format(vm.name))
        LOG.info("The current powerState is: {0}".format(vm.runtime.powerState))
        LOG.info("Attempting to power on {0}".format(vm.name))

        task = vm.PowerOnVM_Task()
        LOG.info("Waiting for a task to complete")
        WaitForTask(task, onProgressUpdate=self._task_status)

        self.get_power_status()

    @retry((ConnectionResetError,), delay=10, tries=3, logger=LOG)
    def reboot(self):
        instance_name = self.get_k8s_node_name()
        client = self._cluster.provider_resources.client
        vm: vim.VirtualMachine = client.get_object_by_property(
            property_name='name',
            property_value=instance_name,
            obj_type=vim.VirtualMachine
        )
        LOG.info("Found: {0}".format(vm.name))
        LOG.info("The current powerState is: {0}".format(vm.runtime.powerState))
        LOG.info("Attempting to reboot {0}".format(vm.name))

        task = vm.ResetVM_Task()
        LOG.info("Waiting for a task to complete")
        WaitForTask(task, onProgressUpdate=self._task_status)

    def get_reboot_count(self, verbose=False) -> str:
        """
        Get number of reboots
        Args:
            verbose: verbose command output

        Returns: command output as a string

        """
        # W\A for RedHat issue https://access.redhat.com/solutions/3162941
        # This issue is known to Red Hat and is tracked by RH BZ 1364092
        # This issue was fixed with systemd-219-45 and later.

        system_version = utils.get_system_version(self)
        if 'Red Hat Enterprise' in system_version and '7.9' in system_version:
            LOG.info("Use 'last reboot' command for RHEL 7.9")
            output = self.exec_pod_cmd("last reboot | wc -l", verbose=verbose)['logs']
            LOG.info(f"Machine '{self.name}' reboots:\n{output}")
            return output
        else:
            super().get_reboot_count()

    def get_reboot_list(self, verbose=False):
        # W\A for RedHat issue https://access.redhat.com/solutions/3162941
        # This issue is known to Red Hat and is tracked by RH BZ 1364092
        # This issue was fixed with systemd-219-45 and later.

        system_version = utils.get_system_version(self)
        if 'Red Hat Enterprise' in system_version and '7.9' in system_version:
            LOG.info("Use 'last reboot' command for RHEL 7.9")
            output = self.exec_pod_cmd("last reboot", verbose=verbose)['logs']
            LOG.info(f"Machine '{self.name}' reboots:\n{output}")
            return output
        else:
            super().get_reboot_list()


class EquinixMetalProviderMachine(Machine):
    """EquinixMetal provider specific methods"""
    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.EQUINIX_PROVIDER_NAME

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        # Using private IP addresses in Equinix starting from 5.16
        # Checking internal node IP to avoid wrong IP address detection
        # after fixing PRODX-14498 (node IP address is still public after the
        # upgrade from 5.15 to 5.16 while in pure 5.16 deployment it is
        # private)
        # TODO: remove getting node IP after 2.9 release
        if self.metadata['annotations']:
            nodes = self._cluster.k8sclient.nodes.list_raw().to_dict()['items']
            node_name = 'kaas-node-' + self.metadata['annotations'][
                'kaas.mirantis.com/uid']
            for node in nodes:
                if node['metadata']['name'] == node_name:
                    return [ip['address'] for ip in node['status']
                            ['addresses'] if 'InternalIP' in ip['type']][0]

        if self.__internal_ip is not None:
            return self.__internal_ip
        if self.data and self.data['status'] and \
                self.data['status']['providerStatus'] and \
                self.data['status']['providerStatus']['privateIp']:
            provider_status = self.data['status']['providerStatus']
            self.__internal_ip = provider_status.get("privateIp")
            return self.__internal_ip
        else:
            return None

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def public_ip(self):
        if self.__public_ip is not None:
            return self.__public_ip
        if self.data and self.data['status'] and \
                self.data['status']['providerStatus'] and \
                self.data['status']['providerStatus']['publicIp']:
            provider_status = self.data['status']['providerStatus']
            self.__public_ip = provider_status.get("publicIp")
            return self.__public_ip
        else:
            return None


class EquinixMetalV2ProviderMachine(Machine):
    """EquinixMetal v2 provider specific methods"""
    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return utils.Provider.equinixmetalv2.provider_name

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        # Using private IP addresses in Equinix starting from 5.16
        # Checking internal node IP to avoid wrong IP address detection
        # after fixing PRODX-14498 (node IP address is still public after the
        # upgrade from 5.15 to 5.16 while in pure 5.16 deployment it is
        # private)
        # TODO: remove getting node IP after 2.9 release
        if self.metadata['annotations']:
            nodes = self._cluster.k8sclient.nodes.list_raw().to_dict()['items']
            node_name = 'kaas-node-' + self.metadata['annotations'][
                'kaas.mirantis.com/uid']
            for node in nodes:
                if node['metadata']['name'] == node_name:
                    return [ip['address'] for ip in node['status']
                            ['addresses'] if 'InternalIP' in ip['type']][0]

        if self.__internal_ip is not None:
            return self.__internal_ip
        if self.data and self.data['status'] and \
                self.data['status']['providerStatus'] and \
                self.data['status']['providerStatus']['privateIp']:
            provider_status = self.data['status']['providerStatus']
            self.__internal_ip = provider_status.get("privateIp")
            return self.__internal_ip
        else:
            return None

    @property
    def public_ip(self):
        # No any public IP expected for EQv2, for machines as well.
        return None


class AzureProviderMachine(Machine):
    """Azure provider specific methods"""
    __internal_ip = None
    __public_ip = None

    @property
    def provider(self):
        return settings.AZURE_PROVIDER_NAME

    @property
    @cachetools_func.ttl_cache(ttl=3600)
    def internal_ip(self):
        if self.__internal_ip is not None:
            return self.__internal_ip

        try:
            self.__internal_ip = self.data['status'][
                'providerStatus']['privateIp']
        except KeyError as e:
            LOG.warning(f"failed to get privateIp from providerStatus: {e}")
            return None

        return self.__internal_ip

    @property
    def public_ip(self):
        LOG.debug("public_ip() is not implemented for Azure provider")
        return None
